/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.fukurou.util.TagBuffer;						// 6.1.1.0 (2015/01/17)
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;		// 6.4.0.5 (2016/01/09)
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.DBEditConfig;
// import org.opengion.hayabusa.db.DBEditConfigManager;			// 6.4.5.0 (2016/04/08) , 6.9.2.1 (2018/03/12) 廃止

import static org.opengion.fukurou.util.StringUtil.nval;

/**
 * query.jsp での検索ﾎﾞﾀﾝを表示します。
 * 検索ﾎﾞﾀﾝ以外に、pageSize、maxRowCount、prevNext、timeView などの
 * 項目も表示可能です。
 * また、BODY部に登録した値は、table 要素として、td 部に追記されます。
 *
 * @og.formSample
 * ●形式：&lt;og:queryButton /&gt;
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し、{&#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *   &lt;og:queryButton
 *       command            【TAG】ﾘｸｴｽﾄとして送信するｺﾏﾝﾄﾞを指定します(初期値:NEW)
 *       pageSize           【TAG】pageSize ﾌﾟﾙﾀﾞｳﾝﾒﾆｭｰの初期値を指定します (初期値:HTML_PAGESIZE[=100])
 *       maxRowCount        【TAG】maxRowCount ﾌﾟﾙﾀﾞｳﾝﾒﾆｭｰの初期値を指定します (初期値:DB_MAX_ROW_COUNT[=1000])
 *       type               【TAG】典型的なﾎﾞﾀﾝの形状に設定します(初期値:null)
 *       scope              【TAG】ﾘｸｴｽﾄとして送信するｽｺｰﾌﾟを指定します(初期値:null)
 *       lbl                【TAG】ﾗﾍﾞﾙﾘｿｰｽのﾗﾍﾞﾙIDを指定します
 *       accesskey          【TAG】検索ﾎﾞﾀﾝに使用するｼｮｰﾄｶｯﾄｷｰ文字を指定します(初期値:F)
 *       usePrevNext        【TAG】進む／戻るのprev_next ﾘﾝｸを表示するかどうかを指定します(初期値:true)
 *       usePageSize        【TAG】pageSize 引数を使用するかどうかを指定します(初期値:true)
 *       useRowCount        【TAG】maxRowCount 引数を使用するかどうかを指定します(初期値:true)
 *       useTimeView        【TAG】処理時間を表示する TimeView を表示するかどうかを指定します
 *										(初期値:VIEW_USE_TIMEBAR[={@og.value SystemData#VIEW_USE_TIMEBAR}])。
 *       clearAccesskey     【TAG】ｸﾘｱﾎﾞﾀﾝに使用するｼｮｰﾄｶｯﾄｷｰ文字を指定します(初期値:C)
 *       useDummy           【TAG】IE用ﾀﾞﾐｰinputを出力するかどうか
 *       useEditMenu        【TAG】編集ﾌﾟﾙﾀﾞｳﾝを出すかどうか(初期値:true)
 *       blockUIMsg         【TAG】検索中表示を画面上に表示するﾒｯｾｰｼﾞﾘｿｰｽIDを指定します (初期値:MSG0090(検索中です･･･)) 6.4.8.0 (2016/06/24)
 *       useClear           【TAG】JavaScriptによる初期化(ｸﾘｱ)ﾎﾞﾀﾝを使用するかどうか[true/false/none]を指定します(初期値:true) 6.4.3.4 (2016/03/11)
 *       clearTarget        【TAG】useClear=true時のﾀｰｹﾞｯﾄ名（ﾌﾚｰﾑ名）を指定します(初期値:QUERY) 5.9.21.1 (2017/06/16)  。
 *       clearHref          【TAG】useClear=true時の遷移先JSPを指定します(初期値:query.jsp) 5.9.21.1 (2017/06/16)。
 *       clearCommand       【TAG】useClear=true時のcommandを指定します(初期値:NEW) 5.9.21.1 (2017/06/16) 。
 *       clearKeys          【TAG】useClear=true時の遷移先に渡すｷｰをCSV形式で複数指定します  7.0.3.0 (2019/05/13)
 *       clearVals          【TAG】useClear=true時の遷移先に渡す値をCSV形式で複数指定します    7.0.3.0 (2019/05/13)
 *       title              【HTML】要素に対する補足的情報(title)を設定します
 *       style              【HTML】この要素に対して適用させるｽﾀｲﾙｼｰﾄ(style)を設定します
 *       disabled           【HTML】その部品に対して、選択や変更が出来ないように(disabled)指定します(ｻｰﾊﾞｰに送信されない)
 *       tabindex           【HTML】ﾀﾌﾞの移動順(tabindex)を指定します(0 ～ 32767)
 *       clazz              【HTML】要素に対して class 属性を設定します
 *       onClick            【HTML】JavaScriptのｲﾍﾞﾝﾄ onClick を設定します(例:onClick="renew('query.jsp','QUERY');")
 *       onBlur             【HTML】JavaScriptのｲﾍﾞﾝﾄ onBlur を設定します(例:onBlur="this.value=value.toUpperCase();")
 *       onFocus            【HTML】JavaScriptのｲﾍﾞﾝﾄ onFocus を設定します
 *       ondblClick         【HTML】JavaScriptのｲﾍﾞﾝﾄ ondblClick を設定します
 *       onMouseDown        【HTML】JavaScriptのｲﾍﾞﾝﾄ onMouseDown を設定します
 *       onMouseUp          【HTML】JavaScriptのｲﾍﾞﾝﾄ onMouseUp を設定します
 *       onMouseMove        【HTML】JavaScriptのｲﾍﾞﾝﾄ onMouseMove を設定します
 *       onMouseOut         【HTML】JavaScriptのｲﾍﾞﾝﾄ onMouseOut を設定します
 *       onMouseOver        【HTML】JavaScriptのｲﾍﾞﾝﾄ onMouseOver を設定します
 *       autofocus          【HTML5】指定した入力欄にｶｰｿﾙが当たって自動的にﾌｫｰｶｽされます。
 *       caseKey            【TAG】このﾀｸﾞ自体を利用するかどうかの条件ｷｰを指定します(初期値:null)
 *       caseVal            【TAG】このﾀｸﾞ自体を利用するかどうかの条件値を指定します(初期値:null)
 *       caseNN             【TAG】指定の値が、null/ｾﾞﾛ文字列 でない場合(Not Null=NN)は、このﾀｸﾞは使用されます(初期値:判定しない)
 *       caseNull           【TAG】指定の値が、null/ｾﾞﾛ文字列 の場合は、このﾀｸﾞは使用されます(初期値:判定しない)
 *       caseIf             【TAG】指定の値が、true/TRUE文字列の場合は、このﾀｸﾞは使用されます(初期値:判定しない)
 *       debug              【TAG】ﾃﾞﾊﾞｯｸﾞ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:queryButton&gt;
 *
 * 【廃止】6.4.3.4 (2016/03/11) つづり間違いを訂正。useCrear → useClear。
 *   //  useCrear           【TAG】JavaScriptによる初期化(ｸﾘｱ)ﾎﾞﾀﾝを使用するかどうか[true/false/none]を指定します(初期値:true)
 *
 * ●使用例
 *    &lt;og:queryButton
 *        command     = "NEW"      NEW:通常の検索(初期値)／ENTRY:ｴﾝﾄﾘ登録時
 *        type        = "ENTRY"    null:標準 , SELECT:検索ﾎﾞﾀﾝのみ , ENTRY:登録ﾎﾞﾀﾝのみ
 *        pageSize    = "100"      pageSize の初期値設定(初期値:HTML_PAGESIZE)
 *        maxRowCount = "100"      maxRowCount の初期値設定(初期値:DB_MAX_ROW_COUNT)
 *        useClear    = "true"     true:JavaScriptによる初期化(ｸﾘｱ)／false:通常のresetﾎﾞﾀﾝ／none:表示しない
 *        scope       = "request"  null(初期値)／session:ｾｯｼｮﾝ登録／request:ﾘｸｴｽﾄ登録
 *        lbl         = "MSG0010"  検索ﾎﾞﾀﾝに表示するﾒｯｾｰｼﾞID(初期値:MSG0010)
 *        accesskey   = "F"        ｼｮｰﾄｶｯﾄｷｰ文字(初期値:F)
 *        usePrevNext = "true"     Prev-Nextﾘﾝｸを使用するかどうか(初期値:true)
 *        usePageSize = "true"     pageSize 引数を使用するかどうか(初期値:true)
 *        useRowCount = "true"     maxRowCount 引数を使用するかどうか(初期値:true)
 *        useTimeView = "true"     timeView表示(実行時間表示)を使用するかどうか(初期値:VIEW_USE_TIMEBAR)
 *        useEditMenu = "true"     編集ﾌﾟﾙﾀﾞｳﾝを出すかどうか(初期値:true)
 *    /&gt;
 *
 *  標準的な検索ﾎﾞﾀﾝ
 *    &lt;og:queryButton /&gt;
 *
 *  標準的な登録ﾎﾞﾀﾝ(command="NEW" で、query.jsp で使用するﾎﾞﾀﾝ)
 *    &lt;og:queryButton type="ENTRY" /&gt;
 *
 *  BODY 部に追加要素を表示
 *    &lt;og:queryButton &gt;
 *        &lt;td&gt;&lt;jsp:directive.include file="/jsp/common/timeViewImage.jsp" /&gt;&lt;/td&gt;
 *    &lt;/og:queryButton &gt;
 *
 * 7.4.2.1 (2021/05/21)
 *   ｼｽﾃﾑ定数 USE_ACCESSKEY を false に設定すると、accesskey は使用されません。
 *
 * @og.rev 4.0.0.0 (2005/01/31) 新規作成
 * @og.rev 5.7.1.0 (2013/12/06) HTMLTagSupport を継承し、ﾎﾞﾀﾝに付ける属性を増やします。
 * @og.rev 7.3.2.3 (2021/04/09) ｼｽﾃﾑ定数のICON_DIRやIMAGE_DIRを使用します。
 * @og.rev 8.2.0.2 (2022/06/24) HTML5廃止対応
 * @og.group 画面制御
 *
 * @version	4.0
 * @author	Kazuhiko Hasegawa
 * @since	JDK5.0,
 */
public class QueryButtonTag extends HTMLTagSupport {
	/** このﾌﾟﾛｸﾞﾗﾑのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "8.5.2.0 (2023/07/14)" ;
	private static final long serialVersionUID = 852020230714L ;

	// 7.0.1.0 (2018/10/15) XHTML → HTML5 対応(空要素の、"／＞" 止めを、"＞" に変更します)。
	private static final String COMMAND	=
//		"<input type=\"hidden\" name=\"command\" value=\"NEW\" id=\"h_cmd\" />";
		"<input type=\"hidden\" name=\"command\" value=\"NEW\" id=\"h_cmd\" >";	// 7.0.1.0 (2018/10/15)

	private static final String JSP = HybsSystem.sys( "JSP" );
	// 7.3.2.3 (2021/04/09) ｼｽﾃﾑ定数のICON_DIRやIMAGE_DIRを使用します。
	private static final String JSP_ICON = HybsSystem.sys( "JSP_ICON" ) ;

	// 4.2.1.0 (2008/04/07) TDにIDを追加
	// 4.3.1.0 (2008/09/03) 戻る・進むﾎﾞﾀﾝをsubmitにより実行する。
	// 6.0.2.4 (2014/10/17) img ﾀｸﾞに、title 属性追記
	// 7.0.1.0 (2018/10/15) XHTML → HTML5 対応(空要素の、"／＞" 止めを、"＞" に変更します)。
//	private static final String PREV_NEXT =
//		"<td id=\"prev-next\"><a href=\"forward.jsp?command=FIRST\" target=\"RESULT\" >"		+ CR +
//		"  <img src=\"" + JSP + "/image/FIRST.gif\" alt=\"FIRST\" title=\"FIRST\" border=\"0px\" " + CR +
////		"    onclick=\"return submitPrevNextForm('FIRST');\"/>"	+ CR +
//		"    onclick=\"return submitPrevNextForm('FIRST');\">"	+ CR +					// 7.0.1.0 (2018/10/15)
//		"</a>"																+ CR +
//		"<a href=\"forward.jsp?command=PREV\" target=\"RESULT\" >"			+ CR +
//		"  <img src=\"" + JSP + "/image/PREV.gif\" alt=\"PREV\" title=\"PREV\" border=\"0px\" "	+ CR +
////		"    onclick=\"return submitPrevNextForm('PREV');\"/>"	+ CR +
//		"    onclick=\"return submitPrevNextForm('PREV');\">"	+ CR +					// 7.0.1.0 (2018/10/15)
//		"</a>"																+ CR +
//		"<a href=\"forward.jsp?command=NEXT\" target=\"RESULT\" >"			+ CR +
//		"  <img src=\"" + JSP + "/image/NEXT.gif\" alt=\"NEXT\" title=\"NEXT\" border=\"0px\" "	+ CR +
////		"    onclick=\"return submitPrevNextForm('NEXT');\"/>"	+ CR +
//		"    onclick=\"return submitPrevNextForm('NEXT');\">"	+ CR +					// 7.0.1.0 (2018/10/15)
//		"</a>"																+ CR +
//		"<a href=\"forward.jsp?command=LAST\" target=\"RESULT\" >"			+ CR +
//		"  <img src=\"" + JSP + "/image/LAST.gif\" alt=\"LAST\" title=\"LAST\" border=\"0px\" "	+ CR +
////		"    onclick=\"return submitPrevNextForm('LAST');\"/>"	+ CR +
//		"    onclick=\"return submitPrevNextForm('LAST');\">"	+ CR +					// 7.0.1.0 (2018/10/15)
//		"</a></td>" + CR ;

	// 7.0.6.3 (2019/11/08) submitPrevNextForm で submit されるので、aﾀｸﾞは不要
	// 7.3.2.3 (2021/04/09) ｼｽﾃﾑ定数のICON_DIRやIMAGE_DIRを使用します。
	private static final String PREV_NEXT =
		"<td id=\"prev-next\">"		+ CR +
//		"  <img src=\"" + JSP + "/image/FIRST.gif\" alt=\"FIRST\" title=\"FIRST\" border=\"0px\" " + CR +
		"  <img src=\"" + JSP_ICON + "/FIRST.gif\" alt=\"FIRST\" title=\"FIRST\" border=\"0px\" " + CR +
		"    onclick=\"submitPrevNextForm('FIRST');\">"	+ CR +
//		"  <img src=\"" + JSP + "/image/PREV.gif\" alt=\"PREV\" title=\"PREV\" border=\"0px\" "		+ CR +
		"  <img src=\"" + JSP_ICON + "/PREV.gif\" alt=\"PREV\" title=\"PREV\" border=\"0px\" "	+ CR +
		"    onclick=\"submitPrevNextForm('PREV');\">"	+ CR +
//		"  <img src=\"" + JSP + "/image/NEXT.gif\" alt=\"NEXT\" title=\"NEXT\" border=\"0px\" "	+ CR +
		"  <img src=\"" + JSP_ICON + "/NEXT.gif\" alt=\"NEXT\" title=\"NEXT\" border=\"0px\" "	+ CR +
		"    onclick=\"submitPrevNextForm('NEXT');\">"	+ CR +
//		"  <img src=\"" + JSP + "/image/LAST.gif\" alt=\"LAST\" title=\"LAST\" border=\"0px\" "	+ CR +
		"  <img src=\"" + JSP_ICON + "/LAST.gif\" alt=\"LAST\" title=\"LAST\" border=\"0px\" "	+ CR +
		"    onclick=\"submitPrevNextForm('LAST');\">"	+ CR +
		"</td>" + CR ;

	private static final String TIME_VIEW =		// 4.2.1.0 (2008/04/07) 直接色指定をやめる。
		// 8.2.0.2 (2022/06/24) Modify
//		"<td><table cellspacing=\"0\" cellpadding=\"0\">"	+ CR +
//		"<tr>"												+ CR +
//		"<td width=\"20px\"> </td>"							+ CR +
//		"<td width=\"0px\" id=\"queryTbl\" title=\"\" ></td>" + CR +			// #ff0000
//		"<td width=\"0px\" id=\"viewTbl\"  title=\"\" ></td>" + CR +			// #0000ff
//		"<td width=\"0px\" id=\"pageTbl\"  title=\"\" ></td>" + CR +			// #00ff00
//		"<td width=\"10px\"></td>"						+ CR +
//		"<td style=\"background-color:transparent;\" id=\"totalTbl\"></td>" + CR +
		"<td><table style=\"border-spacing:0px;\" >"	+ CR +
		"<tr>"											+ CR +
		"<td style=\"width:20px; padding:0px;\"> </td>"	+ CR +
		"<td id=\"queryTbl\" title=\"\" ></td>"			+ CR +					// #ff0000
		"<td id=\"viewTbl\"  title=\"\" ></td>"			+ CR +					// #0000ff
		"<td id=\"pageTbl\"  title=\"\" ></td>"			+ CR +					// #00ff00
		"<td style=\"width:10px; padding:0px;\"></td>"	+ CR +
		"<td id=\"totalTbl\"></td>"						+ CR +
		"</tr>"											+ CR +
		"</table></td>" + CR ;

	// 4.3.6.0 (2009/04/01) 件数をﾎﾞﾀﾝで開閉させる
	// 6.0.2.4 (2014/10/17) img ﾀｸﾞに title 属性追記
	// 7.3.2.3 (2021/04/09) ｼｽﾃﾑ定数のICON_DIRやIMAGE_DIRを使用します。
	private static final String COUNT_SWITCH=
		"<td>" +
//		"  <img id=\"queryHide\" src=\"" + JSP + "/image/expand_r.png\" alt=\"pageSize\" title=\"pageSize\" border=\"0px\" "+	// 5.5.3.2 (2012/06/08)
		"  <img id=\"queryHide\" src=\"" + JSP_ICON + "/expand_r.png\" alt=\"pageSize\" title=\"pageSize\" border=\"0px\" "+	// 5.5.3.2 (2012/06/08)
				" onClick=\"hideClass( event, this, 'hide', 'expand_r', 'shrink_r' )\"> "	+ CR +
		"</td>";

	// 6.1.1.0 (2015/01/17) DUMMY_INPUT_TAG は、ｼｽﾃﾑ共通で固定値でよい。
	// 5.6.1.0 (2013/02/01) display:noneのﾀﾞﾐｰ出力(IEのEnterﾎﾞﾀﾝのための隠しinput出力)
	private static final String DUMMY_INPUT_TAG =
		new TagBuffer( "input" )
			.add( "name"	, "HYBS_DUMMY_NPUT" )
			.add( "style"	, "display:none;" )
			.makeTag() ;

//	// 5.7.6.2 (2014/05/16) IEのHTML5機能が有効か無効かの判定値を飛ばす、hidden ﾀｸﾞを作成します。
//	// 判定方法は、HTML5で新規追加された canvas ﾀｸﾞが、有効かどうか(内部ﾒｿｯﾄﾞの戻り値があるかどうか)で判定します。
//	// 7.0.1.0 (2018/10/15) XHTML → HTML5 対応(空要素の、"／＞" 止めを、"＞" に変更します)。
//	// 8.5.2.0 (2023/07/14) Delete
//	private static final String IE_HTML5_CHECK =
//			// 8.1.0.0 (2021/12/28) HTML5 準拠に見直し(<script> type属性削除)
////			"<script type=\"text/javascript\">" + CR
//			"<script>" + CR
//		+	"  (function($){" + CR
//		+	"    var test = document.createElement('canvas');" + CR
//		+	"    $.support.canvas = !(!test || !test.getContext);" + CR
//		+	"  })(jQuery);" + CR
////		+	"  $('<input/>').attr('type','hidden').attr('name','"
//		+	"  $('<input>').attr('type','hidden').attr('name','"				// 7.0.1.0 (2018/10/15)
//		+			HybsSystem.IE_HTML5_KEY
//		+		"').val( $.support.canvas ).appendTo('form');" + CR
//		+	"</script>" + CR ;

	// 7.4.2.1 (2021/05/21) accesskey の設定の有効(true)/無効(false)を設定します(要再起動)
	// 8.0.2.0 (2021/11/30) useAccessKey → USE_ACCESSKEY 変更
//	private static final boolean useAccessKey = HybsSystem.sysBool( "USE_ACCESSKEY" );
	private static final boolean USE_ACCESSKEY = HybsSystem.sysBool( "USE_ACCESSKEY" );

	/** ｺﾏﾝﾄﾞ */
	private String	command		= "NEW";
	/** 表示件数 */
	private String	pageSize	;
	/** 最大件数 */
	private String	maxRowCount	;
	/** ﾎﾞﾀﾝに使用するｼｮｰﾄｶｯﾄｷｰ文字 */
	private String	accesskey	= "F";
	private String	useClear	= "true";					// 6.4.3.4 (2016/03/11) つづり間違いを訂正。useCrear → useClear

	/** ﾎﾞﾀﾝの形状(null:標準 , SELECT:検索ﾎﾞﾀﾝのみ , ENTRY:登録ﾎﾞﾀﾝのみ) */
	private String	type		;
	private String	scope		;
	private boolean	usePrevNext	= true;
	private boolean	usePageSize	= true;
	private boolean	useRowCount	= true;
	private boolean	useTimeView	= HybsSystem.sysBool( "VIEW_USE_TIMEBAR" );		// 6.3.6.0 (2015/08/16)
	private String	body		;
	private String	clearAcskey	= "C";
	// 6.9.5.0 (2018/04/23) USE_DUMMY_INPUT 廃止(false固定)
//	private boolean useDummy	= HybsSystem.sysBool( "USE_DUMMY_INPUT" );		// 5.6.1.0 (2013/02/01)
	private boolean useDummy	;												// 6.9.5.0 (2018/04/23) USE_DUMMY_INPUT 廃止(false固定)
	/** 編集ﾌﾟﾙﾀﾞｳﾝを出すかどうか(初期値:true) */
	private boolean useEditMenu	= true;											// 5.7.5.0 (2014/04/04)
	/** 検索中です･･･ */
	private String	blockUIMsg	= "MSG0090";									// 6.4.8.0 (2016/06/24)

	/** ｸﾘｱﾎﾞﾀﾝのﾀｰｹﾞｯﾄ */
	private String	clearTarget		= "QUERY";									// 5.9.21.1 (2017/06/16)
	/** ｸﾘｱﾎﾞﾀﾝの遷移ｱﾄﾞﾚｽ */
	private String	clearHref		= "query.jsp";								// 5.9.21.1 (2017/06/16)
	/** ｸﾘｱ時のｺﾏﾝﾄﾞ */
	private String	clearCommand	= "NEW";									// 5.9.21.1 (2017/06/16)
	/** ｸﾘｱ時の遷移先に渡すｷｰ */
	private String[]	clearKeys	;											// 7.0.3.0 (2019/05/13)
	/** ｸﾘｱ時の遷移先に渡す値 */
	private String[]	clearVals	;											// 7.0.3.0 (2019/05/13)

	/**
	 * ﾃﾞﾌｫﾙﾄｺﾝｽﾄﾗｸﾀｰ
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public QueryButtonTag() { super(); }		// これも、自動的に呼ばれるが、空のﾒｿｯﾄﾞを作成すると警告されるので、明示的にしておきます。

	/**
	 * Taglibの開始ﾀｸﾞが見つかったときに処理する doStartTag() を ｵｰﾊﾞｰﾗｲﾄﾞします。
	 *
	 * @og.rev 6.3.4.0 (2015/08/01) caseKey,caseVal,caseNN,caseNull,caseIf 属性対応
	 *
	 * @return	後続処理の指示( EVAL_BODY_BUFFERED )
	 */
	@Override
	public int doStartTag() {
		// 6.4.1.1 (2016/01/16) PMD refactoring. A method should have only one exit point, and that should be the last statement in the method
		// 反転注意
		return useTag() ? EVAL_BODY_BUFFERED : SKIP_BODY ;
	}

	/**
	 * Taglibのﾀｸﾞ本体を処理する doAfterBody() を ｵｰﾊﾞｰﾗｲﾄﾞします。
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語(国際化)対応 ｴﾝｺｰﾄﾞの取得方法変更
	 * @og.rev 3.0.0.0 (2002/12/25) StringUtil#changeString 廃止
	 * @og.rev 3.1.1.0 (2003/03/28) ﾎﾞﾃﾞｨの内容を取得する処理を、CommonTagSupport で行う。
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		body = nval( getBodyString(),body );

		return SKIP_BODY ;
	}

	/**
	 * Taglibの終了ﾀｸﾞが見つかったときに処理する doEndTag() を ｵｰﾊﾞｰﾗｲﾄﾞします。
	 *
	 * @og.rev 5.3.6.0 (2011/06/01) 集計、合計などのEdit機能に対応します。
	 * @og.rev 5.6.5.1 (2013/06/14) debugﾌﾟﾙﾀﾞｳﾝの追加。ついでに、PageSize と MaxRowCount の 表示条件を、整備します。
	 * @og.rev 5.6.9.4 (2013/10/31) editNameの初期値を外部から与えられるようにする
	 * @og.rev 5.7.1.0 (2013/12/06) HTMLTagSupportを継承する為、ﾒｿｯﾄﾞを、getSubmit() ⇒ makeTag() に変更
	 * @og.rev 5.7.1.2 (2013/12/20) USE_DEFAULT_EDITNAMEでeditNameの初期値変更
	 * @og.rev 5.7.5.0 (2014/04/04) useEditMenu 追加
	 * @og.rev 5.7.6.2 (2014/05/16) IEのHTML5機能が有効か無効かの判定値を飛ばす、hidden ﾀｸﾞを作成します。
	 * @og.rev 6.3.4.0 (2015/08/01) caseKey,caseVal,caseNN,caseNull,caseIf 属性対応
	 * @og.rev 6.4.5.0 (2016/04/08) UserInfo のEditConfig関連機能を、DBEditConfigManagerに移植します。
	 * @og.rev 5.9.12.0 (2016/09/02) tableのframe属性を削除する
	 * @og.rev 6.9.2.1 (2018/03/12) DBEditConfigManagerを直接取り出す処理を、廃止します。
	 * @og.rev 8.2.0.2 (2022/06/24) HTML5廃止対応
	 * @og.rev 8.5.2.0 (2023/07/14) 一部の機能廃止による修正(問合・ﾄﾗﾌﾞﾙ 0200010980)
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		if( !useTag() ) { return EVAL_PAGE ; }	// 6.3.4.0 (2015/08/01)

		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
			.append( getHiddenScope() ).append( CR )
//			.append( "<table border=\"0\" summary=\"submit\" ><tr>" )					// 5.9.12.0 (2016/090/02)
			.append( "<table border=\"0\" ><tr>" )										// 8.2.0.2 (2022/06/24) Modify
			.append( CR )
			.append( "<td>" ).append( makeTag()		).append( "</td>" ).append( CR )	// 5.7.1.0 (2013/12/06)
			.append( "<td>" ).append( getReset()	).append( "</td>" ).append( CR )
			.append( "<td>" ).append( getCommand()	).append( "</td>" ).append( CR )
			.append( "<td>" ).append( getGamenId()	).append( "</td>" ).append( CR );
		if( usePrevNext ) { buf.append( PREV_NEXT	).append( CR ); }
		buf.append( "<td><span id=\"query-button-append\"></span></td>").append( CR );	//4 3.6.0 (2009/04/01)
		// if( usePageSize ) { buf.append( getPageSize()	).append( CR ); }
		// if( useRowCount ) { buf.append( getMaxRowCount()).append( CR ); }
		if( body != null ) {
			buf.append( "<td>" ).append( body ).append( "</td>" ).append( CR );
		}

		// 5.7.5.0 (2014/04/04) useEditMenu 追加。trueで、編集ﾌﾟﾙﾀﾞｳﾝを出力します。
		if( useEditMenu ) {
			// 6.9.2.1 (2018/03/12) DBEditConfigManagerを直接取り出す処理を、廃止します。
//			final DBEditConfigManager dbConfMgr = getUser().getEditConfigManager();		// 6.4.5.0 (2016/04/08)

			// 5.3.6.0 (2011/06/01) 集計、合計などのEdit機能に対応します。
			final String guikey = getGUIInfoAttri( "KEY" );
			// 6.1.0.0 (2014/12/26) findBugs: null ではなく長さが0の配列を返す。
			// 6.9.2.1 (2018/03/12) DBEditConfigManagerを直接取り出す処理を、廃止します。
//			final DBEditConfig[] configs = dbConfMgr.getEditConfigs( guikey );
			final DBEditConfig[] configs = getUser().getEditConfigs( guikey );
			if( configs.length > 0 ) {
	//			String selEditName = getUser().getSelectedEdit( getGUIInfoAttri( "KEY" ) );
	//			String selEditName = nval( getRequestValue( "editName" ) , getUser().getSelectedEdit( getGUIInfoAttri( "KEY" ) ));	// 5.6.9.4 (2013/10/31)
				String selEditName ; // 5.7.1.2 (2013/12/20)
				if( "FALSE".equalsIgnoreCase( sys( "USE_DEFAULT_EDITNAME" ) ) ){
					selEditName = getRequestValue( "editName" );
				}
				else{
					selEditName = nval( getRequestValue( "editName" ) , getUser().getSelectedEdit( guikey ));	// 5.6.9.4 (2013/10/31)
	//				selEditName = nval( getRequestValue( "editName" ) , dbConfMgr.getSelectedEdit( guikey ) );	// 6.4.5.0 (2016/04/08)
				}
				buf.append( getEditSelect( configs, selEditName ) ).append( CR );
			}
		}

		// 5.6.5.1 (2013/06/14) debugﾌﾟﾙﾀﾞｳﾝの追加。ついでに、PageSize と MaxRowCount の 表示条件を、整備します。
		if( usePageSize || useRowCount) {
			buf.append( COUNT_SWITCH ).append( CR );
			if( usePageSize ) { buf.append( getPageSize()	).append( CR ); }
			if( useRowCount ) { buf.append( getMaxRowCount()).append( CR ); }
			// ﾃﾞﾊﾞｯｸﾞは、ﾕｰｻﾞｰのﾛｰﾙが root の場合のみ表示するようにします。
			if( "root".equalsIgnoreCase( getUser().getRoles() ) ) {
				buf.append( getDebugClm() ).append( CR );						// 5.6.5.1 (2013/06/14) debugﾌﾟﾙﾀﾞｳﾝの追加
			}
		}

		if( useTimeView ) { buf.append( TIME_VIEW ).append( CR ); }

		buf.append( "</tr></table>" ).append( CR );

//		// 5.7.6.2 (2014/05/16) IEのHTML5機能が有効か無効かの判定値を飛ばす、hidden ﾀｸﾞを作成します。
//		// IE_HTML5_KEY をｷｰとする設定値がなければ、まだ未設定なので、hidden ﾀｸﾞを作成します。
//		// 8.5.2.0 (2023/07/14) Delete
//		if( getSessionAttribute( HybsSystem.IE_HTML5_KEY ) == null ) {
//			buf.append( IE_HTML5_CHECK );
//		}

		jspPrint( buf.toString() );
		if( useDummy ){ jspPrint( DUMMY_INPUT_TAG ); }		// 6.1.1.0 (2015/01/17) 固定値でよい

		return EVAL_PAGE ;				// ﾍﾟｰｼﾞの残りを評価する。
	}

	/**
	 * ﾀｸﾞﾘﾌﾞｵﾌﾞｼﾞｪｸﾄをﾘﾘｰｽします。
	 * ｷｬｯｼｭされて再利用されるので、ﾌｨｰﾙﾄﾞの初期設定を行います。
	 *
	 * @og.rev 5.6.1.0 (2013/02/01) useDummy 追加
	 * @og.rev 5.7.5.0 (2014/04/04) useEditMenu 追加
	 * @og.rev 6.3.6.0 (2015/08/16) useTimeView の初期値を、VIEW_USE_TIMEBAR にする。
	 * @og.rev 6.4.0.5 (2016/01/09) true/false に加えて、none:表示しない を追加。
	 * @og.rev 6.4.8.0 (2016/06/24) blockUIMsg を追加。 検索中です･･･
	 * @og.rev 5.9.21.1 (2017/06/16) clearTarget , clearHref , clearCommand追加
	 * @og.rev 6.9.5.0 (2018/04/23) USE_DUMMY_INPUT 廃止(false固定)
	 * @og.rev 7.0.3.0 (2019/05/13) clearKeys , clearVals属性を追加
	 */
	@Override
	protected void release2() {
		super.release2();
		command			= "NEW";
		pageSize		= null;
		maxRowCount		= null;
		accesskey		= "F";
		useClear		= "true";												// 6.4.0.5 (2016/01/09) true/false に加えて、none:表示しない を追加。
		scope			= null;
		usePrevNext		= true;
		usePageSize		= true;
		useRowCount		= true;
		useTimeView		= HybsSystem.sysBool( "VIEW_USE_TIMEBAR" );				// 6.3.6.0 (2015/08/16)
		clearAcskey		= "C";													// 5.3.5.0 (2011/05/01) ｸﾘｱﾎﾞﾀﾝｱｸｾｽｷｰ対応
//		useDummy		= HybsSystem.sysBool( "USE_DUMMY_INPUT" );				// 5.6.1.0 (2013/02/01)
		useDummy		= false;												// 6.9.5.0 (2018/04/23) USE_DUMMY_INPUT 廃止(false固定)
		useEditMenu		= true;													// 5.7.5.0 (2014/04/04) 編集ﾌﾟﾙﾀﾞｳﾝを出すかどうか(初期値:true)。
		blockUIMsg		= "MSG0090";											// 6.4.8.0 (2016/06/24) 検索中です･･･
		clearTarget		= "QUERY";												// 5.9.21.1 (2017/06/16) ｸﾘｱﾎﾞﾀﾝのﾀｰｹﾞｯﾄ
		clearHref		= "query.jsp";											// 5.9.21.1 (2017/06/16)
		clearCommand	= "NEW";												// 5.9.21.1 (2017/06/16)
		clearKeys		= null;													// 7.0.3.0 (2019/05/13) ｸﾘｱ時の遷移先に渡すｷｰ
		clearVals		= null;													// 7.0.3.0 (2019/05/13) ｸﾘｱ時の遷移先に渡す値
	}

	/**
	 * ｻﾌﾞﾐｯﾄﾎﾞﾀﾝを作成します。
	 *
	 * @og.rev 5.3.6.0 (2011/06/01) accessKeyを削除できるように対応
	 * @og.rev 5.5.7.1 (2012/10/05) 不要な括弧出力を削除
	 * @og.rev 5.6.9.3 (2013/10/25) IDを出力する
	 * @og.rev 5.7.1.0 (2013/12/06) HTMLTagSupportを継承する為、ﾒｿｯﾄﾞを、getSubmit() ⇒ makeTag() に変更
	 * @og.rev 6.2.2.0 (2015/03/27) 互換ﾓｰﾄﾞ／ﾈｲﾃｨﾌﾞﾓｰﾄﾞ対応。検索ﾎﾞﾀﾝをbutton→input に変更
	 * @og.rev 6.4.8.0 (2016/06/24) blockUIMsg を追加。 検索中です･･･
	 * @og.rev 7.4.2.1 (2021/05/21) accesskey の設定の有効(true)/無効(false)を設定
	 * @og.rev 8.0.2.0 (2021/11/30) useAccessKey → USE_ACCESSKEY 変更
	 *
	 * @return	ｻﾌﾞﾐｯﾄﾎﾞﾀﾝ
	 * @og.rtnNotNull
	 */
	@Override
	protected String makeTag() {
		String msglbl = getLongLabel();		// 4.0.0 (2005/01/31)
		if( msglbl == null ) {
			if( "ENTRY".equalsIgnoreCase( type ) ) {
				msglbl = getLabel( "ENTRY" );
			}
			else {
				msglbl = getLabel( "QUERY" );
			}
		}

		// 5.7.1.0 (2013/12/06) XHTMLTag を使用することで、各種属性を追加できるようにします。
//		if( accesskey != null ) {
//		if( useAccessKey && accesskey != null ) {			// 7.4.2.1 (2021/05/21)
		if( USE_ACCESSKEY && accesskey != null ) {			// 8.0.2.0 (2021/11/30)
			set( "accesskey" , accesskey );
			msglbl = msglbl + "(" + accesskey + ")" ;
		}

		// 6.4.8.0 (2016/06/24) blockUIMsg を追加。検索中です・・・
		if( blockUIMsg != null && !blockUIMsg.isEmpty() ) {
			final String uiMsg = getResource().getLabel( blockUIMsg );
			set( "alt" , uiMsg );
		}

		set( "type"	, "submit" );
		set( "id"	, "queryButtonSubmit" );
		// 6.2.2.0 (2015/03/27) 互換ﾓｰﾄﾞ／ﾈｲﾃｨﾌﾞﾓｰﾄﾞ対応。検索ﾎﾞﾀﾝをbutton→input に変更
		set( "value", msglbl );								// 6.2.2.0 (2015/03/27)

		return XHTMLTag.input( getAttributes() );			// 6.2.2.0 (2015/03/27)
	}

	/**
	 * ﾘｾｯﾄﾎﾞﾀﾝを作成します。
	 *
	 * @og.rev 4.0.0.0 (2007/10/18) ﾒｯｾｰｼﾞﾘｿｰｽ統合( getResource().getMessage ⇒ getResource().getLabel )
	 * @og.rev 5.0.2.0 (2009/11/01) 戻るﾎﾞﾀﾝが出力されている状態で、ｸﾘｱﾎﾞﾀﾝを押すと、戻るﾎﾞﾀﾝが消える不具合を修正
	 * @og.rev 5.3.5.0 (2011/05/01) ｸﾘｱﾎﾞﾀﾝにもｱｸｾｽｷｰが付加できるように対応
	 * @og.rev 5.3.6.0 (2011/06/01) accessKeyを削除できるように対応
	 * @og.rev 6.2.2.0 (2015/03/27) 互換ﾓｰﾄﾞ／ﾈｲﾃｨﾌﾞﾓｰﾄﾞ対応。ｸﾘｱﾎﾞﾀﾝをbutton→input に変更
	 * @og.rev 6.4.0.5 (2016/01/09) true/false に加えて、none:表示しない を追加。
	 * @og.rev 5.9.21.1 (2017/06/16) ｸﾘｱﾎﾞﾀﾝのﾀｰｹﾞｯﾄとﾍﾟｰｼﾞを変更可能にする
	 * @og.rev 7.0.1.0 (2018/10/15) XHTML → HTML5 対応(空要素の、"／＞" 止めを、"＞" に変更します)。
	 * @og.rev 7.0.3.0 (2019/05/13) clearKeys , clearVals属性を追加
	 * @og.rev 7.4.2.1 (2021/05/21) accesskey の設定の有効(true)/無効(false)を設定
	 * @og.rev 8.0.2.0 (2021/11/30) useAccessKey → USE_ACCESSKEY 変更
	 *
	 * @return	ﾘｾｯﾄﾎﾞﾀﾝ
	 * @og.rtnNotNull
	 */
	private String getReset() {
		final String rtn ;

//		final boolean noAccs = !useAccessKey || clearAcskey == null ;				// 7.4.2.1 (2021/05/21)
		final boolean noAccs = !USE_ACCESSKEY || clearAcskey == null ;				// 8.0.2.0 (2021/11/30)

		// 6.4.0.5 (2016/01/09) true/false に加えて、none:表示しない を追加。
		if( "true".equalsIgnoreCase( useClear ) ) {
			final String urlEnc = XHTMLTag.urlEncode( clearKeys,clearVals );		// 7.0.3.0 (2019/05/13)

			final char ch = clearHref.indexOf( '?' ) < 0 ? '?' : '&' ;				// 連結文字列

			rtn = "<input type=\"button\" onClick=\"window.open('" + JSP + '/' + getGUIInfoAttri( "ADDRESS" ) +
				'/' + clearHref + ch + "GAMENID=" + getGUIInfoAttri( "KEY" ) +
				( urlEnc.isEmpty() ? "" : "&" + urlEnc ) +							// 7.0.3.0 (2019/05/13)
				"&command=" + clearCommand + "','" + clearTarget +"')\" " +			// 5.9.21.1 (2017/06/16)
//				( clearAcskey == null ? "" : "accesskey=\"" + clearAcskey + '"' ) +
				( noAccs ? "" : "accesskey=\"" + clearAcskey + '"' ) +				// 7.4.2.1 (2021/05/21)
				" value=\"" + getResource().getLabel( "MSG0011" ) +
//				( clearAcskey == null ? "" : '(' + clearAcskey + ')' ) +
				( noAccs ? "" : '(' + clearAcskey + ')' ) +							// 7.4.2.1 (2021/05/21)
//				"\" />" ;
				"\" >" ;			// 7.0.1.0 (2018/10/15)
		}
		else if( "false".equalsIgnoreCase( useClear ) ) {
			rtn = "<input type=\"reset\" " +
//				( clearAcskey == null ? "" : "accesskey=\"" + clearAcskey + '"' ) +
				( noAccs ? "" : "accesskey=\"" + clearAcskey + '"' ) +				// 7.4.2.1 (2021/05/21)
				" value=\"" + getResource().getLabel( "MSG0011" ) +
//				( clearAcskey == null ? "" : '(' + clearAcskey + ')' ) +
				( noAccs ? "" : '(' + clearAcskey + ')' ) +							// 7.4.2.1 (2021/05/21)
//				"\" />" ;
				"\" >" ;				// 7.0.1.0 (2018/10/15)
		}
		else {							// "none" を想定
			rtn = "" ;					// none:表示しない
		}

		return rtn;
	}

	/**
	 * ｺﾏﾝﾄﾞを出力する hidden を作成します。
	 *
	 * @og.rev 7.0.1.0 (2018/10/15) XHTML → HTML5 対応(空要素の、"／＞" 止めを、"＞" に変更します)。
	 *
	 * @return	ｺﾏﾝﾄﾞを出力する hidden
	 * @og.rtnNotNull
	 */
	private String getCommand() {
		// 6.1.1.0 (2015/01/17) refactoring. ﾛｼﾞｯｸを見直します。
		return "NEW".equals( command )
					? COMMAND
					: "<input type=\"hidden\" name=\"command\" value=\"" +
//						command + "\" id=\"h_cmd\" />" ;
						command + "\" id=\"h_cmd\" >" ;							// 7.0.1.0 (2018/10/15)
	}

	/**
	 * ｽｺｰﾌﾟを出力する hidden を作成します。
	 *
	 * @og.rev 7.0.1.0 (2018/10/15) XHTML → HTML5 対応(空要素の、"／＞" 止めを、"＞" に変更します)。
	 *
	 * @return	ｽｺｰﾌﾟを出力する hidden
	 * @og.rtnNotNull
	 */
	private String getHiddenScope() {
		// 6.1.1.0 (2015/01/17) refactoring. ﾛｼﾞｯｸを見直します。
		return scope == null
				? ""
				: "<input type=\"hidden\" name=\"scope\" value=\"" +
//						scope + "\" />" ;
						scope + "\" >" ;										// 7.0.1.0 (2018/10/15)
	}

	/**
	 * 画面IDを出力する hidden を作成します。
	 *
	 * @og.rev 7.0.1.0 (2018/10/15) XHTML → HTML5 対応(空要素の、"／＞" 止めを、"＞" に変更します)。
	 *
	 * @return	画面IDを出力する hidden
	 * @og.rtnNotNull
	 */
	private String getGamenId() {
		// 6.1.1.0 (2015/01/17) refactoring. ロジックを見直します。
		return "<input type=\"hidden\" name=\"GAMENID\" value=\"" +
//						getGUIInfoAttri("KEY") + "\" />";
						getGUIInfoAttri("KEY") + "\" >";						// 7.0.1.0 (2018/10/15)
	}

	/**
	 * ﾍﾟｰｼﾞｻｲｽﾞﾌﾟﾙﾀﾞｳﾝﾒﾆｭｰを作成します。
	 *
	 * @og.rev 4.3.6.0 (2008/04/01) idとclassの変更
	 *
	 * @return	ﾍﾟｰｼﾞｻｲｽﾞﾌﾟﾙﾀﾞｳﾝﾒﾆｭｰ
	 * @og.rtnNotNull
	 */
	private String getPageSize() {
		if( pageSize == null ) {
			final String PAGESIZE  = sys( "HTML_PAGESIZE" );
			pageSize = nval( getRequestValue( "pageSize" ),PAGESIZE ) ;
		}

		final DBColumn column = getDBColumn( "pageSize" );
		final String tag = column.getEditorValue( pageSize );

		// return ( "<td id=\"label\">" +
		return "<td class=\"label pageSize hide\">"								// 4.3.6.0 (2008/04/01) class出力
				+	column.getLongLabel()
				+	":</td><td class=\"pageSize hide\">" + tag + "</td>" ;
	}

	/**
	 * 検索時の最大検索数のﾌﾟﾙﾀﾞｳﾝﾒﾆｭｰを作成します。
	 *
	 * @og.rev 4.3.6.0 (2008/04/01) idとclassの変更
	 *
	 * @return	最大検索数のﾌﾟﾙﾀﾞｳﾝﾒﾆｭｰ
	 * @og.rtnNotNull
	 */
	private String getMaxRowCount() {
		if( maxRowCount == null ) {
			final String ROW_COUNT = sys( "DB_MAX_ROW_COUNT" );
			maxRowCount = nval( getRequestValue( "maxRowCount" ), ROW_COUNT ) ;
		}

		final DBColumn column = getDBColumn( "maxRowCount" );
		final String tag = column.getEditorValue( maxRowCount );

		// return ( "<td id=\"label\">" +
		return "<td class=\"label maxRowCount hide\">"							// 4.3.6.0 (2008/04/01) class出力
				+	column.getLongLabel()
				+	":</td><td class=\"maxRowCount hide\">" + tag + "</td>" ;
	}

	/**
	 * debugﾌﾟﾙﾀﾞｳﾝﾒﾆｭｰを作成します。
	 *
	 * @og.rev 5.6.5.1 (2013/06/14) 新規追加
	 *
	 * @return	debugﾌﾟﾙﾀﾞｳﾝﾒﾆｭｰ
	 * @og.rtnNotNull
	 */
	private String getDebugClm() {
		final DBColumn column = getDBColumn( "debug" );
		final String tag = column.getEditorValue( "false" );

		return "<td class=\"label debug hide\">"
				+	column.getLongLabel()
				+	":</td><td class=\"debug hide\">" + tag + "</td>" ;
	}

	/**
	 * 編集名一覧のﾌﾟﾙﾀﾞｳﾝﾒﾆｭｰを作成します。
	 *
	 * @og.rev 5.3.6.0 (2011/06/01) 新規追加
	 * @og.rev 5.9.1.2 (2015/10/23) 自己終了ﾀｸﾞ修正
	 *
	 * @param configs 編集名一覧(配列)
	 * @param selEditName 選択中の編集名
	 *
	 * @return	編集名一覧のﾌﾟﾙﾀﾞｳﾝ
	 * @og.rtnNotNull
	 */
	private String getEditSelect( final DBEditConfig[] configs, final String selEditName ) {
		final DBColumn column = getDBColumn( "editName" );

		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
			.append( "<td class=\"label editName\">" )
			.append( column.getLongLabel() )
			.append( ":</td><td class=\"editName\">" )
			.append( "<select name=\"editName\">" )
			.append( "<option><!-- --></option>" );			// 5.9.1.2 (2015/10/23)

		for( final DBEditConfig config : configs ) {
			final String name = config.getEditName();
			if( name != null && name.length() > 0 ) {
				buf.append( "<option value=\"" ).append( name ).append( '"' );	// 6.0.2.5 (2014/10/31) char を append する。
				if( config.isCommon() ) {
					buf.append( " class=\"commonEdit\"" );
				}
				if( name.equals( selEditName ) ) {
					buf.append( " selected" );
				}
				buf.append( '>' ).append( name ).append( "</option>" );			// 6.0.2.5 (2014/10/31) char を append する。
			}
		}
		buf.append( "</select></td>" );
		return buf.toString();
	}

	/**
	 * 【TAG】ﾘｸｴｽﾄとして送信するｺﾏﾝﾄﾞを指定します(初期値:NEW)。
	 *
	 * @og.tag
	 *  NEW:通常の検索(初期値)／ENTRY:ｴﾝﾄﾘ登録時
	 *
	 * @param	cmd	送信するｺﾏﾝﾄﾞ
	 */
	public void setCommand( final String cmd ) {
		command = nval( getRequestParameter( cmd ),command );
	}

	/**
	 * 【TAG】JavaScriptによる初期化(ｸﾘｱ)ﾎﾞﾀﾝを使用するかどうか[true/false/none]を指定します(初期値:true)。
	 *
	 * @og.tag
	 * 通常の初期化(ｸﾘｱ)ﾎﾞﾀﾝでは、HTMLが作成された状態に戻すだけです。
	 * 例えば、引数付きで設定された場合、その引数が設定されている状態へ戻します。
	 * このJavaScriptによる初期化ﾎﾞﾀﾝの場合(trueの場合)は、JavaScriptにより
	 * INIT状態でもう一度QUERYﾌﾚｰﾑを呼び出しますので、指定された引数もｸﾘｱされます。
	 * none を指定すると、ｸﾘｱﾎﾞﾀﾝを表示しません。
	 *
	 * true:  JavaScriptによる初期化ﾎﾞﾀﾝ(すべての引数がｸﾘｱされます)
	 * false: HTMLによる RESET処理(type="reset" のinputﾀｸﾞが出力されます。)
	 * none:  ｸﾘｱﾎﾞﾀﾝを表示しません。 (※ 6.4.0.5 (2016/01/09) 以降 )
	 *
	 * 初期値は、true:JavaScriptによる初期化です。
	 *
	 * @og.rev 6.4.0.5 (2016/01/09) true/false に加えて、none:表示しない を追加。
	 * @og.rev 6.4.3.4 (2016/03/11) つづり間違いを訂正。useCrear → useClear。
	 *
	 * @param	flag	JavaScriptによる初期化ﾎﾞﾀﾝ使用可否 [true:JavaScript初期化ﾎﾞﾀﾝ/false:通常のresetﾎﾞﾀﾝ]
	 */
	public void setUseClear( final String flag ) {
		useClear = nval( getRequestParameter( flag ),useClear );

		// 6.4.0.5 (2016/01/09) true/false に加えて、none:表示しない を追加。
		if( useClear != null &&
			!"true".equalsIgnoreCase( useClear ) &&
			!"false".equalsIgnoreCase( useClear ) &&
			!"none".equalsIgnoreCase( useClear ) ) {
				final String errMsg = "useClear は、[true:全クリア/false:リセット/none:表示しない] から選んでください。"
								+ "useClear=[" + flag + "]" + CR ;
				throw new HybsSystemException( errMsg );
		}
	}

//	/**
//	 * 【廃止】JavaScriptによる初期化(ｸﾘｱ)ﾎﾞﾀﾝを使用するかどうか[true/false/none]を指定します(初期値:true)。
//	 *
//	 * @og.tag
//	 * 通常の初期化(ｸﾘｱ)ﾎﾞﾀﾝでは、HTMLが作成された状態に戻すだけです。
//	 * 例えば、引数付きで設定された場合、その引数が設定されている状態へ戻します。
//	 * このJavaScriptによる初期化ﾎﾞﾀﾝの場合(trueの場合)は、JavaScriptにより
//	 * INIT状態でもう一度QUERYﾌﾚｰﾑを呼び出しますので、指定された引数もｸﾘｱされます。
//	 * none を指定すると、ｸﾘｱﾎﾞﾀﾝを表示しません。
//	 *
//	 * true:  JavaScriptによる初期化ﾎﾞﾀﾝ(すべての引数がｸﾘｱされます)
//	 * false: HTMLによる RESET処理(type="reset" のinputﾀｸﾞが出力されます。)
//	 * none:  ｸﾘｱﾎﾞﾀﾝを表示しません。 (※ 6.4.0.5 (2016/01/09) 以降 )
//	 *
//	 * 初期値は、true:JavaScriptによる初期化です。
//	 *
//	 * @og.rev 6.4.0.5 (2016/01/09) true/false に加えて、none:表示しない を追加。
//	 * @og.rev 6.4.3.4 (2016/03/11) つづり間違いを訂正。useCrear → useClear。
//	 *
//	 * @param	flag	JavaScriptによる初期化ﾎﾞﾀﾝ使用可否 [true:JavaScript初期化ﾎﾞﾀﾝ/false:通常のresetﾎﾞﾀﾝ]
//	 */
//	public void setUseCrear( final String flag ) {
//		setUseClear( flag );
//	}

	/**
	 * 【TAG】useClear時のﾀｰｹﾞｯﾄを指定します(初期値=QUERY)。
	 *
	 * @og.tag
	 * useClearをtrueにしている場合、window.openを利用した初期化を行います。
	 * この際に検索部分の名称がQUERY以外の場合にこの属性で指定します。
	 * 初期値はQUERYです。
	 *
	 * @og.rev 5.9.21.1 (2017/06/16) 新規追加
	 *
	 * @param	tgt	ﾀｰｹﾞｯﾄ名
	 */
	public void setClearTarget( final String tgt ) {
		clearTarget = nval( getRequestParameter( tgt ), clearTarget );
	}

	/**
	 * 【TAG】useClear時の遷移先を指定します(初期値=query.jsp)。
	 *
	 * @og.tag
	 * useClearをtrueにしている場合、window.openを利用した初期化を行います。
	 * この際に検索画面のURLがquery.jspでない場合にこの属性で指定します。
	 * また、query.jsp でも、特定の値を再設定したい場合、query.jsp?KEY=VALUE という
	 * 記述が可能です。指定のｷｰだけ、ｸﾘｱしないということが出来るようになります。
	 * 初期値はquery.jspです。
	 *
	 * @og.rev 5.9.21.1 (2017/06/16) 新規追加
	 *
	 * @param	page	遷移先
	 */
	public void setClearHref( final String page ) {
		clearHref = nval( getRequestParameter( page ), clearHref );
	}

	/**
	 * 【TAG】useClear時のcommandを指定します(初期値=NEW)。
	 *
	 * @og.tag
	 * useClearをtrueにしている場合、window.openを利用した初期化を行います。
	 * この際に読み込む画面に対してcommand=NEW以外を送信したい場合にこの属性で指定します。
	 * 初期値はNEWです。
	 *
	 * @og.rev 5.9.21.1 (2017/06/16) 新規追加
	 *
	 * @param	cmd	ｺﾏﾝﾄﾞ
	 */
	public void setClearCommand( final String cmd ) {
		clearCommand = nval( getRequestParameter( cmd ), clearCommand );
	}

	/**
	 * 【TAG】useClear=true時の遷移先に渡すｷｰをCSV形式で複数指定します。
	 *
	 * @og.tag 遷移先に渡すｷｰを指定します。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとﾃﾞｰﾀ自身にｶﾝﾏを持っている場合に分解をﾐｽる為です。
	 *
	 * @og.rev 7.0.3.0 (2019/05/13) clearKeys , clearVals属性を追加
	 *
	 * @param	key	ｷｰ(CSV形式)
	 */
	public void setClearKeys( final String key ) {
		clearKeys = getCSVParameter( key );
	}

	/**
	 * 【TAG】useClear=true時の遷移先に渡す値をCSV形式で複数指定します。
	 *
	 * @og.tag 遷移先に渡す値を指定します。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとﾃﾞｰﾀ自身にｶﾝﾏを持っている場合に分解をﾐｽる為です。
	 * keys,vals を使用すると、URLｴﾝｺｰﾄﾞされます。
	 *
	 * @og.rev 7.0.3.0 (2019/05/13) clearKeys , clearVals属性を追加
	 *
	 * @param	val	値(CSV形式)
	 */
	public void setClearVals( final String val ) {
		clearVals = getCSVParameter( val );
	}

	/**
	 * 【TAG】典型的なﾎﾞﾀﾝの形状に設定します(初期値:null)。
	 *
	 * @og.tag
	 * ｸｴﾘｰﾎﾞﾀﾝでは、検索ﾎﾞﾀﾝ以外に、pageSize、maxRowCount、
	 * prevNext、timeView などの項目も表示可能です。
	 * これらを、個別に設定可能ですが、通常使用するﾀｲﾌﾟとして、3ﾀｲﾌﾟを
	 * 用意します。これは、設定状態の初期値といえます。
	 * null  :通常の検索ﾎﾞﾀﾝで、ﾌﾙ装備の指定です。
	 * SELECT:1件のみ検索するようなｹｰｽで使用します。
	 *        検索ﾎﾞﾀﾝとｸﾘｱﾎﾞﾀﾝのみを表示します。
	 * ENTRY :いきなり登録するようなｹｰｽで使用します。
	 *        登録ﾎﾞﾀﾝとｸﾘｱﾎﾞﾀﾝのみを表示します。(command="NEW" です。)
	 *
	 * @og.rev 6.4.0.5 (2016/01/09) accesskey="" が効かなかったので、対応。
	 * @og.rev 7.4.2.1 (2021/05/21) accesskey の設定の有効(true)/無効(false)を設定
	 * @og.rev 8.0.2.0 (2021/11/30) useAccessKey → USE_ACCESSKEY 変更
	 *
	 * @param	tp	ﾀｲﾌﾟ指定 [null/SELECT/ENTRY]
	 */
	public void setType( final String tp ) {
		type = nval( getRequestParameter( tp ),type );

		// 6.4.0.5 (2016/01/09) accesskey="" が効かなかったので、対応。
		// accesskey の初期値が "F" なので、ENTRYの場合のみ、"F" のときは "W" に変更します。
		if( "SELECT".equalsIgnoreCase( type ) ) {
			usePrevNext	= false;
			usePageSize	= false;
			useRowCount	= false;
			useTimeView	= false;
		}
		else if( "ENTRY".equalsIgnoreCase( type ) ) {
//			if( "F".equals( accesskey ) ) { accesskey = "W"; }						// ENTRYの場合は、"F" の場合のみ、"W" に置き換えます。
//			if( useAccessKey && "F".equals( accesskey ) ) { accesskey = "W"; }		// ENTRYの場合は、"F" の場合のみ、"W" に置き換えます。
			if( USE_ACCESSKEY && "F".equals( accesskey ) ) { accesskey = "W"; }		// ENTRYの場合は、"F" の場合のみ、"W" に置き換えます。
			usePrevNext	= false;
			usePageSize	= false;
			useRowCount	= false;
			useTimeView	= false;
		}
	}

	/**
	 * 【TAG】ﾘｸｴｽﾄとして送信するｽｺｰﾌﾟを指定します(初期値:null)。
	 *
	 * @og.tag
	 * ﾘｸｴｽﾄ時に、付与する引数情報のうちの、ｽｺｰﾌﾟを指定します。
	 * これは、検索専用画面の場合は、scope="request" にする事で、ﾒﾓﾘの
	 * 無駄な確保を行わずに、検索結果を表示できます。
	 * ただし、その結果に対して、NEXT-PREV や、ﾍｯﾀﾞｰｿｰﾄﾘﾝｸなどの
	 * 機能が使えなくなります。
	 * 通常は、何も指定しない状態(null)で、RESULT画面の設定に任せるのが良いでしょう。
	 * 初期値は、null = session(ｾｯｼｮﾝ登録) です。
	 *
	 * @param	scp	ｽｺｰﾌﾟ [session:ｾｯｼｮﾝ登録/request:ﾘｸｴｽﾄ登録]
	 */
	@Override
	public void setScope( final String scp ) {
		scope = nval( getRequestParameter( scp ),scope );
	}

	/**
	 * 【TAG】検索ﾎﾞﾀﾝに使用するｼｮｰﾄｶｯﾄｷｰ文字を指定します(初期値:F)。
	 *
	 * @og.tag
	 * ﾎﾞﾀﾝを、Alt-XX で実行できるようにするｼｮｰﾄｶｯﾄｷｰを設定します。
	 * 標準では、検索ﾎﾞﾀﾝは、"F" に設定していますので、Alt-F となります。
	 * ｾﾞﾛ文字列を指定すると、ｼｮｰﾄｶｯﾄｷｰなしに設定できます。
	 * 初期値は、Find の "F" です。
	 *
	 * 7.4.2.1 (2021/05/21)
	 *   ｼｽﾃﾑ定数 USE_ACCESSKEY を false に設定すると、accesskey は使用されません。
	 *
	 * @og.rev 5.3.6.0 (2011/06/01) accessKeyを削除できるように対応
	 * @og.rev 5.6.0.3 (2012/01/24) getRequestParameter を何度も呼んでいるので修正。
	 * @og.rev 7.4.2.1 (2021/05/21) accesskey の設定の有効(true)/無効(false)を設定
	 * @og.rev 8.0.2.0 (2021/11/30) useAccessKey → USE_ACCESSKEY 変更
	 *
	 * @param	accsKey	ｼｮｰﾄｶｯﾄｷｰ文字(初期値:F)
	 */
	@Override
	public void setAccesskey( final String accsKey ) {
//		if( useAccessKey ) {
		if( USE_ACCESSKEY ) {
			final String val = getRequestParameter( accsKey );
			if( val == null || val.isEmpty() ) {
				accesskey = null;
			}
			else {
				accesskey = val;		// 5.6.0.3 (2012/01/24)
			}
		}
	}

	/**
	 * 【TAG】ｸﾘｱﾎﾞﾀﾝに使用するｼｮｰﾄｶｯﾄｷｰ文字を指定します(初期値:C)。
	 *
	 * @og.tag
	 * ﾎﾞﾀﾝを、Alt-XX で実行できるようにするｼｮｰﾄｶｯﾄｷｰを設定します。
	 * 標準では、検索ﾎﾞﾀﾝは、"C" に設定していますので、Alt-C となります。
	 * ｾﾞﾛ文字列を指定すると、ｼｮｰﾄｶｯﾄｷｰなしに設定できます。
	 * 初期値は、Clear の "C" です。
	 *
	 * 7.4.2.1 (2021/05/21)
	 *   ｼｽﾃﾑ定数 USE_ACCESSKEY を false に設定すると、accesskey は使用されません。
	 *
	 * @og.rev 5.3.6.0 (2011/06/01) accessKeyを削除できるように対応
	 * @og.rev 5.6.0.3 (2012/01/24) getRequestParameter を何度も呼んでいるので修正。
	 * @og.rev 7.4.2.1 (2021/05/21) accesskey の設定の有効(true)/無効(false)を設定
	 * @og.rev 8.0.2.0 (2021/11/30) useAccessKey → USE_ACCESSKEY 変更
	 *
	 * @param	accsKey	ｸﾘｱﾎﾞﾀﾝのｼｮｰﾄｶｯﾄｷｰ文字(初期値:C)
	 */
	public void setClearAccesskey( final String accsKey ) {
//		if( useAccessKey ) {
		if( USE_ACCESSKEY ) {
			final String val = getRequestParameter( accsKey );
			if( val == null || val.isEmpty() ) {
				clearAcskey = null;
			}
			else {
				clearAcskey = val;		// 5.6.0.3 (2012/01/24)
			}
		}
	}

	/**
	 * 【TAG】進む／戻るのprev_next ﾘﾝｸを表示するかどうかを指定します(初期値:true)。
	 *
	 * @og.tag
	 * 進む／戻るﾘﾝｸは、RESULT画面にｺﾏﾝﾄﾞ (FIRST,PREV,NEXT,LAST) という
	 * ｺﾏﾝﾄﾞを送る事で実現している画像ﾘﾝｸです。
	 * PREV_NEXT が存在しない画面では、消去しておく必要があります。
	 * 初期値は、true(表示する)です。
	 *
	 * @param	flag	prev_nextﾘﾝｸ表示 [true:する/false:しない]
	 */
	public void setUsePrevNext( final String flag ) {
		usePrevNext = nval( getRequestParameter( flag ),usePrevNext );
	}

	/**
	 * 【TAG】pageSize 引数を使用するかどうかを指定します(初期値:true)。
	 *
	 * @og.tag
	 * pageSize 引数は、RESULT画面のviewﾀｸﾞで使用する 一回の表示に
	 * 何件のﾃﾞｰﾀを表示させるかを指定する引数です。
	 * この件数が多いと、多くのﾃﾞｰﾀを見ることが出来ますが、その分表示が
	 * 遅くなります。逆に、少ないと、PREV_NEXT を多用することになり、
	 * 目的の行を探し出しにくくなります。
	 * pageSize の値は、pageSizeﾊﾟﾗﾒｰﾀで画面単位に指定するか、ｼｽﾃﾑ
	 * ﾊﾟﾗﾒｰﾀの HTML_PAGESIZE でｼｽﾃﾑ単位に指定できます。
	 * 初期値は、true(使用する)です。
	 *
	 * @param	flag	pageSize引数使用 [true:する/false:しない]
	 * @see	#setPageSize( String )
	 */
	public void setUsePageSize( final String flag ) {
		usePageSize = nval( getRequestParameter( flag ),usePageSize );
	}

	/**
	 * 【TAG】maxRowCount 引数を使用するかどうかを指定します(初期値:true)。
	 *
	 * @og.tag
	 * maxRowCount 引数は、RESULT画面のqueryﾀｸﾞで使用する 一回の検索時に
	 * 何件のﾃﾞｰﾀを取得させるかを指定する引数です。
	 * この件数が多いと、多くのﾃﾞｰﾀを検索することが出来ますが、その分表示が
	 * 遅くなります。逆に、少ないと、目的の行まで探し出せないかもしれません。
	 * maxRowCount の値は、rowCountﾊﾟﾗﾒｰﾀで画面単位に指定するか、ｼｽﾃﾑ
	 * ﾊﾟﾗﾒｰﾀの DB_MAX_ROW_COUNT でｼｽﾃﾑ単位に指定できます。
	 *
	 * 初期値は、true(使用する)です。
	 *
	 * @param	flag	maxRowCount引数使用 [true:する/false:しない]
	 * @see	#setMaxRowCount( String )
	 */
	public void setUseRowCount( final String flag ) {
		useRowCount = nval( getRequestParameter( flag ),useRowCount );
	}

	/**
	 * 【TAG】処理時間を表示する TimeView を表示するかどうか[true:する/false:しない]を指定します
	 *        (初期値:VIEW_USE_TIMEBAR[={@og.value SystemData#VIEW_USE_TIMEBAR}])。
	 *
	 * @og.tag
	 * true に設定すると、処理時間を表示するﾊﾞｰｲﾒｰｼﾞが表示されます。
	 * これは、DB検索、APｻｰﾊﾞｰ処理、画面表示の各処理時間をﾊﾞｰｲﾒｰｼﾞで
	 * 表示させる機能です。処理時間の目安になります。
	 * (初期値:VIEW_USE_TIMEBAR[={@og.value SystemData#VIEW_USE_TIMEBAR}])。
	 *
	 * @og.rev 6.3.6.0 (2015/08/16) useTimeView の初期値を、VIEW_USE_TIMEBAR にする。
	 *
	 * @param	flag	処理時間を表示 [true:する/false:しない]
	 */
	public void setUseTimeView( final String flag ) {
		useTimeView = nval( getRequestParameter( flag ),useTimeView );
	}

	/**
	 * 【TAG】pageSize ﾌﾟﾙﾀﾞｳﾝﾒﾆｭｰの初期表示件数を指定します
	 *        (初期値:HTML_PAGESIZE[={@og.value SystemData#HTML_PAGESIZE}])。
	 *
	 * @og.tag
	 * ﾌﾟﾙﾀﾞｳﾝﾒﾆｭｰを表示する場合の初期値を指定します。
	 * (初期値:ﾕｰｻﾞｰ定数のHTML_PAGESIZE[={@og.value SystemData#HTML_PAGESIZE}])。
	 *
	 * @param	pSize	表示件数
	 * @see		#setUsePageSize( String )
	 * @see		org.opengion.hayabusa.common.SystemData#HTML_PAGESIZE
	 */
	public void setPageSize( final String pSize ) {
		pageSize = nval( getRequestParameter( pSize ),pageSize );
	}

	/**
	 * 【TAG】maxRowCount ﾌﾟﾙﾀﾞｳﾝﾒﾆｭｰの初期値を指定します
	 *        (初期値:DB_MAX_ROW_COUNT[={@og.value SystemData#DB_MAX_ROW_COUNT}])。
	 *
	 * @og.tag
	 * ﾌﾟﾙﾀﾞｳﾝﾒﾆｭｰを表示する場合のmaxRowCountの初期値を指定します。
	 * (初期値:ﾕｰｻﾞｰ定数のDB_MAX_ROW_COUNT[={@og.value SystemData#DB_MAX_ROW_COUNT}])。
	 *
	 * @param	maxSize	最大件数
	 * @see		#setUseRowCount( String )
	 * @see		org.opengion.hayabusa.common.SystemData#DB_MAX_ROW_COUNT
	 */
	public void setMaxRowCount( final String maxSize ) {
		maxRowCount = nval( getRequestParameter( maxSize ),maxRowCount );
	}

	/**
	 * 【TAG】ﾀﾞﾐｰのinputﾀｸﾞ(display:none)を出力するか設定します(初期値:false)。
	 *
	 * @og.tag
	 * ﾀﾞﾐｰのinputﾀｸﾞ(display:none)を出力するか設定します。
	 * IEではinputが1つだけの場合にｴﾝﾀｰでのsubmitが上手くいかないため、ﾀﾞﾐｰを配置する必要があります。
	 * trueに設定するとWriteCheckﾀｸﾞがﾀﾞﾐｰを（常に）出力します。
	 * (初期値:false)。
	 *
	 * @og.rev 5.6.1.0 (2013/02/01) 新規追加
	 *
	 * @param	flag	出力可否 [true:する/false:しない]
	 */
	public void setUseDummy( final String flag ) {
		useDummy = nval( getRequestParameter( flag ), useDummy );
	}

	/**
	 * 【TAG】query画面等で編集ﾌﾟﾙﾀﾞｳﾝを出力するかどうか[true:する/false:しない]を指定します(初期値:true)。
	 *
	 * @og.tag
	 * falseを指定すると編集ﾌﾟﾙﾀﾞｳﾝを出さない。
	 * popup画面等、画面ID単位では編集機能は使うが、部分的に使いたくない場合に
	 * ﾀｸﾞのﾊﾟﾗﾒｰﾀとして個別に指定できるようにします。
	 * 初期値は、互換性の関係から、編集ﾌﾟﾙﾀﾞｳﾝを出力する(=true)です。
	 *
	 * @og.rev 5.7.5.0 (2014/04/04) 新規追加
	 *
	 * @param	flag	編集ﾒﾆｭｰ出力 [true:する/false:しない]
	 * @see		org.opengion.hayabusa.taglib.FileUDTag#setUseEditLink(String)
	 */
	public void setUseEditMenu( final String flag ) {
		useEditMenu = nval( getRequestParameter( flag ),useEditMenu );
	}

	/**
	 * 【TAG】検索ﾎﾞﾀﾝを押下した際に表示されるﾒｯｾｰｼﾞﾘｿｰｽIDを指定します
	 *        (初期値:MSG0090[検索中です･･･])。
	 *
	 * @og.tag
	 * headﾀｸﾞのuseBlockUI=true のときに、検索ﾎﾞﾀﾝを押下すると、｢検索中です･･･｣と表示されます。
	 * このﾒｯｾｰｼﾞのﾘｿｰｽIDを指定します。
	 * 表示させたくない場合は、blockUIMsg = "" をｾｯﾄするか、headﾀｸﾞのuseBlockUI=false をｾｯﾄしてください。
	 *
	 * @og.rev 6.4.8.0 (2016/06/24) blockUIMsg を追加。 検索中です･･･
	 *
	 * @param	id	検索中ですﾒｯｾｰｼﾞID
	 */
	public void setBlockUIMsg( final String id ) {
		final String ids = getRequestParameter( id );
		if( ids != null ) { blockUIMsg = ids; }
	}

	/**
	 * このｵﾌﾞｼﾞｪｸﾄの文字列表現を返します。
	 * 基本的にﾃﾞﾊﾞｯｸﾞ目的に使用します。
	 *
	 * @return	このｸﾗｽの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"			, VERSION		)
				.println( "command"			, command		)
				.println( "pageSize"		, pageSize		)
				.println( "maxRowCount"		, maxRowCount	)
				.println( "accesskey"		, accesskey		)
				.println( "scope"			, scope			)
				.println( "useClear"		, useClear		)
				.println( "usePrevNext"		, usePrevNext	)
				.println( "usePageSize"		, usePageSize	)
				.println( "useRowCount"		, useRowCount	)
				.println( "useTimeView"		, useTimeView	)
				.println( "clearTarget"		, clearTarget	)
				.println( "clearHref"		, clearHref		)
				.println( "clearCommand"	, clearCommand	)
				.println( "body"			, body			)
				.println( "Other..."		, getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
