/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.fukurou.util.Attributes;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)
import static org.opengion.fukurou.system.HybsConst.CR ;			// 6.2.4.0 (2015/05/15)

import java.util.List;
import java.util.ArrayList;

/**
 * and タグで指定された条件を元に、WHERE句を作成します。
 *
 * 各属性は、{&#064;XXXX} 変数が使用できます。
 * これは、ServletRequest から、XXXX をキーに値を取り出し、この変数に
 * 割り当てます。つまり、このXXXXをキーにリクエストすれば、
 * この変数に値をセットすることができます。
 *
 * @og.formSample
 * ●形式：&lt;og:where startKey="…" &gt; &lt;og:and startKey="…" ... /&gt; &lt;/og:where&gt;
 * ●body：あり(EVAL_BODY_INCLUDE:BODYをインクルードし、{&#064;XXXX} は解析しません)
 *
 * ●Tag定義：
 *   &lt;og:where
 *       startKey           【TAG】SQL条件句の最初の演算子を指定します(初期値:where)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:where&gt;
 *
 * ●使用例
 *     &lt;og:query command="NEW"&gt;
 *             select PN,YOBI,NMEN,HINM from XX01 where PN = '{&#064;PN}' order by PN
 *     &lt;/og:query&gt;
 *
 *          ・検索条件が入力された時({&#064;PN}がNOT NULLのとき)
 *            作成されるSQL文⇒select PN,YOBI,NMEN,HINM from XX01 where PN = 'AAA' order by PN
 *          ・検索条件が入力されなかった時({&#064;PN}がNULLのとき)
 *            作成されるSQL文⇒select PN,YOBI,NMEN,HINM from XX01 where PN = '' order by PN
 *
 *     &lt;og:query command="NEW"&gt;
 *             select PN,YOBI,NMEN,HINM from XX01
 *         &lt;og:where&gt;
 *             &lt;og:and value="PN = '{&#064;PN}%'" /&gt;
 *             &lt;og:and value="YOBI like '{&#064;YOBI}%'" /&gt;
 *         &lt;/og:where&gt;
 *             order by PN
 *     &lt;/og:query&gt;
 *
 *          ・検索条件が入力された時({&#064;PN}がNOT NULLのとき)
 *            作成されるSQL文⇒select PN,YOBI,NMEN,HINM from XX01 PN = 'AAA%' and YOBI like 'BBB%' order by PN
 *          ・検索条件が入力されなかった時({&#064;PN}がNULLのとき) WHERE句がなくなる。
 *            作成されるSQL文⇒select PN,YOBI,NMEN,HINM from XX01 order by PN
 *
 *        注意:WhereTagを使った場合、下のようにはなりません。
 *            select PN,YOBI,NMEN,HINM from XX01 PN = '' and YOBI like '%' order by PN
 *
 *     &lt;og:query command="NEW"&gt;
 *             select PN,YOBI,NMEN,HINM from XX01 where PN="11111"
 *         &lt;og:where startKey="and"&gt;
 *             &lt;og:and value="YOBI like '{&#064;PN}%'" /&gt;
 *         &lt;/og:where&gt;
 *             order by PN
 *     &lt;/og:query&gt;
 *
 *          ・検索条件が入力された時({&#064;YOBI}がNOT NULLのとき)
 *            作成されるSQL文⇒select PN,YOBI,NMEN,HINM from XX01 PN = '11111' and YOBI like 'BBB%' order by PN
 *          ・検索条件が入力されなかった時({&#064;YOBI}がNULLのとき) WHERE句がなくなる。
 *            作成されるSQL文⇒select PN,YOBI,NMEN,HINM from XX01 PN = '11111' order by PN
 *
 * @og.group 画面部品
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class SqlWhereTag extends CommonTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.2.0 (2016/01/29)" ;
	private static final long serialVersionUID = 642020160129L ;

	private transient List<Attributes>	values	;

	private String	startKey	= "where";
	private String	whereWord	= "";

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public SqlWhereTag() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @return	後続処理の指示( EVAL_BODY_INCLUDE )
	 */
	@Override
	public int doStartTag() {
		return EVAL_BODY_INCLUDE ;	// Body インクルード( extends TagSupport 時)
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		jspPrint( makeSqlWhere() );

		return EVAL_PAGE ;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		values		= null;
		startKey	= "where";
		whereWord	= "";
	}

	/**
	 * 内部タグの SqlAndTag / SqlAndTag より、個々のカラムの値を書き換える
	 * 為のアトリビュートを受け取る。
	 *
	 * 複数の値を受け取って、後ほど、すべてのカラムに対して処理を行います。
	 *
	 * @og.rev 3.1.0.0 (2003/03/20) Vector を使用している箇所で、非同期でも構わない箇所を、ArrayList に置換え。
	 * @og.rev 3.1.2.0 (2003/04/07) taglib パッケージ内部で使用している箇所を protected 化する。
	 *
	 * @param   attri	属性リスト
	 */
	protected void setAttributes( final Attributes attri ) {
		if( values == null ) { values = new ArrayList<>(); }
		values.add( attri );
	}

	/**
	 * 【TAG】SQL条件句の最初の演算子を指定します(初期値:where)。
	 *
	 * @og.tag SQL条件句の最初の演算子を指定します。
	 *
	 * @param	skey 条件句の最初の演算子
	 */
	public void setStartKey( final String skey ) {
		final String key = getRequestParameter( skey );
		if( key != null && key.length() > 0 ) { startKey = key; }
	}

	/**
	 * SqlAndTag / SqlOrTag で指定された条件を元に、WHERE句を作成します。
	 *
	 * @og.rev 6.2.4.0 (2015/05/15) 各andの終わりに、改行とTABを出力します。
	 *
	 * @return	WHERE句の文字列
	 */
	private String makeSqlWhere() {
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
		if( values != null ) {
			for( int i=0; i<values.size(); i++ ) {
				final Attributes attri = values.get( i );
				String keyWord = attri.get( "keyWord" );
				final String value = attri.get( "value" );
				if( buf.length() == 0 ) { keyWord = this.startKey; }
				buf.append( '\t' )			// 6.2.4.0 (2015/05/15) debug時のレイアウト考慮
					.append( keyWord )
					.append( ' ' )			// 6.0.2.5 (2014/10/31) char を append する。
					.append( value )
					.append( CR );			// 6.2.4.0 (2015/05/15) debug時のレイアウト考慮
			}
		}
		whereWord = buf.toString();
		return whereWord;
	}

	/**
	 * タグの名称を、返します。
	 * 自分自身のクラス名より、自動的に取り出せないため、このメソッドをオーバーライドします。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規追加
	 *
	 * @return  タグの名称
	 * @og.rtnNotNull
	 */
	@Override
	protected String getTagName() {
		return "where" ;
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION	)
				.println( "startKey"		,startKey	)
				.println( "whereWord"		,whereWord	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
