/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.common.SystemManager;					// 7.3.2.0 (2021/03/19)
import org.opengion.hayabusa.resource.GUIInfo;
import org.opengion.hayabusa.resource.UserInfo;
import org.opengion.hayabusa.resource.CodeData;
import org.opengion.hayabusa.resource.FavoriteGUIData;
import org.opengion.hayabusa.resource.ResourceManager;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.fukurou.util.FileMap;
import org.opengion.fukurou.util.TagBuffer;
import org.opengion.fukurou.util.ToString;							// 6.1.1.0 (2015/01/17)
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.ArraySet;							// 6.4.3.4 (2016/03/11)
import org.opengion.fukurou.util.Cleanable;							// 7.3.2.0 (2021/03/19)

import org.opengion.fukurou.system.OgBuilder;						// 6.4.4.1 (2016/03/18)
import java.util.concurrent.ConcurrentMap;							// 7.3.2.0 (2021/03/19)
import java.util.concurrent.ConcurrentHashMap;						// 7.3.2.0 (2021/03/19)

import static org.opengion.fukurou.util.StringUtil.nval;
import static org.opengion.fukurou.system.HybsConst.BUFFER_LARGE;	// 6.1.0.0 (2014/12/26) refactoring
import static org.opengion.fukurou.system.HybsConst.BR;				// 6.1.0.0 (2014/12/26) refactoring

import java.util.Set;
import java.util.LinkedHashSet;
import java.util.TreeSet;											// 6.8.3.1 (2017/12/01)
import java.util.Iterator;
import java.util.Map;

/**
 * 画面ｱｸｾｽﾒﾆｭｰを作成します｡
 *
 * 画面ﾘｿｰｽの階層番号(ﾚﾍﾞﾙ)は､
 * 　0:ｸﾞﾙｰﾌﾟ分類ﾒﾆｭｰ(class="GUI_GRP"のtdﾀｸﾞで囲われます)
 * 　1:ﾄｯﾌﾟ階層(【分類名称】)
 * 　2:選択階層(通常の折りたたみﾒﾆｭｰ)
 * 　3:選択非表示(通常は､隠してあります)
 * です｡
 *
 * ※ 6.9.4.1 (2018/04/09) HYBS_BR の廃止
 *    以前から､画面ﾒﾆｭｰを少し空けるために､HYBS_BR が使われていましたが､廃止しました｡
 *
 * @og.formSample
 * ●形式：&lt;og:topMenu /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:topMenu
 *       menuType           【TAG】作成するﾒﾆｭｰの種類(NORMAL,GROUP,ONELEVEL,NEXTGUI,MATRIX,MATRIX2,NONE,TILE)を指定します(初期値:NORMAL)
 *       expand             【TAG】折り返しﾒﾆｭｰを構築するかどうかを指定します(初期値:true)
 *       groups             【TAG】表示対象となるｸﾞﾙｰﾌﾟをCSV形式で指定します
 *       classify           【TAG】表示対象となる分類(classify)を指定します
 *       href               【TAG】ｸﾞﾙｰﾌﾟﾒﾆｭｰの表示対象となるｿｰｽ名(href)を指定します(初期値:menu.jsp)
 *       target             【TAG】ｸﾞﾙｰﾌﾟﾒﾆｭｰの表示対象となるﾌﾚｰﾑ名(target)を指定します(初期値:MENU)
 *       imageOnly          【TAG】ｸﾞﾙｰﾌﾟﾒﾆｭｰの表示に､画像のみかどうか[true:画像のみ/false:画像＋ﾗﾍﾞﾙ]を指定します(初期値:false)
 *       sideCount          【TAG】ｸﾞﾙｰﾌﾟﾒﾆｭｰの表示対象となるﾒﾆｭｰを横並びさせる数を指定します
 *       minCellCount       【TAG】表形式ﾒﾆｭｰ(MATRIX)の一つのｾﾙに含まれる最小行数を指定します(初期値:8)
 *       maxCellCount       【TAG】表形式ﾒﾆｭｰ(MATRIX)の一つのｾﾙに含まれる最大行数を指定します(初期値:8)
 *       cache              【TAG】ｸﾞﾙｰﾌﾟﾒﾆｭｰのｷｬｯｼｭを使用するかどうか[true/false]を指定します(初期値:true)
 *       match              【TAG】正判定(ﾏｯﾁする場合に､ﾒﾆｭｰに出す)条件を設定します
 *       unmatch            【TAG】逆判定(ﾏｯﾁする場合に､ﾒﾆｭｰに出さない)条件を設定します
 *       useButton          【TAG】画面ﾘﾝｸにﾎﾞﾀﾝを使用するかどうか[true/false]を指定します(初期値:false)
 *       buttonRequest      【TAG】ﾏﾄﾘｸｽからの遷移先でﾎﾞﾀﾝﾒﾆｭｰ表示するかどうか[true/false]を指定します(初期値:false)
 *       inlineStyle        【TAG】標準画面を初期状態で表示するかどうか[true/false]を指定します(初期値:false)
 *       useDivOneLevel     【TAG】ONELEVEL,NEXTGUI で､DIVﾀｸﾞ＋design-onelevel 処理をするかどうか[true/false]を指定します(初期値:false)
 *       useLongName        【TAG】画面名称に､名前(長) を使う場合は､trueを指定します(初期値:false) 8.2.1.0 (2022/07/15)
 *       caseKey            【TAG】このﾀｸﾞ自体を利用するかどうかの条件ｷｰを指定します(初期値:null)
 *       caseVal            【TAG】このﾀｸﾞ自体を利用するかどうかの条件値を指定します(初期値:null)
 *       caseNN             【TAG】指定の値が､null/ｾﾞﾛ文字列 でない場合(Not Null=NN)は､このﾀｸﾞは使用されます(初期値:判定しない)
 *       caseNull           【TAG】指定の値が､null/ｾﾞﾛ文字列 の場合は､このﾀｸﾞは使用されます(初期値:判定しない)
 *       caseIf             【TAG】指定の値が､true/TRUE文字列の場合は､このﾀｸﾞは使用されます(初期値:判定しない)
 *       debug              【TAG】ﾃﾞﾊﾞｯｸﾞ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:topMenu&gt;
 *
 * ●使用例
 *    &lt;og:topMenu /&gt;
 *
 *    &lt;og:topMenu
 *          menuType     = "NORMAL"      NORMAL:通常 / GROUP:ｸﾞﾙｰﾌﾟﾒﾆｭｰ / ONELEVEL:１ﾚﾍﾞﾙ / NEXTGUI:次ｱｸｾｽ先
 *                                           / MATRIX:一覧表ﾒﾆｭｰ / MATRIX2:一覧表ﾒﾆｭｰ(大分類なし版) / NONE:表示しない / TILE:ﾀｲﾙ表示
 *          expand       = "true"        true:折り返しﾒﾆｭｰ / false:階層ﾒﾆｭｰ
 *          groups       = "AA,BB,CC"    表示対象となるｸﾞﾙｰﾌﾟをCSV形式で指定します｡
 *          classify     = "ABC"         表示対象となる分類を指定します｡
 *          href         = "menu.jsp"    ｸﾞﾙｰﾌﾟﾒﾆｭｰの対象ｿｰｽ名(href)を指定します｡
 *          target       = "MENU"        ｸﾞﾙｰﾌﾟﾒﾆｭｰの対象ﾌﾚｰﾑ名(target)を指定します｡
 *          imageOnly    = "false"       ｸﾞﾙｰﾌﾟﾒﾆｭｰに､ true:画像のみ / false:画像＋ﾗﾍﾞﾙ を指定します｡
 *          sideCount    = "6"           ｸﾞﾙｰﾌﾟﾒﾆｭｰの表示対象となるﾒﾆｭｰを横並びさせる数を指定します｡
 *          minCellCount = "8"           表形式ﾒﾆｭｰの１ｾﾙの最小行数を指定します｡
 *          maxCellCount = "8"           表形式ﾒﾆｭｰの１ｾﾙの最大行数を指定します｡
 *          cache        = "true"        ｸﾞﾙｰﾌﾟﾒﾆｭｰのｷｬｯｼｭを使用するかどうか指定します｡
 *          match        = "正規表現"    正判定(ﾏｯﾁする場合に､ﾒﾆｭｰに出す)条件を設定します｡
 *          unmatch      = "正規表現"    逆判定(ﾏｯﾁする場合に､ﾒﾆｭｰに出さない)条件を設定します｡
 *          useButton    = "false"       ﾎﾞﾀﾝ形式のﾘﾝｸを使用するかを指定します｡
 *          useDivOneLevel  = "false"    ONELEVEL,NEXTGUI で､DIVﾀｸﾞ＋design-onelevel 処理をするかどうかを指定します｡
 *    /&gt;
 *
 * @og.rev 3.5.5.3 (2004/04/09) 新規作成
 * @og.group ﾒﾆｭｰ制御
 *
 * @version  4.0
 * @author   Kohei Naruse
 * @since    JDK5.0,
 */
public class TopMenuTag extends CommonTagSupport {
	/** このﾌﾟﾛｸﾞﾗﾑのVERSION文字列を設定します｡	{@value} */
	private static final String VERSION = "8.4.2.1 (2023/03/10)" ;
	private static final long serialVersionUID = 842120230310L ;

//	private static final String FIELD_IN  = "<fieldset style=\"display:inline;\">";
	private static final String FIELD_IN  = "<fieldset style=\"display:inline;\" class=\"menu_field\">";	// 8.4.1.2 (2023/03/03)
	private static final String FIELD_OUT = "</fieldset>" + BR;
//	private static final String JSP		 = HybsSystem.sys( "JSP" );
//	private static final String CNTX	 = HybsSystem.sys( "CONTEXT_NAME" );	// 5.5.4.2 (2012/07/13) META-INF/resources 対応
	private static final String MENU_IMG = "/jsp/menuImage/" ;					// 5.5.4.2 (2012/07/13) META-INF/resources 対応

	// 6.9.5.0 (2018/04/23) multiSessionCheck 廃止(true固定)
//	private boolean multiSessionCheck = HybsSystem.sysBool( "USE_MULTI_SESSION_CHECK" );

	// 6.7.5.0 (2017/03/10) TILE表示追加
	private static final Set<String> MENU_TYPE_SET = new ArraySet<>( "NORMAL","GROUP","ONELEVEL","NEXTGUI","MATRIX","MATRIX2","NONE","TILE" );

	private String   menuType	= "NORMAL" ;		// NORMAL,GROUP,ONELEVEL,NEXTGUI,MATRIX,MATRIX2,NONE,TILE
	private boolean  expand		= true;				// true:折り返しﾒﾆｭｰ / false:階層ﾒﾆｭｰ
	private String[] groups		;					// 表示対象となるｸﾞﾙｰﾌﾟをCSV形式で指定します｡
	private String   selClassify;					// 表示対象となる分類を指定します｡
	private String   href		= "menu.jsp";		// ｸﾞﾙｰﾌﾟﾒﾆｭｰの対象ｿｰｽ名(href)を指定します｡
	private String   target		= "MENU";			// ｸﾞﾙｰﾌﾟﾒﾆｭｰの対象ﾌﾚｰﾑ名(target)を指定します｡
	private boolean  imageOnly	;					// ｸﾞﾙｰﾌﾟﾒﾆｭｰに､ true:画像のみ / false:画像＋ﾗﾍﾞﾙ を指定します
	private int      sideCount	= -1;				// 5.2.3.0 (2010/12/01) 表示対象となるﾒﾆｭｰを横並びさせる数を指定します｡
	private int      minCellCount	= 8;			// 表形式ﾒﾆｭｰの１ｾﾙの最小行数を指定します｡
	private int      maxCellCount	= 8;			// 表形式ﾒﾆｭｰの１ｾﾙの最大行数を指定します｡
	private boolean  cache		= true;				// ｸﾞﾙｰﾌﾟﾒﾆｭｰのｷｬｯｼｭを使用するかどうか指定します｡
	// 3.8.8.7 (2007/05/01) ﾒﾆｭｰの表示条件指定追加
	private String match		;					// 正判定(ﾏｯﾁする場合に､ﾒﾆｭｰに出す)条件を設定
	private String unmatch		;					// 逆判定(ﾏｯﾁする場合に､ﾒﾆｭｰに出さない)条件を設定
	private String imageDir		= MENU_IMG;			// 6.7.5.0 (2017/03/10) TILE表示追加

	private final String JSP	 = HybsSystem.sys( "JSP" );				// 6.9.9.3 (2018/09/25) JavaDoc ｴﾗｰ対応

	// MULTI_SESSION_CHECK 時のﾘﾝｸに付加する情報
	private String	mscKey		;

	private boolean  useButton		;				// ﾎﾞﾀﾝ形式のﾘﾝｸを使用するか 4.2.1.0 (2008/04/01)
	private boolean  buttonRequest	;				// ﾏﾄﾘｸｽ２からの遷移でﾎﾞﾀﾝ形式にするかのﾘｸｴｽﾄ変数 4.2.1.0 (2008/04/17)
	private boolean  excludeButton	;				// 4.3.3.0 (2008/10/01) ﾎﾞﾀﾝﾒﾆｭｰの場合でも強制的に従来のﾌﾟﾙﾀﾞｳﾝにします｡
	private boolean  inlineStyle	;				// 4.3.3.0 (2008/10/01) trueでLv3の画面にstyle属性でinlineを付加

	private boolean  useDivOneLevel	;				// 5.5.2.3 (2012/05/15) ONELEVEL,NEXTGUI で､DIVﾀｸﾞ＋design-onelevel 処理をするかどうかを指定します｡
	private boolean  useLongName	;				// 8.2.1.0 (2022/07/15) 画面名称に､名前(長) を使う場合は､trueを指定します(初期値:false)

	// 7.3.2.0 (2021/03/19) FileMapは､static化して使いまわす｡
//	private final transient FileMap imgFileMap = new FileMap() ;		// 6.3.9.0 (2015/11/06) helpMapの初期化を､initﾒｿｯﾄﾞに変更する｡
	private FileMap imgFileMap ;										// いろんな箇所で使用するので､ｷｬｯｼｭ代わりに用意しておく
	private static final ConcurrentMap<String,FileMap> IMG_MAP = new ConcurrentHashMap<>();

	static {
		final Cleanable clr = new Cleanable() {
			/**
			 * 初期化(ｸﾘｱ)します｡
			 * 主に､ｷｬｯｼｭｸﾘｱで利用します｡
			 */
			public void clear() {
				IMG_MAP.forEach( (key,immap) -> immap.clear() );
			}
		};
		SystemManager.addCleanable( clr );
	}

	/**
	 * ﾃﾞﾌｫﾙﾄｺﾝｽﾄﾗｸﾀｰ
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public TopMenuTag() { super(); }		// これも､自動的に呼ばれるが､空のﾒｿｯﾄﾞを作成すると警告されるので､明示的にしておきます｡

	/**
	 * Taglibの終了ﾀｸﾞが見つかったときに処理する doEndTag() を ｵｰﾊﾞｰﾗｲﾄﾞします｡
	 *
	 * @og.rev 4.0.0.0 (2007/10/31) 1ﾚﾍﾞﾙﾒﾆｭｰの見直し
	 * @og.rev 4.2.1.0 (2008/04/01) ﾏﾄﾘｸｽﾒﾆｭｰ2追加
	 * @og.rev 5.2.3.0 (2010/12/01) NEXTGUI 追加
	 * @og.rev 5.3.0.0 (2010/11/22) NEXTGUI もﾏﾙﾁｾｯｼｮﾝﾁｪｯｸをしない｡
	 * @og.rev 5.3.9.0 (2011/09/01) ﾒﾆｭｰでのﾍﾙﾌﾟｱｲｺﾝ対応
	 * @og.rev 5.5.0.4 (2012/03/12) FAQ表示対応
	 * @og.rev 5.5.2.5 (2012/05/21) FAQ対応 廃止
	 * @og.rev 5.5.4.2 (2012/07/13) META-INF/resources からの読み取り対応
	 * @og.rev 6.3.8.3 (2015/10/03) NONE(表示しない) 追加｡
	 * @og.rev 6.3.8.4 (2015/10/09) topMenu 内でのHelp機能を廃止します｡
	 * @og.rev 6.3.8.4 (2015/10/09) FileMap のｺﾝｽﾄﾗｸﾀ変更に伴う対応｡
	 * @og.rev 6.3.9.0 (2015/11/06) helpMapの初期化を､initﾒｿｯﾄﾞに変更する｡
	 * @og.rev 6.7.5.0 (2017/03/10) TILE表示追加(imageDir 属性追加)
	 * @og.rev 6.9.4.0 (2018/04/02) caseKey ､caseVal 属性対応
	 * @og.rev 6.9.5.0 (2018/04/23) multiSessionCheck 廃止(true固定)
	 * @og.rev 6.9.5.0 (2018/04/23) 6.9.9.3 (2018/09/25) JavaDoc ｴﾗｰ対応
	 * @og.rev 7.3.2.0 (2021/03/19) FileMapは､static化して使いまわす｡
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		if( useTag() ) {		// 6.9.4.0 (2018/04/02) caseKey ､caseVal 属性対応

			// 6.3.8.3 (2015/10/03) NONE(表示しない) 追加｡
			if( "NONE".equals( menuType ) ) { return EVAL_PAGE; }

			// Ver 4.0.0 2007/09/04
			// ONELEVEL, NEXTGUI の場合は､multiSessionCheck を行いません｡
//			if( multiSessionCheck && !"ONELEVEL".equals( menuType ) && !"NEXTGUI".equals( menuType ) ) {
			if( !"ONELEVEL".equals( menuType ) && !"NEXTGUI".equals( menuType ) ) {							// 6.9.5.0 (2018/04/23) multiSessionCheck 廃止(true固定)
				final String cnt = (String)getSessionAttribute( HybsSystem.MULTI_SESSION_CHECK );
				if( cnt == null ) {
					final String errMsg = "ﾏﾙﾁｾｯｼｮﾝﾁｪｯｸのｷｰが存在しません｡"
								+ "topMenu ﾀｸﾞを実行する前に､必ず jspInit ﾀｸﾞを"
								+ "実行しておいてください｡" ;
					jspPrint( errMsg );
					return SKIP_PAGE ;		// ﾍﾟｰｼﾞの残りの処理を行わない｡
				}
				mscKey = HybsSystem.MULTI_SESSION_CHECK + "=" + cnt ;
			}

			// 7.3.2.0 (2021/03/19) FileMapは､static化して使いまわす｡
			imgFileMap = IMG_MAP.computeIfAbsent( imageDir, k -> new FileMap() ) ;
			if( !imgFileMap.isInit() ) {
				final Set<String> pathSet = pageContext.getServletContext().getResourcePaths( imageDir );
				if( pathSet != null ) {
					imgFileMap.init( imageDir , "/" + HybsSystem.getContextName() + imageDir , pathSet );	// 6.9.9.3 (2018/09/25) JavaDoc ｴﾗｰ対応
				}
			}

//			// 6.7.5.0 (2017/03/10) TILE表示追加(imageDir 属性追加)
//	//		final Set<?> set = pageContext.getServletContext().getResourcePaths( imageDir );
//			// 6.3.8.4 (2015/10/09) FileMap のｺﾝｽﾄﾗｸﾀ変更に伴う対応｡
//			// 6.3.9.0 (2015/11/06) helpMapの初期化を､initﾒｿｯﾄﾞに変更する｡
////		imgFileMap.init( imageDir , "/" + CNTX + imageDir , set );
//			imgFileMap.init( imageDir , "/" + HybsSystem.getContextName() + imageDir , set );		// 6.9.9.3 (2018/09/25) JavaDoc ｴﾗｰ対応

			if( "NORMAL".equals( menuType ) ) {
				jspPrint( "<div class=\"forpos\" id=\"dummy\"></div>" + CR );
				if( ! useButton ){ // 4.2.1.0 (2008/04/26) ﾎﾞﾀﾝ時にはお気に入りを出さない
					jspPrint( makeFavoriteMenu() );
				}
				jspPrint( makeMenu() );
			}
			else if( "GROUP".equals( menuType ) ) {
				jspPrint( makeGroupMenu() );
			}
		//	else if( "CLASSIFY".equals( menuType ) ) {
		//		jspPrint( makeClassifyMenu() );
		//	}
			else if( "ONELEVEL".equals( menuType ) && selClassify != null ) {
				jspPrint( makeOneLevelMenu() );
			}
			// 5.2.3.0 (2010/12/01) NEXTGUI 追加
			else if( "NEXTGUI".equals( menuType ) ) {
				jspPrint( makeNextguiMenu() );
			}
			else if( "MATRIX".equals( menuType ) ) {
				jspPrint( makeMatrixMenu() );
			}
			else if( "MATRIX2".equals( menuType ) ) {	// 4.2.1.0 (2008/04/01) 大分類なし版追加
				jspPrint( makeMatrixMenu2() );
			}
			else if( "TILE".equals( menuType ) ) {		// 6.7.5.0 (2017/03/10) TILE表示追加
				jspPrint( makeTileMenu() );
			}
		//	else {
		//		jspPrint( "menuType が想定外です｡menuType=[" + menuType + "]" );
		//	}
		}

		return EVAL_PAGE ;		// ﾍﾟｰｼﾞの残りを評価する｡
	}

	/**
	 * ﾀｸﾞﾘﾌﾞｵﾌﾞｼﾞｪｸﾄをﾘﾘｰｽします｡
	 * ｷｬｯｼｭされて再利用されるので､ﾌｨｰﾙﾄﾞの初期設定を行います｡
	 *
	 * @og.rev 5.2.3.0 (2010/12/01) sideCountの初期値を -1(無制限)に変更
	 * @og.rev 5.3.9.0 (2011/09/01) ﾒﾆｭｰでのﾍﾙﾌﾟｱｲｺﾝ対応
	 * @og.rev 5.5.2.3 (2012/05/15) ONELEVEL,NEXTGUI で､DIVﾀｸﾞ＋design-onelevel 処理をするかどうかを指定します｡
	 * @og.rev 6.3.8.4 (2015/10/09) topMenu 内でのHelp機能を廃止します｡
	 * @og.rev 6.3.9.0 (2015/11/06) helpMapの初期化を､initﾒｿｯﾄﾞに変更する｡
	 * @og.rev 6.7.5.0 (2017/03/10) TILE表示追加(imageDir 属性追加)
	 * @og.rev 6.9.5.0 (2018/04/23) multiSessionCheck 廃止(true固定)
	 * @og.rev 7.3.2.0 (2021/03/19) FileMapは､static化して使いまわす｡
	 * @og.rev 8.2.1.0 (2022/07/15) useLongName 属性追加
	 */
	@Override
	protected void release2() {
		super.release2();
		expand				= true;
		menuType			= "NORMAL";
		groups				= null;
		selClassify			= null;
//		multiSessionCheck	= HybsSystem.sysBool( "USE_MULTI_SESSION_CHECK" );
		href				= "menu.jsp";		// ｸﾞﾙｰﾌﾟﾒﾆｭｰの対象ｿｰｽ名(href)を指定します｡
		imageOnly			= false;			// ｸﾞﾙｰﾌﾟﾒﾆｭｰに､ true:画像のみ / false:画像＋ﾗﾍﾞﾙ を指定します
		target				= "MENU";			// ｸﾞﾙｰﾌﾟﾒﾆｭｰの対象ﾌﾚｰﾑ名(target)を指定します｡
		sideCount			= -1;				// 5.2.3.0 (2010/12/01) 表示対象となるﾒﾆｭｰを横並びさせる数を指定します｡
		minCellCount		= 8;				// 表形式ﾒﾆｭｰの１ｾﾙの最小行数を指定します｡
		maxCellCount		= 8;				// 表形式ﾒﾆｭｰの１ｾﾙの最大行数を指定します｡
		mscKey				= null;				// MULTI_SESSION_CHECK 時のﾘﾝｸに付加する情報
//		imgFileMap.clear();						// 6.3.9.0 (2015/11/06) helpMapの初期化を､initﾒｿｯﾄﾞに変更する｡
		imgFileMap			= null;				// 7.3.2.0 (2021/03/19) FileMapは､static化して使いまわす｡imgFileMapはMapから取り出したｷｬｯｼｭ
		cache				= true;				// ｸﾞﾙｰﾌﾟﾒﾆｭｰのｷｬｯｼｭを使用するかどうか指定します｡
		match				= null;				// 正判定(ﾏｯﾁする場合に､ﾒﾆｭｰに出す)条件を設定
		unmatch				= null;				// 逆判定(ﾏｯﾁする場合に､ﾒﾆｭｰに出さない)条件を設定
		imageDir			= MENU_IMG;			// 6.7.5.0 (2017/03/10) TILE表示追加
		useButton			= false;			// ﾎﾞﾀﾝ形式のﾘﾝｸを使用するか
		buttonRequest		= false;			// trueでﾎﾞﾀﾝﾒﾆｭｰの表示を行う
		excludeButton		= false;			// trueでﾎﾞﾀﾝﾒﾆｭｰ時でも強制的に従来のﾌﾟﾙﾀﾞｳﾝにする｡
		inlineStyle			= false;			// trueの場合は標準画面(Lv3)にdisplay:inlineを付加
		useDivOneLevel		= false;			// 5.5.2.3 (2012/05/15) ONELEVEL,NEXTGUI で､DIVﾀｸﾞ＋design-onelevel 処理をするかどうかを指定します｡
		useLongName			= false;			// 8.2.1.0 (2022/07/15) 画面名称に､名前(長) を使う場合は､trueを指定します(初期値:false)
	}

	/**
	 * ﾒﾆｭｰを表示する為のHTMLを作成します(折り返しJavaScript対応版)｡
	 *
	 * @og.rev 3.5.6.5 (2004/08/09) GUIInfo の comments 属性を title にｾｯﾄする｡
	 * @og.rev 3.6.0.9 (2004/12/03) ﾘｱﾙｱﾄﾞﾚｽ設定時に､{&#064;XXXX}処理を追加
	 * @og.rev 3.8.0.0 (2005/06/07) 同一ｾｯｼｮﾝでのﾏﾙﾁ起動対策を行います｡
	 * @og.rev 4.0.0.0 (2005/01/31) GUIInfoの実ｱﾄﾞﾚｽのﾊﾟﾗﾒｰﾀを考慮する｡
	 * @og.rev 4.0.0.0 (2007/10/31) 分類の廃止に伴い､全面見直し
	 * @og.rev 4.2.1.0 (2008/04/01) 小分類指定での表示対応(↑の対応での再実装漏れ)
	 * @og.rev 4.2.1.0 (2008/04/11) 小分類をexpandしない場合にはspanﾀｸﾞで囲う｡
	 * @og.rev 5.5.2.5 (2012/05/21) ｲﾒｰｼﾞｱｲｺﾝ対応(makeTagMenuのﾊﾟﾗﾒｰﾀ変更)
	 * @og.rev 6.9.4.1 (2018/04/09) HYBS_BR の廃止 , gamenId の変数設定
	 * @og.rev 8.2.1.0 (2022/07/15) useLongName 属性追加
	 *
	 * @return  ﾒﾆｭｰ
	 * @og.rtnNotNull
	 */
	private String makeMenu() {
		final StringBuilder rtn = new StringBuilder( BUFFER_LARGE );

		final UserInfo userInfo = getUser();

		// 4.0.0 (2005/01/31)

		final GUIInfo[] guiInfos = userInfo.getGUIInfos();
//		String gamenId		;			// 画面ID
		int menuNo = 11;
		int kmokuNo = 0;

		boolean isInFieldset	= false;		// 大分類の中か？
		boolean isInClassify	= false;		// 小分類の中か？
		boolean isHiddenMenu	= false;		// 隠しﾒﾆｭｰが存在するか？	6.0.2.5 (2014/10/31) refactoring
		boolean isClassHidden	= false;		// 分類隠しが存在するか？		6.0.2.5 (2014/10/31) refactoring
		int level ;
		int preLevel = 0;

		for( int i=0; i<guiInfos.length; i++ ) {
			if( guiInfos[i].isRead() ) {		// 4.0.0 (2005/01/31)
				final String gamenId = guiInfos[i].getKey() ;

//				if( match   != null && !gamenId.matches( match  ) ) { continue; }
//				if( unmatch != null && gamenId.matches( unmatch ) ) { continue; }
				// 6.9.7.0 (2018/05/14) PMD
				if(		match  != null && !gamenId.matches( match )
					|| unmatch != null && gamenId.matches( unmatch ) ) { continue; }

				// 4.2.1.0 (2008/04/01)
				// 6.9.7.0 (2018/05/14) PMD These nested if statements could be combined
//				if( selClassify != null ) {
////					if( !selClassify.equals( guiInfos[i].getKey() ) && !selClassify.equals( guiInfos[i].getClassify() ) ) {
//					if( !selClassify.equals( gamenId ) && !selClassify.equals( guiInfos[i].getClassify() ) ) {
				if( selClassify != null && !selClassify.equals( gamenId ) && !selClassify.equals( guiInfos[i].getClassify() ) ) {
						continue; // 分類(classify) に含まれない
//					}
				}

				// 処理すべき画面かのﾁｪｯｸ
				final int guiFlg = guiCheck( guiInfos, i );
				if( guiFlg == 0 ) { continue; }

				level = guiInfos[i].getLevel();		// 4.0.0 (2005/01/31)

				// 隠しﾒﾆｭｰ展開用
				if( preLevel >= 3 && level < 3 ) {
					if( isHiddenMenu ) {			// 6.0.2.5 (2014/10/31) refactoring
						rtn.append( makeEllipses( menuNo,kmokuNo,3 ) );
						kmokuNo++;
					}
					isHiddenMenu = false;			// 6.0.2.5 (2014/10/31) refactoring
				}

				// 大分類(ﾌｨｰﾙﾄﾞﾒﾆｭｰ)
				if( level == 1 ) {
					if( isInFieldset ) {
						rtn.append( FIELD_OUT )
							.append( "</div>" );
					}
					isInFieldset = true;

					// 隠しﾒﾆｭｰ用
					if( expand ) {
						rtn.append( "<div class=\"expand1 " );

						if( guiFlg == 1 ) {
							rtn.append( "unpopular ");
						}

						rtn.append( "\">" );
					}
//					rtn.append( makeTagMenu( guiInfos[i],guiInfos[i].getName(),level ) );			// 5.5.2.5 (2012/05/21) ｲﾒｰｼﾞｱｲｺﾝ対応
					rtn.append( makeTagMenu( guiInfos[i],guiInfos[i].getName(useLongName),level ) );	// 8.2.1.0 (2022/07/15) true:名前(長)/false:名前(短)

					isInClassify = false;
				}
				// 小分類・直ﾘﾝｸ
				else if( level == 2 ) {
					menuNo++;
					kmokuNo = 0;
					if( expand ) {
						rtn.append( "<div class=\"expand1 " );

						// 隠しﾒﾆｭｰ用
						if( guiFlg == 1 ) {
							rtn.append( "unpopular " );
							isClassHidden = true;			// 6.0.2.5 (2014/10/31) refactoring
						}

						// rtn.append( "\" id=\"menu" + menuNo + "\">" );
						// 4.3.3.0 (2008/10/01) useButton=true時は表示しない
						rtn.append( "\" id=\"menu" + menuNo + "\" ");
						if( useButton ){
							rtn.append( "style=\"display:none\"");
						}
						rtn.append( '>');		// 6.0.2.5 (2014/10/31) char を append する｡
					}
					else{	// 4.2.1.0 (2008/04/11)expandしない場合にはspan要素で囲ってnon-expandｸﾗｽを指定｡
						rtn.append( "<span class=\"non-expand\">" );
					}

//					rtn.append( makeTagMenu( guiInfos[i],guiInfos[i].getName(),level ) );		// 5.5.2.5 (2012/05/21) ｲﾒｰｼﾞｱｲｺﾝ対応
					rtn.append( makeTagMenu( guiInfos[i],guiInfos[i].getName(useLongName),level ) );	// 8.2.1.0 (2022/07/15) true:名前(長)/false:名前(短)
					rtn.append( BR );
					if( expand ) {
						rtn.append( "</div>" );
					}
					else{	// 4.2.1.0 (2008/04/11)
						rtn.append( "</span>");
					}

					isInClassify = true;
				}
				// 通常ﾒﾆｭｰ・隠しﾒﾆｭｰ
				// 6.9.7.0 (2018/05/14) PMD These nested if statements could be combined
//				else if( level >= 3 ) {
//					if( isInClassify ) {
				else if( level >= 3 && isInClassify ) {
						if( expand ) {
							rtn.append( "<div class=\"expand2 " );

							// 隠しﾒﾆｭｰ用
							if( level == 4 ) {
								rtn.append( "unpopular " );
								isHiddenMenu = true;			// 6.0.2.5 (2014/10/31) refactoring
							}

							// rtn.append( "\" id=\"menu" + menuNo + "_" + kmokuNo + "\">" );
							// 4.3.3.0 (2008/10/01) ﾒﾆｭｰを初期状態で開けるようにする
							// 6.0.2.5 (2014/10/31) char を append する｡
							rtn.append( "\" id=\"menu" ).append( menuNo ).append( '_' ).append( kmokuNo ).append( "\" " );
							if( inlineStyle && level == 3 ) {
								rtn.append( " style=\"display:inline\"" );
							}
							rtn.append( '>' );		// 6.0.2.5 (2014/10/31) char を append する｡
						}
			//			// 6.9.4.1 (2018/04/09) HYBS_BR 対応
			//			if( "HYBS_BR".equals( guiInfos[i].getAddress() ) ) {
			//				rtn.append( BR );
			//			}
			//			else {
//			//			// 画面IDが"HYBS_BR"の時は処理しない
//			//			if( !"HYBS_BR".equals( guiInfos[i].getKey() ) ) {
//							rtn.append( makeTagMenu( guiInfos[i],guiInfos[i].getName(),level ) );		// 5.5.2.5 (2012/05/21) ｲﾒｰｼﾞｱｲｺﾝ対応
							rtn.append( makeTagMenu( guiInfos[i],guiInfos[i].getName(useLongName),level ) );	// 8.2.1.0 (2022/07/15) true:名前(長)/false:名前(短)
			//			}
						rtn.append( BR );
						if( expand ) {
							rtn.append( "</div>" );
						}
						kmokuNo++;
//					}
				}

				rtn.append( CR );

				preLevel = level;
			}
		}

		// 終了処理
		if( isHiddenMenu ) {										// 6.0.2.5 (2014/10/31) refactoring
			rtn.append( makeEllipses( menuNo,kmokuNo,3 ) );
		}

		if( isInFieldset ) {
			rtn.append( FIELD_OUT );
			rtn.append( "</div>" );
		}

		if( isClassHidden ) {										// 6.0.2.5 (2014/10/31) refactoring
			rtn.append( makeEllipses( menuNo,0,2 ) );
		}

		return rtn.toString();
	}

	/**
	 * TILEﾒﾆｭｰを表示する為のHTMLを作成します(折り返しJavaScript対応版)｡
	 *
	 * @og.rev 6.7.5.0 (2017/03/10) TILE表示追加
	 * @og.rev 6.9.4.0 (2018/04/02) HYBS_BR(ﾌﾞﾚｲｸ)の条件を緩める｡
	 * @og.rev 6.9.4.1 (2018/04/09) HYBS_BR の廃止 , gamenId の変数設定
	 * @og.rev 8.2.1.0 (2022/07/15) useLongName 属性追加
	 *
	 * @return  ﾒﾆｭｰ
	 * @og.rtnNotNull
	 */
	private String makeTileMenu() {
		final StringBuilder rtn = new StringBuilder( BUFFER_LARGE );

		final UserInfo userInfo = getUser();

		// 4.0.0 (2005/01/31)

		final GUIInfo[] guiInfos = userInfo.getGUIInfos();
//		String gamenId		;			// 画面ID
		int menuNo = 11;
		int kmokuNo = 0;

		boolean isInFieldset	= false;		// 大分類の中か？
		boolean isInClassify	= false;		// 小分類の中か？
		boolean isHiddenMenu	= false;		// 隠しﾒﾆｭｰが存在するか？	6.0.2.5 (2014/10/31) refactoring
		boolean isClassHidden	= false;		// 分類隠しが存在するか？		6.0.2.5 (2014/10/31) refactoring
		int level ;
		int preLevel = 0;

		String hybsBR = "";						// 6.7.5.0 (2017/03/10)
		for( int i=0; i<guiInfos.length; i++ ) {
			if( guiInfos[i].isRead() ) {		// 4.0.0 (2005/01/31)
				final String gamenId = guiInfos[i].getKey() ;
				if( match   != null && !gamenId.matches( match  ) ) { continue; }
				if( unmatch != null && gamenId.matches( unmatch ) ) { continue; }

				// 4.2.1.0 (2008/04/01)
				// 6.9.7.0 (2018/05/14) PMD These nested if statements could be combined
//				if( selClassify != null ) {
//					if( !selClassify.equals( gamenId ) && !selClassify.equals( guiInfos[i].getClassify() ) ) {
				if( selClassify != null && !selClassify.equals( gamenId ) && !selClassify.equals( guiInfos[i].getClassify() ) ) {
						continue; // 分類(classify) に含まれない
//					}
				}

				// 6.9.4.0 (2018/04/02) HYBS_BR(ﾌﾞﾚｲｸ)の条件を緩める｡
				if( isInFieldset && isInClassify && "HYBS_BR".equals( guiInfos[i].getAddress() ) ) {		// 6.7.5.0 (2017/03/10)
//				if( "HYBS_BR".equals( gamenId ) ) {										// 6.9.4.0 (2018/04/02)
					hybsBR = "</tr><tr>" ;
				}

				// 処理すべき画面かのﾁｪｯｸ
				final int guiFlg = guiCheck( guiInfos, i );
				if( guiFlg == 0 ) { continue; }

				level = guiInfos[i].getLevel();		// 4.0.0 (2005/01/31)

				// 隠しﾒﾆｭｰ展開用
				if( preLevel >= 3 && level < 3 ) {
					if( isHiddenMenu ) {			// 6.0.2.5 (2014/10/31) refactoring
						rtn.append( makeEllipses( menuNo,kmokuNo,3 ) );
						kmokuNo++;
					}
					isHiddenMenu = false;			// 6.0.2.5 (2014/10/31) refactoring
				}

				// 大分類(ﾌｨｰﾙﾄﾞﾒﾆｭｰ)
				if( level == 1 ) {
					if( isInFieldset ) {
						rtn.append( "</td></tr></table>" )		// 6.7.5.0 (2017/03/10)
							.append( FIELD_OUT )
							.append( "</div>" );
					}
					isInFieldset = true;

					// 隠しﾒﾆｭｰ用
					if( expand ) {
						rtn.append( "<div class=\"expand1 " );

						if( guiFlg == 1 ) {
							rtn.append( "unpopular ");
						}
						rtn.append( "\">" );
					}
//					rtn.append( makeTagMenu( guiInfos[i],guiInfos[i].getName(),level ) )		// 5.5.2.5 (2012/05/21) ｲﾒｰｼﾞｱｲｺﾝ対応
					rtn.append( makeTagMenu( guiInfos[i],guiInfos[i].getName(useLongName),level ) )	// 8.2.1.0 (2022/07/15) true:名前(長)/false:名前(短)
						.append( "<table><tr>" );		// 6.7.5.0 (2017/03/10)

					isInClassify = false;
				}
				// 小分類・直ﾘﾝｸ
				else if( level == 2 ) {
					menuNo++;
					kmokuNo = 0;
					if( expand ) {
						if( isInFieldset ) {		// 6.7.5.0 (2017/03/10)
							if( isInClassify ) {
								rtn.append( "</td>" ).append( hybsBR ).append( "<td>" );
								hybsBR = "";
							}
							else {
								rtn.append( "<td>" );
							}
						}
						rtn.append( "<div class=\"expand1 " );

						// 隠しﾒﾆｭｰ用
						if( guiFlg == 1 ) {
							rtn.append( "unpopular " );
							isClassHidden = true;			// 6.0.2.5 (2014/10/31) refactoring
						}

						// rtn.append( "\" id=\"menu" + menuNo + "\">" );
						// 4.3.3.0 (2008/10/01) useButton=true時は表示しない
						rtn.append( "\" id=\"menu" + menuNo + "\" ");
						if( useButton ){
							rtn.append( "style=\"display:none\"");
						}
						rtn.append( '>');		// 6.0.2.5 (2014/10/31) char を append する｡
					}
					else{	// 4.2.1.0 (2008/04/11)expandしない場合にはspan要素で囲ってnon-expandｸﾗｽを指定｡
						rtn.append( "<span class=\"non-expand\">" );
					}

//					rtn.append( makeTagMenu( guiInfos[i],guiInfos[i].getName(),-6 ) );			// 6.7.5.0 (2017/03/10) TILE表示
					rtn.append( makeTagMenu( guiInfos[i],guiInfos[i].getName(useLongName),-6 ) );	// 8.2.1.0 (2022/07/15) true:名前(長)/false:名前(短)
					rtn.append( BR );
					if( expand ) {
						rtn.append( "</div>" );
					}
					else{	// 4.2.1.0 (2008/04/11)
						rtn.append( "</span>");
					}

					isInClassify = true;
				}
				// 通常ﾒﾆｭｰ・隠しﾒﾆｭｰ
				// 6.9.7.0 (2018/05/14) PMD These nested if statements could be combined
//				else if( level >= 3 ) {
//					if( isInClassify ) {
				else if( level >= 3 && isInClassify ) {
						if( expand ) {
							rtn.append( "<div class=\"expand2 " );

							// 隠しﾒﾆｭｰ用
							if( level == 4 ) {
								rtn.append( "unpopular " );
								isHiddenMenu = true;			// 6.0.2.5 (2014/10/31) refactoring
							}

							// rtn.append( "\" id=\"menu" + menuNo + "_" + kmokuNo + "\">" );
							// 4.3.3.0 (2008/10/01) ﾒﾆｭｰを初期状態で開けるようにする
							// 6.0.2.5 (2014/10/31) char を append する｡
							rtn.append( "\" id=\"menu" ).append( menuNo ).append( '_' ).append( kmokuNo ).append( "\" " );
							if( inlineStyle && level == 3 ) {
								rtn.append( " style=\"display:inline\"" );
							}
							rtn.append( '>' );		// 6.0.2.5 (2014/10/31) char を append する｡
						}

			//			// 6.9.4.1 (2018/04/09) HYBS_BR 対応
			//			if( "HYBS_BR".equals( guiInfos[i].getAddress() ) ) {					// 6.9.4.1 (2018/04/09) HYBS_BR は､ｱﾄﾞﾚｽに設定
			//				rtn.append( BR );
			//			}
			//			else {
//			//			// 画面IDが"HYBS_BR"の時は処理しない
//			//			if( !"HYBS_BR".equals( gamenId ) ) {
//							rtn.append( makeTagMenu( guiInfos[i],guiInfos[i].getName(),level ) );		// 5.5.2.5 (2012/05/21) ｲﾒｰｼﾞｱｲｺﾝ対応
							rtn.append( makeTagMenu( guiInfos[i],guiInfos[i].getName(useLongName),level ) );	// 8.2.1.0 (2022/07/15) true:名前(長)/false:名前(短)
			//			}
						rtn.append( BR );
						if( expand ) {
							rtn.append( "</div>" );
						}
						kmokuNo++;
//					}
				}

				rtn.append( CR );
				preLevel = level;
			}
		}

		// 終了処理
		if( isHiddenMenu ) {										// 6.0.2.5 (2014/10/31) refactoring
			rtn.append( makeEllipses( menuNo,kmokuNo,3 ) );
		}

		if( isInFieldset ) {
			rtn.append( "</td></tr></table>" )						// 6.7.5.0 (2017/03/10)
				.append( FIELD_OUT )
				.append( "</div>" );
		}

		if( isClassHidden ) {										// 6.0.2.5 (2014/10/31) refactoring
			rtn.append( makeEllipses( menuNo,0,2 ) );
		}

		return rtn.toString();
	}

	/**
	 * ﾒﾆｭｰを表示する為のHTMLを作成します(折り返しJavaScript対応版)｡
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 個人別のお気に入りﾒﾆｭｰを作成します｡
	 * @og.rev 4.0.0.0 (2007/10/31) 分類項目の廃止
	 * @og.rev 4.1.1.0 (2008/02/05) お気に入り情報はGEA09から取得するように変更
	 * @og.rev 5.5.2.5 (2012/05/21) ｲﾒｰｼﾞｱｲｺﾝ対応(makeTagMenuのﾊﾟﾗﾒｰﾀ変更)
	 * @og.rev 6.4.4.1 (2016/03/18) StringBuilderの代わりに､OgBuilderを使用する｡
	 * @og.rev 6.9.9.1 (2018/08/27) お気に入り編集ﾒﾆｭｰ(GE0014→GFX000)変更｡
	 *
	 * @return  個人別のお気に入りﾒﾆｭｰ
	 * @og.rtnNotNull
	 */
	private String makeFavoriteMenu() {

		final UserInfo userInfo = getUser();
		final Map<String,FavoriteGUIData> favoriteMap = userInfo.getFavoriteMap();

		if( favoriteMap.isEmpty() ) { return ""; }

		final ResourceManager resource = getResource();	// ﾘｿｰｽ参照
		final String	largeClassify	 = resource.getLabelData( "FAVORITE_MENU" ).getShortLabel(); // お気に入り
		String	lastClassify	 = "";				// 前方画面の分類
		int		menuNo			 = 99999;
		int		kmokuNo			 = 0;

		// fieldSetﾀｸﾞ､お気に入り編集画面ﾘﾝｸの出力
//		final GUIInfo editFavorite = userInfo.getGUIInfo( "GE0014" );
		final GUIInfo editFavorite = userInfo.getGUIInfo( "GFX000" );		// 6.9.9.1 (2018/08/27)

		final OgBuilder rtn = new OgBuilder()
			.append( "<div class=\"expand1\"> "
					, makeTagMenuString( null,null,largeClassify,null,1 )
					, makeTagMenu( editFavorite,resource.getLabelData( "EDIT" ).getShortLabel(),2 )		// 5.5.2.5 (2012/05/21) ｲﾒｰｼﾞｱｲｺﾝ対応
					, BR );

		String thisClassify = null;
//		String gamenId = null;
		FavoriteGUIData favoriteGui = null;
		// 4.3.4.4 (2009/01/01) findBugs警告対応
		for( final Map.Entry<String, FavoriteGUIData> entry : favoriteMap.entrySet() ) {
			final String gamenId     = entry.getKey();
			favoriteGui = entry.getValue();
			final GUIInfo guiInfo = userInfo.getGUIInfo( gamenId );

			if( match   != null && !gamenId.matches( match  ) ) { continue; }
			if( unmatch != null && gamenId.matches( unmatch ) ) { continue; }

			thisClassify = favoriteGui.getClassify();
			if( !lastClassify.equals( thisClassify ) ) {
				menuNo++;
				kmokuNo = 0;
				rtn.appendIf( expand , "<div class=\"expand1\" id=\"menu" , String.valueOf( menuNo ) , "\">" )
					.append( makeTagMenuString( null,null,thisClassify,thisClassify,2 ) )
					.append( BR )
					.appendIfCR( expand , "</div>" );
				lastClassify = thisClassify;
			}

			final String linkLabel = favoriteGui.getName();
			rtn.appendIf( expand
							, "<div class=\"expand2\" id=\"menu"
							, String.valueOf( menuNo ) , "_" , String.valueOf( kmokuNo ) , "\">" )
				.append( makeTagMenu( guiInfo,linkLabel,3 ) )		// 5.5.2.5 (2012/05/21) ｲﾒｰｼﾞｱｲｺﾝ対応
				.append( BR )
				.appendIfCR( expand , "</div>" );
			kmokuNo++;
		}
		return rtn.append( FIELD_OUT , "</div>" ).toString();
	}

	/**
	 * 折りたたみﾒﾆｭｰで､非標準ﾒﾆｭｰの 表示に使う､"←･･･→" を作成します｡
	 *
	 * @param	menuNo	階層番号
	 * @param	kmokuNo	階層項目番号
	 * @param	type	ﾀｲﾌﾟ(1,2限定)
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規作成
	 * @og.rev 6.4.4.1 (2016/03/18) StringBuilderの代わりに､OgBuilderを使用する｡
	 *
	 * @return  ﾒﾆｭｰ
	 * @og.rtnNotNull
	 */
	private String makeEllipses( final int menuNo,final int kmokuNo,final int type ) {
		final int tmpType = type-1; // 超暫定対応 4.0.0.0 (2007/10/31)
		final String kmkNo = ( tmpType == 1 ) ? "" : "_" + kmokuNo ;

		// 4.3.3.0 (2008/10/01) inlineStyleがtrueの場合は←･･･→をinlineで表示する
		return new OgBuilder()
				.append( "<div class=\"expand"		, String.valueOf( tmpType ) )
				.append( " ellipses\" id=\"menu"	, String.valueOf( menuNo ) , kmkNo )
				.append( "\" " )
				.appendIf( inlineStyle , "style=\"display:inline\" " )				// if
				.appendCR( ">" , makeTagMenuString( null,null,"←･･･→",null,type )
							, BR , "</div>" )
				.toString();
	}

	/**
	 * menuType="GROUP" 時に作成するｸﾞﾙｰﾌﾟﾒﾆｭｰ｡
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規作成
	 * @og.rev 4.0.0.0 (2007/10/05) ｸﾞﾙｰﾌﾟのｺｰﾄﾞﾘｿｰｽが存在しない場合
	 * @og.rev 4.2.2.0 (2008/05/14) buttonRequestがtrueの場合はTOPへのﾘﾝｸを表示しない
	 * @og.rev 4.2.2.0 (2008/05/15) ｸﾞﾙｰﾌﾟ絞込解除(GUI_ALL)の表示文字にﾗﾍﾞﾙﾘｿｰｽを使う｡
	 * @og.rev 5.0.0.3 (2009/09/22) ｸﾞﾙｰﾌﾟが1件の場合に表示されないﾊﾞｸﾞを修正
	 * @og.rev 5.1.8.0 (2010/07/01) ｺｰﾄﾞﾘｿｰｽのｷｰが存在しない場合にｴﾗｰとなるﾊﾞｸﾞを修正
	 * @og.rev 5.9.7.1 (2016/04/06) GROPUにgroupsが効くようにする
	 * @og.rev 6.4.5.0 (2016/04/08) GROPUにgroupsが効くようにする(効率化､ﾙｰﾌﾟから出す)
	 * @og.rev 6.8.3.1 (2017/12/01) GROPUの表示順を､ｺｰﾄﾞﾘｿｰｽの並び順にします｡
	 * @og.rev 5.10.15.2 (2019/09/20) GROUPのﾏﾙﾁﾊﾞｲﾄ文字対応
	 *
	 * @return  ﾒﾆｭｰ
	 */
	private String makeGroupMenu() {

		// 画面のｸﾞﾙｰﾌﾟﾒﾆｭｰのｷｬｯｼｭを取得します｡
		String groupMenu ;
		final String cacheKey  = HybsSystem.GUI_GR_MENU_CACHE_KEY + href + target ;
		if( cache ) {
			groupMenu = (String)getSessionAttribute( cacheKey );
			if( groupMenu != null ) { return groupMenu; }	// ｷｬｯｼｭを返します｡
		}
		else {
			removeSessionAttribute( cacheKey );
		}

		final UserInfo userInfo = getUser();

		// 6.8.3.1 (2017/12/01) GROPUの表示順を､ｺｰﾄﾞﾘｿｰｽの並び順にします｡
		final CodeData groupCode = getResource().getCodeData( "GROUPS" ) ;
		final Set<String> groupSet = groupCode == null ? new LinkedHashSet<>()				// 要素がｾｯﾄに挿入された順序です(挿入順)｡
													   : new TreeSet<>( (v1,v2) -> groupCode.getAddress( v1 ) - groupCode.getAddress( v2 ) );	// ｺｰﾄﾞﾘｿｰｽの並び順

		// そのﾕｰｻﾞｰで使用できる画面をすべてﾋﾟｯｸｱｯﾌﾟします｡
		// その上で､読取可能なﾒﾆｭｰを含むｸﾞﾙｰﾌﾟを順番に Set にｾｯﾄしていきます｡
		final GUIInfo[] guiInfos = userInfo.getGUIInfos();

		// 7.2.9.4 (2020/11/20) PMD:This for loop can be replaced by a foreach loop
		for( final GUIInfo guiInfo : guiInfos ) {
			if( guiInfo.isRead() ) {
//		for( int i=0; i<guiInfos.length; i++ ) {
//			if( guiInfos[i].isRead() ) {
				final String gamenId = guiInfo.getKey() ;
//				final String gamenId = guiInfos[i].getKey() ;
				if( match   != null && !gamenId.matches( match  ) ) { continue; }
				if( unmatch != null && gamenId.matches( unmatch ) ) { continue; }
				// 6.4.5.0 (2016/04/08) 元のﾙｰﾌﾟを､拡張for文に変更しただけ｡
				final String[] guiGroups = StringUtil.csv2Array( guiInfo.getGroups() );
//				final String[] guiGroups = StringUtil.csv2Array( guiInfos[i].getGroups() );
				for( final String grp : guiGroups ) { groupSet.add( grp ); }
			}
		}

		// 6.4.5.0 (2016/04/08) groupsの削除は､後でまとめて行う｡
		if( groups != null ) {
			for( final String grp : groups ) { groupSet.remove( grp ); }
		}

		// 5.0.0.3 (2009/09/22)
		// 6.1.1.0 (2015/01/17) refactoring ｡size()判定をisEmpty() にすると同時に､if...else を反転する｡
		if( groupSet.isEmpty() ) {
			groupMenu = "";
		}
		else {
			int sideTmpCnt = 1;
			final StringBuilder rtn = new StringBuilder( BUFFER_LARGE );
			rtn.append( "<tr>" );

			final String allMenu = getLabel( "ALL_MENU" ); // 4.2.2.0 (2008/05/15) ALLはﾗﾍﾞﾙﾘｿｰｽ使うように変更
			rtn.append( makeTagMenuString( href,target,allMenu,"GUI_ALL",-1 ) );
			if( sideCount > 0 && sideTmpCnt % sideCount == 0 ) { rtn.append("</tr><tr>"); }
			sideTmpCnt++ ;

			final Iterator<String> ite = groupSet.iterator() ;
			if( groupCode != null ) {
				while( ite.hasNext() ) {
					final String group = ite.next();
					final int cdAdrs = groupCode.getAddress( group ) ;
					// 4.0.0.0 (2007/10/05) ｸﾞﾙｰﾌﾟのｺｰﾄﾞﾘｿｰｽが存在しない場合
					String groupLabel = "";
					if( cdAdrs >= 0 ) { groupLabel = groupCode.getLongLabel( cdAdrs ); }

					// 5.1.8.0 (2010/07/01) ｺｰﾄﾞﾘｿｰｽのｷｰが存在しない場合にｴﾗｰとなるﾊﾞｸﾞを修正
					if( groupLabel.isEmpty() && group != null && group.length() > 0 ) {
						groupLabel = group;
					}

//					final String src = XHTMLTag.addUrlEncode( href,"group=" + group );
					final String src = XHTMLTag.addUrlEncode( href,"group=" + StringUtil.urlEncode( group ) ); // 5.10.15.2 (2019/09/20)
					rtn.append( makeTagMenuString( src,target,groupLabel,group,-1 ) );
					if( sideCount > 0 && sideTmpCnt % sideCount == 0 ) { rtn.append("</tr><tr>"); }
					sideTmpCnt++ ;
				}
			}

			if( ! imageOnly && ! buttonRequest ) { // 4.2.2.0 (2008/05/14) ﾎﾞﾀﾝﾒﾆｭｰ時はﾄｯﾌﾟﾒﾆｭｰ
				rtn.append( makeTagMenuString( JSP + "/index.jsp",target,"Top","GUI_TOP",-1 ) );
			}
			rtn.append( "</tr>" );

			// 画面のｸﾞﾙｰﾌﾟﾒﾆｭｰのｷｬｯｼｭをｾｯﾄします｡
			groupMenu = rtn.toString() ;
		}

		if( cache ) {
			setSessionAttribute( cacheKey,groupMenu );
		}

		return groupMenu;
	}

	/**
	 * ﾒﾆｭｰを表示する為のHTMLを作成します(折り返しJavaScript対応版)｡
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) GUIInfoの実ｱﾄﾞﾚｽのﾊﾟﾗﾒｰﾀを考慮する｡
	 * @og.rev 4.0.0.0 (2007/10/31) 一旦廃止
	 * @og.rev 5.5.2.3 (2012/05/15) useDivOneLevel 対応
	 * @og.rev 5.5.2.5 (2012/05/21) ｲﾒｰｼﾞｱｲｺﾝ対応(makeTagMenuのﾊﾟﾗﾒｰﾀ変更)
	 * @og.rev 5.5.6.0 (2013/01/07) useDivOneLevelの折り返し方法変更
	 * @og.rev 6.2.6.0 (2015/06/19) QUERYに表示される１行ﾒﾆｭは､説明なしの画面名称のみ表示する｡
	 * @og.rev 5.9.6.1 (2016/03/04) ﾌﾚｰﾑ対応
	 * @og.rev 6.9.4.1 (2018/04/09) HYBS_BR の廃止 , gamenId の変数設定
	 * @og.rev 8.4.2.1 (2023/03/10) 画面名称は Long → short をｾｯﾄ
	 *
	 * @return  ﾒﾆｭｰ
	 * @og.rtnNotNull
	 */
	private String makeOneLevelMenu() {
		final StringBuilder rtn = new StringBuilder( BUFFER_LARGE );

		// 5.5.2.3 (2012/05/15) useDivOneLevel 対応
		if( useDivOneLevel ) { rtn.append("<div id=\"design-onelevel\">"); }

		final UserInfo userInfo = getUser();

		// 4.0.0 (2005/01/31)
		final GUIInfo[] guiInfos = userInfo.getGUIInfos();
//		String gamenId  ;			// 画面ID
		int sideTmpCnt = 1;

		boolean isInClassify = false;

		// 7.2.9.4 (2020/11/20) PMD:This for loop can be replaced by a foreach loop
		for( final GUIInfo guiInfo : guiInfos ) {
			final String gamenId = guiInfo.getKey() ;				// 6.9.4.1 (2018/04/09)
			final int    level   = guiInfo.getLevel();
//		for( int i=0; i<guiInfos.length; i++ ) {
//			final String gamenId = guiInfos[i].getKey() ;			// 6.9.4.1 (2018/04/09)
//			final int level = guiInfos[i].getLevel();

			if( level == 2 ) {
//				isInClassify = selClassify.equals( guiInfos[i].getKey() );
				// 6.9.8.0 (2018/05/28) FindBugs:ｺﾝｽﾄﾗｸﾀで初期化されていないﾌｨｰﾙﾄﾞを null ﾁｪｯｸなしで null 値を利用している
				// makeOneLevelMenu() ﾒｿｯﾄﾞが呼ばれる条件に､selClassify != null があるのだが｡
//				isInClassify = selClassify.equals( gamenId );
				isInClassify = gamenId.equals( selClassify );			// 6.9.8.0 (2018/05/28) とりあえず比較方法を入れ替えます｡
			}

			if( guiInfo.isRead() && level == 3 && isInClassify ) {		// 4.0.0 (2005/01/31)
//			if( guiInfos[i].isRead() && level == 3 && isInClassify ) {		// 4.0.0 (2005/01/31)
//				gamenId = guiInfos[i].getKey() ;
				if( match   != null && !gamenId.matches( match  ) ) { continue; }
				if( unmatch != null && gamenId.matches( unmatch ) ) { continue; }

//				final String guiLabel = guiInfo.getLabel();			// 6.2.6.0 (2015/06/19) 画面名称(title無)
				final String guiLabel = guiInfo.getName();			// 8.4.2.1 (2023/03/10) Modify
//				final String guiLabel = guiInfos[i].getLabel();		// 6.2.6.0 (2015/06/19) 画面名称(title無)

				final String thisGamenId = getGUIInfoAttri( "KEY" );
				if( gamenId.equals( thisGamenId ) ) {
					// 5.5.2.3 (2012/05/15) useDivOneLevel 対応
					if( useDivOneLevel ) {
						rtn.append("<span class=\"design-onelevel\">").append( guiLabel ).append( "</span>" );
					}
					else {
						rtn.append( '[' ).append( guiLabel ).append( "] " );		// 6.0.2.5 (2014/10/31) char を append する｡
					}
				}
				else {
					// 5.5.2.3 (2012/05/15) useDivOneLevel 対応
					if( useDivOneLevel ) {
						// 5.5.2.5 (2012/05/21) ｲﾒｰｼﾞｱｲｺﾝ対応
						rtn.append("<span class=\"design-onelevel\">").append( makeTagMenu( guiInfo,guiLabel,-5 ) ).append( "</span>" ); // 5.9.6.1
//						rtn.append("<span class=\"design-onelevel\">").append( makeTagMenu( guiInfos[i],guiLabel,-5 ) ).append( "</span>" ); // 5.9.6.1
					}
					else {
						rtn.append( makeTagMenu( guiInfo,guiLabel,-2 ) );		// 5.5.2.5 (2012/05/21) ｲﾒｰｼﾞｱｲｺﾝ対応
//						rtn.append( makeTagMenu( guiInfos[i],guiLabel,-2 ) );		// 5.5.2.5 (2012/05/21) ｲﾒｰｼﾞｱｲｺﾝ対応
					}
				}

				if( sideCount > 0 ) {
					// 全件数をｻｲﾄﾞｶｳﾝﾄ数で割った値が最大行数の場合は､終了する｡
					if( sideTmpCnt / sideCount >= maxCellCount ) {
						//						rtn.append("･･･");
						break;
					}
					// 全件数をｻｲﾄﾞｶｳﾝﾄ数で割った余りがｾﾞﾛの場合は､行ﾌﾞﾚｲｸする｡
					// 5.5.2.3 (2012/05/15) useDivOneLevel 対応
					if( sideTmpCnt % sideCount == 0 ) {
						// 5.5.2.3 (2012/05/15) useDivOneLevel 対応
						// 6.0.2.5 (2014/10/31) refactoring: findBugs対応｡分岐もｺﾒﾝﾄｱｳﾄする｡
				//		if( useDivOneLevel ) {
				//			rtn.append( BR ); // 5.6.0.0 (2013/01/07) BRにする(条件分岐は残しておく)
				//			// rtn.append("<span style=\"clear: both;\" />");
				//		}
				//		else {
							rtn.append( BR );
				//		}
					}
					sideTmpCnt++ ;
				}
			}
		}

		// 5.5.2.3 (2012/05/15) useDivOneLevel 対応
		if( useDivOneLevel ) {
			rtn.append("<span style=\"clear: both;\" ><!-- --></span></div>");	// 5.9.1.2 (2015/10/23)
		}

		return rtn.toString();
	}

	/**
	 * 既存のﾍﾟｰｼﾞの次にｱｸｾｽされる画面郡のﾘﾝｸを作成します｡
	 *
	 * これは､現時点の画面に対して､次にｱｸｾｽされる画面の候補を
	 * ﾋﾟｯｸｱｯﾌﾟしておく機能です｡
	 * 実際には､過去にｱｸｾｽされた結果より取得しています｡
	 * これは､ONELEVEL と置き換えることになる機能です｡
	 *
	 * @og.rev 5.2.3.0 (2010/12/01) NEXTGUI 追加
	 * @og.rev 5.5.2.3 (2012/05/15) useDivOneLevel 対応
	 * @og.rev 5.5.2.5 (2012/05/21) ｲﾒｰｼﾞｱｲｺﾝ対応(makeTagMenuのﾊﾟﾗﾒｰﾀ変更)
	 * @og.rev 6.2.6.0 (2015/06/19) QUERYに表示される１行ﾒﾆｭは､説明なしの画面名称のみ表示する｡
	 * @og.rev 6.3.1.0 (2015/06/28) useDivOneLevel がうまく動かなかったので､修正｡
	 * @og.rev 5.9.6.1 (2016/03/04) ﾌﾚｰﾑ対応
	 * @og.rev 7.0.1.0 (2018/10/15) XHTML → HTML5 対応(空要素の､"／＞" 止めを､"＞" に変更します)｡
	 *
	 * @return  ﾒﾆｭｰ
	 * @og.rtnNotNull
	 */
	private String makeNextguiMenu() {
		final StringBuilder rtn = new StringBuilder( BUFFER_LARGE );

		// 今､ｱｸｾｽしている画面
		final GUIInfo thisGamen = (GUIInfo)getSessionAttribute( HybsSystem.GUIINFO_KEY );
		final String[] nextGuis = thisGamen.getNextGuiArray();

		final UserInfo userInfo = getUser();
		int sideTmpCnt = 1;

		// 5.5.2.3 (2012/05/15) useDivOneLevel 対応
		if( useDivOneLevel ) { rtn.append("<div id=\"design-onelevel\">"); }

		GUIInfo guiInfo = null;
		// 7.2.9.4 (2020/11/20) PMD:This for loop can be replaced by a foreach loop
		for( final String nxtGrp : nextGuis ) {
			guiInfo = userInfo.getGUIInfo( nxtGrp );
//		for( int i=0; i<nextGuis.length; i++ ) {
//			guiInfo = userInfo.getGUIInfo( nextGuis[i] );
			if( guiInfo == null ) { continue; }		// 存在しない､またはｱｸｾｽ拒否の場合は､無視する｡

			if( guiInfo.isRead() ) {
				final String guiLabel = guiInfo.getLabel();		// 6.2.6.0 (2015/06/19) 画面名称(title無)

				// 5.5.2.3 (2012/05/15) useDivOneLevel 対応
				if( useDivOneLevel ) {
					// 5.5.2.5 (2012/05/21) ｲﾒｰｼﾞｱｲｺﾝ対応
					rtn.append("<span class=\"design-onelevel\">").append( makeTagMenu( guiInfo,guiLabel,-5 ) ).append( "</span>" );
				}
				else {
					rtn.append( makeTagMenu( guiInfo,guiLabel,-2 ) );		// 5.5.2.5 (2012/05/21) ｲﾒｰｼﾞｱｲｺﾝ対応
				}

				if( sideCount > 0 ) {
					// 全件数をｻｲﾄﾞｶｳﾝﾄ数で割った値が最大行数の場合は､終了する｡
					if( sideTmpCnt / sideCount >= maxCellCount ) {
						break;
					}
					// 全件数をｻｲﾄﾞｶｳﾝﾄ数で割った余りがｾﾞﾛの場合は､行ﾌﾞﾚｲｸする｡
					// 5.5.2.3 (2012/05/15) useDivOneLevel 対応
					if( sideTmpCnt % sideCount == 0 ) {
						// 6.3.1.0 (2015/06/28) useDivOneLevel がうまく動かなかったので､修正｡
			//			if( useDivOneLevel ) {
			//				rtn.append("<span style=\"clear: both;\" />");
			//			}
			//			else {
							rtn.append( BR );
			//			}
					}
					sideTmpCnt++ ;
				}
			}
		}

		// 5.5.2.3 (2012/05/15) useDivOneLevel 対応
		// 6.3.1.0 (2015/06/28) useDivOneLevel がうまく動かなかったので､修正｡
		if( useDivOneLevel ) {
//			rtn.append("<span style=\"clear: both;\" /></div>");
			rtn.append("<span style=\"clear: both;\" ></span></div>");			// 7.0.1.0 (2018/10/15)
		}

		return rtn.toString();
	}

	/**
	 * ﾒﾆｭｰを表示する為のHTMLを作成します(ﾏﾄﾘｸｽﾒﾆｭｰ)｡
	 *
	 * 分類まとめ､ｸﾗｽ色指定､最小行数設定､最大行数設定の機能があります｡
	 * 《分類まとめ》 最大行数設定(maxCellCount)に達したｾﾙは､一つ右に新たにｾﾙを作成
	 * します｡このとき､ﾍｯﾀﾞｰの CLASSIFY を同じにして､colspan でまとめます｡
	 * 《ｸﾗｽ色指定》ﾍｯﾀﾞｰ毎に 順次 CLR0,CLR1,････ というｸﾗｽ属性を付与します｡
	 * ﾍｯﾀﾞｰには､MENU_H も出力されています｡CLR0 は､大分類ごとに加算されていきますので､
	 * 繰り返して同じ色を指定する場合は､CSSﾌｧｲﾙでまとめて指定してください｡
	 * 《最小行数設定》minCellCount 属性を指定することで､１ｾﾙに含まれる最小行数を指定できます｡
	 * これは､ｾﾙに入れる &lt;br /&gt; の個数を指定することと同じです｡
	 * 《最大行数設定》maxCellCount 属性を指定することで､１ｾﾙに含まれる最大行数を指定できます｡
	 * 分類まとめでも説明しましたように､最大値をｵｰﾊﾞｰすると次のｾﾙから書き始めます｡
	 *
	 * @og.rev 4.0.0.0 (2005/11/30) 新規追加
	 * @og.rev 4.0.0.0 (2007/10/05) 分類のｺｰﾄﾞﾘｿｰｽが存在しない場合
	 * @og.rev 5.2.3.0 (2010/12/01) sideCount対応
	 * @og.rev 5.5.2.5 (2012/05/21) ｲﾒｰｼﾞｱｲｺﾝ対応(makeTagMenuのﾊﾟﾗﾒｰﾀ変更)
	 * @og.rev 5.5.5.3 (2012/08/17) th,tdにﾍｯﾀﾞの画面IDをｸﾗｽとして出力
	 * @og.rev 6.8.2.3 (2017/11/10) minCellCountの数が､一つ足りない事の修正
	 * @og.rev 6.9.4.1 (2018/04/09) HYBS_BR の廃止 , gamenId の変数設定
	 * @og.rev 8.2.1.0 (2022/07/15) useLongName 属性追加
	 *
	 * @return  ﾏﾄﾘｸｽﾒﾆｭｰ
	 * @og.rtnNotNull
	 */
	private String makeMatrixMenu() {
		final StringBuilder rtn = new StringBuilder( BUFFER_LARGE );

		final UserInfo userInfo = getUser();

		// 4.0.0 (2005/01/31)
		final GUIInfo[] guiInfos = userInfo.getGUIInfos();
//		String gamenId		;			// 画面ID
		String bkClassifyKey	= null;		// 分類ｺｰﾄﾞ(旧)
		String bkClassifyName	= null;		// 分類名称(旧)
		int	   cellColorCnt = 0;		// MENU_H に 追加する CLR ｸﾗｽ属性の連番

		String headGuikey = "";				//  5.5.5.3 (2012/08/17) ﾍｯﾀﾞのgamenId

		final StringBuilder rtnH = new StringBuilder( BUFFER_LARGE );
		final StringBuilder rtnB = new StringBuilder( BUFFER_LARGE );

		int level ;
		int preLevel = 0;

		int lineTmpCnt = 0;				// ｾﾙ中の行ｶｳﾝﾄ
		int cellTmpCnt = 0;				// 1つの分類中のｾﾙｶｳﾝﾄ
		int cellTmpAllCnt = 0;			// 5.2.3.0 (2010/12/01) 該当行のｾﾙｶｳﾝﾄ

		boolean isInClassify  = false;	// 分類の中か？
		boolean isChangeLevel = false;	// 直ﾘﾝｸ用(無理やり通常画面の階層として扱うので)
		boolean isDummyMENU   = false;	// 5.2.3.0 (2010/12/01) sideCountﾌﾞﾚｰｸ時の大分類領域の出力可否		7.2.9.4 (2020/11/20) isDummyMENU_S → isDummyMENU

		for( int i=0; i<guiInfos.length; i++ ) {
			if( guiInfos[i].isRead() ) {		// 4.0.0 (2005/01/31)
				final String gamenId = guiInfos[i].getKey() ;
				if( match   != null && !gamenId.matches( match  ) ) { continue; }
				if( unmatch != null && gamenId.matches( unmatch ) ) { continue; }

				// 処理すべき画面かのﾁｪｯｸ
				final int guiFlg = guiCheck( guiInfos, i );
				if( guiFlg == 0 ) { continue; }

				level = guiInfos[i].getLevel();		// 4.0.0 (2005/01/31)

				// 大分類が来るまでは処理しない
				if( level > 1 && rtnH == null ) { continue; }

				// 直ﾘﾝｸの場合､無理やり通常画面に変換
				if( level == 2 && guiInfos[i].getAddress() != null && guiInfos[i].getAddress().length() != 0 ) {
					level = 3;
					if( !isChangeLevel ) {
						isChangeLevel = true;
						isInClassify = false;
					}
				}
				else {
					isChangeLevel = false;
				}

				// 分類のﾌﾞﾚｲｸ処理
				if( preLevel >= 3 && level < 3 || !isInClassify ) {								// 6.9.7.0 (2018/05/14) PMD Useless parentheses.
					if(  lineTmpCnt != 0 ) {
						for( int j=lineTmpCnt; j<minCellCount; j++ ) { rtnB.append( BR ); }		// 6.8.2.3 (2017/11/10)
						rtnB.append( "</td>" ).append( CR );
					}

					if( bkClassifyKey != null ) {
						rtnH.append( "<th colspan=\"" ).append( cellTmpCnt ).append( "\" class=\"MENU_H CLR" )
							.append( cellColorCnt ).append( ' ' ).append( headGuikey ).append( "\">" ); // 5.5.5.3 (2012/08/17)
						if( "_SPACE".equals( bkClassifyKey ) ) {
							rtnH.append( '　' );		// 6.0.2.5 (2014/10/31) char を append する｡
						}
						else {
							rtnH.append( makeTagMenuString( null,null,bkClassifyName,bkClassifyKey,-3 ) );
						}
						rtnH.append( "</th>" );

						// 5.2.3.0 (2010/12/01) sideCount によるｾﾙの改行
						cellTmpAllCnt += cellTmpCnt;
						if( sideCount > 0 && cellTmpAllCnt >= sideCount ) {
							rtn.append( rtnH ).append( "</tr><tr>" ).append( rtnB ).append( "</tr>" );

							rtnH.setLength(0);		// 6.1.0.0 (2014/12/26) refactoring
							rtnB.setLength(0);		// 6.1.0.0 (2014/12/26) refactoring
							cellTmpAllCnt = 0;
							isDummyMENU   = true;	// 出力予約
						}
					}

					bkClassifyKey = null;
					isInClassify = false;
					lineTmpCnt = 0;
					cellTmpCnt = 0;
				}

				// 大分類(ﾌｨｰﾙﾄﾞﾒﾆｭｰ)
				if( level == 1 ) {
					headGuikey = gamenId; // 5.5.5.3 (2012/08/17)
					if( preLevel > 0 ) {
						cellColorCnt++ ;
						rtn.append( rtnH ).append( "</tr><tr>" ).append( rtnB ).append( "</tr>" );
					}

					rtn.append( "<tr><td rowspan=\"2\" class=\"MENU_S "+ headGuikey +"\">" )	// 5.5.5.3 (2012/08/17)
//						.append( makeTagMenuString( null,null,guiInfos[i].getName(),gamenId,-3 ) )
						.append( makeTagMenuString( null,null,guiInfos[i].getName(useLongName),gamenId,-3 ) )	// 8.2.1.0 (2022/07/15) true:名前(長)/false:名前(短)
						.append( "</td>" ).append( CR );

					rtnH.setLength(0);		// 6.1.0.0 (2014/12/26) refactoring
					rtnB.setLength(0);		// 6.1.0.0 (2014/12/26) refactoring
					cellTmpAllCnt = 0;		// 5.2.3.0 (2010/12/01)
					isDummyMENU   = false;	// 5.2.3.0 (2010/12/01) 出力予約解除
				}
				// 分類(直ﾘﾝｸの場合は､level=3で処理)
				else if( level == 2 ) {
					isInClassify = true;
//					bkClassifyKey = guiInfos[i].getKey();
					bkClassifyKey = gamenId;						// 6.9.4.1 (2018/04/09)
//					bkClassifyName = guiInfos[i].getName();
					bkClassifyName = guiInfos[i].getName(useLongName);	// 8.2.1.0 (2022/07/15) true:名前(長)/false:名前(短)
				}
				// 通常ﾒﾆｭｰ・隠しﾒﾆｭｰ
				else if( level >= 3 ) {

					// 元のMENU_Sに戻り､rowspan を書き換える･･･のが邪魔くさいので､td のみ出力しておく｡
					// 5.2.3.0 (2010/12/01) sideCount によるｾﾙの改行
					if( isDummyMENU ) {
						rtn.append( "<tr><td rowspan=\"2\" class=\"MENU_S " )
							.append( headGuikey ).append( "\"></td>" )	// 5.5.5.3 (2012/08/17)
							.append( CR );
						isDummyMENU = false;		// 出力予約解除
					}

					// 画面ID="HYBS_BR"の場合は､ｾﾙを変える｡
//					if( "HYBS_BR".equals( guiInfos[i].getKey() ) ) {
					if( "HYBS_BR".equals( guiInfos[i].getAddress() ) ) {					// 6.9.4.1 (2018/04/09) HYBS_BR は､ｱﾄﾞﾚｽに設定
						if( lineTmpCnt != 0 ) {
							for( int j=lineTmpCnt; j<minCellCount; j++ ) { rtnB.append( BR ); }		// 6.8.2.3 (2017/11/10)
							lineTmpCnt = 0;
						}
					}
					// 通常画面
					else {
						if( lineTmpCnt == 0 ) {
							rtnB.append( "<td class=\"MENU_B " + headGuikey + "\">" ); // 5.5.5.3 (2012/08/17)
							cellTmpCnt++;
						}

//						rtnB.append( makeTagMenu( guiInfos[i],guiInfos[i].getName(),-3 ) );		// 5.5.2.5 (2012/05/21) ｲﾒｰｼﾞｱｲｺﾝ対応
						rtnB.append( makeTagMenu( guiInfos[i],guiInfos[i].getName(useLongName),-3 ) );	// 8.2.1.0 (2022/07/15) true:名前(長)/false:名前(短)
						lineTmpCnt++;

						if( lineTmpCnt >= maxCellCount ) {
							rtnB.append( "</td>" );
							lineTmpCnt = 0;
						}
						else {
							rtnB.append( BR );
						}

						// 分類の中に入っていない通常画面
						if( !isInClassify ) {
							bkClassifyKey = "_SPACE";
							isInClassify = true;
						}
					}
				}

				preLevel = level;
			}
		}

		// 終了処理
		for( int j=lineTmpCnt; j<minCellCount; j++ ) { rtnB.append( BR ); }		// 6.8.2.3 (2017/11/10)
		rtnB.append( "</td>" ).append( CR );

		if( bkClassifyKey != null ) {
			rtnH.append( "<th colspan=\"" ).append( cellTmpCnt )
				.append( "\" class=\"MENU_H CLR" ) .append(  cellColorCnt )
				.append( ' ' ).append(  headGuikey ).append( "\">" )			// 5.5.5.3 (2012/08/17)
				.append( makeTagMenuString( null,null,bkClassifyName,bkClassifyKey,-3 ) )
				.append( "</th>" );
		}

		rtn.append( rtnH ).append( "</tr><tr>" ).append( rtnB ).append( "</tr>" );

		return rtn.toString();
	}

	/**
	 * ﾒﾆｭｰを表示する為のHTMLを作成します(ﾏﾄﾘｸｽﾒﾆｭｰ2)｡
	 *
	 * ﾏﾄﾘｸｽﾒﾆｭｰをﾍﾞｰｽとした特別ﾊﾞｰｼﾞｮﾝです｡
	 * 通常のﾏﾄﾘｸｽﾒﾆｭｰからの変更点は以下の通りです｡
	 * ①大分類が表示されない
	 *   ⇒ 変わりに､1行に表示されているｾﾙ数がsideCountで指定した数を超えた場合に
	 *      自動的に改行されます｡
	 * ②画面ﾘﾝｸのﾀｰｹﾞｯﾄ
	 *   ⇒ 自分自身のﾌﾚｰﾑに対してﾘﾝｸ先を表示します｡
	 *      ﾘﾝｸ先は､通常ﾒﾆｭｰ構成ですが左ﾒﾆｭｰには該当する小分類の画面しか表示されません｡
	 * ③小分類でのﾘﾝｸ
	 *   ⇒ 小分類をｸﾘｯｸした際に､通常のﾒﾆｭｰ構成画面にﾘﾝｸします｡
	 *      但し､②と同様に左ﾒﾆｭｰには該当する小分類の画面しか表示されません｡
	 *
	 * @og.rev 4.2.1.0 (2008/04/01) 新規追加
	 * @og.rev 4.2.1.1 (2008/05/02) ｶﾃｺﾞﾘｰﾘﾝｸで一番上の画面のﾓｰﾄﾞが-wとなっている場合に､
	 *                               その画面が立ち上がってしまうﾊﾞｸﾞを修正
	 * @og.rev 4.2.2.0 (2008/05/14) buttonRequestの付加をmakeTagMenuString()に変更
	 * @og.rev 5.5.2.5 (2012/05/21) ｲﾒｰｼﾞｱｲｺﾝ対応(makeTagMenuのﾊﾟﾗﾒｰﾀ変更)
	 * @og.rev 5.5.5.3 (2012/08/17) ﾍｯﾀﾞ部のgamenIdをth,tdのｸﾗｽに追加
	 * @og.rev 6.8.2.3 (2017/11/10) minCellCountの数が､一つ足りない事の修正
	 * @og.rev 6.9.4.1 (2018/04/09) HYBS_BR の廃止 , gamenId の変数設定
	 * @og.rev 7.4.2.2 (2021/05/28) ﾏﾄﾘｸｽﾒﾆｭｰ(MATRIX)で､ｸﾞﾙｰﾌﾟのﾎﾞﾀﾝﾘﾝｸを止めます｡
	 * @og.rev 8.2.1.0 (2022/07/15) useLongName 属性追加
	 *
	 * @return  ﾏﾄﾘｸｽﾒﾆｭｰ
	 * @og.rtnNotNull
	 */
	private String makeMatrixMenu2() {
		final StringBuilder rtn = new StringBuilder( BUFFER_LARGE );

		final UserInfo userInfo = getUser();

		final GUIInfo[] guiInfos = userInfo.getGUIInfos();
//		String gamenId		;			// 画面ID
		String bkClassifyKey	= null;		// 分類ｺｰﾄﾞ(旧)
	//	String nextKey			= null;		// 分類ﾎﾞﾀﾝを押した時に最初の画面が開くようにする		8.0.0.0 (2021/07/31) Delete
		String bkClassifyName	= null;		// 分類名称(旧)
		final int	   cellColorCnt = 0;		// MENU_H に 追加する CLR ｸﾗｽ属性の連番

		final StringBuilder rtnH = new StringBuilder( BUFFER_LARGE );		// 分類部分の出力用
		final StringBuilder rtnB = new StringBuilder( BUFFER_LARGE );		// 実画面のﾘﾝｸ部分の出力用

		int level ;
		int preLevel = 0;

		int lineTmpCnt = 0;				// ｾﾙ中の行ｶｳﾝﾄ
		int cellTmpCnt = 0;				// 1つの分類中のｾﾙｶｳﾝﾄ
		int cellTmpAllCnt = 0;			// 該当行のｾﾙｶｳﾝﾄ

		boolean isInClassify = false;	// 分類の中か？
		boolean isChangeLevel = false;	// 直ﾘﾝｸ用(無理やり通常画面の階層として扱うので)

		String headGuikey = "" ;		// 5.5.5.3 (2012/08/17)

		rtn.append( "<tr>" ).append( CR );

		for( int i=0; i<guiInfos.length; i++ ) {
			if( guiInfos[i].isRead() ) {		// 4.0.0 (2005/01/31)
				final String gamenId = guiInfos[i].getKey() ;
				if( match   != null && !gamenId.matches( match  ) ) { continue; }
				if( unmatch != null && gamenId.matches( unmatch ) ) { continue; }

				// 処理すべき画面かのﾁｪｯｸ
				final int guiFlg = guiCheck( guiInfos, i );
				if( guiFlg == 0 ) { continue; }

				level = guiInfos[i].getLevel();		// 4.0.0 (2005/01/31)

				// 直ﾘﾝｸの場合､無理やり通常画面に変換
				if( level == 2 && guiInfos[i].getAddress() != null && guiInfos[i].getAddress().length() != 0 ) {
					level = 3;
					if( !isChangeLevel ) {
						isChangeLevel = true;
						isInClassify = false;
					}
				}
				else {
					isChangeLevel = false;
				}

				// 分類のﾌﾞﾚｲｸ処理
				if( preLevel >= 3 && level < 3 || !isInClassify ) {								// 6.9.7.0 (2018/05/14) PMD Useless parentheses.
					if(  lineTmpCnt != 0 ) {
						for( int j=lineTmpCnt; j<minCellCount; j++ ) { rtnB.append( BR ); }		// 6.8.2.3 (2017/11/10)
						//rtnB.append( "</td>" ).append( CR ); // 4.2.1.0 (2008/04/03) ﾘｽﾄ
						rtnB.append( "</ul></div></td>" ).append( CR );
					}

					if( bkClassifyKey != null ) {
						// 6.0.2.5 (2014/10/31) char を append する｡
						rtnH.append( "<th colspan=\"" ).append( cellTmpCnt ).append( "\" class=\"MENU_H CLR" )
							.append( cellColorCnt ).append( ' ' ).append( headGuikey ).append( "\">" ).append( "<div>" );		// 5.5.5.3 (2012/08/17)
						if( "_SPACE".equals( bkClassifyKey ) ) {
							rtnH.append( '　' );					// 6.0.2.5 (2014/10/31) refactoring
						}
						else {
							// 6.4.2.1 (2016/02/05) PMD refactoring.
				//			final String classifyHref = JSP + "/index.jsp?classify=" + bkClassifyKey + "&amp;GAMENID="+nextKey;		// ﾎﾞﾀﾝを押した場合に最初の画面が現れる
				//			rtnH.append( makeTagMenuString( classifyHref,"_self",bkClassifyName,bkClassifyKey,-3 ) );
							rtnH.append( makeTagMenuString( null,null,bkClassifyName,bkClassifyKey,-3 ) );					// 7.4.2.2 (2021/05/28)
						}
						rtnH.append( "</div>" ).append( "</th>" );

						cellTmpAllCnt += cellTmpCnt;

						if( sideCount > 0 && cellTmpAllCnt >= sideCount ) {
							rtn.append( rtnH ).append( "</tr><tr>" ).append( rtnB ).append( "</tr>" );

							rtnH.setLength(0);				// 6.1.0.0 (2014/12/26) refactoring
							rtnB.setLength(0);				// 6.1.0.0 (2014/12/26) refactoring
							cellTmpAllCnt = 0;
						}
					}

					bkClassifyKey = null;
		//			nextKey		  = null;
					isInClassify = false;
					lineTmpCnt = 0;
					cellTmpCnt = 0;
				}

				// 5.5.5.3 (2012/08/17) 大分類(ﾌｨｰﾙﾄﾞﾒﾆｭｰ)
				if( level == 1 ) {
					headGuikey = gamenId;
				}
				// 分類(直ﾘﾝｸの場合は､level=3で処理)
				else if( level == 2 ) {
					isInClassify = true;
					if( guiInfos[i].isPulldown() ){ // 4.3.3.0 (2008/10/01) ﾌﾟﾙﾀﾞｳﾝ化ﾁｪｯｸ
						excludeButton = true;
					}
					else{
						excludeButton = false;
					}
//					bkClassifyKey = guiInfos[i].getKey();
					bkClassifyKey = gamenId;					// 6.9.4.1 (2018/04/09)
//					bkClassifyName = guiInfos[i].getName();
					bkClassifyName = guiInfos[i].getName(useLongName);	// 8.2.1.0 (2022/07/15) true:名前(長)/false:名前(短)
			//		for( int j=i+1; j<guiInfos.length; j++ ) {
			//			if( guiInfos[j] != null && guiInfos[j].isRead() ) {
			//				nextKey = guiInfos[j].getKey();
			//				break;
			//			}
			//		}
				}
				// 通常ﾒﾆｭｰ・隠しﾒﾆｭｰ
				else if( level >= 3 ) {
					// 画面ID="HYBS_BR"の場合は､ｾﾙを変える｡
//					if( "HYBS_BR".equals( guiInfos[i].getKey() ) ) {
					if( "HYBS_BR".equals( guiInfos[i].getAddress() ) ) {					// 6.9.4.1 (2018/04/09) HYBS_BR は､ｱﾄﾞﾚｽに設定
						if( lineTmpCnt != 0 ) {
							for( int j=lineTmpCnt; j<minCellCount; j++ ) { rtnB.append( BR ); }		// 6.8.2.3 (2017/11/10)
							lineTmpCnt = 0;
						}
					}
					// 通常画面
					else {
						if( lineTmpCnt == 0 ) {
							// rtnB.append( "<td class=\"MENU_B\">" );
							// 4.2.1.0 (2008/04/03) ﾘﾝｸをﾘｽﾄ形式で出す案
							rtnB.append( "<td class=\"MENU_B "+ headGuikey +"\"> <div class=\"gamen-menu-wrap\"><ul class=\"gamen-menu\">" ); // 5.5.5.3 (2012/08/17)
							cellTmpCnt++;
						}

						// 5.5.2.5 (2012/05/21) ﾘｽﾄは､ｲﾒｰｼﾞを設定するので､ﾒｿｯﾄﾞ側で付与します｡
//						rtnB.append( makeTagMenu( guiInfos[i],guiInfos[i].getName(),-4 ) );		// 5.5.2.5 (2012/05/21) ｲﾒｰｼﾞｱｲｺﾝ対応
						rtnB.append( makeTagMenu( guiInfos[i],guiInfos[i].getName(useLongName),-4 ) );	// 8.2.1.0 (2022/07/15) true:名前(長)/false:名前(短)
						lineTmpCnt++;

						if( lineTmpCnt >= maxCellCount ) {
							// rtnB.append( "</td>" );
							rtnB.append( " </ul> </div></td>" ); // 4.2.1.0 (008/04/03) ﾘｽﾄ
							lineTmpCnt = 0;
						}
				//		else {
				//			rtnB.append( BR ); // 4.2.1.0 (2008/04/03) ﾘｽﾄ｡改行不要｡
				//		}

						// 分類の中に入っていない通常画面
						if( !isInClassify ) {
							bkClassifyKey = "_SPACE";
							isInClassify = true;
						}
					}
				}
				preLevel = level;
			}
		}

		// 終了処理
		// 4.3.3.0 (2008/10/01) 終了処理は0の時行はない
		if( lineTmpCnt != 0){
			for( int j=lineTmpCnt; j<minCellCount; j++ ) { rtnB.append( BR ); }			// 6.8.2.3 (2017/11/10)
			rtnB.append( "</td>" ).append( CR );
		}

		if( bkClassifyKey != null ) {
			rtnH.append( "<th colspan=\"" + cellTmpCnt + "\" class=\"MENU_H CLR" + cellColorCnt + " " + headGuikey + "\">" ).append( "<div>" ); // 5.5.5.3 (2012/08/17)

			// 6.4.2.1 (2016/02/05) PMD refactoring.
	//		final String classifyHref = JSP + "/index.jsp?classify=" + bkClassifyKey + "&amp;GAMENID="+nextKey;
	//		rtnH.append( makeTagMenuString( classifyHref,"_self",bkClassifyName,bkClassifyKey,-3 ) );
			rtnH.append( makeTagMenuString( null,null,bkClassifyName,bkClassifyKey,-3 ) );							// 7.4.2.2 (2021/05/28)
			rtnH.append( "</div>" ).append( "</th>" );
		}

		rtn.append( rtnH ).append( "</tr><tr>" ).append( rtnB ).append( "</tr>" );

		return rtn.toString();
	}

	/**
	 * ﾀｲﾌﾟに応じたﾒﾆｭｰを表示する為の文字列を作成します｡
	 * 引数の GUIInfo より､ｱｸｾｽすべきﾒﾆｭｰのhrefと､targetを求めます｡
	 * type = -3 (ﾏﾄﾘｸｽﾒﾆｭｰ) の場合だけ､ﾀｸﾞ属性の target を使用します｡
	 *
	 * 　0:ﾌｨｰﾙﾄﾞｾｯﾄﾒﾆｭｰ
	 * 　1:ﾄｯﾌﾟ階層(【分類名称】)
	 * 　2:選択階層(通常の折りたたみﾒﾆｭｰ)
	 * 　3:選択非表示(通常は､隠してあります)
	 * 　-1:ｸﾞﾙｰﾌﾟ分類ﾒﾆｭｰ(class="GUI_GRP"のtdﾀｸﾞで囲われます)
	 * 　-2:ﾗｲﾝﾒﾆｭｰ([画面名称]　)
	 * 　-3:ﾏﾄﾘｸｽﾒﾆｭｰ(一覧)
	 * 　-4:ﾏﾄﾘｸｽﾒﾆｭｰ２(一覧)
	 * 　-5:1ﾚﾍﾞﾙ表示(後続処理は-3と同じ)
	 * 　-6:TILE表示時の選択階層(通常の折りたたみﾒﾆｭｰ,2:と同じ)
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規登録(makeTopMenu､makeSecondMenuの置き換え)
	 * @og.rev 4.1.0.1 (2008/01/22) ｱﾄﾞﾚｽが[..]から始まる画面はﾏﾙﾁｾｯｼｮﾝﾁｪｯｸを行わない｡
	 * @og.rev 4.2.1.0 (2008/04/01) ﾏﾄﾘｸｽﾒﾆｭｰ2用のﾀｸﾞ作成処理追加
	 * @og.rev 4.2.1.0 (2008/04/17) ﾏﾄﾘｸｽ２からの遷移先でﾎﾞﾀﾝﾒﾆｭｰにする処理
	 * @og.rev 4.2.2.0 (2008/05/14) buttonRequestの付加をmakeTagMenuString()に変更
	 * @og.rev 4.3.3.7 (2008/11/22) https対応
	 * @og.rev 5.5.2.5 (2012/05/21) ｲﾒｰｼﾞｱｲｺﾝ対応(画面ID単位ではなく､IMAGE_KEYﾊﾟﾗﾒｰﾀで指定可能にする｡)
	 * @og.rev 5.7.8.1 (2014/07/18) realAddress の "/" 対応
	 * @og.rev 6.3.8.4 (2015/10/09) KBLINK(ﾘﾝｸ区分)を画面のｲﾒｰｼﾞﾌｧｲﾙに割り当てます｡
	 * @og.rev 5.9.6.1 (2016/03/04) -5を追加｡-3との違いはresult.jspに飛ばない事
	 * @og.rev 6.9.5.0 (2018/04/23) multiSessionCheck 廃止(true固定)
	 *
	 * @param	guiInfo GUIInfoﾘﾝｸ
	 * @param	bodyIn	BODY部(表示)
	 * @param	type	ﾀｲﾌﾟ
	 *
	 * @return  階層別ﾒﾆｭｰ文字列
	 * @og.rtnNotNull
	 */
	private String makeTagMenu( final GUIInfo guiInfo, final String bodyIn, final int type ) {

		String href		 = null;
		String tmpTarget = null;
		final  String gamenId	= guiInfo.getKey();		// 5.5.2.5 (2012/05/21) ｲﾒｰｼﾞｱｲｺﾝ対応
		int	   		  tmpType	= type;					// 5.9.6.1

		if( guiInfo.getAddress() != null && guiInfo.getAddress().length() > 0 ) {
			final String readAdrs = guiInfo.getRealAddress( "index.jsp" );
			final String reqParam = getRequestParameter( guiInfo.getParam() );
			href = XHTMLTag.addUrlEncode( readAdrs,reqParam );
			tmpTarget = guiInfo.getTarget();

			// 4.3.3.7 (2008/11/22) https対応
			// 5.7.8.1 (2014/07/18) realAddress の "/" 対応
			final String kblink = guiInfo.getKblink();
			if( ! readAdrs.startsWith( "http://" ) && ! readAdrs.startsWith( "https://" ) && !"/".equals( kblink ) ) {
				if( type == -3 ) {
					tmpTarget = target;
					href = XHTMLTag.addUrlEncode( JSP + "/result.jsp",reqParam );	// 6.2.0.0 (2015/02/27)
				}
				else if( type == -4 ) {		// 4.2.1.0 (2008/04/01)
					tmpTarget = "_self";
					href = XHTMLTag.addUrlEncode( JSP + "/index.jsp",reqParam );	// 6.2.0.0 (2015/02/27)
					href = XHTMLTag.addUrlEncode( href,"classify=" + guiInfo.getClassify() );
				}
				else if( type == -5 ){		// 5.9.6.1 (2016/03/04)
					tmpTarget = target;
					tmpType = -3;
				}
				href = XHTMLTag.addUrlEncode( href,"GAMENID=" + gamenId );			// 5.5.2.5 (2012/05/21) せっかくなので､利用する｡

				// 3.8.0.0 (2005/06/07) 同一ｾｯｼｮﾝでのﾏﾙﾁ起動対策を行います｡
				// 4.1.0.1 (2008/01/22) ｱﾄﾞﾚｽが[..]から始まる画面はﾏﾙﾁｾｯｼｮﾝﾁｪｯｸを行いません｡
				// 4.1.0.1 (2008/04/01) ﾏﾄﾘｸｽﾒﾆｭｰ2はﾏﾙﾁｾｯｼｮﾝﾁｪｯｸを行いません｡
//				if( multiSessionCheck && !guiInfo.getAddress().startsWith( ".." ) && type != -4 ) {
				if( !guiInfo.getAddress().startsWith( ".." ) && type != -4 ) {		// 6.9.5.0 (2018/04/23) multiSessionCheck 廃止(true固定)
					href = XHTMLTag.addUrlEncode( href,mscKey );
				}
			}
		}
		// 5.5.2.5 (2012/05/21) ｲﾒｰｼﾞｱｲｺﾝ対応(画面ID単位ではなく､IMAGE_KEYﾊﾟﾗﾒｰﾀで指定可能にする｡)
		final String imgKey = guiInfo.getImageKey() ;
		return makeTagMenuString( href,tmpTarget,bodyIn,gamenId,imgKey,tmpType ) ; // 5.9.6.1
	}

	/**
	 * ﾀｲﾌﾟに応じたﾒﾆｭｰを表示する為の文字列を作成します｡
	 * 従来からのﾒｿｯﾄﾞの引数のままで､新しい gamenId は､imgKey をそのままｾｯﾄします｡
	 *
	 * 　0:ﾌｨｰﾙﾄﾞｾｯﾄﾒﾆｭｰ
	 * 　1:ﾄｯﾌﾟ階層(【分類名称】)
	 * 　2:選択階層(通常の折りたたみﾒﾆｭｰ)
	 * 　3:選択非表示(通常は､隠してあります)
	 * 　-1:ｸﾞﾙｰﾌﾟ分類ﾒﾆｭｰ(class="GUI_GRP"のtdﾀｸﾞで囲われます)
	 * 　-2:ﾗｲﾝﾒﾆｭｰ([画面名称]　)
	 * 　-3:ﾏﾄﾘｸｽﾒﾆｭｰ(一覧)
	 * 　-4:ﾏﾄﾘｸｽﾒﾆｭｰ2(一覧)
	 * 　-5:1ﾚﾍﾞﾙ表示(後続処理は-3と同じだが､絶対ﾊﾟｽの場合は､画面ﾘｿｰｽのtargetを使用する｡)
	 * 　-6:TILE表示時の選択階層(通常の折りたたみﾒﾆｭｰ,2:と同じ)
	 *
	 * @og.rev 5.5.2.5 (2012/05/21) ｲﾒｰｼﾞｱｲｺﾝ対応(画面ID単位ではなく､IMAGE_KEYﾊﾟﾗﾒｰﾀで指定可能にする｡)
	 *
	 * @param	href	ﾘﾝｸ
	 * @param	target	ﾀｰｹﾞｯﾄ
	 * @param	bodyIn	BODY部(表示)
	 * @param	imgKey	imageﾌｧｲﾙの検索ｷｰ
	 * @param	type	ﾀｲﾌﾟ
	 *
	 * @return  階層別ﾒﾆｭｰ文字列
	 * @og.rtnNotNull
	 */
	private String makeTagMenuString( final String href,final String target, final String bodyIn,
								final String imgKey,final int type ) {
		return makeTagMenuString( href,target,bodyIn,imgKey,imgKey,type );	// gamenId の代わりに､imgKey をｾｯﾄする｡
	}

	/**
	 * ﾀｲﾌﾟに応じたﾒﾆｭｰを表示する為の文字列を作成します｡
	 *
	 * ﾀｲﾌﾟは､階層別になっていますが､ﾏｲﾅｽは､内部で処理するための記号です｡
	 *
	 * 　0:ﾌｨｰﾙﾄﾞｾｯﾄﾒﾆｭｰ
	 * 　1:ﾄｯﾌﾟ階層(【分類名称】)
	 * 　2:選択階層(通常の折りたたみﾒﾆｭｰ)
	 * 　3:選択非表示(通常は､隠してあります)
	 * 　-1:ｸﾞﾙｰﾌﾟ分類ﾒﾆｭｰ(class="GUI_GRP"のtdﾀｸﾞで囲われます)
	 * 　-2:ﾗｲﾝﾒﾆｭｰ([画面名称]　)
	 * 　-3:ﾏﾄﾘｸｽﾒﾆｭｰ(一覧)
	 * 　-4:ﾏﾄﾘｸｽﾒﾆｭｰ2(一覧)
	 * 　-5:1ﾚﾍﾞﾙ表示(後続処理は-3と同じだが､絶対ﾊﾟｽの場合は､画面ﾘｿｰｽのtargetを使用する｡)
	 * 　-6:TILE表示時の選択階層(通常の折りたたみﾒﾆｭｰ,2:と同じ)
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規登録(makeTopMenu､makeSecondMenuの置き換え)
	 * @og.rev 4.0.0.0 (2007/11/28) switch文の中の二重ｺｰﾄﾞを統一します｡
	 * @og.rev 4.2.1.0 (2008/04/01) ﾎﾞﾀﾝ形式のﾘﾝｸ対応､ﾏﾄﾘｸｽﾒﾆｭｰ2対応
	 * @og.rev 4.2.2.0 (2008/05/14) buttonRequestの付加をここで行うようにする
	 * @og.rev 4.3.5.0 (2009/02/01) switch の case で重複項目をまとめます｡
	 * @og.rev 5.1.4.0 (2010/03/01) onClick,onMouseOver,onMouseOutの処理は､外部のJavaScriptﾌｧｲﾙで記述します｡
	 * @og.rev 5.1.8.0 (2010/07/01) ｺｰﾄﾞﾘｿｰｽのｷｰが存在しない場合にｴﾗｰとなるﾊﾞｸﾞを修正
	 * @og.rev 5.3.9.0 (2011/09/01) ﾒﾆｭｰでのﾍﾙﾌﾟｱｲｺﾝ対応
	 * @og.rev 5.4.4.4 (2012/02/15) ﾍﾙﾌﾟｱｲｺﾝはimgKeyがnullでない場合のみとする
	 * @og.rev 5.5.0.4 (2012/03/14) FAQ対応
	 * @og.rev 5.5.2.5 (2012/05/21) ｲﾒｰｼﾞｱｲｺﾝ対応(画面ID単位ではなく､IMAGE_KEYﾊﾟﾗﾒｰﾀで指定可能にする｡)
	 * @og.rev 5.5.4.2 (2012/07/13) JSP(/gf/jsp/) ではなく､CNTX(gf)＋MENU_IMG(/jsp/menuImage/) で処理するように変更
	 * @og.rev 6.2.2.0 (2015/03/27) BRと\nを相互に変換する処理を追加
	 * @og.rev 6.2.2.3 (2015/04/10) htmlﾌｨﾙﾀｰに､BR→改行処理機能を追加｡
	 * @og.rev 6.3.8.4 (2015/10/09) topMenu 内でのHelp機能を廃止します｡
	 * @og.rev 6.3.8.4 (2015/10/09) FileMap のｺﾝｽﾄﾗｸﾀ変更に伴う対応｡
	 * @og.rev 6.7.5.0 (2017/03/10) TILE表示追加
	 * @og.rev 6.8.0.1 (2017/06/30) -5:1ﾚﾍﾞﾙ表示追加(元は､相対ﾊﾟｽのみ対応していたが､絶対ﾊﾟｽにも対応)
	 * @og.rev 6.8.1.6 (2017/09/29) imageOnly="true" で､ｲﾒｰｼﾞにﾘﾝｸが付かないﾊﾞｸﾞ修正｡
	 * @og.rev 7.0.1.0 (2018/10/15) XHTML → HTML5 対応(空要素の､"／＞" 止めを､"＞" に変更します)｡
	 * @og.rev 7.2.9.4 (2020/11/20) spotbugs:switch 文の2つの case のために同じｺｰﾄﾞを使用しているﾒｿｯﾄﾞ
	 * @og.rev 7.3.2.0 (2021/03/19) FileMapは､static化して使いまわす｡
	 *
	 * @param	href	ﾘﾝｸ
	 * @param	target	ﾀｰｹﾞｯﾄ
	 * @param	bodyIn	BODY部(表示)
	 * @param	gamenId	画面ID/ｸﾞﾙｰﾌﾟIDなど
	 * @param	imgKey	imageﾌｧｲﾙの検索ｷｰ
	 * @param	type	ﾀｲﾌﾟ
	 *
	 * @return  階層別ﾒﾆｭｰ文字列
	 * @og.rtnNotNull
	 */
	private String makeTagMenuString( final String href,final String target, final String bodyIn,
								final String gamenId,final String imgKey,final int type ) {

		final String body ;
		final String img ;			// 6.7.5.0 (2017/03/10) ｲﾒｰｼﾞを､【】の外に出す｡
		String listStyle = "<li>";
		// 6.3.8.4 (2015/10/09) FileMap のｺﾝｽﾄﾗｸﾀ変更に伴う対応｡
		// 最初に見つけたｷｰが優先されるので､画面IDのﾌｧｲﾙを優先します｡
//		final String imgFile = imgFileMap.getFilename( gamenId,imgKey );	// 5.5.2.5 (2012/05/21) 属性名変更
		final String imgFile = imgFileMap == null ? null : imgFileMap.getFilename( gamenId,imgKey );	// 7.3.2.0 (2021/03/19) 一応､ﾁｪｯｸ
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
		if( imgFile == null ) {
			img = "";
			if( imageOnly && bodyIn != null && bodyIn.length() > 0 ) {
				body = "<span title=\"" + StringUtil.htmlFilter( bodyIn,true ) + "\" >" + bodyIn.charAt(0) + "</span>" ;
			}
			else {
				body = bodyIn ;
			}
		}
		else {
			if( imageOnly ) {
				// 6.8.1.6 (2017/09/29) imageOnly="true" で､ｲﾒｰｼﾞにﾘﾝｸが付かないﾊﾞｸﾞ修正｡ﾘﾝｸは､body に付く
				img = "";
				body = "<img src=\"" + imgFile + "\" class=\"ICON1\" title=\""
//							 + StringUtil.htmlFilter( bodyIn,true ) + "\" />" ;
							 + StringUtil.htmlFilter( bodyIn,true ) + "\" >" ;			// 7.0.1.0 (2018/10/15)
			}
			// 5.5.2.5 (2012/05/21) -4(MATRIX Menu2)の場合は､<li style="list-style:url(画像URL);"> で処理する｡
			else if( type == -4 ) {
				listStyle = "<li style=\"list-style:url(" + imgFile + ");\">" ;
				body = bodyIn ;
				img = "";
			}
			else {
				// 6.7.5.0 (2017/03/10) TILE表示のﾚﾍﾞﾙ-6では､ｲﾒｰｼﾞ直後に BR を入れる｡(暫定処置)
				if( type == -6 ) {
//					img = "<img src=\"" + imgFile + "\" class=\"ICON2\" /><br />" ;		// ICON2 + BR
					img = "<img src=\"" + imgFile + "\" class=\"ICON2\" ><br>" ;		// ICON2 + BR	7.0.1.0 (2018/10/15)
				}
				else {
//					img = "<img src=\"" + imgFile + "\" class=\"ICON1\" />" ;			// ICON1
					img = "<img src=\"" + imgFile + "\" class=\"ICON1\" >" ;			// ICON1		7.0.1.0 (2018/10/15)
				}
				body = bodyIn;
			}
		}

		final String in ;
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
		if( href == null ) {
			in = body;
		}
		else {
			if( useButton ) { // 4.2.1.0 (2008/04/01)
				// 6.1.1.0 (2015/01/17) TagBufferの連結記述
				in = new TagBuffer( "button" )
						.add( "onclick"	, "top."+target+".location.href=\'"+ href + "\';" )		// 5.1.4.0 (2010/03/01)
						.add( "class"	, "buttonMenu" ) 		// 4.2.1.0 (2008/04/03) classを追加
						.add( "id"		, gamenId )				// 5.5.2.5 (2012/05/21) idを本当の画面IDで設定
						.addBody( body )
						.makeTag();
			}
			else {
				// 4.2.2.0 (2008/05/14) ﾎﾞﾀﾝ式にする場合はﾘｸｴｽﾄ変数付加
				// 4.3.3.0 (2008/10/01) 強制ﾌﾟﾙﾀﾞｳﾝ化の場合はfalseで渡す(excludeButton の値の反転)
				final String href2 = buttonRequest
											? XHTMLTag.addUrlEncode( href,"buttonRequest=" + !excludeButton)
											: href ;
				// 6.1.1.0 (2015/01/17) TagBufferの連結記述
				in = new TagBuffer( "a" )
						.add( "class"	, "buttonClassify" , buttonRequest )
						.add( "href"	, href2 )
						.add( "target"	, target )
						.addBody( body )
						.makeTag();
			}
		}

		// 6.7.5.0 (2017/03/10) ｲﾒｰｼﾞを､【】の外に出す｡
		String rtn = null;
		switch( type ) {
			// 6.9.8.0 (2018/05/28) FindBugs:switch 文の2つの case のために同じｺｰﾄﾞを使用しているﾒｿｯﾄﾞ
			// 偶然同じになったのと､これを対応するとｿｰｽがみにくくなるので､このままとします｡
			case 1:  rtn = FIELD_IN + "<legend>" + img + in + "</legend>" ; break;	// 大分類
	//		case 2:  rtn = img + "【" + in + "】" 	; break;		// 小分類・直ﾘﾝｸ
			case 3: 												// 通常ﾒﾆｭｰ
			case 4:  rtn = "　" + img + in 			; break;		// 隠れﾒﾆｭｰ
			case -1: rtn = "<td class=\"MENU_G " + gamenId + "\">" + img + in + "</td>" ; break;	// 5.5.2.5 (2012/05/21) ｸﾞﾙｰﾌﾟﾒﾆｭｰ
			case -2: rtn = img + "[" + in + "] "	; break;		// ﾗｲﾝﾒﾆｭｰ
	//		case -3: rtn = img + in					; break;		// ﾏﾄﾘｸｽﾒﾆｭｰ
			case -4: rtn = listStyle + in + "</li>"	; break;		// 4.2.1.0 (2008/04/03) ﾘｽﾄ｡ﾘｽﾄのため■はいらない
			case -3:												// 7.2.9.4 (2020/11/20) ﾏﾄﾘｸｽﾒﾆｭｰ
			case -5: rtn = img + in					; break;		// 6.8.0.1 (2017/06/30) 1ﾚﾍﾞﾙ表示絶対ﾊﾟｽ
			case  2:												// 7.2.9.4 (2020/11/20) 小分類・直ﾘﾝｸ
			case -6: rtn = img + "【" + in + "】"	; break;		// 小分類・直ﾘﾝｸ(TILE)
			default :
				rtn = "X_" + in ;  break;							// 6.0.2.5 (2014/10/31) break追記
		}

		return rtn ;
	}

	/**
	 * 【TAG】作成するﾒﾆｭｰの種類[NORMAL/GROUP/ONELEVEL/NEXTGUI/MATRIX/MATRIX2]を指定します(初期値:NORMAL)｡
	 *
	 * @og.tag
	 * 作成するﾒﾆｭｰには､複数の種類があります｡
	 * <table class="plain">
	 *   <caption>ﾒﾆｭｰの種類</caption>
	 *   <tr><th>種類    </th><th>説明  </th></tr>
	 *   <tr><td>NORMAL  </td><td>通常の階層ﾒﾆｭｰ</td></tr>
	 *   <tr><td>GROUP   </td><td>GROUPのみを取り出してﾘﾝｸを作成します｡(topMenuに利用)</td></tr>
	 *   <tr><td>ONELEVEL</td><td>指定のclassify のﾒﾆｭｰのみを取り出してﾘﾝｸを作成します｡(lineMenuに利用)</td></tr>
	 *   <tr><td>NEXTGUI </td><td>既存のﾍﾟｰｼﾞの次にｱｸｾｽされる画面郡のﾘﾝｸを作成します｡</td></tr>
	 *   <tr><td>MATRIX  </td><td>一覧表形式のﾒﾆｭｰを作成します｡(大分類付きﾏﾙﾁﾒﾆｭｰ)</td></tr>
	 *   <tr><td>MATRIX2 </td><td>一覧表形式のﾒﾆｭｰを作成します｡(大分類なしﾎﾞﾀﾝﾒﾆｭｰ)</td></tr>
	 *   <tr><td>NONE    </td><td>表示しない(ﾍｯﾀﾞｰﾒﾆｭｰを表示しない時などに使用)</td></tr>
	 *   <tr><td>TILE    </td><td>ｲﾒｰｼﾞを使った全面ｱｲｺﾝ画面</td></tr>
	 * </table>
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規登録
	 * @og.rev 5.2.3.0 (2010/12/01) NEXTGUI 追加
	 * @og.rev 6.3.4.0 (2015/08/01) Arrays.toString から String.join に置き換え｡
	 * @og.rev 6.3.8.3 (2015/10/03) NONE(表示しない) 追加(JavaDoc追記のみ)｡
	 * @og.rev 6.4.3.4 (2016/03/11) String配列 から､Setに置き換えます｡
	 * @og.rev 6.7.5.0 (2017/03/10) TILE表示追加
	 *
	 * @param	type ﾒﾆｭｰの種類 [NORMAL/GROUP/ONELEVEL/NEXTGUI/MATRIX/MATRIX2/NONE/TILE]
	 */
	public void setMenuType( final String type ) {
		menuType = nval( getRequestParameter( type ),menuType );
		if( ! check( menuType, MENU_TYPE_SET ) ) {
			final String errMsg = "menuType に､指定できない種類の文字が指定されました｡"	+ CR
							+ "menuType=[" + menuType + "] " 								+ CR
							+ "menuType List=" + String.join( ", " , MENU_TYPE_SET ) ;
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【TAG】折り返しﾒﾆｭｰを構築するかどうかを指定します(初期値:true)｡
	 *
	 * @og.tag
	 * trueを設定すると､JavaScriptによる折り返しﾒﾆｭｰを構築します｡
	 * false の場合は､通常のHTMLのみで､階層ﾒﾆｭｰを構築します｡
	 * 初期値は､true(折り返しﾒﾆｭｰ)です｡
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規登録
	 *
	 * @param	flag 折り返しﾒﾆｭｰ [true:する/false:しない]
	 */
	public void setExpand( final String flag ) {
		expand = nval( getRequestParameter( flag ),expand );
	}

	/**
	 * 【TAG】表示対象となるｸﾞﾙｰﾌﾟをCSV形式で指定します｡
	 *
	 * @og.tag
	 * ﾒﾆｭｰの表示対象ｸﾞﾙｰﾌﾟをCSV形式で複数指定できます｡
	 * 指定のｸﾞﾙｰﾌﾟのﾒﾆｭｰだけが､表示対象になります｡
	 * ﾒﾆｭｰにも､複数のｸﾞﾙｰﾌﾟを指定できるため､１ｸﾞﾙｰﾌﾟの指定で､
	 * 複数のくくりを表示することも可能です｡
	 * ｸﾞﾙｰﾌﾟを指定しない場合は､全ｸﾞﾙｰﾌﾟが対象になります｡
	 * また､ﾒﾆｭｰ側にｸﾞﾙｰﾌﾟ指定がない場合は､ｸﾞﾙｰﾌﾟ指定に
	 * 関係なく､対象になります｡
	 * 初期値は､未指定(全ﾒﾆｭｰが対象)です｡
	 * 分解方法は､通常のﾊﾟﾗﾒｰﾀ取得後に､CSV分解します｡
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規登録
	 *
	 * @param	menuGroups 表示対象ｸﾞﾙｰﾌﾟ (CSV形式)
	 */
	public void setGroups( final String menuGroups ) {
		groups = StringUtil.csv2Array( getRequestParameter( menuGroups ) );
		if( groups.length == 0 ) { groups = null; }
	}

	/**
	 * 【TAG】表示対象となる分類(classify)を指定します｡
	 *
	 * @og.tag
	 * ﾒﾆｭｰの表示対象となる分類(classify)を指定することで､一まとまりの
	 * ﾒﾆｭｰを作成します｡これは､３段階ﾒﾆｭｰの最終ﾒﾆｭｰを求める場合に
	 * 指定します｡
	 * 最終ﾒﾆｭｰは､画面上部に設ける予定のﾒﾆｭｰで､上下ﾌﾚｰﾑ分割での
	 * 運用時に使用します｡
	 * 分類の指定がない場合は､すべてが表示対象になります｡
	 * 初期値は､未指定(全ﾒﾆｭｰが対象)です｡
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規登録
	 *
	 * @param	classify 表示対象分類 (classify)
	 */
	public void setClassify( final String classify ) {
		selClassify = nval( getRequestParameter( classify ),selClassify );
	}

	/**
	 * 【TAG】ｸﾞﾙｰﾌﾟﾒﾆｭｰの表示対象となるｿｰｽ名(href)を指定します(初期値:menu.jsp)｡
	 *
	 * @og.tag
	 * GROUPﾒﾆｭｰの表示対象となるｿｰｽ名(href)を指定することで､
	 * ｻﾌﾞﾒﾆｭｰを自分自身のﾌﾚｰﾑに対して割り当てるのか､ﾌﾚｰﾑを分けて
	 * 管理するのかを自由に設定できます｡
	 * 初期値は､menu.jspです｡
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規登録
	 *
	 * @param	inHref 表示対象ｿｰｽ名 (href)
	 */
	public void setHref( final String inHref ) {
		href = nval( getRequestParameter( inHref ),href );
	}

	/**
	 * 【TAG】ｸﾞﾙｰﾌﾟﾒﾆｭｰの表示対象となるﾌﾚｰﾑ名(target)を指定します(初期値:MENU)｡
	 *
	 * @og.tag
	 * GROUPﾒﾆｭｰの表示対象となるﾌﾚｰﾑ名(target)を指定することで､
	 * ｻﾌﾞﾒﾆｭｰを自分自身のﾌﾚｰﾑに対して割り当てるのか､ﾌﾚｰﾑを分けて
	 * 管理するのかを自由に設定できます｡
	 * ﾌﾚｰﾑ分割を行うと､変更箇所は､ｻﾌﾞﾒﾆｭｰのみになる為､動きに無駄が
	 * なくなりますが､ｸﾞﾙｰﾌﾟﾒﾆｭｰの大きさが固定されてしまいます｡
	 * 自分自身にすると､ｸﾞﾙｰﾌﾟﾒﾆｭｰとｻﾌﾞﾒﾆｭｰを一つのﾌﾚｰﾑに
	 * 入れることで､更新時の画面のちらつきは発生しますが､無駄なｽﾍﾟｰｽは
	 * 省くことが可能になります｡
	 * 初期値は､MENU(通常のﾒﾆｭｰﾌﾚｰﾑ)です｡
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規登録
	 *
	 * @param	inTarget 表示対象ﾌﾚｰﾑ名 (target)
	 */
	public void setTarget( final String inTarget ) {
		target = nval( getRequestParameter( inTarget ),target );
	}

	/**
	 * 【TAG】ｸﾞﾙｰﾌﾟﾒﾆｭｰの表示に､画像のみかどうか[true:画像のみ/false:画像＋ﾗﾍﾞﾙ]を指定します(初期値:false)｡
	 *
	 * @og.tag
	 * GROUPﾒﾆｭｰの表示対象として､jsp/menuImage 以下に ｸﾞﾙｰﾌﾟ名と同一の
	 * 画像ﾌｧｲﾙが存在する場合は､画像を使用します｡
	 * このﾌﾗｸﾞを､true に設定すると､画像のみを使用します｡
	 * false の場合は､画像＋ｸﾞﾙｰﾌﾟ名のﾗﾍﾞﾙを使用します｡
	 * 画像のみの場合でも､title 属性にｸﾞﾙｰﾌﾟ名のﾗﾍﾞﾙをｾｯﾄしますので､
	 * ﾏｳｽｶｰｿﾙをｵｰﾊﾞｰすれば､名称がTips表示されます｡
	 * 画像が存在しない場合に､true(画像のみ)に設定した場合は､ﾗﾍﾞﾙの最初の１文字
	 * のみを出力します｡
	 * 初期値は､false(画像＋ﾗﾍﾞﾙ)です｡
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規登録
	 *
	 * @param	flag ｸﾞﾙｰﾌﾟﾒﾆｭｰ表示 [true:画像のみ/false:画像＋ﾗﾍﾞﾙ]
	 */
	public void setImageOnly( final String flag ) {
		imageOnly = nval( getRequestParameter( flag ),imageOnly );
	}

	/**
	 * 【TAG】ｸﾞﾙｰﾌﾟﾒﾆｭｰの表示対象となるﾒﾆｭｰを横並びさせる数を指定します｡
	 *
	 * @og.tag
	 * GROUPﾒﾆｭｰなどの表示を行う場合に､横方向に何個のﾒﾆｭｰを表示させるかを
	 * 指定します｡例えば､画像のみのﾘﾝｸと組み合わせれば､より､多くのｸﾞﾙｰﾌﾟを
	 * 横方向に並べることで､小領域に多くの情報を詰めることが可能になります｡
	 * 0 を設定すると､横方向にのみ並べる(折り返さない)ﾒﾆｭｰを作ることが
	 * 可能になります｡
	 * 初期値は､無制限です｡
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規登録
	 *
	 * @param	count 横並び数
	 */
	public void setSideCount( final String count ) {
		sideCount = nval( getRequestParameter( count ),sideCount );
	}

	/**
	 * 【TAG】表形式ﾒﾆｭｰ(MATRIX)の一つのｾﾙに含まれる最小行数を指定します(初期値:8)｡
	 *
	 * @og.tag
	 * 表形式ﾒﾆｭｰ(MATRIX)では､一つのｾﾙの高さを同一にする為､&lt;br /&gt;ﾀｸﾞを挿入します｡
	 * このﾀｸﾞの挿入する個数を指定します｡
	 * この個数だけ､ﾒﾆｭｰの数があろうとなかろうと行を確保します｡
	 * 指定の値が､実際の行数より少ない場合は､実際の行数分だけ拡張されます｡
	 * 初期値は､8 です｡
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規登録
	 *
	 * @param	count ｾﾙ内最小行数
	 * @see		#setMaxCellCount( String )
	 */
	public void setMinCellCount( final String count ) {
		minCellCount = nval( getRequestParameter( count ),minCellCount );
	}

	/**
	 * 【TAG】表形式ﾒﾆｭｰ(MATRIX)の一つのｾﾙに含まれる最大行数を指定します(初期値:8)｡
	 *
	 * @og.tag
	 * 表形式ﾒﾆｭｰ(MATRIX)では､一つのｾﾙの高さを同一にする為､指定の行数で
	 * 新たな ｾﾙを作成して､ｾﾙを横方向に連結します｡
	 * 初期値は､8 です｡
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規登録
	 *
	 * @param	count ｾﾙ内最大行数
	 * @see		#setMinCellCount( String )
	 */
	public void setMaxCellCount( final String count ) {
		maxCellCount = nval( getRequestParameter( count ),maxCellCount );
	}

	/**
	 * 【TAG】ｸﾞﾙｰﾌﾟﾒﾆｭｰのｷｬｯｼｭを使用するかどうか[true/false]を指定します(初期値:true)｡
	 *
	 * @og.tag
	 * GROUPﾒﾆｭｰの表示は､一旦作成すると､ほとんど書き換えることがありません｡
	 * 作成は､使用できる全ﾒﾆｭｰをｽｷｬﾝして､その中からｸﾞﾙｰﾌﾟ属性をﾋﾟｯｸｱｯﾌﾟ
	 * するという処理を行っている為､明らかに無駄な処理です｡
	 * そこで､jsp/index.jsp が実行された場合のみｷｬｯｼｭをｸﾘｱして､内部では
	 * ｷｬｯｼｭがなければ作成し､あればｷｬｯｼｭを使うﾛｼﾞｯｸになっています｡
	 * ここでは､ｷｬｯｼｭを使用するのか､毎回作成しなおすのかを指定します｡
	 * 対象として､jsp/menuImage 以下に ｸﾞﾙｰﾌﾟ名と同一の
	 * 画像ﾌｧｲﾙが存在する場合は､画像を使用します｡
	 * このﾌﾗｸﾞを､true に設定すると､画像のみを使用します｡
	 * false の場合は､画像＋ｸﾞﾙｰﾌﾟ名のﾗﾍﾞﾙを使用します｡
	 * 画像のみの場合でも､title 属性にｸﾞﾙｰﾌﾟ名のﾗﾍﾞﾙをｾｯﾄしますので､
	 * ﾏｳｽｶｰｿﾙをｵｰﾊﾞｰすれば､名称がTips表示されます｡
	 * 画像が存在しない場合は､たとえ､true(画像のみ)に設定しても､ﾗﾍﾞﾙを出力します｡
	 * 初期値は､true(ｷｬｯｼｭする)です｡
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規登録
	 *
	 * @param	flag ｷｬｯｼｭ可否 [true:ｷｬｯｼｭする/false:ｷｬｯｼｭしない]
	 */
	public void setCache( final String flag ) {
		cache = nval( getRequestParameter( flag ),cache );
	}

	/**
	 * 【TAG】正判定(ﾏｯﾁする場合に､ﾒﾆｭｰに出す)条件を設定します｡
	 *
	 * @og.tag
	 * ﾒﾆｭｰを表示する/しないの判定を､画面IDのﾏｯﾁﾝｸﾞで判断します｡
	 * これは､正規表現で表される引数と､画面IDがﾏｯﾁする場合に､
	 * ﾒﾆｭｰを表示させます｡
	 * ﾏｯﾁしない場合は､表示されません｡
	 * 何も指定しない場合は､ﾛｰﾙの判定みの行われます｡
	 *
	 * @param   mt 正判定条件 (ﾏｯﾁする場合に､ﾒﾆｭｰに出す)
	 */
	public void setMatch( final String mt ) {
		match = nval( getRequestParameter( mt ),match );
	}

	/**
	 * 【TAG】逆判定(ﾏｯﾁする場合に､ﾒﾆｭｰに出さない)条件を設定します｡
	 *
	 * @og.tag
	 * ﾒﾆｭｰを表示する/しないの判定を､画面IDのﾏｯﾁﾝｸﾞで判断します｡
	 * これは､正規表現で表される引数と､画面IDがﾏｯﾁする場合に､
	 * ﾒﾆｭｰを表示させません｡
	 * ﾏｯﾁしない場合は､表示されます｡
	 * 何も指定しない場合は､ﾛｰﾙの判定みの行われます｡
	 *
	 * @param   umt 逆判定条件 (ﾏｯﾁする場合に､ﾒﾆｭｰに出さない)
	 */
	public void setUnmatch( final String umt ) {
		unmatch = nval( getRequestParameter( umt ),unmatch );
	}

	/**
	 * 【TAG】ﾒﾆｭｰに使用する画像ﾌｧｲﾙのﾌｫﾙﾀﾞを指定します(初期値:{@og.value #MENU_IMG})｡
	 *
	 * @og.tag
	 * ﾒﾆｭｰに使用する画像ﾌｧｲﾙのﾌｫﾙﾀﾞを指定します｡
	 * 画面IDのｷｰと一致する画像ﾌｧｲﾙを使用します｡
	 * 初期値は､{@og.value #MENU_IMG} です｡
	 * /jspからのﾌｫﾙﾀﾞ指定が必要です｡
	 *
	 * @og.rev 6.7.5.0 (2017/03/10) TILE表示追加(imageDir 属性追加)
	 *
	 * @param   dir ﾒﾆｭｰに使用する画像ﾌｧｲﾙのﾌｫﾙﾀﾞ
	 */
	public void setImageDir( final String dir ) {
		imageDir = nval( getRequestParameter( dir ),imageDir );
	}

	/**
	 * 【TAG】画面ﾘﾝｸにﾎﾞﾀﾝを使用するかどうか[true/false]を指定します(初期値:false)｡
	 *
	 * @og.tag
	 * 画面ﾘﾝｸをﾎﾞﾀﾝﾘﾝｸ形式で表示するかを指定します｡
	 * falseの場合は､通常のﾘﾝｸになります｡
	 * 初期値は､false(通常ﾘﾝｸ)です｡
	 *
	 * @og.rev 4.2.1.0 (2008/04/01) 新規登録
	 *
	 * @param	flag ﾎﾞﾀﾝﾘﾝｸ [true:ﾎﾞﾀﾝ形式ﾘﾝｸ/false:通常ﾘﾝｸ]
	 */
	public void setUseButton( final String flag ) {
		useButton = nval( getRequestParameter( flag ),useButton );
	}

	/**
	 * 【TAG】ﾏﾄﾘｸｽからの遷移先でﾎﾞﾀﾝﾒﾆｭｰ表示するかどうか[true/false]を指定します(初期値:false)｡
	 *
	 * @og.tag
	 * ﾏﾄﾘｸｽﾒﾆｭｰからの遷移先でﾎﾞﾀﾝﾒﾆｭｰを表示させるために
	 * ｱﾄﾞﾚｽに付加するﾘｸｴｽﾄ変数を指定します｡
	 * trueにするとbuttonRequest=trueのﾘｸｴｽﾄ変数を付けます｡
	 * falseの場合はﾘｸｴｽﾄ変数を付けません｡
	 * 初期値は､false(ﾌﾟﾙﾀﾞｳﾝ形式で表示)です｡
	 *
	 * @og.rev 4.2.1.0 (2008/04/17) 新規登録
	 *
	 * @param	flag ﾒﾆｭｰ形式 [true:ﾎﾞﾀﾝ形式/false:通常形式]
	 */
	public void setButtonRequest( final String flag ) {
		buttonRequest = nval( getRequestParameter( flag ),buttonRequest );
	}

	/**
	 * 【TAG】標準画面を初期状態で開いた状態で表示するかどうか[true/false]を指定します(初期値:false)｡
	 *
	 * @og.tag
	 * 通常expand=trueの状態では､ﾒﾆｭｰが折りたたまれています｡
	 * このinlineStyle属性をtrueにすると標準画面のｽﾀｲﾙにdisplay:inlineを
	 * 付加する事で､初期状態でﾒﾆｭｰが開いた状態になります｡
	 * expand=falseとの違いは､隠しﾒﾆｭｰ及び分類の折りたたみ機能が利用できる事です｡
	 * 初期値は､false(折りたたまれた状態)です｡
	 *
	 * @og.rev 4.3.3.0 (2008/10/01) 新規
	 *
	 * @param	flag ﾒﾆｭｰｽﾀｲﾙ [true:開いた状態/false:閉じた状態]
	 */
	public void setInlineStyle( final String flag ) {
		inlineStyle = nval( getRequestParameter( flag ),inlineStyle );
	}

	/**
	 * 【TAG】ONELEVEL,NEXTGUI で､DIVﾀｸﾞ＋design-onelevel 処理をするかどうか[true/false]を指定します(初期値:false)｡
	 *
	 * @og.tag
	 * ONELEVEL,NEXTGUI は､QUERY 画面の上部に表示される簡易ﾒﾆｭｰです｡
	 * この表記は､[画面ﾘﾝｸ] 形式のﾃｷｽﾄ表示されていますが､画面名称を固定長にするなどの処理を入れるため､
	 * DIVでﾌｫｰﾏｯﾄします｡
	 * 作りは､出力される HTML を確認いただきたいと思います｡
	 * 全体を､&lt;div id="design-onelevel" &gt; で､囲い､画面名称は､&lt;span clas="design-onelevel" &gt; で､囲います｡
	 * これを､標準CSSで､固定幅と背景色､ﾘﾝｸの文字色など書き換えて､体裁を整えます｡
	 * 初期値は､false(従来と同じ)です｡
	 *
	 * @og.rev 5.5.2.3 (2012/05/15) 新規追加
	 *
	 * @param	flag DIVﾀｸﾞ処理するか [true:DIVﾀｸﾞ処理をする/false:しない]
	 */
	public void setUseDivOneLevel( final String flag ) {
		useDivOneLevel = nval( getRequestParameter( flag ),useDivOneLevel );
	}

	/**
	 * 【TAG】画面名称に､名前(長) を使う場合は､trueを指定します(初期値:false)｡
	 *
	 * @og.tag
	 * 画面の名称は､画面ﾘｿｰｽの GUIInfo#getName() を使用しますが､通常､画面名称(short) です｡
	 * 画面名称(long) を使いたい場合は､このﾌﾗｸﾞを true に設定すると､GUIInfo#getLongName() を使用します｡
	 * 初期値は､false(従来と同じ)です｡
	 *
	 * menuType が NORMAL,MATRIX,MATRIX2,TILE の場合のみ効きます｡
	 *
	 * @og.rev 8.2.1.0 (2022/07/15) useLongName 属性追加
	 *
	 * @param	flag 画面の名称 [true:名前(長)/false:名前(短)]
	 */
	public void setUseLongName( final String flag ) {
		useLongName = nval( getRequestParameter( flag ),useLongName );
	}

	/**
	 * 画面ｵﾌﾞｼﾞｪｸﾄ一覧より､指定されたｲﾝﾃﾞｯｸｽより後に実体となる画面ID
	 * (直ﾘﾝｸ､通常ﾒﾆｭｰ､隠しﾒﾆｭｰ※改行ﾒﾆｭｰは除く)が存在するかを判定します｡
	 * ｸﾞﾙｰﾌﾟｽが指定されている場合は､ｸﾞﾙｰﾌﾟで絞り込まれた結果に対して判定を行います｡
	 *
	 * @og.rev 4.0.0.0 (2007/10/30) 新規追加
	 * @og.rev 6.9.8.0 (2018/05/28) FindBugs 対応で､groupCheckの判定方法に､groups == null も入れる｡
	 *
	 * @param guiInfos 画面ｵﾌﾞｼﾞｪｸﾄ一覧
	 * @param idx 検索を開始するｲﾝﾃﾞｯｸｽ番号
	 *
	 * @return 0:実体画面が存在せず 1:全て隠しの分類 2:実体画面 or 通常画面を含む分類
	 */
	private int guiCheck( final GUIInfo[] guiInfos, final int idx ) {
		int flg = 0;

		// 実態探し
		if( levelCheck( guiInfos[idx] ) > 0 ) {
			// ｸﾞﾙｰﾌﾟﾒﾆｭｰﾘﾝｸ時の処理
			// 6.9.8.0 (2018/05/28) FindBugs 対応で､groupCheckの判定方法に､groups == null も入れる｡
//			if( groups == null ) { flg = 2; }
//			else { flg = groupCheck( guiInfos[idx] ) ? 2 : 0; }
			flg = groupCheck( guiInfos[idx] ) ? 2 : 0;
		}
		else {
			for( int j=idx+1; j<guiInfos.length; j++ ) {
				if( !guiInfos[j].isRead() ) { continue; }
				if( flg > 1 || guiInfos[j].getLevel() <= guiInfos[idx].getLevel() ) { break; }

				// 6.9.8.0 (2018/05/28) FindBugs 対応で､groupCheckの判定方法に､groups == null も入れる｡
//				if( groups == null || groupCheck( guiInfos[j] ) ) {
				if( groupCheck( guiInfos[j] ) ) {
					if( levelCheck( guiInfos[j] ) > 0 ) {
						flg = levelCheck( guiInfos[j] );
					}
				}
			}
		}

		return flg;
	}

	/**
	 * 指定された画面IDが実体であるか(直ﾘﾝｸ､通常ﾒﾆｭｰ､隠しﾒﾆｭｰ※改行ﾒﾆｭｰは除く)を判定します｡
	 *
	 * @og.rev 4.0.0.0 (2007/10/30) 新規追加
	 * @og.rev 4.0.0.0 (2007/11/30) switch に default 追加
	 * @og.rev 6.9.4.1 (2018/04/09) HYBS_BR の廃止
	 *
	 * @param guiInfo 画面ｵﾌﾞｼﾞｪｸﾄ
	 *
	 * @return 0:分類 1:実体画面(隠し) 2:実体画面(通常) or 直ﾘﾝｸ
	 */
	private int levelCheck( final GUIInfo guiInfo ) {
		int flg = 0;

		final String adrs = guiInfo.getAddress();	// 6.9.4.1 (2018/04/09) HYBS_BR は､ｱﾄﾞﾚｽに設定

		switch ( guiInfo.getLevel() ) {
			case 2:
//				final String adrs = guiInfo.getAddress();
				if( adrs != null && adrs.length() > 0 ) {
					flg = 2;
				}
				break;
			case 3:
//				flg = "HYBS_BR".equals( guiInfo.getKey() ) ? flg : 2; break;
				flg = "HYBS_BR".equals( adrs ) ? flg : 2; break;
			case 4:
//				flg = "HYBS_BR".equals( guiInfo.getKey() ) ? flg : 1; break;
				flg = "HYBS_BR".equals( adrs ) ? flg : 1; break;
			default :
				flg = 0; break;
		}

		return flg;
	}

	/**
	 * 指定された画面IDが設定されたｸﾞﾙｰﾌﾟｽに含まれるかを判定します｡
	 *
	 * groups == null か､または､画面IDが設定されたｸﾞﾙｰﾌﾟｽに含まれる場合､true を返します｡
	 *
	 * @og.rev 4.0.0.0 (2007/10/30) 新規追加
	 * @og.rev 6.9.8.0 (2018/05/28) FindBugs 対応で､groupCheckの判定方法に､groups == null も入れる｡
	 *
	 * @param guiInfo 画面ｵﾌﾞｼﾞｪｸﾄ
	 *
	 * @return groupsが nullか､ｸﾞﾙｰﾌﾟｽに含まれるかどうか
	 */
	private boolean groupCheck( final GUIInfo guiInfo ) {
		// 6.9.8.0 (2018/05/28) FindBugs:ｺﾝｽﾄﾗｸﾀで初期化されていないﾌｨｰﾙﾄﾞを null ﾁｪｯｸなしで null 値を利用している
		if( groups == null ) { return true; }
		else {
			boolean flg = false;

			// 7.2.9.4 (2020/11/20) PMD:This for loop can be replaced by a foreach loop
			for( final String grp : groups ) {
				if( guiInfo.isGroupIn( grp ) ) {
//			for( int k=0; k<groups.length; k++ ) {
//				if( guiInfo.isGroupIn( groups[k] ) ) {
					flg = true;
					break;
				}
			}

			return flg;
		}
	}

	/**
	 * このｵﾌﾞｼﾞｪｸﾄの文字列表現を返します｡
	 * 基本的にﾃﾞﾊﾞｯｸﾞ目的に使用します｡
	 *
	 * @return このｸﾗｽの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"				,VERSION			)
				.println( "menuType"			,menuType			)
				.println( "expand"				,expand				)
				.println( "groups"				,groups				)
				.println( "selClassify"			,selClassify		)
				.println( "href"				,href				)
				.println( "target"				,target				)
				.println( "imageOnly"			,imageOnly			)
				.println( "sideCount"			,sideCount			)
				.println( "minCellCount"		,minCellCount		)
				.println( "maxCellCount"		,maxCellCount		)
				.println( "cache"				,cache				)
				.println( "mscKey"				,mscKey				)
//				.println( "multiSessionCheck"	,multiSessionCheck	)	// 6.9.5.0 (2018/04/23) multiSessionCheck 廃止(true固定)
				.println( "useButton"			,useButton			)
				.println( "buttonRequest"		,buttonRequest		)
				.println( "MENU_TYPE"			,MENU_TYPE_SET		)	// 6.4.3.4 (2016/03/11)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
