/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.html.ViewStackTableParam;

import static org.opengion.fukurou.util.StringUtil.nval ;

/**
 * viewﾀｸﾞの viewFormType が HTMLStackedGanttTable の場合にﾊﾟﾗﾒｰﾀを設定します。
 *
 * ｶﾞﾝﾄﾁｬｰﾄを表示する ViewForm_HTMLStackedGanttTable ｸﾗｽに対して、各種ﾊﾟﾗﾒｰﾀを
 * 設定します。
 * ﾊﾟﾗﾒｰﾀが設定されていない場合は、ViewStackTableParam の初期値が
 * 使用されます。
 * (ﾊﾟﾗﾒｰﾀを使用するには、viewﾀｸﾞのuseParam 属性をtrueに設定する必要があります。)
 *
 * @og.formSample
 * ●形式：&lt;og:stackParam stackColumns="PRJCODE,RESOURCE_ID" /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:stackParam
 *       stackColumns     ○【TAG】積上単位とするｶﾗﾑ名(CSV形式)をｾｯﾄします (必須)
 *       costColumn         【TAG】内部積上げする場合の工数ｶﾗﾑをｾｯﾄします
 *       innerStack         【TAG】工数の内部積み上げをするかどうか[true/false]指定します(初期値:false)
 *       debug              【TAG】ﾃﾞﾊﾞｯｸﾞ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * @og.rev 5.5.7.0 (2012/10/01) 新規作成
 * @og.group 画面表示
 *
 * @version	5.0
 * @author	Takahashi Masakazu
 * @since	JDK5.0,
 */
public class ViewStackParamTag extends ViewParamImpl {
	/** このﾌﾟﾛｸﾞﾗﾑのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.2.0 (2016/01/29)" ;
	private static final long serialVersionUID = 642020160129L ;

	/**
	 * ﾃﾞﾌｫﾙﾄｺﾝｽﾄﾗｸﾀｰ
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public ViewStackParamTag() { super(); }		// これも、自動的に呼ばれるが、空のﾒｿｯﾄﾞを作成すると警告されるので、明示的にしておきます。

	/**
	 * 【TAG】積上単位とするｶﾗﾑ名(CSV形式)をｾｯﾄします。
	 *
	 * @og.tag
	 * 工数積上げのｷｰとなるｶﾗﾑ名を指定します。
	 *
	 * 複数ある場合は、CSV形式で指定できます。
	 * ﾃｰﾌﾞﾙﾓﾃﾞﾙは積上単位＋ﾀｽｸ単位でOrderByがかかっている必要があります。
	 *
	 * @param	stackColumns	積上ｶﾗﾑ名 (CSV形式)
	 */
	public void setStackColumns( final String stackColumns ) {
		putParam( ViewStackTableParam.STACK_COLUMNS_KEY ,
				  nval( getRequestParameter( stackColumns ),null ) );
	}

	/**
	 * 【TAG】工数関係のｶﾗﾑ（工数,開始日,終了日）を指定します。
	 *
	 * @og.tag
	 * 工数関係のｶﾗﾑ名を指定します。
	 * 工数(x.y)、開始日(yyyyMMdd)、終了日(yyyyMMdd)
	 * の３つのｶﾗﾑをCSV形式で設定して下さい。
	 * ｴﾝｼﾞﾝ内部で工数積上げを行う場合は必須です。
	 *
	 * ﾃｰﾌﾞﾙﾓﾃﾞﾙは積上単位＋ﾀｽｸ単位でOrderByがかかっている必要があります。
	 *
	 * @og.rev 5.5.8.3 (2012/11/17)
	 *
	 * @param	costColumns	工数のｶﾗﾑ名
	 */
	public void setCostColumns( final String costColumns ) {
		putParam( ViewStackTableParam.COST_COLUMNS_KEY ,
				  nval( getRequestParameter( costColumns ),null ) );
	}

	/**
	 * 【TAG】ｴﾝｼﾞﾝ内部で積上げするかどうかを指定します
	 *		(初期値:INNER_STACK_VALUE[={@og.value org.opengion.hayabusa.html.ViewStackTableParam#INNER_STACK_VALUE}])。
	 *
	 * @og.tag
	 * ｴﾝｼﾞﾝ内部で積上げを行うかどうかを指定します。
	 * 内部で積上げを行った場合は、HTMLStackGanttTableではｶﾞﾝﾄ行の出力を行いません。
	 * 初期値はfalse（積上げしない）です。
	 *
	 * @og.rev 5.5.8.3 (2012/11/17)
	 *
	 * @param	flag	内部積上げ可否 [true:する/false:しない]
	 */
	public void setInnerStack( final String flag ) {
		putParam( ViewStackTableParam.INNER_STACK_KEY ,
				  nval( getRequestParameter( flag ),ViewStackTableParam.INNER_STACK_VALUE ) );
	}

	/**
	 * 【TAG】ｴﾝｼﾞﾝ内部で積上げする場合に休日も積上げるかどうかを指定します
	 *		(初期値:STACK_HOLIDAY_VALUE[={@og.value org.opengion.hayabusa.html.ViewStackTableParam#STACK_HOLIDAY_VALUE}])。
	 *
	 * @og.tag
	 * ｴﾝｼﾞﾝ内部で積上げを行う場合に休日も積上げるかどうかを指定します。
	 * zoomがDAYの場合のみ有効です。
	 * 初期値はtrue（積上げする）です。
	 *
	 * @og.rev 5.5.8.3 (2012/11/17)
	 *
	 * @param	flag	休日積上げ可否 [true:する/false:しない]
	 */
	public void setStackHoliday( final String flag ) {
		putParam( ViewStackTableParam.STACK_HOLIDAY_KEY ,
				  nval( getRequestParameter( flag ),ViewStackTableParam.STACK_HOLIDAY_VALUE ) );
	}

	/**
	 * 【TAG】その行の能力値ｶﾗﾑを指定します
	 *		(初期値:CAP_COLUMN_VALUE[={@og.value org.opengion.hayabusa.html.ViewStackTableParam#CAP_COLUMN_VALUE}])。
	 *
	 * @og.tag
	 * 内部積上げをする際、各行の能力のｶﾗﾑ名を指定します。
	 * ｶﾗﾑを指定しない場合は1となります。
	 *
	 * @og.rev 5.6.1.2 (2013/02/22)
	 *
	 * @param	capColumn	能力値のｶﾗﾑ名
	 */
	public void setCapacityColumn( final String capColumn ) {
		putParam( ViewStackTableParam.CAP_COLUMN_KEY ,
				  nval( getRequestParameter( capColumn ),ViewStackTableParam.CAP_COLUMN_VALUE  ) );
	}

	/**
	 * ﾀｸﾞの名称を、返します。
	 * 自分自身のｸﾗｽ名より、自動的に取り出せないため、このﾒｿｯﾄﾞをｵｰﾊﾞｰﾗｲﾄﾞします。
	 *
	 * @return	ﾀｸﾞの名称
	 * @og.rtnNotNull
	 */
	@Override
	protected String getTagName() {
		return "stackParam" ;
	}
}
