/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.db.AbstractEditor;
import org.opengion.hayabusa.db.CellEditor;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.SelectionCellEditor;					// 6.2.2.0 (2015/03/27)
import org.opengion.hayabusa.db.Selection;
import org.opengion.hayabusa.db.SelectionFactory;						// 5.7.3.0 (2014/02/07)
import org.opengion.fukurou.util.Attributes;
import org.opengion.fukurou.util.TagBuffer;
import org.opengion.fukurou.util.XHTMLTag;

/**
 * RADIO エディターは、カラムのデータをコードリソースに対応したラジオボタンで編集する
 * 場合に使用するクラスです。
 * ラジオボタンは、ラベルがない状態で表示される為、運用時には、ラベルに
 * 値(選択肢のラベル)を作成して、columnEditor 等で登録して置いてください。
 *
 * 一覧検索明細登録画面等で、ラベル表示が必要な場合は、編集パラメータに、"useLabel"と
 * 記述しておくことで、ラベルを出力することが可能です。
 *
 * このエディタはeventColumnに対応していません。
 *
 * カラムの表示に必要な属性は、DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.rev 3.5.1.0 (2003/10/03) 新規作成
 * @og.rev 6.2.2.0 (2015/03/27) SelectionCellEditor I/Fを追加
 * @og.group データ編集
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Editor_RADIO extends AbstractEditor implements SelectionCellEditor {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "7.0.5.1 (2019/09/27)" ;

	private final Selection	selection ;
	private final boolean writable	;
	private final boolean useLabel	;		// 4.3.3.0 (2008/10/01)

	private String errMsg	;				// 6.0.4.0 (2014/11/28)

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 */
	public Editor_RADIO() {
		super();		// 6.4.1.1 (2016/01/16) PMD refactoring. It is a good practice to call super() in a constructor
		// 4.3.4.4 (2009/01/01)
		selection = null;
		writable  = false;
		useLabel  = false; // 4.3.3.0 (2008/10/01)
	}

	/**
	 * DBColumnオブジェクトを指定したprivateコンストラクター。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) makeCodeSelection メソッドを CodeSelectionクラスに変更。
	 * @og.rev 3.5.5.7 (2004/05/10) SelectionFactory を使用して、オブジェクト作成
	 * @og.rev 4.0.0.0 (2005/01/31) SelectionFactory ではなく、直接 Selection_RADIO を作成。
	 * @og.rev 4.3.3.0 (2008/10/01) 編集パラメーターで明細表示時でもラベルが使えるように対応
	 * @og.rev 5.7.3.0 (2014/02/07) SelectionFactory 対応
	 * @og.rev 6.0.4.0 (2014/11/28) selection が null の場合、警告表示します。
	 * @og.rev 6.0.4.0 (2014/11/28) optionAttributes は、コンストラクタで設定します。
	 * @og.rev 6.2.0.0 (2015/02/27) キー:ラベル形式で表示するかどうかを、指定できるようにします。
	 * @og.rev 6.4.4.2 (2016/04/01) nameがnullになってしまった。
	 * @og.rev 7.0.5.1 (2019/09/27) optionAttributes が二重に設定されていたため、削除
	 *
	 * @param	clm	DBColumnオブジェクト
	 */
	private Editor_RADIO( final DBColumn clm ) {
	//	super();		// 6.4.1.1 (2016/01/16) PMD refactoring. It is a good practice to call super() in a constructor
		super( clm );	// 6.4.4.2 (2016/04/01) nameがnullになってしまった。
	//	name  = clm.getName();

		final String addKeyLabel = clm.getAddKeyLabel();		// 6.2.0.0 (2015/02/27) キー:ラベル形式

		// 5.7.3.0 (2014/02/07) SelectionFactory 対応
		selection = SelectionFactory.newSelection( "RADIO" , clm.getCodeData(), addKeyLabel );

		// 6.0.4.0 (2014/11/28) selection が null の場合、警告表示します。
		if( selection == null ) {
			errMsg = "codeData が未設定です。"
								+ " name="  + name
								+ " label=" + clm.getLabel()
								+ " editType="  + clm.getEditor() ;
			System.out.println( errMsg );
		}

		writable = clm.isWritable();

		// 6.1.1.0 (2015/01/17) Attributesの連結記述
		attributes = new Attributes()
					.set( clm.getEditorAttributes() )				// #addAttributes( Attributes ) の代替え
					.add( "class","RADIO" );						// (2003/10/08 修正分)

		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		// 7.0.5.1 (2019/09/27) optionAttributes が二重に設定されていたため、削除
		tagBuffer.add( XHTMLTag.inputAttri( attributes ) );
//				 .add( attributes.get( "optionAttributes" ) );		// 6.0.4.0 (2014/11/28)

		// 4.3.3.0 (2008/10/01)
		useLabel = "useLabel".equalsIgnoreCase( clm.getEditorParam() ) ;
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellEditorオブジェクト
	 * @og.rtnNotNull
	 */
	public CellEditor newInstance( final DBColumn clm ) {
		return new Editor_RADIO( clm );
	}

	/**
	 * データの編集用文字列を返します。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) selection が null の場合、警告表示します。
	 * @og.rev 6.2.2.4 (2015/04/24) getRadio廃止。getOption として、引数違いとして用意する。
	 *
	 * @param	value 入力値
	 *
	 * @return	データの編集用文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final String value ) {
		// 6.0.4.0 (2014/11/28) selection が null の場合、警告表示します。
		if( selection == null ) {
			return "<span class=\"error\">" + errMsg + " value=" + value + "</span>";
		}

		final String radio ;
		if( writable ) {
			radio = selection.getOption( name,value,true );			// 6.2.2.4 (2015/04/24)
		}
		else {
			radio = selection.getValueLabel( value,true );			// 6.2.2.4 (2015/04/24) useLabel
		}

		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		return new TagBuffer( "pre" )
					.add( tagBuffer.makeTag() )
					.addBody( radio )
					.makeTag();
	}

	/**
	 * name属性を変えた、データ表示/編集用のHTML文字列を作成します。
	 * テーブル上の name に 行番号を付加して、名前_行番号 で登録するキーを作成し、
	 * リクエスト情報を１つ毎のフィールドで処理できます。
	 *
	 * @og.rev 3.5.5.0 (2004/03/12) 名前と行番号の区切り記号("__")を、HybsSystem.JOINT_STRING  に変更。
	 * @og.rev 3.5.5.5 (2004/04/23) changeOptionAttributes を廃止します。
	 * @og.rev 4.3.3.0 (2008/10/01) 編集パラメーターで明細表示時でもラベルが使えるように対応
	 * @og.rev 6.0.4.0 (2014/11/28) selection が null の場合、警告表示します。
	 * @og.rev 6.2.2.4 (2015/04/24) getRadio廃止。getOption として、引数違いとして用意する。
	 *
	 * @param	row   行番号
	 * @param	value 入力値
	 *
	 * @return	データ表示/編集用の文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final int row,final String value ) {
		// 6.0.4.0 (2014/11/28) selection が null の場合、警告表示します。
		if( selection == null ) {
			return "<span class=\"error\">" + errMsg + " value=" + value + " row=" + row + "</span>";
		}

		final String radio ;
		if( writable ) {
			// 4.3.3.0 (2008/10/01)
			radio = selection.getOption( name + HybsSystem.JOINT_STRING + row,value,useLabel );		// 6.2.2.4 (2015/04/24)
		}
		else {
			radio = selection.getValueLabel( value,useLabel );		// 6.2.2.4 (2015/04/24)
		}

		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		return new TagBuffer( "pre" )
					.add( tagBuffer.makeTag() )
					.addBody( radio )
					.makeTag( row,value );
	}
}
