/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.common.HybsSystem;									// 8.0.2.0 (2021/11/30) Add
import org.opengion.hayabusa.db.AbstractEditor;
import org.opengion.hayabusa.db.CellEditor;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.fukurou.util.TagBuffer;										// 8.0.2.0 (2021/11/30) Add
import org.opengion.fukurou.util.StringUtil;									// 8.0.2.0 (2021/11/30) Add
import org.opengion.fukurou.util.XHTMLTag;

import static org.opengion.fukurou.util.StringUtil.isNull;						// 8.0.2.0 (2021/11/30) Add

/**
 * YM エディターは、カラムのデータを日付(年/月)編集する場合に使用するクラスです。
 * 元の値が8桁の場合は先頭6桁にsubstringされます。
 *
 *  カラムの表示に必要な属性は、DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * 8.0.2.0 (2021/11/30)
 * カラムのパラメータの情報より、日付送り戻しの機能が使えます。
 * 例：M-1,M0,M+1 or M-1,M0,M+1,true
 *
 * 第一引数は、日付戻しを指定します。
 * 第二引数は、初期化します。
 * 第三引数は、日付送りを指定します。
 * 第四引数は、検索ボタンを押すか押さないか[true/false]を指定します。 (初期値:false)
 *
 * 日付についての加減算処理を行うためのコマンドを指定します。
 *  ・SYXX ：年の最初の日付を指定の分だけ進めます。(SY-1なら先年の1月1日、SY1なら翌年の1月1日)
 *  ・SDXX ：月の最初の日付を指定の分だけ進めます。(SD-1なら先月の1日、SD1なら翌月の1日)
 *  ・SWXX ：週初め(月曜日)を指定の分だけ進めます。(SW-1なら先週の月曜日、SW1なら翌週の月曜日)
 *  ・EYXX ：年の最後の日付を指定の分だけ進めます。(EY-1なら先年の年末、EY1なら翌年の年末)
 *  ・EDXX ：月の最後の日付を指定の分だけ進めます。(ED-1なら先月の月末、ED1なら翌月の月末)
 *  ・EWXX ：週末(日曜日)を指定の分だけ進めます。(EW-1なら先週の日曜日、EW1なら翌週の日曜日)
 *  ・YXX  ：年を指定の分だけ進めます。(Y-1なら1年前、Y1なら1年後)
 *  ・MXX  ：月を指定の分だけ進めます。(M-1なら1月前、M1なら1月後)
 *  ・DXX  ：日を指定の分だけ進めます。(D-1なら1日前、D1なら1日後)
 *  ※ 数字がゼロのコマンドは初期化します。
 *  ※ 数字がないコマンドはサーバー上のシステム日付をセットします。
 *
 * @og.rev 5.4.3.6 (2012/01/19) コメント修正
 * @og.rev 8.0.2.0 (2021/11/30) 日付送り戻し対応
 * @og.rev 8.1.2.3 (2022/05/20) 日付送り戻し不具合対応(useDateFeed 属性追加)
 *
 * @og.group データ編集
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Editor_YM extends AbstractEditor {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "8.1.2.3 (2022/05/20)" ;

	// 8.0.2.0 (2021/11/30) 日付送り戻し対応
	private static final int CNT_ARY		= 3;
	private static final String JSP_ICON	= HybsSystem.sys( "JSP_ICON" ) ;
	private static final String FR_STR		= "<a href=\"#\" onClick=\"dateFeedRtn('%1$s','%2$s','%3$s','%4$s','%5$s');\" style=\"margin:0 5px 0 3px\" >";
	private static final String FR_END		= "<img src=\"" + JSP_ICON + "/%6$s\" alt=\"%4$s\" title=\"%4$s\" /></a>";
	private static final String[] FR_IMG	= { "FR_PREV.png", "FR_CIRCLE.png", "FR_NEXT.png" };

	private String isSubm					= "false";							// 検索ボタンを押すか押さないか
	private String errMsg;														// エラーメッセージ
	private String[] prmAry;													// 編集パラメータ

	private final boolean hidden;
	private final boolean useDateFeed;											// 日付送り戻し機能の有効/無効 8.1.2.3 (2022/05/20)

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 8.0.2.0 (2021/11/30) 日付送り戻し対応
	 * @og.rev 8.1.2.3 (2022/05/20) 日付送り戻し不具合対応(useDateFeed 属性追加)
	 *
	 */
//	public Editor_YM() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。
	// 8.0.2.0 (2021/11/30) Modify
	public Editor_YM() {
		super();
		isSubm		= "false";
		errMsg		= null;
		prmAry		= null;
		hidden		= true;
		useDateFeed	= true;														// 8.1.2.3 (2022/05/20)
	}

	/**
	 * DBColumnオブジェクトを指定したprivateコンストラクター。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 3.5.6.0 (2004/06/18) XHTMLTag の 内部配列 INPUT_KEY を隠蔽します。
	 * @og.rev 8.0.2.0 (2021/11/30) 日付送り戻し対応
	 * @og.rev 8.1.2.3 (2022/05/20) 日付送り戻し不具合対応(useDateFeed 属性追加)
	 *
	 * @param	clm	DBColumnオブジェクト
	 */
	private Editor_YM( final DBColumn clm ) {
		super( clm );
		tagBuffer.add( XHTMLTag.inputAttri( attributes ) );

		final boolean disabled = "disabled".equalsIgnoreCase( attributes.get( "disabled" ) );
		final boolean readonly = "readonly".equalsIgnoreCase( attributes.get( "readonly" ) );	// 8.0.2.0 (2021/11/30)
		useDateFeed = clm.isDateFeed();															// 8.1.2.3 (2022/05/20)

		// 8.0.2.0 (2021/11/30) Add 日付送り戻し対応
		hidden = disabled || readonly ;
//		if( !hidden ) {															// 8.1.2.3 (2022/05/20) Modify
		if( !hidden && useDateFeed ) {
			// 例：M-1,M0,M+1 or M-1,M0,M+1,true
			final String prmStr = clm.getEditorParam();
			if( prmStr != null ) {
				prmAry = StringUtil.csv2Array( prmStr );
				// パラメータの第四引数がある場合
				if( prmAry.length > CNT_ARY ) {
					isSubm = prmAry[CNT_ARY];
				} else if( prmAry.length < CNT_ARY ) {
					errMsg = "editorParam の設定が不足です。"
										+ " name="  + name
										+ " label=" + clm.getLabel()
										+ " editorParam=" + clm.getEditorParam();
					System.out.println( errMsg );
				}
			}
		}
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 3.1.2.1 (2003/04/10) synchronized を、削除します。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellEditorオブジェクト
	 * @og.rtnNotNull
	 */
	public CellEditor newInstance( final DBColumn clm ) {
		return new Editor_YM( clm );
	}

	/**
	 * データの編集用文字列を返します。
	 *
	 * @og.rev 8.0.2.0 (2021/11/30) 日付送り戻し対応
	 * @og.rev 8.0.3.0 (2021/12/17) ｲﾒｰｼﾞにname属性を付けるため、spanﾀｸﾞで囲います。
	 * @og.rev 8.1.2.3 (2022/05/20) 日付送り戻し不具合対応(useDateFeed 属性追加)
	 *
	 * @param	value 入力値
	 *
	 * @return	データの編集用文字列
	 */
	@Override
	public String getValue( final String value ) {
		// 8.0.2.0 (2021/11/30) Add
		if( prmAry != null && prmAry.length < CNT_ARY ) {
			return "<span class=\"error\">" + errMsg + "</span>";
		}

		// 6.4.1.1 (2016/01/16) PMD refactoring. A method should have only one exit point, and that should be the last statement in the method
		// 8.0.2.0 (2021/11/30) Delete
//		return value != null && value.length() == 8
//					? super.getValue( value.substring( 0,6 ) )
//					: super.getValue( value );

		// 8.0.2.0 (2021/11/30) Modify
		final String val = ( value != null && value.length() == 8 ) ? value.substring( 0,6 ) : value ;

		final String tag = new TagBuffer( "input" )
						.add( "name"	, name )
						.add( "id"		, name , isNull( attributes.get( "id" ) ) )
						.add( "value"	, val )
						.add( "size"	, size1 )
						.add( tagBuffer.makeTag() )
						.makeTag();

		if( hidden ) {
			return tag;
		} else {
			final StringBuilder buf = new StringBuilder(BUFFER_MIDDLE)
				.append( tag );

			// 日付送り戻し対応
//			if( prmAry != null && prmAry.length > 0 ) {							// 8.1.2.3 (2022/05/20) Modify
			if( useDateFeed && prmAry != null && prmAry.length > 0 ) {
				final String nmid = isNull( attributes.get( "id" ) ) ? name : attributes.get( "id" );
				// 第一回目の処理は、日付戻しを指定します。
				// 第二回目の処理は、初期化します。
				// 第三回目の処理は、日付送りを指定します。
				buf.append( "<span name=\"img" ).append( name ).append( "\">" );	// 8.0.3.0 (2021/12/17)
				for( int i=0; i<CNT_ARY; i++ ) {
					buf.append( String.format( FR_STR + FR_END, "YM", nmid, val, prmAry[i], isSubm, FR_IMG[i] ) );
				}
				buf.append( "</span>" );	// 8.0.3.0 (2021/12/17)
			}
			return buf.toString();
		}
	}

	/**
	 * name属性を変えた、データ表示/編集用のHTML文字列を作成します。
	 * テーブル上の name に 行番号を付加して、名前_行番号 で登録するキーを作成し、
	 * リクエスト情報を１つ毎のフィールドで処理できます。
	 *
	 * @og.rev 7.2.9.4 (2020/11/20) spotbugs:メソッド呼び出しは非 null パラメータに null を渡している
	 * @og.rev 8.0.2.0 (2021/11/30) 日付送り戻し対応
	 * @og.rev 8.0.3.0 (2021/12/17) ｲﾒｰｼﾞにname属性を付けるため、spanﾀｸﾞで囲います。
	 * @og.rev 8.1.2.3 (2022/05/20) 日付送り戻し不具合対応(useDateFeed 属性追加)
	 *
	 * @param	row		行番号
	 * @param	value	入力値
	 *
	 * @return  データ表示/編集用の文字列
	 */
	@Override
	public String getValue( final int row,final String value ) {
		// 8.0.2.0 (2021/11/30) Add
		if( prmAry != null && prmAry.length < CNT_ARY ) {
			return "<span class=\"error\">" + errMsg + " row=" + row + "</span>";
		}

		// 7.2.9.4 (2020/11/20) spotbugs:メソッド呼び出しは非 null パラメータに null を渡している
		final String val = value == null ? ""
										 : value.length() > 6 ? value.substring( 0,6 )
															  : value ;

//		return super.getValue( row,val );										// 8.0.2.0 (2021/11/30) Delete

//		// 6.4.1.1 (2016/01/16) PMD refactoring. A method should have only one exit point, and that should be the last statement in the method
//		return value != null && value.length() == 8
//					? super.getValue( row,value.substring( 0,6 ) )
//					: super.getValue( row,value );

		// 8.0.2.0 (2021/11/30) Modify
		final String name2 = name + HybsSystem.JOINT_STRING + row ;

		final String tag = new TagBuffer( "input" )
						.add( "name"	, name2 )
						.add( "id"		, name2 , isNull( attributes.get( "id" ) ) )
						.add( "value"	, val )
						.add( "size"	, size2 )
						.add( tagBuffer.makeTag() )
						.makeTag( row,val );

		if( hidden ) {
			return tag;
		} else {
			final StringBuilder buf = new StringBuilder(BUFFER_MIDDLE)
				.append( tag );

			// 日付送り戻し対応
//			if( prmAry != null && prmAry.length > 0 ) {							// 8.1.2.3 (2022/05/20) Modify
			if( useDateFeed && prmAry != null && prmAry.length > 0 ) {
				final String nmid = isNull( attributes.get( "id" ) ) ? name2 : attributes.get( "id" );
				// 第一回目の処理は、日付戻しを指定します。
				// 第二回目の処理は、初期化します。
				// 第三回目の処理は、日付送りを指定します。
				buf.append( "<span name=\"img" ).append( name2 ).append( "\">" );	// 8.0.3.0 (2021/12/17)
				for( int i=0; i<CNT_ARY; i++ ) {
					buf.append( String.format( FR_STR + FR_END, "YM", nmid, val, prmAry[i], isSubm, FR_IMG[i] ) );
				}
				buf.append( "</span>" );	// 8.0.3.0 (2021/12/17)
			}
			return buf.toString();
		}
	}
}
