/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.db.AbstractRenderer;
import org.opengion.hayabusa.db.CellRenderer;
import org.opengion.hayabusa.db.DBColumn;

/**
 * MONEY レンデラーは、カラムのデータを金額表示する場合に使用するクラスです。
 *
 * マイナス時の表示は、id="minus" をキーに CSSファイルで指定しています。
 * 通貨は、標準では、￥ですが、値:記号 という形式で指定すれば、各値ごとに
 * 通貨を指定できます。(ただし、通貨変換は、サポートしていません。)
 * 負数の場合はspanタグclass="minus"を付けて出力します。
 *
 * カラムの表示に必要な属性は、DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * 6.8.3.1 (2017/12/01)
 *   小数点で桁ぞろえするため、小数点以下、４桁のスペースと、pre 属性、右寄せを標準に入れます。
 *
 * 6.9.7.0 (2018/05/14)
 *   RendererParam で、記号(￥など)を指定できるようにします。
 *
 * @og.group データ表示
 * @og.rev 5.4.3.6 (2012/01/19) コメント修正
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Renderer_MONEY extends AbstractRenderer {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.9.7.0 (2018/05/14)" ;

	private static final String DEF_KIGO  = "￥" ;	// 6.9.7.0 (2018/05/14)
	private static final String MINUS_MRK = "▲" ;	// 6.9.7.0 (2018/05/14)

	private static final String[] SPACE = {			// 6.8.3.1 (2017/12/01) 追加
							"     ",	// size == 0
							"   ",		// size == 1
							"  ",		// size == 2
							" ",		// size == 3
							""			// size == 4
	};

	private static final CellRenderer[] DB_CELL = {
							new Renderer_MONEY(),
							new Renderer_MONEY("",1,null,DEF_KIGO),
							new Renderer_MONEY("",2,null,DEF_KIGO),
							new Renderer_MONEY("",3,null,DEF_KIGO),		// 6.8.3.1 (2017/12/01) 追加
							new Renderer_MONEY("",4,null,DEF_KIGO)		// 6.8.3.1 (2017/12/01) 追加
	};

	private final String	defValue ;			// 6.8.3.1 (2017/12/01) 追加
	private final int		minFraction;
	private final String	noDisplayVal ;		// 5.6.2.3 (2013/03/22)
	private final String	kigo ;				// 6.9.7.0 (2018/05/14)

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 3.3.0.0 (2003/06/23) 初期値設定追加。
	 * @og.rev 5.6.2.3 (2013/03/22) noDisplayVal 変数初期化
	 * @og.rev 6.9.7.0 (2018/05/14) 記号追加
	 *
	 */
	public Renderer_MONEY() {
//		this( "",0,null );				// 6.0.2.4 (2014/10/17)
		this( "",0,null,DEF_KIGO );		// 6.9.7.0 (2018/05/14) 記号追加
	}

	/**
	 * コンストラクター
	 *
	 * @og.rev 6.0.2.4 (2014/10/17) noDisplayVal 対応漏れのため、追加
	 * @og.rev 6.9.7.0 (2018/05/14) 記号追加
	 *
	 * @param	defval		初期値
	 * @param	size  		小数点
	 * @param	noDispVal  	非表示文字の設定
	 * @param	mkigo	  	マネー記号
	 */
//	private Renderer_MONEY( final String defval , final int size , final String noDispVal ) {
	private Renderer_MONEY( final String defval , final int size , final String noDispVal , final String mkigo ) {
		super();		// 6.4.1.1 (2016/01/16) PMD refactoring. It is a good practice to call super() in a constructor
		defValue     = defval;
		minFraction  = size ;
		noDisplayVal = noDispVal;		// 5.5.1.0 (2012/04/03)
		kigo		 = mkigo;			// 6.9.7.0 (2018/05/14)
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 3.1.2.1 (2003/04/10) synchronized を、削除します。
	 * @og.rev 6.0.4.0 (2014/11/28) 表示は、ViewLength属性を元に行う。
	 * @og.rev 6.8.3.1 (2017/12/01) size（minFraction）の最大値は、4（DB_CELL.length-1） とします。
	 * @og.rev 6.9.7.0 (2018/05/14) 記号追加
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellRendererオブジェクト
	 * @og.rtnNotNull
	 */
	public CellRenderer newInstance( final DBColumn clm ) {
		final String defval = clm.getDefault();

		// 6.3.9.1 (2015/11/27) Found 'DD'-anomaly for variable(PMD)

		// 6.0.4.0 (2014/11/28) 表示は、ViewLength属性があれば、それを使う。
		final String viewLength = clm.getViewLength();
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
		int size ;
		if( viewLength == null ) {
			size = clm.getSizeY();			// 6.3.9.1 (2015/11/27)
		}
		else {
			final int ch = viewLength.indexOf( ',' ) ;		// DBColumn で、"." を "," に変換済み
			size = ch > 0 ? Integer.parseInt( viewLength.substring( ch+1 ) ) : 0 ;
		}

		// 6.8.3.1 (2017/12/01) size（minFraction）の最大値は、4 とします。
		if( size >= DB_CELL.length ) { size = DB_CELL.length - 1; }

		final String noDispVal = clm.getNoDisplayVal();		// 6.0.2.4 (2014/10/17)

		// 6.9.7.0 (2018/05/14) 記号追加
		final String mkigo = StringUtil.nval( clm.getRendererParam() , DEF_KIGO );

		// 6.4.1.1 (2016/01/16) PMD refactoring. A method should have only one exit point, and that should be the last statement in the method
		return noDispVal == null && ( defval == null || defval.isEmpty() )
										? DB_CELL[size]
//										: new Renderer_MONEY( defval,size,noDispVal );
										: new Renderer_MONEY( defval,size,noDispVal,mkigo );		// 6.9.7.0 (2018/05/14) 記号追加
	}

	/**
	 * データの表示用文字列を返します。
	 *
	 * 引数の値が、『数字型文字列:通貨』という値を渡すことで、通貨を
	 * 頭につけて通貨ごとに異なる値を表示させることができる。
	 *
	 * @og.rev 3.1.0.0 (2003/03/20) 内部に、DBColumn オブジェクトをキープしないように変更
	 * @og.rev 3.3.0.0 (2003/06/23) NumberFormatクラスは、廃止します。
	 * @og.rev 5.6.2.3 (2013/03/22) noDisplayVal 変数追加
	 * @og.rev 6.0.4.0 (2014/11/28) ロジックの共通化
	 *
	 * @param	value 入力値(『数字型文字列』 または『数字型文字列:通貨』)
	 *
	 * @return	データの表示用文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final String value ) {
		return getValue( value , true );
	}

	/**
	 * データ出力用の文字列を作成します。
	 * ファイル等に出力する形式を想定しますので、HTMLタグを含まない
	 * データを返します。
	 * 基本は、#getValue( String ) をそのまま返します。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) データ出力用のレンデラー
	 * @og.rev 6.2.0.0 (2015/02/27) そのまま数値化できるように、数値部のみを返します。
	 *
	 * @param	value 入力値(『数字型文字列』 または『数字型文字列:通貨』)
	 *
	 * @return  データ出力用の文字列(数字型文字列 のみ)
	 * @og.rtnNotNull
	 * @see		#getValue( String )
	 */
	@Override
	public String getWriteValue( final String value ) {
		return getValue( value , false );
	}

	/**
	 * データ表示用/出力用の文字列を作成します。
	 * 第二引数の isView == true で、データ表示用文字列を、false で
	 * データ出力用の文字列を作成します。
	 * 処理の共通化を行うためのメソッドです。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) ロジックの共通化
	 * @og.rev 6.2.0.0 (2015/02/27) マイナス金額を、"-" ではなく、"▲" にします。
	 * @og.rev 6.8.3.1 (2017/12/01) 小数点で桁ぞろえするため、小数点以下、４桁のスペースと、pre 属性、右寄せを標準に入れます。
	 * @og.rev 6.9.7.0 (2018/05/14) 記号追加
	 *
	 * @param	value 入力値(『数字型文字列』 または『数字型文字列:通貨』)
	 * @param   isView データ表示用かどうか(true:表示用/false:出力用)
	 *
	 * @return  データ表示用/出力用の文字列
	 * @og.rtnNotNull
	 * @see		#getValue( String )
	 */
	private String getValue( final String value , final boolean isView ) {
		// 5.6.2.3 (2013/03/22) noDisplayVal 変数追加
		if( noDisplayVal != null && noDisplayVal.equalsIgnoreCase( value ) ) { return "" ; }

		String rtn = value == null || value.trim().isEmpty() ? defValue : value ;
		if( rtn == null || rtn.isEmpty() ) { return "" ; }		// 6.8.3.1 (2017/12/01) 

//		String tuuka = "￥";
		String tuuka = kigo;									// 6.9.7.0 (2018/05/14) 記号追加
		final int taniPos = rtn.indexOf( ':' );
		if( taniPos >= 0 ) {
			tuuka = rtn.substring( taniPos+1 );					// rtn に副作用があるため、先に処理
			rtn   = rtn.substring( 0,taniPos );
		}

		boolean minus = false ;
		if( StringUtil.startsChar( rtn , '-' ) ) {				// 6.2.0.0 (2015/02/27) １文字 String.startsWith
			minus = true;
			rtn   = rtn.substring( 1 );							// マイナス記号は取っ払っておきます。
		}

		// 6.2.0.0 (2015/02/27) マイナス金額を、"-" ではなく、"▲" にします。
//		rtn = tuuka + (minus ? "▲" : "" ) + StringUtil.numberFormat( rtn,minFraction ) + "-";
		rtn = tuuka + (minus ? MINUS_MRK : "" ) + StringUtil.numberFormat( rtn,minFraction ) + "-";		// 6.9.7.0 (2018/05/14) "▲" → MINUS_MRK
 
		if( !isView ) { return rtn; }							// 6.2.0.0 (2015/02/27) マイナス記号のまま

		// 6.8.3.1 (2017/12/01) 小数点で桁ぞろえするため、小数点以下、４桁のスペースと、pre 属性、右寄せを標準に入れます。
		rtn = "<pre>" + rtn + SPACE[minFraction] + "</pre>";			// preタグは、custom.css で、固定幅フォントを指定

		if( minus ) {
			rtn = "<span class=\"minus\">" + rtn + "</span>";
		}
		return rtn;
	}

	/**
	 * name属性を変えた、データ表示用のHTML文字列を作成します。
	 * レンデラーのため、row(行番号)は使いません。
	 * 第３引数に、パラメータを渡すことが出来ます。これは、viewMarker で
	 * [$XXXX param] 形式を渡すことで、行単位に表示形式を変更できます。
	 *
	 * @og.rev 6.8.3.1 (2017/12/01) パラメータを渡せるようにします。
	 * @og.rev 6.9.7.0 (2018/05/14) 記号追加
	 *
	 * @param   row   行番号
	 * @param   value 値
	 * @param   param パラメータ
	 *
	 * @return  データ表示/編集用の文字列
	 */
	@Override
	public String getValue( final int row,final String value,final String param ) {
		int size = param == null || param.isEmpty() ? 0 : Integer.parseInt( param );
		if( size >= DB_CELL.length ) { size = DB_CELL.length - 1; }

		final CellRenderer rende = noDisplayVal == null && ( defValue == null || defValue.isEmpty() )
										? DB_CELL[size]
//										: new Renderer_MONEY( defValue,size,noDisplayVal,"￥" );
										: new Renderer_MONEY( defValue,size,noDisplayVal,DEF_KIGO );			// 6.9.7.0 (2018/05/14)

		return rende.getValue( value );
	}
}
