/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.query;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;

import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.AbstractQuery;

/**
 * 指定のSQL文を実行して、検索する Query クラスです。
 *
 * java.sql.Statement を用いて、データベース検索処理を行います。
 * 引数は無しです。(与えられたSQL文を実行します。)
 * 内部変数の受け渡しのデフォルト実装は、AbstractQuery クラスを継承している
 * ため、ここでは、execute() メソッドを実装しています。
 * このクラスでは、ステートメント文を execute() する事により、データベースを
 * 検索した結果を DBTableModel に割り当てます。
 *
 * @og.formSample
 * &lt;og:query command=&quot;{&#064;command}&quot; debug=&quot;false&quot;&gt;
 *     &lt;!-- 先頭のカラム名が、&quot;WRITABLE&quot; の場合、&#39;true&#39; or &#39;1&#39; で、書き込み許可が与えら、&#39;2&#39; でチェック済みになります。--&gt;
 *         select KBSAKU AS WRITABLE,
 *                 CLM,NAME_JA,LABEL_NAME,KBSAKU,SYSTEM_ID,LANG,FGJ
 *         from GE41
 *     &lt;og:where&gt;
 *         &lt;og:and value = &quot;FGJ         in  (&#39;0&#39;,&#39;1&#39;)&quot;         /&gt;
 *         &lt;og:and value = &quot;SYSTEM_ID   like &#39;{&#064;SYSTEM_ID}%&#39;&quot;  /&gt;
 *         &lt;og:and value = &quot;LANG        like &#39;{&#064;LANG}%&#39;&quot;       /&gt;
 *         &lt;og:and value = &quot;CLM         like &#39;{&#064;CLM}%&#39;&quot;        /&gt;
 *         &lt;og:and value = &quot;KBSAKU      =    &#39;{&#064;KBSAKU}&#39;&quot;      /&gt;
 *     &lt;/og:where&gt;
 *     &lt;og:appear startKey = &quot;order by&quot; value = &quot;{&#064;ORDER_BY}&quot;
 *                 defaultVal = &quot;SYSTEM_ID,CLM,LANG&quot; /&gt;
 * &lt;/og:query&gt;
 *
 * @og.group データ表示
 * @og.group データ編集
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Query_JDBC extends AbstractQuery {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.9.3.0 (2018/03/26)" ;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public Query_JDBC() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * クエリーを実行します。
	 * セットされているステートメント文字列とそのタイプが合っていない場合は、
	 * エラーになります。
	 * 実行結果は、DBTableModel にセットされます。
	 *
	 * ※ 6.1.1.0 (2015/01/17) 引数のargsは使いません。
	 *
	 * @og.rev 3.8.0.8 (2005/10/03) エラーメッセージの出力順をメッセージ＋Queryに変更します。
	 * @og.rev 6.1.1.0 (2015/01/17) 引数配列を可変引数にして、execute() を含めて定義します。
	 * @og.rev 6.3.6.1 (2015/08/28) close(),realClose() 廃止。Queryはキャッシュしません。
	 * @og.rev 6.4.2.1 (2016/02/05) try-with-resources 文で記述。
	 * @og.rev 6.9.3.0 (2018/03/26) DB_FETCH_SIZE追加。
	 *
	 * @param   args オブジェクトの引数配列(可変長引数)…ここでは使いません。
	 */
	@Override
	public void execute( final String... args ) {			// 6.1.1.0 (2015/01/17) refactoring
		// 6.4.2.1 (2016/02/05) try-with-resources 文
		try( Statement stmt = getConnection().createStatement() ) {
			stmt.setQueryTimeout( DB_MAX_QUERY_TIMEOUT );
			stmt.setFetchSize( DB_FETCH_SIZE );				// 6.9.3.0 (2018/03/26)

			final boolean status = stmt.execute( getStatement() );

			if( status ) {
				// 6.4.2.1 (2016/02/05) try-with-resources 文
				try( ResultSet resultSet = stmt.getResultSet() ) {
					createTableModel( resultSet );
					setUpdateFlag( false );
				}
			}
			else {
				setExecuteCount( stmt.getUpdateCount() );
			}
			setErrorCode( ErrorMessage.OK );
		}
		catch( final SQLException ex ) {		// catch は、close() されてから呼ばれます。
			setErrorCode( ErrorMessage.EXCEPTION );
			final String errMsg = ex.getMessage() + ":" + ex.getSQLState() + CR
						+ getStatement() + CR;
			throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}
	}
}
