/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.table;

import org.opengion.fukurou.util.StringUtil;				// 8.1.0.3 (2022/01/21)
import org.opengion.fukurou.system.OgBuilder ;				// 8.1.0.3 (2022/01/21)

/**
 * TableFilter_TABLE_ORACLE は、TableUpda インターフェースを継承した、DBTableModel 処理用の
 * 実装クラスです。とくに、ORACLE用のDB定義スクリプトを作成します。
 *
 * ※ 他のデータベースと同じ形で扱いたいため、用意したクラスですが、内容は、TableFilter_TABLE と同じです。
 *
 * ここでは、テーブル一覧の検索結果より、GF05 のテーブルカラム定義テーブルから
 * 必要な情報を取得し、テーブル作成スクリプトを作成します。
 * 出力ファイルは、テーブル名＋"S.sql" という命名規則で作成します。
 * 検索では、(SYSTEM_ID,TBLSYU,TABLE_NAME,NAME_JA,TABLESPACE_NAME,INITIAL_EXTENT,COMMENTS)
 * の項目を取得する必要があります。
 *
 * @og.rev 6.5.0.0 (2016/09/30) 新規作成
 * @og.rev 8.1.0.3 (2022/01/21) TableFilter_TABLE のORACLE部分をTableFilter_TABLE_ORACLEに移動する。
 *
 * @version  6.5.0.0  2016/09/13
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.8,
 */
public class TableFilter_TABLE_ORACLE extends TableFilter_TABLE {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "8.1.0.3 (2022/01/21)" ;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.9.7.0 (2018/05/14) PMD Each class should declare at least one constructor
	 */
	public TableFilter_TABLE_ORACLE() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * ヘッダー部分の処理を実行します。
	 *
	 * @og.rev 8.1.0.3 (2022/01/21) EXEC_SQLに、exists属性追加。
	 *
	 * @param	clmNo	カラム番号配列
	 * @param	data	１行分のデータ配列
	 *
	 * @return	ヘッダー部分の文字列
	 * @og.rtnNotNull
	 */
	@Override
	protected String makeHeadLine( final int[] clmNo,final String[] data ) {
		// 8.1.0.3 (2022/01/21) EXEC_SQLに、exists属性追加。
		execExistsSQL = "SELECT COUNT(*) FROM USER_TABLES WHERE TABLE_NAME=UPPER('" ;	// 大文字で比較

		return super.makeHeadLine( clmNo,data );
	}

	/**
	 * 定義の最後の部分の処理を実行します。
	 *
	 * 6.1.0.0 (2014/12/26) より、
	 *   1．TABLESPACE_NAME を指定しない場合は、TABLESPACE 句を出力しません。
	 *   2．INITIAL_EXTENT を 0 で指定した場合は、STORAGE 句を出力しません。
	 *   3．NEXT と PCTINCREASE は、出力しません。
	 *
	 * @og.rev 6.0.2.3 (2014/10/10) isXml で、CR + EXEC_END_TAG のキャッシュを作成します。
	 * @og.rev 6.1.0.0 (2014/12/26) TABLESPACE_NAME,INITIAL_EXTENT が未設定の場合、設定しません。
	 * @og.rev 6.4.4.1 (2016/03/18) StringBuilderの代わりに、OgBuilderを使用する。
	 * @og.rev 8.1.0.3 (2022/01/21) TableFilter_TABLE のORACLE部分をTableFilter_TABLE_ORACLEに移動する。
	 *
	 * @param	clmNo	カラム番号配列
	 * @param	data	1行分のデータ配列
	 *
	 * @return	定義の最後の部分
	 * @og.rtnNotNull
	 */
	protected String makeEndLine( final int[] clmNo,final String[] data ) {

		// 6.1.0.0 (2014/12/26) TABLESPACE_NAME,INITIAL_EXTENT が未設定の場合、設定しません。
		final String tblSpcse = data[clmNo[TABLESPACE_NAME]] ;
		final String initExt  = data[clmNo[INITIAL_EXTENT]] ;

		return new OgBuilder()
				.append( ")" )
				.appendIf( !StringUtil.isNull( tblSpcse )
							, CR , "TABLESPACE " , tblSpcse )
				.appendIf( !StringUtil.isNull( initExt ) && initExt.charAt(0) != '0'
							, CR , "STORAGE( INITIAL " , initExt , "K )" )
				.append( execEndTag )
				.toString();
	}

	/**
	 * ユニークシーケンスの作成処理を実行します。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) シーケンス名を[TABLE_NAME]S00に変更
	 * @og.rev 6.0.2.3 (2014/10/10) isXml で、CR + EXEC_END_TAG のキャッシュを作成します。
	 * @og.rev 6.4.4.1 (2016/03/18) StringBuilderの代わりに、OgBuilderを使用する。
	 * @og.rev 8.1.0.3 (2022/01/21) 引数にtableNameを渡すように変更。
	 * @og.rev 8.1.0.3 (2022/01/21) TableFilter_TABLE のORACLE部分をTableFilter_TABLE_ORACLEに移動する。
	 *
//	 * @param	clmNo	カラム番号配列
//	 * @param	data	１行分のデータ配列
	 * @param	tableName	ﾃｰﾌﾞﾙ名
	 *
	 * @return	ユニークシーケンス
	 * @og.rtnNotNull
	 */
//	protected String makeUniqSeq( final int[] clmNo,final String[] data ) {
	protected String makeUniqSeq( final String tableName ) {
		final String execSeqSQL = "SELECT COUNT(*) FROM USER_SEQUENCES WHERE SEQUENCE_NAME=UPPER('" ;

		return new OgBuilder()
				.appendCR()
//				.appendIfCR( isXml , EXEC_START_TAG )
//				.appendCR( "CREATE SEQUENCE " , data[clmNo[TABLE_NAME]] , "S00 " )
				.appendIfCR( isXml , EXEC_EXISTS_0_TAG )		//  EXISTS_0 判定は、XML の場合のみ。
				.appendIfCR( isXml , execSeqSQL , tableName , "S00');" )
				.appendCR( "CREATE SEQUENCE "   , tableName , "S00 " )				// 8.1.0.3 (2022/01/21)
				.append( "  INCREMENT BY 1 START WITH 1 MAXVALUE 999999999 CYCLE NOCACHE" )
				.append( execEndTag )
				.toString();
	}

	/**
	 * ユニークシーケンスと関連付けるトリガの作成処理を実行します。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) トリガー名を[TABLE_NAME]T00に変更
	 * @og.rev 6.4.4.1 (2016/03/18) StringBuilderの代わりに、OgBuilderを使用する。
	 * @og.rev 8.1.0.3 (2022/01/21) TableFilter_TABLE のORACLE部分をTableFilter_TABLE_ORACLEに移動する。
	 *
//	 * @param	clmNo	カラム番号配列
//	 * @param	data	１行分のデータ配列
	 * @param	tableName	ﾃｰﾌﾞﾙ名
	 * @param   uniqName	ユニークトリガ名
	 *
	 * @return	ユニークシーケンスと関連付けるトリガ
	 * @og.rtnNotNull
	 */
//	protected String makeUniqTrig( final int[] clmNo,final String[] data, final String uniqName ) {
	protected String makeUniqTrig( final String tableName, final String uniqName ) {
//		final String TBL_NAME = data[clmNo[TABLE_NAME]] ;

		return new OgBuilder()
				.appendCR()
				.appendIfCR( isXml , EXEC_START_TAG )
				.appendCR( "CREATE OR REPLACE TRIGGER "	, tableName , "T00 " )
				.appendCR( "  BEFORE INSERT ON "		, tableName )
				.appendCR( "  FOR EACH ROW " )
				.appendCR( "  BEGIN " )
				.append( "    SELECT "					, tableName , "S00.NEXTVAL INTO :NEW." )
				.appendCR( uniqName , " FROM DUAL; " )
				.appendCR( "  END; " )
				.appendCase( isXml , EXEC_END_TAG , "/" )		// isXmlで選択されます。true/false
				.toString();
	}
}
