/*
 * OpenI2CRADIO
 * UI Handler
 * Copyright (C) 2013-06-10 K.Ohta <whatisthis.sowhat ai gmail.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2,
 *  or (at your option) any later version.
 *  This library / program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  See the GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this library; see the file COPYING. If not, write to the
 *  Free Software Foundation, 51 Franklin Street, Fifth Floor, Boston,
 *  MA 02110-1301, USA.
 *
 *  As a special exception, if you link this(includeed from sdcc) library
 *  with other files, some of which are compiled with SDCC,
 *  to produce an executable, this library does not by itself cause
 *  the resulting executable to be covered by the GNU General Public License.
 *  This exception does not however invalidate any other reasons why
 *  the executable file might be covered by the GNU General Public License.
 */

#include "ioports.h"
#include "ui.h"
#include "idle.h"


const char charcodemap[] = {charcode_0,
                            charcode_1,
                            charcode_4,
                            charcode_7,

                            charcode_f,
                            charcode_2,
                            charcode_5,
                            charcode_8,

                            charcode_e,
                            charcode_3,
                            charcode_6,
                            charcode_9,

                            charcode_d,
                            charcode_c,
                            charcode_b,
                            charcode_a,
};

extern unsigned char pollkeybuf[32];

keyin_defs keyin_old[2];
keyin_defs keyin_now;
char keyin_fifo[32];
char keyin_nowp;
char keyin_readp;
char keyin_counter;

unsigned char cold;

void keyin_init(void)
{
    char i;
    /* Initialize vars*/
    for(i = 0; i < 2; i++) {
        keyin_old[0].byte[i] = 0x00;
        keyin_old[1].byte[i] = 0x00;
        keyin_now.byte[i] = 0x00;
    }
    for(i = 0; i < 32; i++) keyin_fifo[i] = 0x00;
    keyin_nowp = 0;
    keyin_readp = 0;
    keyin_counter = 0;
    cold = charcode_null;
}
/*
 * Push to keyin fifo; not used atomic-writing.
 */
#ifdef __SDCC
void push_keyinfifo(char b) __critical
#else
void push_keyinfifo(char b)
#endif
{
    if(keyin_counter >= 31) {
        return; // Discard data.
    }
    keyin_fifo[keyin_nowp] = b;
    keyin_nowp++;
    keyin_counter++;
    if((keyin_nowp > 31) || (keyin_nowp < 0)) keyin_nowp = 0;
}

/*
 * Pop from keyin fifo; not used atomic-reading.
 */
#ifdef __SDCC
char pop_keyinfifo(void) __critical
#else
char pop_keyinfifo(void)
#endif
{
    char c;
    if(keyin_counter <= 0) {
        keyin_counter = 0;
        return charcode_null ;
    }
    c = keyin_fifo[keyin_readp];
    keyin_readp++;
    keyin_counter--;
    if((keyin_readp > 31) || (keyin_readp < 0)) keyin_readp = 0;
    return c;
}

void printstr(char *s)
{
    int p = 0;
    _CURSOR_RIGHT();
    if(s == NULL) return;
    do {
        if(s[p] == '\0') break;
        _PUTCHAR(s[p]);
        p++;
    } while(p < 255);
}

void print_numeric_nosupress(unsigned int data, unsigned char digit)
{
    unsigned int i;

    if(digit == 0) return;
    if(digit >= 5) digit = 5;
    if(digit == 5){
        i = data / 10000;
        data = data % 10000;
        _PUTCHAR(i + '0');
        digit--;
    }
    if(digit == 4){
        i = data / 1000;
        data = data % 1000;
        _PUTCHAR(i + '0');
        digit--;
    }
    if(digit == 3){
        i = data / 100;
        data = data % 100;
        _PUTCHAR(i + '0');
        digit--;
    }
    if(digit == 2){
        i = data / 10;
        data = data % 10;
        _PUTCHAR(i + '0');
        digit--;
    }
    if(digit == 1){
        i = data;
        _PUTCHAR(i + '0');
        digit--;
    }
}
/*
 * Read Numeric(int)
 */
unsigned int subst_numeric(unsigned int start, unsigned char pos, unsigned int c)
{
    unsigned int val;
    char i;
    unsigned int fact;
    unsigned char bcd[5];

    if(pos > 4) pos = 4;
    if((pos == 4) && (c > 6)) return start;
    val = start;
    // binary to BCD
    fact = 10000;
    for(i = 4; i >= 0; i--){
        bcd[i] = val / fact;
        val = val % fact;
        fact = fact / 10;
    }
    // subst
    bcd[pos] = c;
    val = 0;
    for(i = 3; i >= 0; i--){
        val = val * 10;
        val = val + bcd[i];
    }
    if((bcd[4] == 6) && (val > 5535)) return val;
    val = val + bcd[4] * 10000;
    return val;
}

unsigned int read_numeric(unsigned int initial, unsigned char digit,
        char startx, char starty)
{
    unsigned char c;
    unsigned char n;
    char i;
    unsigned int val;
    char d;

    d = digit - 1;
    val = initial;
    i = d;
    do {
       ClrWdt();
        _LOCATE(startx, starty);
        print_numeric_nosupress(val, digit);
       ClrWdt();
        
       do {
           n = pollkeys(pollkeybuf, 60, 0);
       } while(n == 0);
       c = pollkeybuf[0];

        if(c == charcode_0){
            val = subst_numeric(val, i, 0);
            i--;
        } else if((c >= charcode_1) && (c <= charcode_9)) {
            val = subst_numeric(val, i, c - charcode_1 + 1);
            i--;
        } else if(c == charcode_f) {
            // Enter
            break;
        } else if(c == charcode_a) {
            // Del
            val = val / 10;
            i++;
        } else if(c == charcode_b) {
            // cancel
            val = initial;
            i = d;
            break;
        }  else if(c == charcode_e) {
            i++;
        } else if(c == charcode_d) {
            i--;
        }
       if(i <= 0) i = 0;
       if(i > d) i = d;
    } while(1);
    return val;
}

unsigned char readkey_compare(void)
{
    char b;
    char c;
    char d;
    char e;
    unsigned char shift;
    unsigned char f;
    f = 0;
    e = 0;
    for(d = 0; d < 2; d++) {
        shift = 0x01;
        for(b = 0; b < 8; b++){
            c = 0;
            if((keyin_now.byte[d] & shift) != 0) c++;
            if((keyin_old[0].byte[d] & shift) != 0) c++;
            if((keyin_old[1].byte[d] & shift) != 0) c++;
            if(c >= 2) {
            /*
             * Clear older-inputs on .
             */
                f |= 1;
                keyin_old[0].byte[d] &= ~shift;
                keyin_old[1].byte[d] &= ~shift;
                keyin_now.byte[d] &= ~shift;
                push_keyinfifo(charcodemap[e]);
            }
            shift <<= 1;
            e++;
        }
    }
    /**/
    return f;
}

unsigned char readkey(void)
{
    unsigned char i;
    for(i = 0; i < 9; i++) {
        idle_time_ms(2); // 2ms
        readkey_io(i);
        ClrWdt();
    }
    readkey_compare();
    return pop_keyinfifo();
}

/*
 * Polling key
 * Max = 32bytes;
 * 0 = Timeout
 * 1~32 = Received.
 * if((limit * 23ms) elapsed), break;
 */
unsigned char pollkeys(unsigned char *p, unsigned int limit, unsigned char repeat)
{
    unsigned int count = 0;
    unsigned int lifetime = 0;
    unsigned int penalty = 0;
    unsigned char c;

    do {
        idle_time_ms(5); // 5ms.
        c = readkey(); //
        ClrWdt();
        if(c != charcode_null) {
            push_keyinfifo(c);
            do {
                ClrWdt();
                c = pop_keyinfifo();
                if(c == charcode_null) {
                    break;
                }
                if(c != cold) {
                    p[count++] = c;
                    cold = c;
                }
            } while(count < 32);
            penalty = 0;
        } else {
            penalty++;
            if((limit > 3) && (penalty > (limit >> 2))){
                penalty = 0;
                cold = charcode_null;
            }
        }
        if(limit != 0) lifetime++;
        if(lifetime > limit) break;
    } while(count < 32);
    if(repeat != 0) cold = charcode_null;
    return count;
}

unsigned char pollkey_single(void)
{
    unsigned int penalty = 0;
    unsigned char c;

    cold = charcode_null;
    do {
        idle_time_ms(5); // 0.125 * 4 * 20 = 10ms.
        c = readkey(); // 2 * 9 = 18ms
        ClrWdt();
        if(c != charcode_null) {
            if(cold != c){
                cold = c;
                return c;
            }
        }
        penalty++;
        if(penalty > 20) cold = charcode_null;
    } while(1);
}

/*
 * Notes:
 * Initialize sequence:
 * keyin_init();
 * keyin_ioinit();
 * 
 * Read-key-sequence:
 * In interrupt/unsleep hook(call per Nms):
 * readkey_io();
 * readkey_compare();
 *
 * In application handler:
 * c = pop_keyinfifo();
 * if(c != 0) do(c);
 */