/*
 * 
 * Licensed Materials - Property of IBM
 *
 * Open Platform Trust Services - An open source TCG PTS
 *
 * (C) Copyright International Business Machines Corp. 2007
 *
 */

package com.ibm.trl.tcg.pts.vulnerability.tool;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.TreeMap;
import java.util.Vector;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.ibatis.sqlmap.client.SqlMapClient;
import com.ibm.trl.tcg.pts.ibatis.SqlConfigIidbCreater;
import com.ibm.trl.tcg.pts.ibatis.dto.Measurements;

/**
 * Some Digest Lists (File Path, Digest Value) by openssl sha1 came from
 * makeRpmSha1.sh List the files which have the same name Arrange them in order
 * of the version number Attach the vulnerability info Compare the hash value
 * with historical value.
 * 
 * @author Megumi Nakamura
 * 
 */
public class CheckVulByDigest {

	/* Logger */
	private Log log = LogFactory.getLog(this.getClass());

	private FileWriter fileWriter = null;

	private BufferedWriter bufWriter = null;

	private SqlMapClient sqlMapIidb = null;

	/**
	 * @param args
	 *            Not used
	 */
	public static void main(String[] args) {
		String packageName = "firefox";

		CheckVulByDigest checker = new CheckVulByDigest(1);
		checker.runCheckDigestsetWrite2File(packageName, "firefox/rhel4",
				"firefox/rhel4");
	}

	/**
	 * Run checking the digest.
	 * 
	 * @param packageName
	 *            The name of package
	 */
	public void runCheckDigestWithDB(String packageName) {
		Vector packageSet = searchPackageFromDB(packageName);
		checkDigestWithDB(packageSet);
	}

	/**
	 * Run checking the digest, and writing the result to the file.
	 * 
	 * @param packageName
	 *            The name of package
	 * @param outputDir
	 *            The directory name to store the output files
	 * @param digestDir
	 *            The directory name which contains the files to be checked
	 */
	public void runCheckDigestsetWrite2File(String packageName,
			String outputDir, String digestDir) {
		try {
			if (outputDir != null) {
				writeFileOpen(outputDir + File.separator + packageName + ".txt");
			} else {
				writeFileOpen(packageName + ".txt");
			}
			Vector fileDigestSet = null;
			if (digestDir != null) {
				fileDigestSet = getDigestSetFromDir(digestDir + File.separator,
						packageName);
			} else {
				fileDigestSet = getDigestSetFromDir(packageName
						+ File.separator, packageName);
			}
			HashMap fileSet = searchFileFromDB(packageName);
			checkDigest(fileSet, fileDigestSet);
			writeFileClose();
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Run checking the digest, and writing the result to the file.
	 * 
	 * @param packageName
	 *            The name of package
	 * @param outputDir
	 *            The directory name to store the output files
	 * @param digestFile
	 *            The file name to be checked
	 */
	public void runCheckDigestsWrite2File(String packageName, String outputDir,
			String digestFile) {
		try {
			if (outputDir != null) {
				writeFileOpen(outputDir + File.separator + packageName + ".txt");
			} else {
				writeFileOpen(packageName + ".txt");
			}
			TreeMap fileDigests = getDigestFromFile(digestFile, packageName);
			HashMap fileSet = searchFileFromDB(packageName);
			checkDigest(fileSet, fileDigests);
			writeFileClose();
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Constructor.
	 * 
	 * @param dbIndex
	 *            ID of Integrity database
	 */
	public CheckVulByDigest(int dbIndex) {
		sqlMapIidb = SqlConfigIidbCreater.getSqlMapInstance(dbIndex);
	}

	/**
	 * Setup the file to write.
	 * 
	 * @param outputFile
	 *            The file contains the result
	 */
	private void writeFileOpen(String outputFile) {
		try {
			fileWriter = new FileWriter(outputFile);
			bufWriter = new BufferedWriter(fileWriter);
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Close the file.
	 */
	private void writeFileClose() {
		try {
			bufWriter.flush();
			bufWriter.close();
			fileWriter.close();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Write a line.
	 * 
	 * @param line
	 *            The line to be written in a file
	 */
	private void writeFileLine(String line) {
		try {
			bufWriter.write(line);
			bufWriter.newLine();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Print for debug: dbname/package/cve/version from packageSet,
	 * filename/digest from fileDigests.
	 * 
	 * @param packageSet
	 *            The Vector of PackageDigest class
	 */
	private void checkDigestWithDB(Vector packageSet) {
		String dbname = null;
		String packageName = null;
		String cve = null;
		String version = null;
		String file = null;
		String digest = null;
		for (int i = 0; i < packageSet.size(); i++) {
			PackageDigest packageDigest = (PackageDigest) packageSet.get(i);
			dbname = packageDigest.getDatabaseName();
			packageName = packageDigest.getPackageName();
			cve = packageDigest.getCve();
			version = packageDigest.getVersion();
			if (log.isDebugEnabled()) {
				log.debug("Database\t" + dbname);
				log.debug("Package\t" + packageName);
				log.debug("Cve\t" + cve);
				log.debug("Ver\t" + version);
			}

			TreeMap fileDigestsFromDB = (TreeMap) packageDigest
					.getFileDigests();
			Iterator it = fileDigestsFromDB.keySet().iterator();
			for (int j = 0; j < fileDigestsFromDB.size(); j++) {
				file = it.next().toString();
				digest = fileDigestsFromDB.get(file).toString();
				if (log.isDebugEnabled()) {
					log.debug("Key\t" + file);
					log.debug("Value\t" + digest);
				}
			}
		}
	}

	/**
	 * Count how many times the number appeared, and Write to the file.
	 * 
	 * @param fileSet
	 *            The data from the database. HashMap of {filename, cve} from
	 *            searchFileFromDB
	 * @param fileDigestSet
	 *            The Vector of PackageDigest class for packageName, from the
	 *            files
	 */
	private void checkDigest(HashMap fileSet, Vector fileDigestSet) {
		String filename = null;
		String version = null;
		// String cve = null;
		String digest = null;
		HashMap digests = null; // to count duplicate
		int num = 0;
		Iterator it = fileSet.keySet().iterator();
		for (int i = 0; i < fileSet.size(); i++) {
			filename = it.next().toString();
			// cve = fileSet.get(filename).toString();
			digests = new HashMap();
			if (filename != null) {
				for (int j = 0; j < fileDigestSet.size(); j++) {
					PackageDigest packageDigest = (PackageDigest) fileDigestSet
							.get(j);
					version = packageDigest.getVersion();
					TreeMap fileDigests = packageDigest.getFileDigests();
					digest = (String) fileDigests.get(filename);
					if (digest != null) {
						// writeFileLine("File: "+filename);
						// writeFileLine("Cve: "+cve);
						// System.out.println("File: "+filename);
						// System.out.println("Cve: "+cve);
						if (digests.containsKey(digest)) {
							num = Integer
									.parseInt((String) digests.get(digest)) + 1;
							digests.remove(digest);
							digests.put(digest, Integer.toString(num));
						} else {
							num = 0;
							digests.put(digest, Integer.toString(num));
						}
						writeFileLine(filename + "> " + ":\t" + digest + " ["
								+ num + "] " + version);
						if (log.isDebugEnabled()) {
							log.debug(filename + "> " + ":\t" + digest + " ["
									+ num + "] " + version);
						}
					}
				}
			}
		}
	}

	/**
	 * Print for debug: filename/cve from fileSet, filename/digest from
	 * fileDigests.
	 * 
	 * @param fileSet
	 *            The data from the database. HashMap of {filename, cve} from
	 *            searchFileFromDB
	 * @param fileDigests
	 *            The data from file. The TreeMap of {filename, digest} from
	 *            getDigestFromFile
	 */
	private void checkDigest(HashMap fileSet, TreeMap fileDigests) {
		String filename = null;
		String cve = null;
		String digest = null;
		Iterator it = fileSet.keySet().iterator();
		for (int i = 0; i < fileSet.size(); i++) {
			filename = it.next().toString();
			cve = (String) fileSet.get(filename);
			if (filename != null) {
				digest = (String) fileDigests.get(filename);
				if (digest != null) {
					if (log.isDebugEnabled()) {
						log.debug("File: " + filename + ", Cve: " + cve
								+ ", SHA1: " + digest);
					}
					writeFileLine("File: " + filename + ", Cve: " + cve
							+ ", SHA1: " + digest);
				}
			}
		}
	}

	/**
	 * List the {filename, cve} to HashMap which came from the all database.
	 * 
	 * @param packageName
	 *            The package name
	 * @return The HashMap of {filename, cve} for packageName
	 */
	private HashMap searchFileFromDB(String packageName) {
		HashMap fileSet = new HashMap();

		String filename = null;
		String cve = null;
		try {
			List<Measurements> list = (List<Measurements>) sqlMapIidb
					.queryForList("getPackageDigest", packageName);
			for (Measurements m : list) {
				filename = m.getDigestName();
				cve = m.getPackages().getPackageCve();
				filename = replaceVersion(packageName, filename);
				fileSet.put(filename, cve);
			}
		} catch (SQLException ex) {
			do {
				log.error("SQLSTATE: " + ex.getSQLState());
				log.error("ERR-CODE: " + ex.getErrorCode());
				log.error("ERR-MSEG: " + ex.getMessage());
				ex = ex.getNextException();
			} while (null != ex);
		}
		return fileSet;
	}

	/**
	 * Pack the PackageDigest class which came from the all database.
	 * 
	 * @param packageName
	 *            The package name
	 * @return The Vector of PackageDigest class for packageName
	 */
	private Vector searchPackageFromDB(String packageName) {

		Vector packageSet = new Vector();
		PackageDigest packageDigest = new PackageDigest();

		String filename = null;
		String version = null;
		String digest = null;
		String cve = null;
		try {
			List<Measurements> list = (List<Measurements>) sqlMapIidb
					.queryForList("getPackageDigest", packageName);
			for (Measurements m : list) {
				version = m.getPackages().getPackageVersion();
				cve = m.getPackages().getPackageCve();
				filename = m.getDigestName();
				digest = m.getDigest();
				filename = replaceVersion(packageName, filename);
				packageDigest.setVersion(version);
				packageDigest.setCve(cve);
				packageDigest.setFileDigests(filename, digest);
			}
		} catch (SQLException ex) {
			do {
				log.error("SQLSTATE: " + ex.getSQLState());
				log.error("ERR-CODE: " + ex.getErrorCode());
				log.error("ERR-MSEG: " + ex.getMessage());
				ex = ex.getNextException();
			} while (null != ex);
		}
		packageSet.add(packageDigest);
		return packageSet;
	}

	/**
	 * Remove the version number in the file/directory path.
	 * 
	 * @param packageName
	 *            The package name
	 * @param filename
	 *            The target file
	 * @return The file/directory path without the version number
	 */
	protected String replaceVersion(String packageName, String filename) {
		String separator = "/";
		String[] tmp = filename.split(separator);
		String rtn = null;
		for (int i = 0; i < tmp.length; i++) {
			if (tmp[i].matches(".*-" + packageName + "-[0-9.-]+")
					&& i < tmp.length - 1) { // seamonkey
				tmp[i] = packageName;
			} else if (tmp[i].startsWith(packageName + "-")
					&& i < tmp.length - 1) {
				tmp[i] = packageName;
			} else if (packageName.startsWith("kernel")
					&& tmp[i].matches("[0-9.]+.+")) { // kernel
				tmp[i] = packageName;
			}
			if (i != 0) {
				if (rtn == null) {
					rtn = separator + tmp[i];
				} else {
					rtn = rtn + separator + tmp[i];
				}
			}
		}
		return rtn;
	}

	/**
	 * Pack the PackageDigest class which came from the files in the directory.
	 * 
	 * @param inputDir
	 *            The directory which contains the files of SHA1
	 * @param packageName
	 *            The package name
	 * @return The Vector of PackageDigest class for packageName
	 * @throws FileNotFoundException
	 *             The file to be checked is not found.
	 */
	private Vector getDigestSetFromDir(String inputDir, String packageName)
			throws FileNotFoundException {
		Vector fileDigestSet = new Vector();
		File dir = new File(inputDir);
		File[] files = dir.listFiles();
		TreeMap fileDigests = null;

		for (int i = 0; i < files.length; i++) {
			File file = files[i];
			fileDigests = getDigestFromFile(inputDir + file.getName(),
					packageName);
			PackageDigest packageDigest = new PackageDigest();
			packageDigest.setFileDigests(fileDigests);
			packageDigest.setVersion(file.getName());
			fileDigestSet.add(packageDigest);

		}
		return fileDigestSet;
	}

	/**
	 * Translate one file to TreeMap(file, digest).
	 * 
	 * @param inputFile
	 *            The file which contains filename and SHA1 of one package
	 * @param packageName
	 *            The package name
	 * @return TreeMap(filename, digest)
	 * @throws FileNotFoundException
	 *             The file to be checked is not found.
	 */
	private TreeMap getDigestFromFile(String inputFile, String packageName)
			throws FileNotFoundException {
		TreeMap fileDigests = new TreeMap();
		try {
			BufferedReader bufReader = new BufferedReader(new FileReader(
					inputFile));
			String line = null;
			String file = null;
			String digest = null;
			// lines -> line
			while ((line = bufReader.readLine()) != null) {
				String[] tmp = line.split("=");
				file = tmp[0].replace("SHA1(.", " ").replace(")", " ").trim();
				file = replaceVersion(packageName, file);
				digest = tmp[1].trim();
				fileDigests.put(file, digest);
			}
			if (bufReader != null) {
				bufReader.close();
			}
		} catch (IOException e) {
			e.printStackTrace();
		}
		return fileDigests;
	}

}

/**
 * Information of One Package.
 * 
 * @author nakamegu
 */
class PackageDigest {
	private String databaseName = null;

	private String packageName = null;

	private String cve = null;

	private String version = null;

	private TreeMap fileDigests = new TreeMap(); // filename, digest

	public String getDatabaseName() {
		return databaseName;
	}

	public void setDatabaseName(String name) {
		this.databaseName = name;
	}

	public String getCve() {
		return cve;
	}

	public void setCve(String data) {
		this.cve = data;
	}

	public TreeMap getFileDigests() {
		return fileDigests;
	}

	public void setFileDigests(TreeMap digests) {
		this.fileDigests = digests;
	}

	public void setFileDigests(String filename, String digest) {
		this.fileDigests.put(filename, digest);
	}

	public String getPackageName() {
		return packageName;
	}

	public void setPackageName(String name) {
		this.packageName = name;
	}

	public String getVersion() {
		return version;
	}

	public void setVersion(String ver) {
		this.version = ver;
	}

}
