/*
 * 
 * Licensed Materials - Property of IBM
 *
 * Open Platform Trust Services - An open source TCG PTS
 *
 * (C) Copyright International Business Machines Corp. 2007
 *
 */
package com.ibm.trl.tcg.pts.vulnerability.deb;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.sql.SQLException;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.ibatis.sqlmap.client.SqlMapClient;
import com.ibm.trl.tcg.pts.ibatis.SqlConfigVul;
import com.ibm.trl.tcg.pts.ibatis.dto.CveDefinitions;
import com.ibm.trl.tcg.pts.ibatis.dto.Debians;
import com.ibm.trl.tcg.pts.vulnerability.tool.PackageVersionTool;

/**
 * Check the vulnerability by comparing with database
 * 
 * @author Megumi Nakamura
 * 
 */
public class CheckDpkgVulnerability {

	/* Logger */
	private Log log = LogFactory.getLog(this.getClass());

	// DB
	private SqlMapClient sqlMapVul = SqlConfigVul.getSqlMapInstance();

	// File
	private FileWriter fileWriter = null;

	private BufferedWriter bufWriter = null;

	private static int lineNum = 0;

	private int goodPackageNum = 0;

	private int badPackageNum = 0;

	private int unknownPackageNum = 0;

	private int defaultValue = -2;

	/**
	 * @param args
	 */
	public static void main(String[] args) {
		// String inputFile = null;
		String inputFile = "testdata/debian/debian3.1";
		String outputFile = "sandbox/debian31.out.txt";

		CheckDpkgVulnerability checker = new CheckDpkgVulnerability();
		try {
			checker.run(inputFile, outputFile);
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	/**
	 * kick start.
	 * 
	 * @param inputFile
	 *            The output of dpkg -l
	 * @param outputFile
	 *            The output file
	 * @throws Exception
	 *             All exceptions
	 */
	public void run(String inputFile, String outputFile) throws Exception {
		try {
			sqlMapVul.startTransaction();
		} catch (SQLException e) {
			e.printStackTrace();
		}

		writeFileOpen(outputFile);

		getPackagesFromFile(inputFile);

		if (log.isInfoEnabled()) {
			log.info("=== Summary ===");
			log.info("OK:\t" + goodPackageNum);
			log.info("NG:\t" + badPackageNum);
			log.info("Unknown:\t" + unknownPackageNum);
		}

		writeFileClose();

	}

	/**
	 * Read each line of the input file, and pass to getCompareWithVulDB().
	 * 
	 * @param inputFile
	 *            The output of dpkg -l
	 * @return Exception All exceptions
	 */
	private void getPackagesFromFile(String inputFile) throws Exception {
		boolean start = false;
		BufferedReader bufReader = null;
		try {
			bufReader = new BufferedReader(new FileReader(inputFile));
			String line = null;
			// lines -> line
			while ((line = bufReader.readLine()) != null) {
				if (start) {
					String[] packageVersion = getPackageVersion(line);
					getCompareWithVulDB(packageVersion[0], packageVersion[1]);
					writeFileLineCountup();
				}
				if (!start
						&& line
								.startsWith("+++-===============================")) {
					start = true;
				}
			}
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		} finally {
			bufReader.close();
		}
	}

	/**
	 * Check the vulnerability by comparing with database.
	 * 
	 * @param packageName
	 *            The package name
	 * @param version
	 *            The package version
	 * @return CVE ID
	 */
	protected String getCompareWithVulDB(String packageName, String version) {
		if (fileWriter != null && bufWriter != null) {
			writeFileLine("name=" + packageName);
			writeFileLine("version=" + version);
		}
		try {
			List<Debians> debians = (List<Debians>) sqlMapVul.queryForList(
					"getDebianByPackageName", packageName);
			int etchStatus = defaultValue;
			int sargeStatus = defaultValue;
			int sidStatus = defaultValue;
			String etchFixedVersion = null;
			String sargeFixedVersion = null;
			String sidFixedVersion = null;
			if (debians.size() != 0) {
				Debians d = debians.get(0);
				etchFixedVersion = d.getDebianFixedEtch();
				sargeFixedVersion = d.getDebianFixedSarge();
				sidFixedVersion = d.getDebianFixedSid();
				// compare with database
				etchStatus = checkVulnerability(etchFixedVersion, version);
				sargeStatus = checkVulnerability(sargeFixedVersion, version);
				sidStatus = checkVulnerability(sidFixedVersion, version);

				if ((etchStatus == 0 || etchStatus == defaultValue)
						&& (sargeStatus == 0 || sargeStatus == defaultValue)
						&& (sidStatus == 0 || sidStatus == defaultValue)) {

					if (log.isDebugEnabled()) {
						log.debug("package : " + d.getDebianPackageName());
						log.debug("dsa     : " + d.getDebianDsaId());
						log.debug("etch    : " + etchFixedVersion);
						log.debug("sarge   : " + sargeFixedVersion);
						log.debug("sid     : " + sidStatus);
						log.debug("mine    : " + version);
					}

					writeFileLine("vulnerability=no");
					goodPackageNum++;

				} else {
					// status 0 means "safe" at that version.
					if (etchStatus == 0) {
						if (fileWriter != null && bufWriter != null) {
							writeFileLine("vulnerability=no");
						}
						if (log.isDebugEnabled()) {
							log.debug(packageName + "\t" + version + "\tok.\t"
									+ etchFixedVersion + " (etch)");
						}
					}
					if (sargeStatus == 0) {
						if (fileWriter != null && bufWriter != null) {
							writeFileLine("vulnerability=no");
						}
						if (log.isDebugEnabled()) {
							log.debug(packageName + "\t" + version + "\tok.\t"
									+ sargeFixedVersion + " (sarge)");
						}
					}
					if (sidStatus == 0) {
						if (fileWriter != null && bufWriter != null) {
							writeFileLine("vulnerability=no");
						}
						if (log.isDebugEnabled()) {
							log.debug(packageName + "\t" + version + "\tok.\t"
									+ sidFixedVersion + " (sid)");
						}
					}
					// status 1 means "vulnerable" at that version.
					if (etchStatus == 1) {
						if (fileWriter != null && bufWriter != null) {
							writeFileLine("vulnerability=yes");
							writeFileLine("vulnerability.etch="
									+ etchFixedVersion);
						}
						if (log.isDebugEnabled()) {
							log.debug(packageName + "\t" + version + "\tvul\t"
									+ etchFixedVersion + " (etch)");
						}
					}
					if (sargeStatus == 1) {
						if (fileWriter != null && bufWriter != null) {
							writeFileLine("vulnerability=yes");
							writeFileLine("vulnerability.sarge="
									+ sargeFixedVersion);
						}
						if (log.isDebugEnabled()) {
							log.debug(packageName + "\t" + version + "\tvul\t"
									+ sargeFixedVersion + " (sarge)");
						}
					}
					if (sidStatus == 1) {
						if (fileWriter != null && bufWriter != null) {
							writeFileLine("vulnerability=yes");
							writeFileLine("vulnerability.sid="
									+ sidFixedVersion);
						}
						if (log.isDebugEnabled()) {
							log.debug(packageName + "\t" + version + "\tvul\t"
									+ sidFixedVersion + " (sid)");
						}
					}
					// status -1 means "unknown" (it couldn't be checked by
					// program.)
					if (etchStatus == -1 && etchFixedVersion != null) {
						if (fileWriter != null && bufWriter != null) {
							writeFileLine("vulnerability=unknown");
							writeFileLine("vulnerability.etch="
									+ etchFixedVersion);
						}
						if (log.isDebugEnabled()) {
							log.debug(packageName + "\t" + version + "\t???\t"
									+ etchFixedVersion + " (etch)");
						}
					}
					if (sargeStatus == -1 && sargeFixedVersion != null) {
						if (fileWriter != null && bufWriter != null) {
							writeFileLine("vulnerability=unknown");
							writeFileLine("vulnerability.sarge="
									+ sargeFixedVersion);
						}
						if (log.isDebugEnabled()) {
							log.debug(packageName + "\t" + version + "\t???\t"
									+ sargeFixedVersion + " (sarge)");
						}
					}
					if (sidStatus == -1 && sidFixedVersion != null) {
						if (fileWriter != null && bufWriter != null) {
							writeFileLine("vulnerability=unknown");
							writeFileLine("vulnerability.sid="
									+ sidFixedVersion);
						}
						if (log.isDebugEnabled()) {
							log.debug(packageName + "\t" + version + "\t???\t"
									+ sidFixedVersion + " (sid)");
						}
					}

					String cve = d.getDebianCve();
					String cvss = getCvss(cve);

					if (log.isDebugEnabled()) {
						log.debug("package : " + d.getDebianPackageName());
						log.debug("dsa     : " + d.getDebianDsaId());
						log.debug("cve     : " + cve);
						log.debug("cvss    : " + cvss);
					}
					badPackageNum++;
				}
				return d.getDebianCve();
			} else {
				writeFileLine("vulnerability=unknown");
				unknownPackageNum++;
			}

		} catch (SQLException e) {
			e.printStackTrace();
		}
		return null;
	}

	/**
	 * Get the CVSS Score from CVE_IDs.
	 * 
	 * @param cveId
	 *            CVE ID
	 * @return CVSS Score
	 */
	private String getCvss(String cveId) {
		String cvss = "";
		if (fileWriter != null && bufWriter != null) {
			writeFileLine("vulnerability.cve=" + cveId);
		}

		if (cveId != null) {
			String[] cves = cveId.split(" ");
			for (int i = 0; i < cves.length; i++) {
				String str = getInfoFromVULDB(cves[i].trim());
				if (str == null) {
					cvss += "n/a ";
				} else {
					cvss += str + " ";
				}
			}
		}
		return cvss;
	}

	/**
	 * Get the information about cve_id from the vulnerability database.
	 * 
	 * @param cveId
	 *            The CVE ID
	 * @return CVSS Score
	 */
	protected String getInfoFromVULDB(String cveId) {
		String nvdCvssScore = "n/a";

		if (fileWriter != null && bufWriter != null) {
			writeFileLine("vulnerability.cve=" + cveId);
		}

		try {
			CveDefinitions cveDef = (CveDefinitions) sqlMapVul.queryForObject(
					"getCveDefinitionByCveId", cveId);
			if (cveDef != null) {
				nvdCvssScore = cveDef.getNvdCvssScore();
				if (fileWriter != null && bufWriter != null) {
					writeFileLine("vulnerability.cvss.score=" + nvdCvssScore);
				}
			}
		} catch (SQLException e) {
			e.printStackTrace();
		}
		return nvdCvssScore;
	}

	/**
	 * Compare two versions.
	 * 
	 * @param fixedVersion
	 *            The version number from database
	 * @param currentVersion
	 *            The version number from input file
	 * @return 0(safe), 1(vulnerable), -1(couldn't check), -2(null param)
	 */
	private int checkVulnerability(String fixedVersion, String currentVersion) {
		if (currentVersion == null || fixedVersion == null) {
			return defaultValue; // Null Error
		}
		int status = PackageVersionTool.compareTwoVersion(fixedVersion,
				currentVersion);
		if (status == 0) { // EQUAL
			return 0; // Safe
		} else if (status == 1) { // GREATER
			return 1; // Vulnerable
		} else if (status == -1) { // LESS
			return 0; // Safe
		} else {
			return -1; // Couldn't check
		}
	}

	/**
	 * Get package name and package version from one line from output of dpkg
	 * -l. example: "ii firefox 2.0.0.1+dfsg-1 Transition package for iceweasel
	 * rename"
	 * 
	 * @param line
	 *            One line from output of dpkg -l (input file)
	 * @return String[0]:Package name, String[1]:Package version
	 */
	private String[] getPackageVersion(String line) {
		String[] tmp = line.split(" +");
		String[] rtn = new String[2];
		if (tmp.length > 2) {
			rtn[0] = tmp[1];
			rtn[1] = tmp[2];
		}
		return rtn;
	}

	/**
	 * Setup the file to write.
	 * 
	 * @param outputFile
	 *            which contains the result
	 */
	private void writeFileOpen(String outputFile) {
		try {
			fileWriter = new FileWriter(outputFile);
			bufWriter = new BufferedWriter(fileWriter);
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Close the file.
	 */
	private void writeFileClose() {
		try {
			bufWriter.flush();
			bufWriter.close();
			fileWriter.close();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	/**
	 * The number written in the file. Rpm order.
	 */
	private void writeFileLineCountup() {
		lineNum++;
	}

	/**
	 * Write one line with "package.i.".
	 * 
	 * @param line
	 *            The value that follows after package.i.
	 */
	private void writeFileLine(String line) {
		try {
			if (line != null) {
				bufWriter.write("package." + lineNum + "." + line);
				bufWriter.newLine();
			}
		} catch (IOException e) {
			e.printStackTrace();
		}
	}
}
