/*
 * 
 * Licensed Materials - Property of IBM
 *
 * Open Platform Trust Services - An open source TCG PTS
 *
 * (C) Copyright International Business Machines Corp. 2007
 *
 */
package com.ibm.trl.tcg.pts.vulnerability.deb;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.sql.SQLException;
import java.util.List;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.ibatis.sqlmap.client.SqlMapClient;
import com.ibm.trl.tcg.pts.ibatis.SqlConfigIidbCreater;
import com.ibm.trl.tcg.pts.ibatis.SqlConfigVul;
import com.ibm.trl.tcg.pts.ibatis.dto.Packages;
import com.ibm.trl.tcg.pts.vulnerability.oval.ReadOvalRedhat;
import com.ibm.trl.tcg.pts.vulnerability.tool.MakeTM3File;

/**
 * Make TM3 file for visualization of the vulnerabiltiy map for Debian packages.
 * The input data is from perl script in tools/package/deb/deb-tm3.pl
 * 
 * @author Megumi Nakamura
 */
public class MakeTM3OfDebVulnerability {

	/* Logger */
	private Log log = LogFactory.getLog(this.getClass());

	// For visualize
	private MakeTM3File makeTM3;

	// DB
	private SqlMapClient sqlMapVul = SqlConfigVul.getSqlMapInstance();

	private SqlMapClient sqlMapIidb = null;

	// Input file
	private Properties inputProp = new Properties();

	private InputStream inStream = null;

	// Counter
	private static int lineNum = 0;

	/**
	 * Main method to start to load the xmlfile, parse them, and store into the
	 * database.
	 * 
	 * @param args --dbindex, --dbindex, --output, --dir
	 */
	public static void main(String[] args) {
		String inputFile = null;
		String outputFile = null;
		String outputDir = null;
		int dbIndex = 0;

		for (int i = 0; i < args.length; ++i) {
			if ("--dbindex".equals(args[i])) {
				dbIndex = Integer.valueOf(args[++i]);
			} else if ("--input".equals(args[i]) || "-i".equals(args[i])) {
				inputFile = args[++i];
			} else if ("--output".equals(args[i]) || "-o".equals(args[i])) {
				outputFile = args[++i];
			} else if ("--dir".equals(args[i]) || "-d".equals(args[i])) {
				outputDir = args[++i];
			} else {
				System.err.println("Unknown option " + args[i]);
				usage();
				return;
			}
		}
		if (inputFile == null) {
			usage();
			return;
		}
		if (outputFile == null) {
			outputFile = inputFile + ".tm3";
		}

		try {
			MakeTM3OfDebVulnerability checker = new MakeTM3OfDebVulnerability(dbIndex);
			checker.run(inputFile, outputFile, outputDir);
			checker.finish();
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	/**
	 * @param dbIndex Index for Integrity Info DB
	 */
	public MakeTM3OfDebVulnerability(int dbIndex) {
		sqlMapIidb = SqlConfigIidbCreater.getSqlMapInstance(dbIndex);
	}

	/**
	 * Print the usage for main method.
	 */
	private static void usage() {
		System.out.println("usage: maketm3deb -i tm3-data.txt -o output.tm3");
		System.out.println("--dbindex: (required)");
		System.out.println("--input, -i: (required)");
		System.out.println("	The output from the following command -");
		System.out.println("	$ perl deb-tm3.pl");
		System.out.println("  ex. -i tm3-data.txt");
		System.out.println("--output, -o:");
		System.out.println("	Output file");
		System.out.println("--dir, -d:");
		System.out.println("	Output directory");
		System.out.println(" ");
	}

	/**
	 * Running the all process of preparing the file, checking the version,
	 * writing down in the file.
	 * 
	 * @param inputFile
	 *            Filename which contains the output from the command of rpm -qa
	 * @param outputFile
	 *            Filename for result output
	 * @throws Exception
	 */
	public void run(String inputFile, String outputFile) throws Exception {
		run(inputFile, outputFile, null);
	}

	/**
	 * Running the all process of preparing the file, checking the version,
	 * writing down in the file.
	 * @param inputFile Filename which contains the output from the command of rpm -qa
	 * @param outputFile Filename for result output
	 * @param outputDir Directory for output files
	 * @throws Exception
	 */
	public void run(String inputFile, String outputFile, String outputDir)
			throws Exception {
		try {
			sqlMapVul.startTransaction();
			sqlMapIidb.startTransaction();

		} catch (SQLException ex) {
			do {
				log.error("SQLSTATE: " + ex.getSQLState());
				log.error("ERR-CODE: " + ex.getErrorCode());
				log.error("ERR-MSEG: " + ex.getMessage());
				ex = ex.getNextException();
			} while (null != ex);
		}

		writeFileOpen(outputFile, outputDir);
		getDPKGfromFile(inputFile);
	}

	/**
	 * Close the file stream.
	 */
	public void finish() {
		writeFileClose();

		try {
			sqlMapVul.commitTransaction();
			sqlMapIidb.commitTransaction();

		} catch (SQLException ex) {
			do {
				log.error("SQLSTATE: " + ex.getSQLState());
				log.error("ERR-CODE: " + ex.getErrorCode());
				log.error("ERR-MSEG: " + ex.getMessage());
				ex = ex.getNextException();
			} while (null != ex);
		}
	}

	/**
	 * Read the file, and store each line into Vector.
	 * 
	 * @param inputFile
	 *            Filename contains the output from the command of rpm -qa
	 */
	private void getDPKGfromFile(String inputFile) {
		try {
			inStream = new FileInputStream(new File(inputFile));
			inputProp.load(inStream);

			Packages pkg = null;

			int i = 1;
			String name = null;
			String version = null;
			String size = null;
			String majorGroup = null;
			String requiredNum = null;
			while (!inputProp.isEmpty()) {
				name = inputProp.getProperty("Name." + i);
				if (name == null) {
					break;
				} else {
					name = name.trim();
				}
				version = inputProp.getProperty("Version." + i).trim();
				size = inputProp.getProperty("Size." + i).trim();
				majorGroup = inputProp.getProperty("MajorGroup." + i);
				if (majorGroup != null) {
					majorGroup = majorGroup.trim();
				}
				requiredNum = inputProp.getProperty("Required_Num." + i);
				if (requiredNum != null) {
					requiredNum = requiredNum.trim();
				} else {
					requiredNum = "0";
				}

				// insert num of required-package
				pkg = new Packages();
				pkg.setPackageName(name);
				pkg.setPackageVersion(version);
				pkg.setPackageSize(new Integer(size).intValue());
				pkg
						.setPackageDependencyNum(new Integer(requiredNum)
								.intValue());
				sqlMapIidb.update("updatePackageByNameVersion", pkg);
				sqlMapIidb.commitTransaction();

				i++;

				List<Packages> pkgs = (List<Packages>) sqlMapIidb.queryForList(
						"getPackageByNameVersion", pkg);
				if (pkgs != null) {
					if (pkgs.size() > 0) {
						Packages p = pkgs.get(0);
						int vul = p.getPackageVulnerability();
						if (vul == ReadOvalRedhat.VUL_DISTRO_MATCH) {
							makeTM3.setIntegrity(p.getPackageVulnerability());
							makeTM3.setCvss(p.getPackageCvssMax());
						}
					}
				}

				makeTM3.setLoc(size);
				makeTM3.setLayer("Runtime\t" + majorGroup + "\t" + name + "-"
						+ version);
				makeTM3.setRequiredNum(requiredNum);

				writeFileLineCountup();
			}
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		} catch (SQLException e) {
			e.printStackTrace();
		}

	}

	/**
	 * Setup the file to write.
	 * 
	 * @param outputFile
	 *            Filename for result output
	 * @param outputDir Directory for output file
	 */
	private void writeFileOpen(String outputFile, String outputDir)
			throws Exception {
		makeTM3 = new MakeTM3File(outputFile, outputDir);
	}

	/**
	 * Close the file.
	 */
	private void writeFileClose() {
		makeTM3.writeFileClose();
	}

	/**
	 * The number written in the file. Rpm order.
	 */
	private void writeFileLineCountup() {
		lineNum++;
		makeTM3.writeVariable();
	}
}
