/*
 * 
 * Licensed Materials - Property of IBM
 *
 * Open Platform Trust Services - An open source TCG PTS
 *
 * (C) Copyright International Business Machines Corp. 2007
 *
 */
package com.ibm.trl.tcg.pts.vulnerability.rpm;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.sql.SQLException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.ibatis.sqlmap.client.SqlMapClient;
import com.ibm.trl.tcg.pts.ibatis.SqlConfigIidbCreater;
import com.ibm.trl.tcg.pts.ibatis.dto.Measurements;
import com.ibm.trl.tcg.pts.ibatis.dto.Packages;
import com.ibm.trl.tcg.pts.vulnerability.tool.DbOperations;

/**
 * Copy data files (.metadata, .sha1.filelist, .md5.filelist) to database. These
 * files are made by perl scripts of tools/rpm/(rpm-file.pl, rpm-meta.pl)
 * 
 * @author Megumi Nakamura
 */
public class CopyActualRpmPackageData2DB {

	/* Logger */
	private Log log = LogFactory.getLog(this.getClass());

	private SqlMapClient sqlMapIidb = null;

	private int dbIndex;

	public CopyActualRpmPackageData2DB(int dbIndex) {
		this.dbIndex = dbIndex;
		sqlMapIidb = SqlConfigIidbCreater.getSqlMapInstance(dbIndex);
	}

	/**
	 * Run.
	 * 
	 * @param metaDir
	 *            The directory name of data files
	 * @throws Exception
	 *             All exceptions
	 */
	public void run(String metaDir) throws Exception {
		try {
			sqlMapIidb.startTransaction();
		} catch (SQLException e) {
			e.printStackTrace();
		}

		getFileFromDir(metaDir);

		new DbOperations().updateObsolete(dbIndex);
	}

	/**
	 * Start from Metadata directory.
	 * 
	 * @param dirPath
	 *            Directory path which stores data files
	 * @throws Exception
	 *             All exceptions
	 */
	private void getFileFromDir(String dirPath) throws Exception {
		File dir = new File(dirPath);
		File[] files = dir.listFiles();

		for (int i = 0; i < files.length; i++) {
			File file = files[i];
			if (file.getName().endsWith(".metadata")) {
				if (file.exists() && file.canRead() && file.isFile()) {
					getMetadata(file);
				}
			}
		}
	}

	/**
	 * Read Metadata File and Insert into database.
	 * 
	 * @param file
	 *            The file of metadata
	 */
	private void getMetadata(File file) {
		Packages packages = new Packages();
		try {
			String pkgName = null;
			String pkgVersion = null;
			String pkgRelease = null;

			FileReader fileReader = new FileReader(file);
			BufferedReader bufReader = new BufferedReader(fileReader);
			String line = null;
			while ((line = bufReader.readLine()) != null) {
				if (line.startsWith("Name:")) {
					pkgName = line.substring(5).trim();
					packages.setPackageName(pkgName);

				} else if (line.startsWith("Version:")) {
					pkgVersion = line.substring(8).trim();
					packages.setPackageVersion(pkgVersion);

				} else if (line.startsWith("Release:")) {
					pkgRelease = line.substring(8).trim();
					if (pkgVersion != null) {
						pkgVersion = pkgVersion + "-" + pkgRelease;
					}
					packages.setPackageVersion(pkgVersion);

				} else if (line.startsWith("Vendor:")) {
					packages.setPackageVendor(line.substring(7).trim());

				} else if (line.startsWith("Build:")) {
					long build = changeTime(line.substring(6).trim());
					packages.setPackageBuild(build);

				} else if (line.startsWith("Summary:")) {
					if (line.substring(12).trim().length() > 255) {
						packages.setPackageSummary(line.substring(8, 255)
								.trim());
					} else {
						packages.setPackageSummary(line.substring(8).trim());
					}

				}
			}
			bufReader.close();
			fileReader.close();

			// Insert into database

			packages.setPackageArch("i386");

			// If not exist, Insert. If exist, Update.
			List<Packages> pkgs = (List<Packages>) sqlMapIidb.queryForList(
					"getPackageByNameVerArchBuild", packages);
			if (pkgs.size() > 0) {
				for (Packages p : pkgs) {
					int packageId = p.getPackageId();
					packages.setPackageId(packageId);
				}
				sqlMapIidb.update("updatePackageById", packages);
				sqlMapIidb.commitTransaction();

			} else {
				sqlMapIidb.insert("insertPackage", packages);
				sqlMapIidb.commitTransaction();
				int packageId = ((Packages)sqlMapIidb.queryForObject("getPackageIdInserted", packages)).getPackageId();
				// Measurements
				String sha1filename = file.getPath().replaceAll(".metadata",
						".sha1.filelist");
				String md5filename = file.getPath().replaceAll(".metadata",
						".md5.filelist");
				getMeasurements(sha1filename, packageId);
				getMeasurements(md5filename, packageId);
			}

		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		} catch (SQLException ex) {
			do {
				log.error("SQLSTATE: " + ex.getSQLState());
				log.error("ERR-CODE: " + ex.getErrorCode());
				log.error("ERR-MSEG: " + ex.getMessage());
				ex = ex.getNextException();
			} while (null != ex);
		}
	}

	/**
	 * Get Measurements and Insert into database.
	 * 
	 * @param filename
	 *            The file of sha1.filelist and md5.filelist
	 * @param packageId
	 *            Package_id used in database
	 */
	private void getMeasurements(String filename, int packageId) {
		try {
			File file = new File(filename);
			BufferedReader bufReader = null;
			if (file.exists() && file.canRead() && file.isFile()) {
				bufReader = new BufferedReader(new FileReader(file));
				Measurements measurements = new Measurements();
				String line = null;
				while ((line = bufReader.readLine()) != null) {
					String[] digestPath = line.split(" +");
					measurements.setDigest(digestPath[0]);
					measurements.setDigestName(digestPath[1]);

					if (filename.indexOf("md5") > 0) {
						measurements.setDigestAlg("md5");
					} else if (filename.indexOf("sha1") > 0) {
						measurements.setDigestAlg("sha1");
					}
					measurements.setDigestPackageId(packageId);

					sqlMapIidb.insert("insertDigest", measurements);
					sqlMapIidb.commitTransaction();
				}
			}
			if (bufReader != null) {
				bufReader.close();
			}

		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		} catch (SQLException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Get time(msec) from 1970/1/1 00:00.
	 * 
	 * @param source
	 *            Wed 30 Aug 2006 10:34:15 PM JST
	 * @return Time in long format
	 */
	private long changeTime(String source) {
		try {
			String[] s = source.split(" +");
			SimpleDateFormat dateFormat = new SimpleDateFormat();
			dateFormat.setLenient(true);
			Date dat = null;
			if (s.length == 7) { // Fri 31 Aug 2007 10:29:06 AM JST
				dateFormat.applyPattern("dd MM yyyy kk:mm:ss");
				dat = dateFormat.parse(s[1] + " " + changeMonth(s[2]) + " "
						+ s[3] + " " + changeAmPm(s[4], s[5]));
			} else if (s.length == 5) { // Fri Nov 17 03:14:35 2006
				dateFormat.applyPattern("dd MM yyyy kk:mm:ss");
				dat = dateFormat.parse(s[2] + " " + changeMonth(s[1]) + " "
						+ s[4] + " " + s[3]);
			}
			if (dat != null) {
				return dat.getTime();
			} else {
				return 0;
			}
		} catch (ParseException e) {
			e.printStackTrace();
		}
		return 0;
	}

	/**
	 * Change Time in 12 hours to in 24 hours.
	 * 
	 * @param time
	 *            Time in 12 hours
	 * @param ampm
	 *            AM or PM
	 * @return Time in 24 hours
	 */
	private String changeAmPm(String time, String ampm) {
		String[] times = time.split(":");
		int hour = Integer.valueOf(times[0]).intValue();
		if (ampm.equalsIgnoreCase("PM")) {
			hour += 12;
			times[0] = Integer.toString(hour);
		}
		return times[0] + ":" + times[1] + ":" + times[2];
	}

	/**
	 * Change month of three characters(Jan-Dec) to digit string(01-12).
	 * 
	 * @param month
	 *            The month of three characters(Jan-Dec)
	 * @return The month of digit string(01-12)
	 */
	private String changeMonth(String month) {
		if (month.equalsIgnoreCase("Apr")) {
			return "04";
		} else if (month.equalsIgnoreCase("May")) {
			return "05";
		} else if (month.equalsIgnoreCase("Jun")) {
			return "06";
		} else if (month.equalsIgnoreCase("Jul")) {
			return "07";
		} else if (month.equalsIgnoreCase("Aug")) {
			return "08";
		} else if (month.equalsIgnoreCase("Sep")) {
			return "09";
		} else if (month.equalsIgnoreCase("Oct")) {
			return "10";
		} else if (month.equalsIgnoreCase("Nov")) {
			return "11";
		} else if (month.equalsIgnoreCase("Dec")) {
			return "12";
		} else if (month.equalsIgnoreCase("Jan")) {
			return "01";
		} else if (month.equalsIgnoreCase("Feb")) {
			return "02";
		} else if (month.equalsIgnoreCase("Mar")) {
			return "03";
		}
		return null;
	}

	/**
	 * Main.
	 * 
	 * @param args --dbindex, --inputdir
	 */
	public static void main(String[] args) {
		String inputDir = null;
		int dbIndex = 0;
		
		for (int i = 0; i < args.length; ++i) {
			if ("--dbindex".equals(args[i])) {
				dbIndex = Integer.valueOf(args[++i]);
			} else if ("--inputdir".equals(args[i]) || "-i".equals(args[i])) {
				inputDir = args[++i];
			} else {
				System.err.println("Unknown option " + args[i]);
				usage();
				return;
			}
		}
		try {
			new CopyActualRpmPackageData2DB(dbIndex).run(inputDir);
			
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	/**
	 * Print the usage for main method.
	 */
	private static void usage() {
		System.out.println("usage: java rpmimport --dbindex 1 -i testdata/debian/all");
		System.out.println("--dbindex: (required)");
		System.out.println("--inputdir, -i: (required)");
		System.out.println("	Input file which contains the metadata files");
		System.out.println(" ");
	}

}
