/*
 * 
 * Licensed Materials - Property of IBM
 *
 * Open Platform Trust Services - An open source TCG PTS
 *
 * (C) Copyright International Business Machines Corp. 2011
 *
 */

package com.ibm.trl.tcg.pts.attestation;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;


/**
 * IF-M TLV messages used by OpenPTS
 * 
 * @author Seiji Munetoh (munetoh@users.sourceforge.jp)
 *
 */
public class IFM {
	/* Logger */
	private Log log = LogFactory.getLog(this.getClass());
	
	/* flag */
	
	/* vid (SMI PEN) */
	public final static int PEN_TCG           = 0x005597;
	public final static int PEN_OPENPTS       = 0x000000;  // TBD
	
	/* message type (ref: openpts - include/openpts_ifm.h) */
	public final static int OPENPTS_CAPABILITY           = 0x00000001;
	public final static int DH_NONCE_PARAMETERS_REQUEST  = 0x00000002;
	public final static int DH_NONCE_PARAMETORS_RESPONSE = 0x00000003;
	public final static int DH_NONCE_FINISH              = 0x00000004;
	public final static int REQUEST_RIMM_SET             = 0x00000005;
	public final static int RIMM_SET                     = 0x00000006;
	public final static int REQUEST_INTEGRITY_REPORT     = 0x00000007;
	public final static int INTEGRITY_REPORT             = 0x00000008;
	public final static int VERIFICATION_RESULT          = 0x00000009;	
	public final static int NONCE                        = 0x00000010;
	public final static int REQUEST_NEW_RIMM_SET         = 0x0000000A;
	public final static int NEW_RIMM_SET                 = 0x0000000B;
	public final static int REQUEST_TPM_PUBKEY           = 0x00040000;
	public final static int TPM_PUBKEY                   = 0x00050000;
	public final static int REQUEST_AIDE_DATABASE        = 0x00020000;
	public final static int AIDE_DATABASE                = 0x00030000;

	/* I/O stream */
	private DataInputStream _in;
	private DataOutputStream _out;
	
	/* TLV */
	private int _type;
	private int _vid;
	private int _flag;
	private int _size;
	private byte[] _value;
	//private byte[][] _values;

	/**
	 * 
	 * @param in
	 * @param out
	 */
	public IFM(DataInputStream in, DataOutputStream out) {
		_in = in;
		_out = out;
	}

	/**
	 * Send TLV massage (byte array)
	 * 
	 * Flag|VID
	 * Type
	 * Length
	 * data[length]
	 * 
	 * @param out
	 * @param flag
	 * @param id
	 * @param type
	 * @param msg
	 * @throws IOException 
	 */
	void sendMessage(int flag, int vid, int type,
			byte[] msg) throws IOException {
		int size=0;
		
		log.info("sendMassage type : " + type);
		
		int fvid = ((flag & 0xFF) <<24) + vid;
		_out.writeInt(fvid);
		_out.writeInt(type);
		
		if (msg == null) {
			size=0;
			_out.writeInt(size);
		} else {
			size=msg.length;
			_out.writeInt(size);
			_out.write(msg);
			log.info("sendMassage size : " + size);	
		}
		_out.flush();
	}
	
	/**
	 * Send TLV massage (BASE64 string => byte array)
	 * 
	 * Flag|VID
	 * Type
	 * Length
	 * num
	 * len0
	 * data0[len0]
	 * len1
	 * data1[len1]
	 * 
	 * 
	 * @param flag
	 * @param vid
	 * @param type
	 * @param base64str
	 * @throws IOException 
	 */
	public void sendMessage2(int flag, int vid, int type, String[] base64str) throws IOException {
		int size=0;
		
		log.info("sendMassage : " + type);	
		
		int fvid = ((flag & 0xFF) <<24) + vid;
		_out.writeInt(fvid);
		_out.writeInt(type);
		
		if (base64str == null) {
			size=0;
			_out.writeInt(size);
			_out.flush();
		} else {
			size = 4 + 4*base64str.length;
			for (int i=0;i<base64str.length;i++) {
				size+=base64str[i].length();
			}
			_out.writeInt(size); // total size
			log.info("sendMassage total size =  " + size);
			
			_out.writeInt(base64str.length); // num
			_out.flush();
			
			for (int i=0;i<base64str.length;i++) {	
				size=base64str[i].length();
				_out.writeInt(size); // size
				_out.write(base64str[i].getBytes());
				log.info("sendMassage " + i + " " + size);	
				_out.flush();
			}
		}
	}


	/**
	 * get TLV message
	 * 
	 * 
	 * @throws IOException
	 * @throws InterruptedException
	 */
	public void getMessage() throws IOException, InterruptedException {
		
        int fvid = _in.readInt();
        _flag = (0xFF000000 & fvid) >> 24;
        _vid = (0x00FFFFFF & fvid);
        _type = _in.readInt();
        int len;
        
        _size = _in.readInt();

        log.info("getMessage fvid : " + fvid);
        log.info("getMessage flag : " + _flag);
        log.info("getMessage vid  : " + _vid);
        log.info("getMessage type : " + _type);
        log.info("getMessage size : " + _size);
        
		if (_size > 0) {
			_value = new byte[_size];	        
			int loc=0;
			int size = _size;
			int sum=0;
			while (true) {
				len = _in.read(_value, loc, size);
				sum+=len;
				size = size - len;
				if (size <= 0) break;
				loc += len;
			}

			if (_size != sum) {
				System.err.println("ERROR " + _size + " but " + sum);
				throw new IOException();
			}

			log.info("getMessage sum = :" + sum);
		}
	}
	
	/**
	 * get Type
	 * 
	 * @return
	 */
	public int getType() {
		return _type;
	}
	
	/**
	 * get Value
	 * 
	 * @return
	 */
	public byte[] getMsg() {
		return _value;
	}

	
	/**
	 * get Value in String
	 *  (msg -> String[])
	 * 
	 * UINT32 num_of_msg
	 * UINT32 msg_size    // 0
	 * BYTE*  msg         // 0
	 * UINT32 msg_size    // 1
	 * BYTE*  msg         // 1
	 * ...
	 * 
	 * @return
	 * @throws Exception 
	 */
	public String[] getStrings() throws Exception {
		int loc=0;
		int num = byte2int(_value,loc,4);
		loc+=4;
		
		log.info("getStrings num = "  + num);	
		
		String[] str = new String[num];
		
		
		for (int i=0;i<num;i++){
			int len = byte2int(_value,loc,4);
			loc+=4;
			log.info("getStrings " + i + " len = " + len);	
			
			if (len > _size) {
				System.err.println("len = " + len + " too BIG");
				
				throw new Exception();
			}
			
			str[i] = new String(_value,loc,len);
			loc+=len;
		}
		return str;
	}

	/**
	 * 
	 * get int from byte array (Big Endian)
	 * 
	 * @param msg
	 * @param offset
	 * @param len
	 * @return
	 */
	private int byte2int(byte[] msg, int offset, int len) {
		//ByteBuffer bw = new ByteBuffer(msg,offset,len);		
		int val=0;
		
		switch(len) {
		case 4:
			// 4 bytes  UINT32
			val  = ((int)msg[offset] << 24) & 0xFF000000;
			val += ((int)msg[offset+1] << 16) & 0xFF0000;
			val += ((int)msg[offset+2] << 8) & 0xFF00;
			val += ((int)msg[offset+3]) & 0xFF;
			break;
		case 3:
			// 4 bytes  UINT32
			val += ((int)msg[offset] << 16) & 0xFF0000;
			val += ((int)msg[offset+1] << 8) & 0xFF00;
			val += ((int)msg[offset+2]) & 0xFF;
			break;
		case 2:
			// 4 bytes  UINT32
			val += ((int)msg[offset  ] << 8) & 0xFF00;
			val += ((int)msg[offset+1]) & 0xFF;
			break;
		case 1:
			// 4 bytes  UINT32
			val += ((int)msg[offset]) & 0xFF;
			break;
		default:
			// INTERNAL ERROR
			break;
		}
		if (len == 4) {

		} else 
		
		log.info(
				(byte)msg[offset] + " " + 
				(byte)msg[offset+1] + " " +
				(byte)msg[offset+2] + " " +
				(byte)msg[offset+3] + " " +
				" > " + val);
		
		return val;
	}

	/**
	 * message must be an UTF-8
	 * 
	 * @return
	 * @throws UnsupportedEncodingException
	 */
	public String getString() throws UnsupportedEncodingException {
		String str;

		str = new String(_value, 0, _size, "UTF-8");
	
		return str;
	}
}
