/*
 * 
 * Licensed Materials - Property of IBM
 *
 * Open Platform Trust Services - An open source TCG PTS
 *
 * (C) Copyright International Business Machines Corp. 2011
 *
 */

package com.ibm.trl.tcg.pts.attestation;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
//import java.io.InputStream;
//import java.io.OutputStream;
//import java.net.Socket;
//import java.net.UnknownHostException;
import java.util.Random;
import java.util.UUID;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.ibm.trl.tcg.pts.tools.HexTool;
import com.jcraft.jsch.JSchException;

/**
 * IF-M with ptsc on SSH
 * 
 * @author Seiji Munetoh (munetoh@users.sourceforge.jp)
 *
 */
public class PTS {
	/* Logger */
	private Log log = LogFactory.getLog(this.getClass());
	
	/* */
	private DataOutputStream _out;
	private DataInputStream _in;
	private IFM _msg;
	public int type;
	public byte[] value;
	public int length;
	public UUID colector_uuid = null;
	public UUID rm_uuid = null;
	private byte[] nonce;
	private SSH _ssh;
	
	/**
	 * 
	 * PTS IF-M on SSH
	 * 
	 * @param hostname
	 * @param sshUsername
	 * @param sshPort
	 * @throws Exception
	 */
	public PTS(String hostname, String sshUsername, int sshPort) throws Exception {
		_ssh = new SSH(hostname, sshUsername, sshPort);
	}

	/**
	 * 
	 * Open SSH connection
	 * 
	 * @throws JSchException
	 * @throws IOException
	 */
	public void open() throws JSchException, IOException {
		_ssh.open();
	    _out = new DataOutputStream(new BufferedOutputStream(_ssh.getOutputStream()));
	    _in = new DataInputStream(new BufferedInputStream(_ssh.getInputStream()));
	    _msg = new IFM(_in, _out);
	}
	
	/**
	 * close SSH connection
	 */
	public void close() {
		_ssh.close();
	}

	/**
	 * 
	 * @throws IOException
	 * @throws InterruptedException
	 */
	public void receiveMessage() throws IOException, InterruptedException {
    	_msg.getMessage();
    	type = _msg.getType();
    	value = _msg.getMsg();
    	length = value.length;
	}

	
	/**
	 * send OPENPTS_CAPABILITY message
	 * @throws IOException
	 */
	public void sendCapability() throws IOException {
        int flag = 0;
        int id = 0;
        byte[] capability;
        
        // version >= 0.2.3
        //  0:3   flag[4]
        //  4:7   TPM_VERSION[4]
        //  8:11   TSS_VERSION[4]
        // 12:15  PTS_VERSION[4]
        // 16:31  PLATFORM_UUID[16]
        // 32:47  MANIFEST_UUID[16]
        capability = new byte[48];
                
        log.info("v ----> OPENPTS_CAPABILITY ----> C");
        _msg.sendMessage(flag,id,IFM.OPENPTS_CAPABILITY,capability);		
	}


	/**
	 * 
	 */
	public void getCapablity() {
	    long msb = 0;
	    long lsb = 0;
        // version >= 0.2.3
        //  0:3   TPM_VERSION[4]
        //  4:7   TSS_VERSION[4]
        //  8:11  PTS_VERSION[4]
        // 12:27  PLATFORM_UUID[16]
        // 28:43  MANIFEST_UUID[16]
        // v.0.2.4
        //  0:3   FLAG[4]
        //  4:7   TPM_VERSION[4]
        //  8:11  TSS_VERSION[4]
        // 12:15  PTS_VERSION[4]
        // 16:31  PLATFORM_UUID[16]
        // 32:47  MANIFEST_UUID[16]
	    // TODO check flag
	    // TODO check version
	    for (int i = 16; i < 24; i++)
	            msb = (msb << 8) | (value[i] & 0xff);
	    for (int i = 24; i < 32; i++)
	            lsb = (lsb << 8) | (value[i] & 0xff);
	    colector_uuid = new UUID(msb, lsb);

	    for (int i = 32; i < 40; i++)
            msb = (msb << 8) | (value[i] & 0xff);
	    for (int i = 40; i < 48; i++)
            lsb = (lsb << 8) | (value[i] & 0xff);
	    rm_uuid = new UUID(msb, lsb);

	    log.info("UUID    : " + colector_uuid.toString());
	    log.info("RM UUID : " + rm_uuid.toString());       
	}

	/**
	 * send NONCE to  collector
	 * verify this nonce later at VerificarionResult
	 * @throws IOException 
	 * 
	 */
	public byte[] sendNonce() throws IOException {
        int flag = 0;
        int id = 0;
        
        /* random */
        Random rand = new Random();
        nonce = new byte[20];
        rand.nextBytes(nonce);
        
        log.info("v ----> NONCE ----> C");
        log.trace("  nonce      : " + HexTool.getHexString(nonce));
        
        _msg.sendMessage(flag,id,IFM.NONCE,nonce);
                
        return nonce;
	}

	/**
	 * send REQUEST_INTEGRITY_REPORT message to collector
	 * @throws IOException
	 */
	public void sendRequestIntegrityReport() throws IOException {
        int flag = 0;
        int id = 0;

        log.info("v ----> REQUEST_INTEGRITY_REPORT ----> C");
        _msg.sendMessage(flag,id,IFM.REQUEST_INTEGRITY_REPORT, null);
	}

	/**
	 * send REQUEST_RIMM_SET message to collector
	 * @throws IOException
	 */
	public void sendRequestManifestSets() throws IOException {
        int flag = 0;
        int id = 0;

        log.info("v ----> REQUEST_RIMM_SET ----> C");
        _msg.sendMessage(flag, id, IFM.REQUEST_RIMM_SET, null);
	}

	/**
	 * send REQUEST_TPM_PUBKEY message to collector
	 * @throws IOException
	 */
	public void sendRequestTpmPubkey() throws IOException {
        int flag = 0;
        int id = 0;

        log.info("v ----> REQUEST_TPM_PUBKEY ----> C");
        _msg.sendMessage(flag,id,IFM.REQUEST_TPM_PUBKEY, null);
	}
}
