/*
 * 
 * Licensed Materials - Property of IBM
 *
 * Open Platform Trust Services - An open source TCG PTS
 *
 * (C) Copyright International Business Machines Corp. 2007, 2011
 *
 */

package com.ibm.trl.tcg.pts.engine;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.ibm.trl.tcg.pts.tools.Base64Tool;
import com.ibm.trl.tcg.pts.tools.DigestTool;
import com.ibm.trl.tcg.pts.tools.HexTool;

/**
 * @author Seiji Munetoh (munetoh@users.sourceforge.jp)
 * 
 */
public class Event {

	/* Logger */
	private Log log = LogFactory.getLog(this.getClass());

	/* BIOS */
	private long _pcrIndex;
	private long _eventType;
	private byte[] _digest;
	private byte[] _eventData;

	/* PC Platform (eventType==6) */
	private long _pcEventID;
	private long _pcEventDataSize;
	private byte[] _pcEventData;

	// Transition by
	public static final long DIGEST = 0;
	public static final long EVENTTYPE = 1;
	public long guardType = 0;

	// Digest Value 0
	// Event Type 1
	private boolean _printBase64;

	/* For Event Validation */

	private boolean _enableValidation = false;
	private boolean _integrity = true;
	private String _message;
	private String _str;

	/**
	 * Null event
	 */
	public Event() {
	}
	
	/**
	 * 
	 * @param pcrindex
	 * @param type
	 * @param digest
	 * @param eventdata
	 */
	public Event(int pcrindex, long type, byte[] digest, byte[] eventdata) {
		set(pcrindex,type, digest, eventdata);
	}

	/**
	 * 
	 * @param pcrindex
	 * @param type
	 * @param digest2
	 * @param eventdata
	 */
	public void set(int pcrindex, long type, byte[] digest2, byte[] eventdata) {
		_pcrIndex = pcrindex;
		_eventType = type;
		_digest = digest2;
		
		if (eventdata == null) _eventData = new byte[0];
		else                   _eventData = eventdata;
	
		if (_eventType == 6) {
			_pcEventID = UINT32(_eventData, 0);
			_pcEventData = EventData(_eventData, 4);
			_pcEventDataSize = _pcEventData.length;
		}
	
	}

	/**
	 * Parse BIOS IML Byte[]
	 * 
	 * @param message
	 * @param ptr
	 * @return
	 */
	public int set(byte[] message, int ptr) {
		_pcrIndex = UINT32(message, ptr);
		ptr += 4;
		_eventType = UINT32(message, ptr);
		ptr += 4;
		_digest = SHA1_DIGEST(message, ptr);
		ptr += 20;
		_eventData = EventData(message, ptr);
		ptr += 4 + _eventData.length;
	
		if (_eventType == 6) {
			_pcEventID = UINT32(_eventData, 0);
			_pcEventData = EventData(_eventData, 4);
			_pcEventDataSize = _pcEventData.length;
		}
	
		return ptr;
	}

	/**
	 * 
	 * ASCII string of eventlog like BIOS ASCII Eventlog format and also check
	 * the inetgrity of event.
	 * 
	 * 
	 * Reference P76, Table 11
	 * 
	 */
	
	public String toString() {
		/*
		 * 6 d9be6524a5f5047db5866813acf3277892a7a30a 04 7
		 * d9be6524a5f5047db5866813acf3277892a7a30a 04 4
		 * 38f30a0a967fcf2bfee1e3b2971de540115048c8 05 [Returned INT 19h] 4
		 * 89f0284e00992d067654818a9f2c09bbaa31acde 05 [Booting CD ROM, -
		 * MATSHITADVD-RAM UJ-833S] 4 8dfd8ee51758ac29ccf1bb6eedc2d483acf83a9b
		 * 0d [IPL] 4 1cdac212c5342627905cfcc4931972a8b4a09996 0d [IPL] 4
		 * 2cedbf54913d69d027c5b97e02763f921b16e345 06
		 */
		boolean validateEventdata = false;
		String str = "";
		_message = "";
		int eventdata_len = 0;

		/* PCR index */
		str = " " + _pcrIndex + " ";
		
		/* Event Type (hex) */
		str += "0x" + Long.toString(_eventType, 16) + ",";
		
		/* Digest */
		if (_printBase64) {
			str += getBase64String(_digest) + " ";

		} else {
			str += getHexString(_digest) + ",";
		}

		if (_eventData == null) {
			eventdata_len = 0;
		} else {
			eventdata_len = _eventData.length;
		}
		
		/* Event */
		str += "[";

		/* Event of IMA */
		if (_pcrIndex == 10) { // IMA
			if (_eventType == 0x00) {
				str += "IMA Boot Aggrigate";
			} else if ((_eventType & 0x00ff) == 0x01) { // MASK for user measumrent
				str += "ELF";
				str += "[" + eventdata_len + "]=";
				str += "\"" + new String(_eventData) + "\"";
			} else if ((_eventType & 0x00ff) == 0x02) {
				str += "LKM";
				str += "[" + eventdata_len + "]=";
				str += "\"" + new String(_eventData) + "\"";
			} else {
				str += "unknown";
			}

		}
		/* Event of BIOS and Grub */
		else {
			if (_eventType == 0x01) {
				str += "EV_POST_CODE";
			} else if (_eventType == 0x02) {
				str += "EV_UNUSED";
			} else if (_eventType == 0x04) {
				str += "EV_SEPARATOR";
				if (_eventData.length > 4) {
					str += "=\"" + new String(_eventData) + "\"";
				} else {
					str += "[" + _eventData.length + "]";
				}

				validateEventdata = true; // Check eventdata & digest later
			}
			else if ((_eventType) == 5) {
				str += "EV_ACTION";
				str += "[" + eventdata_len + "]=";
				str += "\"" + new String(_eventData) + "\"";

				validateEventdata = true; // Check eventdata & digest later
			} else if (_eventType == 0x06) {
				str += "EV_EVENT_TAG";
				// byte[] tagEventData = getTagEventData(eventData);
				// int EventID = getDD(eventData,0);
				// int EventDataSize = getDD(eventData,4);
				// str += " ID=" + getHexString(eventData,0,4);
				// str += ",Size=" + getHexString(eventData,4,4);
				if (_pcEventID == 0x01) {
					str += " SMBIOS[" + _pcEventDataSize + "]";
					// log.debug("SMBIOS " + pcEventDataSize);
					// TODO SMBIOS data in the eventlog is hard to parse since it was masked and ASN.1 format was broken.
					// HexTool.printHex(eventData);
					// SMBIOS smbios = new SMBIOS(pcEventData);
					// smbios.print();
				} else if (_pcEventID == 0x02) {
					str += " BIS Cert[" + _pcEventDataSize + "]";
				} else if (_pcEventID == 0x03) {
					str += " POST BIOS ROM Strings[" + _pcEventDataSize + "]=";
					// str += "[" +new String(tagEventData) +"]";
					str += "\"" + getHexString(_pcEventData) + "\"";
				} else if (_pcEventID == 0x04) {
					str += " ESCD[" + _pcEventDataSize + "]";
				} else if (_pcEventID == 0x05) {
					str += " CMOS[" + _pcEventDataSize + "]";
				} else if (_pcEventID == 0x06) {
					str += " NVRAM[" + _pcEventDataSize + "]";
				} else if (_pcEventID == 0x07) {
					str += " Option ROM Execute[" + _pcEventDataSize + "]";
					str += "\"" + getHexString(_pcEventData) + "\"";
				} else if (_pcEventID == 0x08) {
					str += " Option ROM Configuration[" + _pcEventDataSize + "]";
				} else {
					str += " unknown " + _pcEventID + "[" + _pcEventDataSize
							+ "]";
				}
			} else if (_eventType == 0x07) {
				// DELL BIOS use 7 for CRTM
				// EV_S_CRTM_CONTENTS
				str += "EV_S_CRTM_CONTENTS";

			} else if (_eventType == 0x08) {
				str += "EV_S_CRTM_VERSION";
				if (eventdata_len == 8) { // Panasonic use String
					str += "[8]=\"" + new String(_eventData) + "\"";
				} else if (eventdata_len == 5) { // Lenovo
					str += "[5]=\"" + new String(_eventData) + "\"";
				} else { // 32 HP use hex
					str += "[" + eventdata_len + "]=";
					str += getHexString(_eventData) + "";
				}
			} else if (_eventType == 0x09) {
				str += "EV_CPU_MICROCODE";
				str += "[" + eventdata_len + "]=";
				str += "\"" + new String(_eventData) + "\"";
			} else if (_eventType == 0x0a) {
				str += "EV_PLATFORM_CONFIG_FLAGS";
				str += "[" + eventdata_len + "]=";
				str += "\"" + new String(_eventData) + "\""; // Panasonic BIOS

				// TODO digest is not match. check the spec
				// validateEventdata = true; // Check eventdata & digest later
				_message += " TBD";
			} else if (_eventType == 0x0c) {
				// DELL BIOS use 0xc for MBR?
				// EV_COMPACT_HASH
				str += "EV_COMPACT_HASH";

			} else if (_eventType == 0x0d) {
				str += "EV_IPL";
				str += "[" + eventdata_len + "]";
			} else if (_eventType == 0x0e) {
				str += "EV_IPL_PARTITION_DATA";
				str += "[" + eventdata_len + "]";
			} else if (_eventType == 0x11) {
				str += "EV_NOHOST_INFO";
				str += "[" + eventdata_len + "]=";
				str += "\"" + new String(_eventData) + "\"";
				// str += "[" + getHexString(eventData) +"]";
			} else if (_eventType == 0x12) {
				str += "EV_SPECIFICATION_IDENTIFIER";
				str += "[" + eventdata_len + "]";
				str += "\"" + getHexString(_eventData) + "\""; // Panasonic
				// 0000 00 00 01 01 01 02 01 64 00000000000

				validateEventdata = true; // Check eventdata & digest later
			}
			/* Grub */
			else if (_eventType == 0x1005) {
				str += "EV_GRUB_BOOT_MSG";
				str += "[" + eventdata_len + "]=";
				str += "\"" + new String(_eventData) + "\"";

				validateEventdata = true; // Check eventdata & digest later
			} else if (_eventType == 0x1105) {
				str += "EV_GRUB_KERNEL_CMD";
				str += "[" + eventdata_len + "]=";
				str += "\"" + new String(_eventData) + "\"";

				validateEventdata = true; // Check eventdata & digest later

			} else if (_eventType == 0x1205) {
				str += "EV_GRUB_KERNEL";
				str += "[" + _eventData.length + "]=";
				str += "\"" + new String(_eventData) + "\"";
			} else if (_eventType == 0x1305) {
				str += "EV_GRUB_INITRD";
				str += "[" + eventdata_len + "]=";
				str += "\"" + new String(_eventData) + "\"";
			} else if (_eventType == 0x1405) {
				str += "EV_GRUB_MODULE";
				str += "[" + eventdata_len + "]=";
				str += "\"" + new String(_eventData) + "\"";
			} else if (_eventType == 0x1505) {
				str += "EV_GRUB_MODULE_CMD";
				str += "[" + eventdata_len + "]=";
				str += "\"" + new String(_eventData) + "\"";

				validateEventdata = true; // Check eventdata & digest later
			} else if (_eventType == 0x100) {  // Intel TXT, TBOOT
				str += "EV_TBOOT_SINIT_V6";
			} else if (_eventType == 0x101) {  // Intel TXT, TBOOT
				str += "EV_TBOOT_STM_V6";
			} else if (_eventType == 0x102) {  // Intel TXT, TBOOT
				str += "EV_TBOOT_POLCTL";
			} else if (_eventType == 0x103) {  // Intel TXT, TBOOT
				str += "EV_TBOOT_SINIT_V7";
			} else if (_eventType == 0x110) {  // Intel TXT, TBOOT
				str += "EV_TBOOT_MLE_HASH";
			} else if (_eventType == 0x111) {  // Intel TXT, TBOOT
				str += "EV_TBOOT_MODULE";
			} else if (_eventType == 0x80) {  // OPENPTS
				str += "EV_COLLECTOR_START";
			} else if (_eventType == 0x81) {  // OPENPTS
				str += "EV_UPDATE_START";
			} else if (_eventType == 0x82) {  // OPENPTS
				str += "EV_NEW_EVENT";
			} else if (_eventType == 0x83) {  // OPENPTS
				str += "EV_UPDATE_END";
			} else if (_eventType == 0x84) {  // OPENPTS
				str += "EV_FILE_SCAN";
			} else if (_eventType == 0x85) {  // OPENPTS
				str += "EV_COLLECTOR_UPDATE";
			} else if (_eventType == 0x86) {  // OPENPTS
				str += "EV_FILE_SCAN_TSS";
			} else {
				if (_eventData != null)
					str += "unknown[" + eventdata_len + "]";
				else
					str += "unknown[null]";
			}
		}
		str += "]";

		if ((validateEventdata == true) && (_enableValidation == true)) {
			// if ((validateEventdata)) {
			/* Check Digest */
			byte[] d = DigestTool.SHA1(_eventData);
			if (validateDigest(d, _digest)) {
				// OK
				_message += " Good eventdata";
				if (log.isDebugEnabled()) {
					log.debug("Valid Evant Data");
					log.debug(" EventData(hex) : "
							+ HexTool.getHexString(_eventData));
					log.debug(" Digest(hex)    : "
							+ HexTool.getHexString(_digest));
				}
			} else {
				// NG
				if (_eventType == 0x04) {
					_message += " BAD eventdata";
					log.error("Invalid Evant Data, but ignore type=0x04, workaround of XEN TCGBIOS");
				} else if (_eventType == 0x05) {
						_message += " BAD eventdata";
						log.error("Invalid Evant Data, but ignore type=0x05, workaround of IBM System X EV_ACTION Calling INT 19h");
				} else {
					_message += " BAD eventdata";
					_integrity = false;
				}
				if (log.isDebugEnabled()) {
					log.debug("Invalid Evant Data");
					log.debug(" EventData(hex) : "
							+ HexTool.getHexString(_eventData));
					log.debug(" Digest(hex)    : "
							+ HexTool.getHexString(d));
					log.debug(" PCR(hex)       : "
							+ HexTool.getHexString(_digest));
				}
			}
		} else {
			_message += "";
		}

		return str;
	}

	/**
	 * 
	 * Structure view
	 * 
	 * @return
	 */
	public String toStructString() {
		String str = "";


		/* PCR index */
		str = " " + _pcrIndex + " " + _eventType + " ";
		str += getHexString(_digest) + ",";
		str += getHexString(_eventData) + ",";


		return str;
	}

	/**
	 * 
	 * 
	 * @return
	 */
	public String getEventDataString() {
		boolean validateEventdata = false;
		String str = "";
		_message = "";
		
		/* Event */
		str += "";

		/* Event of IMA */
		if (_pcrIndex == 10) { // IMA
			if (_eventType == 0x00) {
				str += "IMA Boot Aggrigate";
			} else if ((_eventType & 0x00ff) == 0x01) {
				str += "ELF";
				str += "[" + _eventData.length + "]=";
				str += "\"" + new String(_eventData) + "\"";
			} else if ((_eventType & 0x00ff) == 0x02) {
				str += "LKM";
				str += "[" + _eventData.length + "]=";
				str += "\"" + new String(_eventData) + "\"";
			} else {
				str += "unknown";
			}

		}
		/* Event of BIOS and Grub */
		else {
			if (_eventType == 0x01) {
				str += "EV_POST_CODE";
			} else if (_eventType == 0x02) {
				str += "EV_UNUSED";
			} else if (_eventType == 0x04) {
				str += "EV_SEPARATOR";
				if (_eventData.length > 4) {
					str += "=\"" + new String(_eventData) + "\"";
				} else {
					str += "[" + _eventData.length + "]";
				}

				validateEventdata = true; // Check eventdata & digest later
			}
			// else if ( (0x0F & eventType) == 5) {
			else if ((_eventType) == 5) {
				str += "EV_ACTION";
				str += "[" + _eventData.length + "]=";
				str += "\"" + new String(_eventData) + "\"";

				validateEventdata = true; // Check eventdata & digest later
			} else if (_eventType == 0x06) {
				str += "EV_EVENT_TAG";
				// byte[] tagEventData = getTagEventData(eventData);
				// int EventID = getDD(eventData,0);
				// int EventDataSize = getDD(eventData,4);
				// str += " ID=" + getHexString(eventData,0,4);
				// str += ",Size=" + getHexString(eventData,4,4);
				if (_pcEventID == 0x01) {
					str += " SMBIOS[" + _pcEventDataSize + "]";
					// log.debug("SMBIOS " + pcEventDataSize);
					// TODO SMBIOS data in the eventlog is hard to parse since it was masked and ASN.1 format was broken.
					// HexTool.printHex(eventData);
					// SMBIOS smbios = new SMBIOS(pcEventData);
					// smbios.print();
				} else if (_pcEventID == 0x02) {
					str += " BIS Cert[" + _pcEventDataSize + "]";
				} else if (_pcEventID == 0x03) {
					str += " POST BIOS ROM Strings[" + _pcEventDataSize + "]=";
					// str += "[" +new String(tagEventData) +"]";
					str += "\"" + getHexString(_pcEventData) + "\"";
				} else if (_pcEventID == 0x04) {
					str += " ESCD[" + _pcEventDataSize + "]";
				} else if (_pcEventID == 0x05) {
					str += " CMOS[" + _pcEventDataSize + "]";
				} else if (_pcEventID == 0x06) {
					str += " NVRAM[" + _pcEventDataSize + "]";
				} else if (_pcEventID == 0x07) {
					str += " Option ROM Execute[" + _pcEventDataSize + "]";
					str += "\"" + getHexString(_pcEventData) + "\"";
				} else if (_pcEventID == 0x08) {
					str += " Option ROM Configuration[" + _pcEventDataSize + "]";
				} else {
					str += " unknown " + _pcEventID + "[" + _pcEventDataSize
							+ "]";
				}
			} else if (_eventType == 0x07) {
				// DELL BIOS use 7 for CRTM
				// EV_S_CRTM_CONTENTS
				str += "EV_S_CRTM_CONTENTS";

			} else if (_eventType == 0x08) {
				str += "EV_S_CRTM_VERSION";
				if (_eventData.length == 8) { // Panasonic use String
					str += "[8]=\"" + new String(_eventData) + "\"";
				} else if (_eventData.length == 5) { // Lenovo
					str += "[5]=\"" + new String(_eventData) + "\"";
				} else { // 32 HP use hex
					str += "[" + _eventData.length + "]=";
					str += getHexString(_eventData) + "";
				}
			} else if (_eventType == 0x09) {
				str += "EV_CPU_MICROCODE";
				str += "[" + _eventData.length + "]=";
				str += "\"" + new String(_eventData) + "\"";
			} else if (_eventType == 0x0a) {
				str += "EV_PLATFORM_CONFIG_FLAGS";
				str += "[" + _eventData.length + "]=";
				str += "\"" + new String(_eventData) + "\""; // Panasonic
				
				// TODO digest is not match. check the spec
				// validateEventdata = true; // Check eventdata & digest later
				_message += " TBD";
			} else if (_eventType == 0x0c) {
				// DELL BIOS use 0xc for MBR?
				// EV_COMPACT_HASH
				str += "EV_COMPACT_HASH";

			} else if (_eventType == 0x0d) {
				str += "EV_IPL";
				str += "[" + _eventData.length + "]";
			} else if (_eventType == 0x0e) {
				str += "EV_IPL_PARTITION_DATA";
				str += "[" + _eventData.length + "]";
			} else if (_eventType == 0x11) {
				str += "EV_NOHOST_INFO";
				str += "[" + _eventData.length + "]=";
				str += "\"" + new String(_eventData) + "\"";
				// str += "[" + getHexString(eventData) +"]";
			} else if (_eventType == 0x12) {
				str += "EV_SPECIFICATION_IDENTIFIER";
				str += "[" + _eventData.length + "]";
				str += "\"" + getHexString(_eventData) + "\""; // Panasonic
				// 0000 00 00 01 01 01 02 01 64 00000000000

				validateEventdata = true; // Check eventdata & digest later
			}
			/* Grub */
			else if (_eventType == 0x1005) {
				str += "EV_GRUB_BOOT_MSG";
				str += "[" + _eventData.length + "]=";
				str += "\"" + new String(_eventData) + "\"";

				validateEventdata = true; // Check eventdata & digest later
			} else if (_eventType == 0x1105) {
				str += "EV_GRUB_KERNEL_CMD";
				str += "[" + _eventData.length + "]=";
				str += "\"" + new String(_eventData) + "\"";

				validateEventdata = true; // Check eventdata & digest later

			} else if (_eventType == 0x1205) {
				str += "EV_GRUB_KERNEL";
				str += "[" + _eventData.length + "]=";
				str += "\"" + new String(_eventData) + "\"";
			} else if (_eventType == 0x1305) {
				str += "EV_GRUB_INITRD";
				str += "[" + _eventData.length + "]=";
				str += "\"" + new String(_eventData) + "\"";
			} else if (_eventType == 0x1405) {
				str += "EV_GRUB_MODULE";
				str += "[" + _eventData.length + "]=";
				str += "\"" + new String(_eventData) + "\"";
			} else if (_eventType == 0x1505) {
				str += "EV_GRUB_MODULE_CMD";
				str += "[" + _eventData.length + "]=";
				str += "\"" + new String(_eventData) + "\"";
				validateEventdata = true; // Check eventdata & digest later
			} else if (_eventType == 0x100) {  // Intel TXT, TBOOT
				str += "EV_TBOOT_SINIT_V6";
			} else if (_eventType == 0x101) {  // Intel TXT, TBOOT
				str += "EV_TBOOT_STM_V6";
			} else if (_eventType == 0x102) {  // Intel TXT, TBOOT
				str += "EV_TBOOT_POLCTL";
			} else if (_eventType == 0x103) {  // Intel TXT, TBOOT
				str += "EV_TBOOT_SINIT_V7";
			} else if (_eventType == 0x110) {  // Intel TXT, TBOOT
				str += "EV_TBOOT_MLE_HASH";
			} else if (_eventType == 0x111) {  // Intel TXT, TBOOT
				str += "EV_TBOOT_MODULE";
			} else if (_eventType == 0x80) {  // OPENPTS
				str += "EV_COLLECTOR_START";
			} else if (_eventType == 0x81) {  // OPENPTS
				str += "EV_UPDATE_START";
			} else if (_eventType == 0x82) {  // OPENPTS
				str += "EV_NEW_EVENT";
			} else if (_eventType == 0x83) {  // OPENPTS
				str += "EV_UPDATE_END";
			} else if (_eventType == 0x84) {  // OPENPTS
				str += "EV_FILE_SCAN";
			} else if (_eventType == 0x85) {  // OPENPTS
				str += "EV_COLLECTOR_UPDATE";
			} else if (_eventType == 0x86) {  // OPENPTS
				str += "EV_FILE_SCAN_TSS";
			} else {
				if (_eventData != null)
					str += "unknown[" + _eventData.length + "]";
				else
					str += "unknown[null]";
				log.error("Missing string of the event at PCR["  +_pcrIndex + "]");
			}
		}
		str += "";

		if ((validateEventdata == true) && (_enableValidation == true)) {
			// if ((validateEventdata)) {
			/* Check Digest */
			byte[] d = DigestTool.SHA1(_eventData);
			if (validateDigest(d, _digest)) {
				// OK
				_message += " Good eventdata";
				if (log.isDebugEnabled()) {
					log.debug("Valid Evant Data");
					log.debug(" EventData(hex) : "
							+ HexTool.getHexString(_eventData));
					log.debug(" Digest(hex)    : "
							+ HexTool.getHexString(_digest));
				}
			} else {
				// NG
				if (_eventType == 0x04) {
					_message += " BAD eventdata";
					log
							.error("Invalid Evant Data, but ignore type=0x04, workaround of XEN TCGBIOS");
				} else {
					_message += " BAD eventdata";
					_integrity = false;
				}
				if (log.isDebugEnabled()) {
					log.debug("Invalid Evant Data");
					log.debug(" EventData(hex) : "
							+ HexTool.getHexString(_eventData));
					log.debug(" Digest(hex)    : "
							+ HexTool.getHexString(_digest));
				}
			}
		} else {
			_message += "";
		}

		return str;
	}

	
	private boolean validateDigest(byte[] d1, byte[] d2) {
		if (d1.length != d2.length) {
			return false;
		}
		for (int i = 0; i < d1.length; i++) {
			if (d1[i] != d2[i])
				return false;
		}
		return true;
	}

	// private byte[] getTagEventData(byte[] data) {
	// byte[] buf = new byte[data.length-8];
	// for (int i=0;i<data.length-8;i++) buf[i] = data[i+8];
	// return buf;
	// }

	public String toStringBase64() {
		_printBase64 = true;
		String rc = toString();
		return rc;
	}

	/*
	private int getDD(byte[] data, int offset) {
		int i;
		i = data[offset + 3];
		i = i << 4;
		i = data[offset + 2];
		i = i << 4;
		i = data[offset + 1];
		i = i << 4;
		i = data[offset];
		return i;
	}
	*/
	
	private String getBase64String(byte[] digest) {
		if (digest != null) {
			return Base64Tool.encode(digest);
		} else {
			return "N/A";
		}
	}

	/**
	 * @param eventdata
	 */
	public static String getHexString(byte data) {
		int d = data;
		String str = "";
		if (d < 0)
			d += 256;
		if (d < 16)
			str = "0";
		str += Integer.toString(d, 16);
		return str;
	}

	/**
	 * @param eventdata
	 */
	public static String getHexString(byte[] data) {
		if (data == null)
			return "na";
		String str = "";
		for (int i = 0; i < data.length; i++) {
			str += getHexString(data[i]);
		}
		return str;
	}

	/**
	 * @return
	 */
	public String getHexString() {
		return getHexString(_digest);
	}

	/**
	 * BIOS Event? String eventdata return just String binary eventdata return
	 * base64(eventdata)
	 * 
	 * @return
	 */
	public String getEventDataStr() {
		String str = "";
		if (_eventType == 4) {
			str += new String(_eventData);
		} else if ((0x05 & _eventType) == 5) {
			str += new String(_eventData);
		} else {
			str += Base64Tool.encode(_eventData);
		}
		return str;
	}

	/**
	 * IMA?
	 * 
	 * @return
	 */
	public String getEventDataStrForce() {
		String str = "";
		str += new String(_eventData);
		return str;
	}

	/**
	 * 
	 * @param b
	 * @param ptr
	 * @return
	 */
	public static long UINT32(byte b[], int ptr) {
		long i = 0;

		i = b[ptr + 3] & 0xFF;
		i = i << 8;
		i += b[ptr + 2] & 0xFF;
		i = i << 8;
		i += b[ptr + 1] & 0xFF;
		i = i << 8;
		i += b[ptr] & 0xFF;
		
		return i;
	}
	
	/**
	 * @param message
	 * @param ptr
	 * @return
	 */
	private static byte[] SHA1_DIGEST(byte[] message, int ptr) {
		byte b[] = new byte[20];
		int i;
		for (i = 0; i < 20; i++) {
			b[i] = message[ptr + i];
		}
		return b;
	}

	/**
	 * @param message
	 * @param ptr
	 * @return
	 */
	private static byte[] EventData(byte[] message, int ptr) {
		// 
		int len = (int) UINT32(message, ptr);

		byte b[] = new byte[len];
		int i;
		for (i = 0; i < len; i++) {
			b[i] = message[ptr + 4 + i];
		}
		return b;
	}

	public byte[] getDigest() {
		return _digest;
	}

	public long getPcrIndex() {
		return _pcrIndex;
	}

	public long getEventType() {
		return _eventType;
	}

	public byte[] getEventData() {
		return _eventData;
	}


	public byte[] getPCEventData() {
		return _pcEventData;
	}

	public boolean isValid() {
		return _integrity;
	}

	public String getMessage() {
		return _message;
	}

	/**
	 * get Event String
	 * 
	 * @return
	 */
	public String getEventString() {
		return _str;
	}

	public boolean validate() {
		_enableValidation = true;
		_str = this.toString();
		_enableValidation = false;
		return _integrity;
	}
}
