/*
 * 
 * Licensed Materials - Property of IBM
 *
 * Open Platform Trust Services - An open source TCG PTS
 *
 * (C) Copyright International Business Machines Corp. 2007
 *
 */
package com.ibm.trl.tcg.pts.integrity;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
//import java.util.Properties;
import java.util.UUID;
import java.util.Vector;
import java.util.zip.GZIPInputStream;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
//import javax.xml.parsers.ParserConfigurationException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
//import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
//import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import com.ibm.trl.tcg.pts.engine.Event;
//import com.ibm.trl.tcg.pts.engine.FiniteStateMachine;
//import com.ibm.trl.tcg.pts.engine.PlatformProperties;
import com.ibm.trl.tcg.pts.eventlog.EventSet;
//import com.ibm.trl.tcg.pts.eventlog.IML;
//import com.ibm.trl.tcg.pts.eventlog.PCRS;
import com.ibm.trl.tcg.pts.tools.Base64Tool;
//import com.ibm.trl.tcg.pts.tools.HexTool;
//import com.ibm.trl.tcg.pts.tools.SignatureTool;
import com.ibm.trl.tcg.pts.tools.XmlTool;

/**
 * Object to handle TCG Integrity Report
 * 
 * Reference: TCG Infrastructure Working Group Integrity Report Schema
 * Specification Specification Version 1.0 Revision 1.0 17 November 2006 FINAL
 * https://www.trustedcomputinggroup.org/specs/IWG/IntegrityReport_Schema_Specification_v1.0.pdf
 * 
 * 
 * @author Seiji Munetoh (munetoh@users.sourceforge.jp)
 * 
 */
public class IntegrityReport {
	/* Logger */
	private Log log = LogFactory.getLog(this.getClass());

	/* IR */
	private String _filename;
	private Document _doc;
	private Element _ir;
	
	/* status */
	private int _eventNum = 0;
	private UUID _uuid = null;
	
	/**/
	private PcrComposite _pcrComposite;


	/**
	 * Create integrity report object from file (XML)
	 * 
	 * @param filename
	 * @throws Exception
	 */
	public IntegrityReport(String filename, byte[] pubkey) throws Exception {
		if (log.isTraceEnabled()) {
			log.trace("IntegrityReport");
			log.trace(" filename : " + filename);
		}

		_filename = filename;
		this.load(filename);
		// this.validateQuoteData(pubkey);
	}

	/**
	 * Create integrity report object from file (XML) and Zipped file (XML)
	 * 
	 * @param irFilename
	 * @param option,
	 *            1:ziped file
	 * @throws Exception
	 */
	public IntegrityReport(String filename, byte[] pubkey, int option) throws Exception {
		if (log.isTraceEnabled()) {
			log.trace("IntegrityReport");
			log.trace(" filename : " + filename);
		}

		_filename = filename;
		if (option == 1) {
			this.loadGzip(filename);
		} else {
			this.load(filename);
		}
	}

	/**
	 * 1) load integrity report (XML file) 2) create DOM tree
	 * 
	 * @param filename
	 */
	private void load(String filename) {
		try {
			DocumentBuilderFactory dbfactory = DocumentBuilderFactory
					.newInstance();
			DocumentBuilder builder = dbfactory.newDocumentBuilder();
			_doc = builder.parse(new File(filename));
			_ir = _doc.getDocumentElement();
		} catch (Exception e) {
			log.error("loadFile " + filename + " failed");
			e.printStackTrace();
		}
	}

	private void loadGzip(String filename) {
		try {
			FileInputStream fis = new FileInputStream(filename);
			BufferedInputStream bis = new BufferedInputStream(fis);
			GZIPInputStream gis = new GZIPInputStream(bis);
	
			DocumentBuilderFactory dbfactory = DocumentBuilderFactory
					.newInstance();
			DocumentBuilder builder = dbfactory.newDocumentBuilder();
			_doc = builder.parse(gis);
			_ir = _doc.getDocumentElement();
		} catch (Exception e) {
			log.error("Load Giped File " + filename + " failed");
			e.printStackTrace();
		}
	}

	/**
	 * save to file
	 * 
	 * @param filename
	 * @throws Exception
	 */
	public void save(String filename) throws Exception {
		XmlTool.saveToFile(filename, _doc, false);
		if (log.isDebugEnabled())
			log.debug("Write IR to : " + filename);
	}

	/**
	 * get events in this report, single snapshot for each PCR
	 * 
	 * @return
	 * @throws Exception
	 */
	public EventSet[] getEventSet() throws Exception {
		EventSet[] events = new EventSet[24]; // TODO 24?
		for (int i = 0; i < 24; i++) {
			events[i] = new EventSet();
			events[i].setPcrIndex(i);
		}

		/* SnapshotCollection per PCR index */
		NodeList nlSC = _ir.getElementsByTagName("SnapshotCollection");
		for (int i = 0; i < nlSC.getLength(); i++) {
			Element elSC = (Element) nlSC.item(i);

			/* Check PCR index of this SnapshotCollection */
			NodeList nlPH = elSC.getElementsByTagName("PcrHash");
			Element elPH = (Element) nlPH.item(0); // TODO must be 0
			int pcrindex = new Integer(elPH.getAttribute("Number"));

			/* fill events */
			NodeList nlSO = elSC.getElementsByTagName("stuff:Objects");
			for (int j = 0; j < nlSO.getLength(); j++) {
				Element elSO = (Element) nlSO.item(j);
				/* Digest */
				NodeList nlSH = elSO.getElementsByTagName("stuff:Hash");
				Element elSH = (Element) nlSH.item(0);
				String b64digest = elSH.getTextContent();
				byte[] digest = Base64Tool.decode(b64digest);
				/* Type */
				NodeList nlET = elSO.getElementsByTagName("eventtype");
				Element elET = (Element) nlET.item(0);
				long type = new Long(elET.getTextContent());
				/* EventData */
				NodeList nlED = elSO.getElementsByTagName("eventdata");
				Element elED = (Element) nlED.item(0);
				if (elED == null) {
					events[pcrindex].addEvent(type, digest, null);
				} else {
					String b64eventdata = elED.getTextContent();
					byte[] eventdata = Base64Tool.decode(b64eventdata);
					events[pcrindex].addEvent(type, digest, eventdata);
				}
				
				_eventNum++;
			}
		}
		return events;
	}

	/**
	 * get events in this report which has given name (SimpleName attribute)
	 * 
	 * @return
	 * @throws Exception
	 */
	public EventSet[] getEventSetByName(String name) throws Exception {
		EventSet[] events = new EventSet[24]; // TODO 24?
		for (int i = 0; i < 24; i++) {
			events[i] = new EventSet();
			events[i].setPcrIndex(i);
		}

		/* SnapshotCollection per PCR index */
		NodeList nlSC = _ir.getElementsByTagName("SnapshotCollection");
		for (int i = 0; i < nlSC.getLength(); i++) {
			Element elSC = (Element) nlSC.item(i);

			/* Check Name */
			NodeList nlCCI = elSC.getElementsByTagName("core:ComponentID");
			Element elCCI = (Element) nlCCI.item(0); // TODO must be 0
			String simpleName = elCCI.getAttribute("SimpleName");
			if (simpleName.equals(name)) {
				/* Check PCR index of this SnapshotCollection */
				NodeList nlPH = elSC.getElementsByTagName("PcrHash");
				Element elPH = (Element) nlPH.item(0); // TODO must be 0
				int pcrindex = new Integer(elPH.getAttribute("Number"));

				/* fill events */
				NodeList nlSO = elSC.getElementsByTagName("stuff:Objects");
				for (int j = 0; j < nlSO.getLength(); j++) {
					Element elSO = (Element) nlSO.item(j);
					/* Digest */
					NodeList nlSH = elSO.getElementsByTagName("stuff:Hash");
					Element elSH = (Element) nlSH.item(0);
					byte[] digest = Base64Tool.decode(elSH.getTextContent());
					/* Type */
					NodeList nlET = elSO.getElementsByTagName("eventtype");
					Element elET = (Element) nlET.item(0);
					long type = new Long(elET.getTextContent());
					/* EventData */
					NodeList nlED = elSO.getElementsByTagName("eventdata");
					Element elED = (Element) nlED.item(0);
					byte[] eventdata = Base64Tool.decode(elED.getTextContent());
					events[pcrindex].addEvent(type, digest, eventdata);
					_eventNum++;
				}
			} // if name
		} // for snapshot
		return events;
	}
	
	/**
	 * 
	 * single list
	 * 
	 * @return
	 * @throws IOException 
	 */
	public Vector<Event>  getEventList() throws IOException {
		Vector<Event> el = new Vector<Event>();

		/* SnapshotCollection per PCR index */
		NodeList nlSC = _ir.getElementsByTagName("SnapshotCollection");
		for (int i = 0; i < nlSC.getLength(); i++) {
			Element elSC = (Element) nlSC.item(i);

			/* Check PCR index of this SnapshotCollection */
			NodeList nlPH = elSC.getElementsByTagName("PcrHash");
			Element elPH = (Element) nlPH.item(0); // TODO must be 0
			int pcrindex = new Integer(elPH.getAttribute("Number"));

			/* fill events */
			NodeList nlSO = elSC.getElementsByTagName("stuff:Objects");
			for (int j = 0; j < nlSO.getLength(); j++) {
				Element elSO = (Element) nlSO.item(j);
				/* Digest */
				NodeList nlSH = elSO.getElementsByTagName("stuff:Hash");
				Element elSH = (Element) nlSH.item(0);
				String b64digest = elSH.getTextContent();
				byte[] digest = Base64Tool.decode(b64digest);
				/* Type */
				NodeList nlET = elSO.getElementsByTagName("eventtype");
				Element elET = (Element) nlET.item(0);
				long type = new Long(elET.getTextContent());
				/* EventData */
				NodeList nlED = elSO.getElementsByTagName("eventdata");
				Element elED = (Element) nlED.item(0);
				if (elED == null) {
					//events[pcrindex].addEvent(type, digest, null);
					Event e = new Event(pcrindex, type, digest, null);
					el.add(e);
				} else {
					String b64eventdata = elED.getTextContent();
					byte[] eventdata = Base64Tool.decode(b64eventdata);
					//events[pcrindex].addEvent(type, digest, eventdata);
					Event e = new Event(pcrindex, type, digest, eventdata);
					el.add(e);
				}
				
				_eventNum++;
			}
		}
		return el;
	}

	/**
	 * get event number in this report
	 * 
	 * @return
	 */
	public int getEventNum() {
		return _eventNum;
	}

	public PcrComposite getPcrComposite() {
		// TODO if (_pcrComposite == null)
		return _pcrComposite;
	}



	//static int MAX_FILE_COUNT = 2;

	public NodeList getElementsByTagName(String string) {
		return _ir.getElementsByTagName(string);
	}

	public boolean validateXML() throws Exception {
		TCGSchemas schema = new TCGSchemas();
		return schema.validate(_filename);
	}

	public String getGipedBase64String() throws Exception {
		return XmlTool.getGipedBase64String(_doc);
	}

	/**
	 * <Report 
	 *   xmlns:core="http://www.trustedcomputinggroup.org/XML/SCHEMA/1_0_1/core_integrity#" 
	 *   xmlns:stuff="http://www.trustedcomputinggroup.org/XML/SCHEMA/1_0/simple_object#" 
	 *   xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" 
	 *   xmlns="http://www.trustedcomputinggroup.org/XML/SCHEMA/1_0/integrity_report#" 
	 *   ID="IR_d076c972-6a76-11e0-a240-001f160c9c28" 
	 *   UUID="d076c972-6a76-11e0-a240-001f160c9c28">
	 * 
	 * @return
	 */
	public UUID getUUID() {
		if (_uuid != null) return _uuid;
		/* get from IR */
		NodeList nl = _doc.getElementsByTagName("Report");
		for (int i = 0; i < nl.getLength(); i++) {  // TODO must be 1
			Element e = (Element) nl.item(i);
			String u = e.getAttribute("UUID");
			log.debug("getUUID return " + u);
			if (u != null) {
				_uuid = UUID.fromString(u);
				return _uuid;
			}
		}
		log.debug("getUUID() return null  UUID");
		_uuid = null;
		return null;
	}

	public String getUuidString() {
		getUUID();
		if (_uuid != null) return _uuid.toString();
		else return "na";
	}

}
