/*
 * 
 * Licensed Materials - Property of IBM
 *
 * Open Platform Trust Services - An open source TCG PTS
 *
 * (C) Copyright International Business Machines Corp. 2007, 2011
 *
 */
package com.ibm.trl.tcg.pts.integrity;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.StringReader;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.Properties;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.w3c.dom.DOMException;
import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;

import com.ibm.trl.tcg.pts.engine.Event;
import com.ibm.trl.tcg.pts.engine.FiniteStateMachine;
import com.ibm.trl.tcg.pts.engine.IntegrityProperties;
import com.ibm.trl.tcg.pts.engine.PlatformProperties;
import com.ibm.trl.tcg.pts.eventlog.EventSet;
import com.ibm.trl.tcg.pts.tools.Base64Tool;
import com.ibm.trl.tcg.pts.tools.HexTool;
import com.ibm.trl.tcg.pts.tools.SignatureTool;
import com.ibm.trl.tcg.pts.tools.XmlTool;

/**
 * 
 * Container of TCG IWG VerificationResult
 * 
 * 
 * @author Seiji Munetoh (munetoh@users.sourceforge.jp)
 * 
 */
public class VerificationResult {
	/* Logger */
	private Log log = LogFactory.getLog(this.getClass());

	// private String _schemasPath = "../../tcg/schemas/";

	// private String[] _schemas = {
	// _schemasPath + "xmldsig-core-schema.xsd",
	// _schemasPath + "SimpleObject_v8_ibm.xsd",
	// _schemasPath + "Core_Integrity_Manifest_v16_ibm.xsd",
	// _schemasPath + "Integrity_Report_Manifest_v19_ibm.xsd",
	// _schemasPath + "Verification_Result_v1_0_ibm.xsd"
	// //_schemasPath + "VerificationResult_v5_ibm.xsd"
	// };

	private String _filename;
	private String _namespace = "http://www.trustedcomputinggroup.org/XML/SCHEMA/1_0/verification_result#";

	/* CML */
	private Document _doc;
	private Element _result;

	
	private boolean _indent = false; // 20071120 SM true->false
	private PcrComposite _pcrComposite;
	private boolean _verbose = false;
	// private int _iidbIndex = -1;

	private IntegrityProperties _integrityProps = null;
	private PlatformProperties _platformProps = null;
	private Properties _policyProp = null;

	/* result */
	public static int VALID = 0;
	public static int INVALID = 1;
	public static int UNVERIFIED = 2;

	// private static boolean useAide = false;

	public VerificationResult() {
		init("UNVERIFIED");
		//_iidbIndex = -1;
	}

	public VerificationResult(String filename) {
		this.loadFile(filename);
	}

	/**
	 * Create new VerificationResult
	 */
	public void init(String resultStr) {
		// http://java.sun.com/j2se/1.5.0/ja/docs/ja/api/java/util/UUID.html
		//UUID uuid = UUID.randomUUID();

		try {
			DocumentBuilderFactory factory = DocumentBuilderFactory
					.newInstance();
			DocumentBuilder builder = factory.newDocumentBuilder();
			DOMImplementation domImpl = builder.getDOMImplementation();
			_doc = domImpl.createDocument(_namespace, "VerifyResult", null);

			_result = _doc.getDocumentElement();
			_result
					.setAttribute("xmlns:core",
							"http://www.trustedcomputinggroup.org/XML/SCHEMA/1_0_1/core_integrity#");
			_result
					.setAttribute("xmlns:ir",
							"http://www.trustedcomputinggroup.org/XML/SCHEMA/1_0/integrity_report#");
			_result.setAttribute("xmlns:xsi",
					"http://www.w3.org/2001/XMLSchema-instance");

			Element ruuid = _doc.createElement("ResultUUID");
			ruuid.setAttribute("DependentResults", "DependentResults");
			ruuid.appendChild(_doc.createTextNode("Demo"));
			_result.appendChild(ruuid);

			Element rs = _doc.createElement("Results");
			rs.setAttribute("RuleUUID", "RuleUUID");
			rs.setAttribute("Result", resultStr);
			rs.setAttribute("ReportUUID", "ReportUUID");
			rs.setAttribute("ReasonStirings", "TEST");

			Element pid = _doc.createElement("PolicyID");
			pid.setAttribute("UriValue", "CSADSAD");
			rs.appendChild(pid);

			_result.appendChild(rs);

			// ruuid.appendChild(_doc.createTextNode("test"));
			// _result.appendChild(ruuid);

		} catch (ParserConfigurationException e) {
			e.printStackTrace();
		}

	}

	/**
	 * 
	 * Print Verification Detail to stdout
	 * 
	 */
	public void setVerbose() {
		_verbose = true;
	
	}

	public void save(String filename) throws Exception {
		_filename = filename;
		XmlTool.saveToFile(filename, _doc, _indent);
	}

	/**
	 * 
	 * 
	 * @param filename
	 */
	public void loadFile(String filename) {
		_filename = filename;
		try {
			DocumentBuilderFactory dbfactory = DocumentBuilderFactory
					.newInstance();
			DocumentBuilder builder = dbfactory.newDocumentBuilder();
			_doc = builder.parse(new File(_filename));
			_result = _doc.getDocumentElement();
			// NodeList nl = _result.getElementsByTagName("core:ComponentID");
			// _cid = (Element) nl.item(0); // TODO check size

			// log.debug(_root.getFirstChild().getNodeValue());
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	public void loadXacmlResponse(String filename) throws Exception {
		/* Load */
		// _filename = filename;
		String result = null;

		try {
			DocumentBuilderFactory dbfactory = DocumentBuilderFactory
					.newInstance();
			DocumentBuilder builder = dbfactory.newDocumentBuilder();
			Document doc = builder.parse(new File(filename));

			Element response = doc.getDocumentElement();
			NodeList nl = response.getElementsByTagName("Decision");
			result = nl.item(0).getTextContent();

		} catch (Exception e) {
			e.printStackTrace();
		}

		if (result != null){
			if (result.equals("Permit")) {
				// OK
				init("VALID");

			} else if (result.equals("NotApplicable")) {
				init("INVALID");
				throw new Exception("NotApplicable");
			}
			
		} else {
			throw new Exception(result);
		}
	}

	// public void setSchemaPath(String path) throws Exception {
	// _tcgSchemas = new TCGSchemas(path);
	// //_tcgSchemas.setSchemaDir(path);
	// }

	//public void setIIDBIndex(int iidbIndex) {
	//	_iidbIndex = iidbIndex;
	//}

	//public void useAide() {
	//	useAide = true;
	//}

	public void setValidationPolicy(String filename) throws Exception {
		File file = new File(filename);
		if (file.exists() == false) {
			throw new Exception("Policy file (properties file) is not found "
					+ filename);
		}
	
		/* Load Properties */
		_policyProp = new Properties();
		_policyProp.load(new FileInputStream(filename));
	}

	/**
	 * 
	 * @param result
	 *            0 VALID 1 INVALID 2 UNVERIFIED
	 * 
	 */
	public void setResult(int result) {
		if (result == VALID)
			init("VALID");
		else if (result == UNVERIFIED)
			init("UNVERIFIED");
		else if (result == INVALID)
			init("INVALID");
		else {
			log.error("Wrong result " + result);
			init("INVALID"); //
		}
	}

	/**
	 * Get result string
	 * 
	 * @return
	 */
	public String getResult() {
		NodeList nl = _result.getElementsByTagName("Results");
		Element r = (Element) nl.item(0);
		return r.getAttribute("Result");
	}

	/**
	 * Set reason string (internal format is base64)
	 * 
	 * @param message
	 */
	public void setReason(String message) {
		// 20071121 SM fixed
		String b64 = Base64Tool.encode(message.getBytes());
		
		NodeList nl = _result.getElementsByTagName("Results");
		Element r = (Element) nl.item(0);
		r.setAttribute("ReasonStirings", b64);
	}

	/**
	 * Get reason string
	 * 
	 * @return
	 * @throws Exception 
	 */
	public String getReason() throws Exception {
		NodeList nl = _result.getElementsByTagName("Results");
		Element r = (Element) nl.item(0);
		String b64 = r.getAttribute("ReasonStirings");
		byte[] b = Base64Tool.decode(b64);
		String  str = new String(b);
		return str;
	}

	public void parseString(String message) throws Exception {
		_filename = null;
		DocumentBuilderFactory dbfactory = DocumentBuilderFactory.newInstance();
		DocumentBuilder builder = dbfactory.newDocumentBuilder();
		_doc = builder.parse(new InputSource(new StringReader(message)));
		_result = _doc.getDocumentElement();
	}

	/**
	 * base64 XML string to VR
	 * 
	 * @param base64
	 * @throws Exception
	 */
	public void parseBase64String(String base64) throws Exception {
		if (base64 == null) {
			throw new Exception("Base64String is null. PTS Server Error");
		}
		parseString(new String(Base64Tool.decode(base64)));
	}

	/**
	 * 
	 * Validate Quote Signature
	 * 
	 * @return 0 valid 1 invalid 2 unverified
	 * @throws Exception
	 * @throws DOMException
	 */
	public int validateQuoteData(IntegrityReport ir, byte[] pubKey) throws Exception {
		byte[] message = new byte[48];
		NodeList nlQI = ir.getElementsByTagName("QuoteInfo");
		Element qi = (Element) nlQI.item(0);
		if (qi == null) {
			/* QuoteInfo is missing */
			return 2; // UNVERIFYED
		}
		
		message[0] =  1;  // TODO get from IR
		message[1] =  1;  // TODO
		message[2] =  0;  // TODO
		message[3] =  0;  // TODO
		message[4] = 'Q'; // TODO
		message[5] = 'U'; // TODO
		message[6] = 'O'; // TODO
		message[7] = 'T'; // TODO
		byte[] dv = Base64Tool.decode(qi.getAttribute("DigestValue"));
		for (int i = 0; i < 20; i++)
			message[8 + i] = dv[i];
		byte[] ed = Base64Tool.decode(qi.getAttribute("ExternalData"));
		for (int i = 0; i < 20; i++)
			message[28 + i] = ed[i];
		// TODO Valdate ED if nonce from server
	
		NodeList nlSV = ir.getElementsByTagName("SignatureValue");
		byte[] signature = Base64Tool.decode(nlSV.item(0).getTextContent());
	
		if (log.isTraceEnabled()) {
			log.trace("validate signature ");
			log.trace("  pubKey    : " + HexTool.getHexString(pubKey));
			log.trace("  message   : " + HexTool.getHexString(message));
			log.trace("  signature : " + HexTool.getHexString(signature));
		}
	
		if (SignatureTool.validate(pubKey, message, signature) == false) {
			if (log.isDebugEnabled()) {
				log.debug("Bad Signature, return 1");
			}
			// TODO 20071119 SM ignoe for a while
			log.error("Bad Signature, BUT CONTINUE");
			// return 1; // INVALID
		}
	
		/* Validate PCR Composite <-> message */
		/* PcrSelection */
	
		NodeList nlPS = ir.getElementsByTagName("PcrSelection");
		Element ps = (Element) nlPS.item(0);
		int sizeOfSelect = new Integer(ps.getAttribute("SizeOfSelect"));
	
		_pcrComposite = new PcrComposite();
		_pcrComposite.setPcrNumber(sizeOfSelect * 8); // TODO
	
		NodeList nlPV = ir.getElementsByTagName("PcrValue");
		for (int i = 0; i < nlPV.getLength(); i++) {
			Element pv = (Element) nlPV.item(i);
			int pcrindex = new Integer(pv.getAttribute("PcrNumber"));
			_pcrComposite.selectPcr(pcrindex, Base64Tool.decode(pv
					.getTextContent()));
		}
	
		_pcrComposite.setExtraData(ed);
	
		if (_pcrComposite.validate(message) == false) {
			if (log.isDebugEnabled()) {
				log.debug("Bad pcrComposite, return 1");
			}
			_platformProps.setProperty("tpm.quote.pcrs", "invalid");
			return INVALID;
		} else {
			_platformProps.setProperty("tpm.quote.pcrs", "valid");
		}
	
		if (log.isDebugEnabled()) {
			log.debug("Good QuoteInfo, return VALID");
		}
		
		/* set prop */
		for (int i = 0; i< 24; i ++) {
			if (_pcrComposite.selected(i)) {
				String name = "tpm.quote.pcr." + i;
				String value = Base64Tool.encode(_pcrComposite.getPcrValue(i));
				_platformProps.setProperty(name, value);
			}
		}
		
		
		return VALID;
	}

	/*
	2011-02-25 SM
	0036
	51555432
	a32a82a50ca3713911d44e2b29c7e3335bbc6c1f
	0003ff0000
	01
	77c75764731dcf87215afa87d13e9cc9aef195bf
	
	/**
	 * 
	 * Validate Quote Signature
	 * 
	 * @return 0 valid 1 invalid 2 unverified
	 * @throws Exception
	 * @throws DOMException
	 */
	public int validateQuote2Data(IntegrityReport ir, byte[] pubKey) throws Exception {
		byte[] message = new byte[52];  // TODO w/o Verfion Info (52->75?)

		
		/* QuoteInfo2 element */
		/* e.g.
		 	<QuoteInfo2 Tag="54" Fixed="QUT2" ExternalData="GWE00CfgBEroxWneIwTt5URsJLY=">
			<PcrInfoShort>
			<PcrSelection SizeOfSelect="3" PcrSelect="/wAA"/>
			<LocalityAtRelease>0</LocalityAtRelease>
			<CompositeHash>dC8NatJkvttL+Dfqt8aD7tr+rO8=</CompositeHash>
			<PcrComposit>
			<PcrSelection SizeOfSelect="3" PcrSelect="/wAA"/>
			<ValueSize>0</ValueSize>
			<PcrValue PcrNumber="0">kTvYo8w6LsZzbP4yV3CDGlpTow0=</PcrValue>
			<PcrValue PcrNumber="1">1X3RhqGU+e3pDYtiLUqO8FptQLU=</PcrValue>
			<PcrValue PcrNumber="2">U95YTc7wP2p9rBokCoNYk4lvIY0=</PcrValue>
			<PcrValue PcrNumber="3">Oj94DxGktJlp/KqAzW45V8M7InU=</PcrValue>
			<PcrValue PcrNumber="4">6C2nztlIcfny8WX+whUSiSm2P+s=</PcrValue>
			<PcrValue PcrNumber="5">rQkyWrN3ELI0lhjL+XIsjw7ggQ8=</PcrValue>
			<PcrValue PcrNumber="6">WF5XnkiZf+6O/SCDDGqEHrNTxig=</PcrValue>
			<PcrValue PcrNumber="7">Oj94DxGktJlp/KqAzW45V8M7InU=</PcrValue>
			</PcrComposit>
			</PcrInfoShort>
			</QuoteInfo2>
		 */
		NodeList nlQI = ir.getElementsByTagName("QuoteInfo2");
		Element qi = (Element) nlQI.item(0);
		if (qi == null) {
			/* QuoteInfo2 is missing */
			return UNVERIFIED;
		}
		int tag = Integer.parseInt(qi.getAttribute("Tag"));
		char[] fixed = qi.getAttribute("Fixed").toCharArray();
		
		/* LocalityAtRelease */
		NodeList nlLAR = ir.getElementsByTagName("LocalityAtRelease");
		Element lar = (Element) nlLAR.item(0);
		if (lar == null) {
			/* LocalityAtRelease is missing */
			return UNVERIFIED;
		}
		int locality = Integer.parseInt(lar.getTextContent());
		
		/* CompositeHash element */
		NodeList nlCH = ir.getElementsByTagName("CompositeHash");
		Element ch = (Element) nlCH.item(0);
		if (ch == null) {
			/* QuoteInfo2 is missing */
			return UNVERIFIED;
		}
		
		/* PcrSelection */
		NodeList nlPS = ir.getElementsByTagName("PcrSelection");
		Element ps = (Element) nlPS.item(0);
		if (ps == null) {
			/* PcrSelection is missing */
			return UNVERIFIED;
		}
		int SizeOfSelect = Integer.parseInt(ps.getAttribute("SizeOfSelect"));
		byte[] PcrSelect = Base64Tool.decode(ps.getAttribute("PcrSelect"));
		if (SizeOfSelect != PcrSelect.length) {
			/* Bad PcrSelection size */
			return UNVERIFIED;
		}
		if (SizeOfSelect != 3) {
			// TODO 24 => 3 only
			/* Bad PcrSelection size */
			return UNVERIFIED;
		}
		
		
		
		/* gen message */
		// TAG[2]
		message[0] = (byte)((tag >> 8) & 0xff); // (byte)0x00
		message[1] = (byte)(tag & 0xff);        // (byte)0x36	
		// QUOT2[4]
		message[2] = (byte)fixed[0]; //'Q'
		message[3] = (byte)fixed[1]; //'U'
		message[4] = (byte)fixed[2]; //'T'
		message[5] = (byte)fixed[3]; //'2'
		// NONCE [20]
		byte[] ed = Base64Tool.decode(qi.getAttribute("ExternalData"));
		for (int i = 0; i < 20; i++) message[6 + i] = ed[i];
		// TPM_PCR_SELECTION[5]
		message[26] = (byte)((SizeOfSelect >> 8) & 0xff); // (byte)0x00
		message[27] = (byte)(SizeOfSelect & 0xff); // (byte)0x03
		message[28] = PcrSelect[0];  // (byte)0xff
		message[29] = PcrSelect[1];  // (byte)0x00
		message[30] = PcrSelect[2];  // (byte)0x00
		// LOCALITY
		message[31] = (byte) locality; //(byte)0x00
		// COPMPOSIT_HASH
		byte[] bCH = Base64Tool.decode(ch.getTextContent());
		for (int i = 0; i < 20; i++) message[32 + i] = bCH[i];	
	
		// SignatureValue
		NodeList nlSV = ir.getElementsByTagName("SignatureValue");
		byte[] signature = Base64Tool.decode(nlSV.item(0).getTextContent());
	
		if (log.isInfoEnabled()) {
			log.info("validate signature ");
			log.info("  pubKey    : " + HexTool.getHexString(pubKey));
			log.info("  message   : " + HexTool.getHexString(message));
			log.info("  signature : " + HexTool.getHexString(signature));
		}
	
		if (SignatureTool.validate(pubKey, message, signature) == false) {
			if (log.isDebugEnabled()) {
				log.debug("Bad Signature, return 1");
			}
			// TODO 20071119 SM ignoe for a while
			log.error("Bad Signature, BUT CONTINUE");
			System.err.println("Bad Signature, BUT CONTINUE");
			// return 1; // INVALID
			
			_platformProps.setProperty("tpm.quote.signature", "invalid");
			
		} else {
			log.debug("Good Signature");
			_platformProps.setProperty("tpm.quote.signature", "valid");
		}
			
		/* Validate PCR Composite <-> message */
		/* PcrSelection */
		//NodeList nlPS = ir.getElementsByTagName("PcrSelection");
		//Element ps = (Element) nlPS.item(0);
		//int sizeOfSelect = new Integer(ps.getAttribute("SizeOfSelect"));
	
		_pcrComposite = new PcrComposite();
		_pcrComposite.setPcrNumber(SizeOfSelect * 8); // TODO
	
		NodeList nlPV = ir.getElementsByTagName("PcrValue");
		for (int i = 0; i < nlPV.getLength(); i++) {
			Element pv = (Element) nlPV.item(i);
			int pcrindex = new Integer(pv.getAttribute("PcrNumber"));
			_pcrComposite.selectPcr(pcrindex, Base64Tool.decode(pv
					.getTextContent()));
		}
	
		_pcrComposite.setExtraData(ed);
	
		if (_pcrComposite.validate2(message) == false) {
			if (log.isDebugEnabled()) {
				log.debug("Bad pcrComposite, return 1");
			}
			_platformProps.setProperty("tpm.quote.pcrs", "invalid");
			return INVALID;
		} else {
			_platformProps.setProperty("tpm.quote.pcrs", "valid");
		}
	
		if (log.isDebugEnabled()) {
			log.debug("Good QuoteInfo, return 0");
		}
		
		/* set prop */
		/* 2011-05-02 BAD INDEXING
		for (int i = 0; i< 24; i ++) {
			if (_pcrComposite.selected(i)) {
				String name = "tpm.quote.pcr." + i;
				String value = Base64Tool.encode(_pcrComposite.getPcrValue(i));
				_platformProps.setProperty(name, value);
			}
		}
		*/
		for (int i = 0; i< _pcrComposite.getSelectedPcrNums(); i ++) {
			int pcrIndex = _pcrComposite.getPcrIndex(i);
			String name = "tpm.quote.pcr." + pcrIndex;
			String value = Base64Tool.encode(_pcrComposite.getPcrValue(i));
			_platformProps.setProperty(name, value);
		}
		
		return VALID;
	}



	/**
	 * IR validation - From file Herem we wet Reason and Reqult into VR
	 * 
	 * @param irFilename
	 * @param rmFiles
	 * @param option
	 *            1:zipped file
	 * @return result
	 * @throws Exception 
	 * @throws Exception
	 */
	public VerificationResult validateIRfromFile(
			String irFilename,
			byte[] nonce,
			byte[] pubkey,
			String[] rmFiles,
			int option) throws Exception  {
		/* log */
		if (log.isTraceEnabled()) {
			log.trace("validateIRfromFile");
			log.trace("  irFilename : " + irFilename);
			if (nonce != null) {
				log.trace("  nonce      : " + HexTool.getHexString(nonce));
			}
			for (int i = 0; i < rmFiles.length; i++) {
				log.trace("  rmFiles[" + i + "] : " + rmFiles[i]);
			}
		}

		int result = UNVERIFIED;
		String reason = "";

		/* Integrity Properties */
		_integrityProps = new IntegrityProperties();
		
		/* Platform Properties */
		_platformProps = new PlatformProperties();
		_platformProps.init();
		
		/* Validate Quote -> Expected PCR Value */
		IntegrityReport ir;
		EventSet[] eventSet;
		try {
			ir = new IntegrityReport(irFilename, pubkey, option);
			// Check Quote2
			result = this.validateQuote2Data(ir, pubkey);
			if (result == VALID) {
				// Good
				if (log.isDebugEnabled()) {
					log.debug("validateQuote2Data ...done, valid ");
				}
			} else 	if (result == UNVERIFIED) {
				// check Quote
				result = this.validateQuoteData(ir, pubkey);
				if (result == VALID) {
					// Good
					if (log.isDebugEnabled()) {
						log.debug("validateQuoteData ...done, valid ");
					}			
				} else {
						/* Wrong Quote/Signature */
						log.error("Wrong or Unknown  Quote/Signature");
					}
			} else {
				/* Wrong Quote/Signature */
				log.error("Wrong Quote2/Signature");
			}
			//_pcrComposite = ir.getPcrComposite();
			eventSet = ir.getEventSet();
			String u = ir.getUuidString();
			_integrityProps.setProperty("uuid", u);
		} catch (Exception e) {
			// Internal Error
			log.error("Exception Error at Validate Quote ", e);
			e.printStackTrace();

			this.setReason("Internal Error");
			this.setResult(UNVERIFIED);
			return this;
		}

		/* init */
		if (log.isTraceEnabled()) {
			log.trace("init PlatformProperties");
		}


		/* Validate nonce */
		if (nonce != null) {
			if (_pcrComposite == null) {
				if (log.isDebugEnabled()) {
					log.debug("_pcrComposite == null ERROR?");
				}
				_platformProps.setProperty("tpm.quote.nonce", "unknown");
			} else {
				// TODO
				byte[] buf1 = _pcrComposite.getExternalData();

				if (log.isDebugEnabled()) {
					log.debug("nonce res:" + HexTool.getHexString(buf1));
					log.debug("nonce exp:" + HexTool.getHexString(nonce));
				}
				
				/* check the nonce */
				if (Arrays.equals(buf1, nonce)) {
					_platformProps.setProperty("tpm.quote.nonce", "valid");
				} else {
					_platformProps.setProperty("tpm.quote.nonce", "invalid");
				}
				
				// _props.setProperty("quote.nonce", Base64Tool.encode(nonce));
				_integrityProps.setProperty("uuid.base64", Base64Tool.encode(nonce));
				_integrityProps.setProperty("uuid.hex", HexTool.getHexString(nonce));
			}
		} else {
			if (log.isDebugEnabled()) {
				log.debug("Validate IR without Nonce");
			}
			// _props.setProperty("quote.nonce", "na");
			_platformProps.setProperty("tpm.quote.nonce", "unknown");
			_integrityProps.setProperty("uuid.base64", "na");
			_integrityProps.setProperty("uuid.hex", "na");
		}

		/* Validate IML */
		if (log.isTraceEnabled()) {
			log.trace("Validate IML");
		}
		int eventIndex = 0;
		int countValid = 0;
		int countInvalid = 0;
		int countUnverified = 0;
		
		for (int rmindex = 0; rmindex < rmFiles.length; rmindex++) {
			if (log.isDebugEnabled()) {
				log.debug("RM " + rmindex);
			}
			
			/* Setup FSM[] from RM */
			// ReferenceManifestGCJ rm = new ReferenceManifestGCJ(
			ReferenceManifest rm = new ReferenceManifest(
					rmFiles[rmindex],
					option);

			FiniteStateMachine[] fsms = null;
			try {
				fsms = rm.getValidationModel();
				if (log.isDebugEnabled()) {
					log.debug("FSM # " + fsms.length);
				}
			} catch (Exception e) {
				log.error("RM does not conains any FSM", e);
				return null;
			}

			/* Validate events by PCR */
			for (int pcrindex = 0; pcrindex < 24; pcrindex++) {
				if (log.isDebugEnabled()) {
					log.debug("RM " + rmindex + " PcrIndex " + pcrindex);
				}

				boolean validate = true;
				boolean noevent = false;
				Event[] eventList = null;
				FiniteStateMachine fsm = null;

				if (eventSet[pcrindex].pcrIndex != pcrindex) {
					log.error("Wrong Event...");
					return null;
				}

				if (eventSet[pcrindex].size == 0) {
					/* no event */
					if (pcrindex == 17) {
						/* PCR17 VMM */
						if (log.isDebugEnabled()) {
							log.debug("VMM Validate without Event in PCR "
									+ pcrindex);
						}
						noevent = true;
						// TODO revise later
						fsm = fsms[pcrindex];
						if (fsm == null) {
							if (log.isDebugEnabled()) {
								log.debug("Missing FSM for pcrindex="
										+ pcrindex + " SKIP");
							}
							validate = false;
						}
					} else {
						/* else */
						if (log.isDebugEnabled()) {
							log.debug("no event in PCR " + pcrindex + " SKIP");
						}
						validate = false;
					}
				} else {
					/* setup event and validation model(fsm) */
					eventList = eventSet[pcrindex].getEventList();
					fsm = fsms[pcrindex];
					if (fsm == null) {
						if (log.isDebugEnabled()) {
							log.debug("Missing FSM for pcrindex=" + pcrindex
									+ " SKIP");
						}
						validate = false;
					}
					fsm.setLevel(rmindex);
				}

				/* OK, Validate events now */
				if (validate) {
					if (_verbose) {
						System.out.println("--- PCR[" + pcrindex + "] ---");
					}
					if (log.isDebugEnabled()) {
						log.debug("Validate PCR " + pcrindex);
					}

					/* Validate */
					boolean rc = false;
					fsm.reset(_platformProps);
					if (_verbose) {
						fsm.setVerbose();
					}

					try {
						/* if validation was faild it issue Exception */
						if (noevent) {
							/* validate without event */
							rc = fsm.validate();
						} else {
							/* validate without IIDB */
							rc = fsm.validate(eventList, _integrityProps);
						}
						reason += fsm.getReason();

					} catch (Exception e) {
						// TODO For IMA event, Keep validation until last event
						log.error("Internal Error?", e);
						// Fail
						e.printStackTrace();
						rc = true; // Keep
						reason += "Inertnal error at PCR[" + pcrindex
								+ "] validation,";
						result = INVALID;
						//setIntegrityProp(eventIndex, eventList, "INVALID");
						// return null;
					}

					/* Final */
					if (rc == false) {
						if (log.isDebugEnabled()) {
							log.debug("Validation was failed");
						}
						reason += "PCR[" + pcrindex
								+ "] measurement is invalid,";
						// result = INVALID;
						countInvalid++;
					} else {
						// VALID
						if (log.isDebugEnabled()) {
							log.debug("VALID!!!");
						}

						_platformProps = fsm.getTveProperties();
						eventList = fsm.getEvents();

						if (eventList.length > 0) {
							/* Update Event List for next RM */
							if (log.isDebugEnabled())
								log.debug("Event remains for PCR " + pcrindex);
							eventSet[pcrindex].setEventList(eventList);
						}
						// reason += "Measurements sounds pretty neat.:-),";
						// result = VALID;
						
						countValid++;
					}
					eventIndex++;
					if (log.isDebugEnabled())
						log.debug("Validation of PCR " + pcrindex
								+ " events are done...");
				} // validate
			} // pcrindex
		} // rmindex

		if (log.isTraceEnabled()) {
			log.trace("Validate IML - done");
		}

		if (countInvalid > 0) {
			result = INVALID;
		} else if (countUnverified > 0) {
			//
		} else {
			/* Seems no bad measurement */
			reason += "Measurements sounds pretty neat.:-),";
			result = VALID;
		}

		/* Validate Properties by given Policy, PDP */
		if (log.isTraceEnabled()) {
			log.trace("Validate Properties by given Policy,  PDP");
		}

		/* */
		// TODO 
		if (_policyProp != null) {
			String name = null;
			String valueRef;
			String valueActual;
			try {
				int count = 0;
				for (Enumeration<?> e = _policyProp.propertyNames(); e
						.hasMoreElements();) {
					name = (String) e.nextElement();
					valueRef = _policyProp.getProperty(name);
					//valueActual = _platformProps.getPropertyValueByName(name);
					valueActual = _platformProps.getProperty(name);
					
					if (valueRef.equals(valueActual)) { // props.verifyProperty(name,
														// value)) {
						// HIT
						if (log.isDebugEnabled()) {
							log.debug("Policy) " + name + " == " + valueRef
									+ " HIT");
						}

					} else {
						// MISS
						if (log.isDebugEnabled()) {
							log.debug("Policy) " + name + " == " + valueActual
									+ " !=" + valueRef + "(ref) MISS");
						}

						reason += "Policy mismatch (" + name + "!=" + valueRef
								+ " but " + valueActual + "),";
						count++;

					}
				} // for
				if (count > 0) {
					reason += "some policy violation exist,";
					result = INVALID;
				} else {
					reason += "no policy violation,";
					// result = INVALID;
				}

			} catch (Exception e1) {
				reason += "Sorry, PTS Internal Error,";
				// Validation Fail
				// e1.printStackTrace();
				result = INVALID;
			}
		}
		
		
		if (result == VALID) {
			/* Validate PcrComposite */
			if (log.isTraceEnabled()) {
				log.trace("Validate PcrComposite");
			}
			if (_pcrComposite != null) {
				int selectedPcrNums = _pcrComposite.getSelectedPcrNums();

				for (int i = 0; i < selectedPcrNums; i++) {
					int pcrindex = _pcrComposite.getPcrIndex(i);
					byte[] pcr = _pcrComposite.getPcrValue(i);

					//PlatformProperty p;
					byte[] pcr2 = null;
					try {
						//p = _platformProps.getPropertyByName("tpm.pcr." + pcrindex);
						//pcr2 = Base64Tool.decode(p.getValue());
						String b64 = _platformProps.getProperty("tpm.pcr." + pcrindex);
						pcr2 = Base64Tool.decode(b64);
					} catch (Exception e) {
						// TODO Auto-generated catch block
						e.printStackTrace();
					}

					int rc = VALID;
					for (int k = 0; k < 20; k++) {
						if (pcr[k] != pcr2[k]) {
							rc = INVALID;
						}
					}

					if (rc == VALID) {
						// OK
						if (log.isDebugEnabled())
							log.debug("good PCR pcrindex " + pcrindex);
						// } else if (rc == 2) {
						// // IML < PCR Selection/Quote
						// if (log.isDebugEnabled()) log.debug("unverified PCR
						// pcrindex " + pcrindex);
						// reason += "PCR[" + pcrindex +"] is unverified,";
					} else {
						// NG
						log.error("wrong PCR pcrindex " + pcrindex);
						log.error("PCR(Quote) " + HexTool.getHexString(pcr));
						log.error("PCR(Calc)  " + HexTool.getHexString(pcr2));
						reason += "PCR[" + pcrindex
								+ "] value does not match with signature,";
					}
				}
			} else { // no QuoteInfo
				log.error("no QuoteInfo? _pcrComposite is null");
				//_platformProps.setProperty("tpm.quote.pcrs", "unverified");
				//_platformProps.setProperty("tpm.quote.signature", "unverified");
			}

		}
		if (log.isDebugEnabled())
			log.debug("SM DEBUG Validate PcrComposite ...done ");

		if (_verbose) {
			System.out.println("--- properties ---");
			_platformProps.printProperties();
			System.out.println("------------------");
		}
		//_platformProps.log();

		this.setResult(result);
		this.setReason(reason);
		if (log.isDebugEnabled()) {
			log.debug("Result : " + this.getResult());
			log.debug("Reason : " + this.getReason());
		}
		return this;
	}

	/**
	 * 
	 * @return VALID, INVALID, UNKNOWN
	 */
	public int validationPropertiesByPolicy() {
		String reason = "";
		int result = VALID;
		
		if (_policyProp != null) {
			String name = null;
			String valueRef;
			String valueActual;
			try {
				int count = 0;
				for (Enumeration<?> e = _policyProp.propertyNames(); e
						.hasMoreElements();) {
					name = (String) e.nextElement();
					valueRef = _policyProp.getProperty(name);
					//valueActual = _platformProps.getPropertyValueByName(name);
					valueActual = _platformProps.getProperty(name);
					
					if (valueRef.equals(valueActual)) { // props.verifyProperty(name,
														// value)) {
						// HIT only shows missed policy
						//if (log.isDebugEnabled()) {
						//	log.debug("Policy : HIT  <= " + name + " == " + valueRef );
						//}
	
					} else {
						// MISS
						if (log.isDebugEnabled()) {
							log.debug("Policy : MISS <= " + name + " == " + valueActual
									+ " != " + valueRef + " (policy)");
						}
	
						reason += "Policy mismatch (" + name + "!=" + valueRef
								+ " but " + valueActual + "),";
						count++;
	
					}
				} // for
				if (count > 0) {
					if (log.isDebugEnabled()) {
						log.debug("Policy check => INVALID");
					}
					reason += "some policy violation exist,";
					result = INVALID;
				} else {
					if (log.isDebugEnabled()) {
						log.debug("Policy check => VALID");
					}
					reason += "no policy violation,";
					// result = INVALID;
				}
	
			} catch (Exception e1) {
				reason += "Sorry, PTS Internal Error,";
				// Validation Fail
				// e1.printStackTrace();
				result = INVALID;
			}
		}
		
		return result;
	}

	/**
	 * 
	 * @return
	 * @throws Exception
	 */
	public boolean validateXML() throws Exception {
		TCGSchemas schema = new TCGSchemas();
		return schema.validate(_filename);
	}

	/**
	 * 
	 * @return
	 */
	public Properties getIntegrityProperties() {
		return _integrityProps;
	}

	/**
	 * 
	 * @return
	 */
	public PlatformProperties getPlatformProperties() {
		return _platformProps;
	}

	/**
	 * 
	 * @param filename
	 * @throws Exception
	 */
	public void saveIntegrityProperties(String filename) throws Exception {
		if (_integrityProps == null) {
			log.error("Internal error, missing Integrity Property");
			return;
		}
		/* add count */
		int num = _integrityProps.getEventNum();
		_integrityProps.setProperty("num", Integer.toString(num));
		_integrityProps.store(new FileOutputStream(filename), "generated by jOpenPTS");
	}

	/**
	 * 
	 * Platform Properties
	 * 
	 * @param filename
	 * @throws Exception
	 */
	public void savePlatformProperties(String filename) throws Exception {
		if (_platformProps == null) {
			log.error("Internal error, missing Property");
			return;
		}
		_platformProps.saveProperties(filename);
	}
}
