/*
 * 
 * Licensed Materials - Property of IBM
 *
 * Open Platform Trust Services - An open source TCG PTS
 *
 * (C) Copyright International Business Machines Corp. 2011
 *
 */

package com.ibm.trl.tcg.pts.verifier;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.UnknownHostException;
import java.util.Properties;
import java.util.UUID;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.ibm.trl.tcg.pts.attestation.IFM;
import com.ibm.trl.tcg.pts.attestation.PTS;
import com.ibm.trl.tcg.pts.engine.IntegrityProperties;
import com.ibm.trl.tcg.pts.engine.PlatformProperties;
import com.ibm.trl.tcg.pts.engine.Reasons;
import com.ibm.trl.tcg.pts.integrity.IntegrityReport;
import com.ibm.trl.tcg.pts.integrity.VerificationResult;
import com.ibm.trl.tcg.pts.tools.Base64Tool;
import com.ibm.trl.tcg.pts.tools.UuidTool;
import com.jcraft.jsch.JSchException;

/**
 * 
 * Class for each Target(PTS Collector).
 * 
 * 
 * @author Seiji Munetoh (munetoh@users.sourceforge.jp)
 *
 */
public class Target {
	/* Logger */
	private Log log = LogFactory.getLog(this.getClass());
	
	/**/
	Properties _target_conf;
	
	public UUID _uuid = null;
	public UUID _rm_uuid = null;
	public UUID _newrm_uuid = null;
	public UUID _oldrm_uuid = null;

	private String _dir= null;
	private String[] _rm_files= null;
	private String _ir_file= null;
	private VerificationResult _vr = null;

	/* target (collector)  */
	public String _hostname = null;
	private byte[] _pubkey = null;
	private int _rm_num;
	public String _prop_filename;
	public String _policy_filename;
	
	private Reasons _reasons= null;


	/* SSH remote exec  */
	public int  _ssh_port = 22;
	public String _ssh_username = null;
	// private SSH _ssh = null;
	private PTS _ptsc = null;

	
	/* IF-M */
	private byte[] _nonce = null;
	private String _integrity = "UNVERIFIED";
	public UUID _ir_uuid;
	
	/* Verify -> properties */
	private String _integrityPropFilename = null;
	private IntegrityProperties _integrityProp = null;


	
	
	/**
	 * 
	 * read UUID info, not RMs at this time 
	 * 
	 * @param dir
	 * @throws FileNotFoundException 
	 * @throws JSchException 
	 * @throws InterruptedException 
	 */
	public Target(File dir) throws FileNotFoundException, InterruptedException, JSchException {

		_dir = dir.getAbsolutePath();
		_reasons = new Reasons();
		
		try {
			// TODO Auto-generated constructor stub
			_uuid = UUID.randomUUID();  // read 
			
			/* read config, target.conf  */
			String target_conf_filename = _dir +  File.separator +  "target.conf";
			
			_target_conf = new Properties();
			_target_conf.load(new FileInputStream(target_conf_filename));
			
			/* target UUID  */
			String target_uuid = _target_conf.getProperty("target.uuid");			
			_uuid = UUID.fromString(target_uuid);
			
			/**/
			_hostname = _target_conf.getProperty("hostname");
			
			/* read *RM UUID file  */
			String rm_uuid_filename    = _dir + File.separator + _target_conf.getProperty("rm.uuid.file");
			_rm_uuid    = UuidTool.readUuidFile(rm_uuid_filename);
			
			/* policy */
			_policy_filename = _dir + File.separator + _target_conf.getProperty("policy.file");
			
			/* integrity prop */
			String ipfn = _target_conf.getProperty("integrity.file");
			if (ipfn != null) {
				_integrityPropFilename = _dir + File.separator + _target_conf.getProperty("integrity.file");
			} else {
				// use default name
				_integrityPropFilename = _dir + File.separator + "./ir.properties";
			}
			/* prop */
			_prop_filename = _dir + File.separator + _target_conf.getProperty("prop.file");
			
			/* SSH */
			String sshPortStr = _target_conf.getProperty("ssh.port");
			if (sshPortStr == null) {
				_ssh_port = 22;
			} else {
				_ssh_port = Integer.parseInt(sshPortStr);
			}
			_ssh_username = _target_conf.getProperty("ssh.username");
			if (_ssh_username == null) {
				_ssh_username = System.getProperty("user.name");
			}
			
			/* read pubkey */
			String pubkey_b64 = _target_conf.getProperty("target.pubkey");
			byte[] pubkey_hex = Base64Tool.decode(pubkey_b64);
			
			 // 00000001
			 // 0001
			 // 0002
			 // 0000000c
			 // 00000800
			 // 00000002
			 // 00000000
			 // 00000100
			 //
			 // 0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF
			 // ----------------------------------------------------------------
			 // 717a67aad618a7d6a51efe46cc7c3c23967bd0da83bc1745407b1cdadd8ac43a
			 // 1dfaf87d7bc0a19ccfbbc1218c10184bfcc35e9b080c46ece5ba02cbc72de1ec
			 // 7b2b9fcf399b10eb7f910db64f0ca4340959d4fe753a8eb9797251a2c7884903
			 // f6beb4fbf2e188e389807315860abc8c74cfb50dd0b6f85e052b7c62eded22d7
			 // 795e25365c8915f658e97e3fed7ce30d3a295a61be20427b3e0705b749362b81
			 // 83803c0482ccb276d486b9d7793fca99e9284433f7fc1512f94a9c8f0742416a
			 // a00f54683c03c131f9c333d26ec933e13e300d87bf4914659d5f0a1298a34bc7
			 // 5c41dd832d90e8380589e22f0363322f2e256f2cc28259c3079a08384cc6080f
			
			/* skip */
			_pubkey = new byte[1 + 256]; // TODO 
			_pubkey[0] = 0; // set MSB = 0
			for (int i = 0;i < 256; i++) {
				_pubkey[i + 1] = pubkey_hex[28 + i];
			}
		} catch (FileNotFoundException e) {
			log.error("Target file is missing at " + _dir);
			e.printStackTrace();
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		
		try {
			/* NEW/OLD RM UUID - Optional */
			String newrm_uuid_filename = dir.getAbsolutePath() + File.separator + _target_conf.getProperty("newrm.uuid.file");
			String oldrm_uuid_filename = dir.getAbsolutePath() + File.separator + _target_conf.getProperty("oldrm.uuid.file");
			_newrm_uuid = UuidTool.readUuidFile(newrm_uuid_filename);		
			_oldrm_uuid = UuidTool.readUuidFile(oldrm_uuid_filename);
		} catch (FileNotFoundException e) {
			/* Do not care */
			// TODO Auto-generated catch block
			// e.printStackTrace();
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		
		/* load the last result */
		loadIntegrityProperties();
		
	}

	
	/**
	 * Enrol new target
	 * 
	 * SSH remote exec
	 * 
	 * Verifier -> sshPort --> SSH tunnel --> port -> Collector
	 * 
	 * @param dir
	 * @param hostname
	 * @param port
	 * @param sshPort
	 * @param username
	 * @throws Exception 
	 */
	public Target(String conf_dirname, String hostname, String ssh_username, int ssh_port) throws Exception {
		_hostname = hostname;
		_ssh_username = ssh_username;
		_ssh_port = ssh_port;
		
		/* PTSC conenction over IF-M/SSH */
		_ptsc = new PTS(_hostname, _ssh_username, _ssh_port);
		
		/* init */
		init(conf_dirname);
	}
	
	public void initSsh() throws Exception {
		// TODO check
		if (_ptsc == null) {
			_ptsc = new PTS(_hostname, _ssh_username, _ssh_port);
		}
	}

	

	/**
	 * init dir
	 * 
	 * @param conf_dirname
	 * @throws Exception
	 */
	private void init(String conf_dirname) throws Exception {
		_reasons = new Reasons();
		
		/* access to target and get UUID */
		
		log.info("Target " + _hostname);
		
		helloUsingPtscIFM();
		
		/* setup Target DIR */
		String dirname = conf_dirname + File.separator + _uuid;
		File dir = new File(dirname);
		dir.mkdirs();
		_dir = dirname;
		log.info("mkdir, Create target dir : " + dirname);
		
		
		/* setup Target RM DIR */
		String rm_dirname = dirname + File.separator + _rm_uuid;
		File rm_dir = new File(rm_dirname);
		rm_dir.mkdirs();
		log.info("mkdir, Create manifest dir : " + rm_dirname);
		
	}

	/**
	 * load manifest files TODO bad name
	 * 
	 * @return
	 * @throws Exception
	 */
	public boolean setManifestFiles() throws Exception {
		_rm_num = Integer.parseInt(_target_conf.getProperty("rm.num"));
		_rm_files = new String[_rm_num];
		
		for (int i = 0; i < _rm_num; i++) {
			/* rm filename */
			_rm_files[i] =   _dir + File.separator + _rm_uuid.toString() + File.separator + "rm" + i + ".xml";
			log.debug("Manifest file : " + _rm_files[i]);			
		}
		return true;
	}

	/**
	 * load integrity report  TODO bad name
	 * 
	 * @return
	 */
	
	public boolean setIntegrityReportFile() {
		_ir_file = _dir + File.separator + "ir.xml";
		log.debug("Integrity Report file : " + _ir_file);
		return true;
	}

	/**
	 * 
	 * Get IR to create the table by GUI
	 * 
	 * TODO But the IR file did not contains validation info. gen ir.properties to save the detail of validation
	 * 
	 * 
	 * @return
	 * @throws Exception
	 */
	public IntegrityReport loadIntegrityReport() throws Exception {
		_ir_file = _dir + File.separator + "ir.xml"; // TODO mue be set by property
		IntegrityReport ir = new IntegrityReport(_ir_file, null);
		return ir;
	}


	public IntegrityProperties loadIntegrityProperties() {
		if (_integrityProp != null) return _integrityProp;
		
		/* load */
		_integrityProp = new IntegrityProperties();
		try {
			_integrityProp.load(new FileInputStream(_integrityPropFilename));
			
			String numStr = _integrityProp.getProperty("num");
			_integrityProp.eventIndex = Integer.parseInt(numStr);
			
			String uuid = _integrityProp.getProperty("uuid");
			_ir_uuid = UUID.fromString(uuid);
			
			return _integrityProp;
		} catch (FileNotFoundException e) {
			// initialize by C version of OpenPTS
			log.error("FileNotFoundException " + _integrityPropFilename);
			_integrityProp.eventIndex = 0;
			_ir_uuid = null;
			//e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
		
		return null;
	}


	/**
	 * 
	 * @return
	 * @throws Exception
	 */
	public boolean validateByManifest() throws Exception {		
		_vr = new VerificationResult();
		
		_vr.validateIRfromFile(_ir_file, _nonce, _pubkey , _rm_files, 0);

		if (_vr.getResult().equals("VALID")) {
			return true;
		}
		
		System.out.println("Result : " + _vr.getResult());
		System.out.println("Reason : " + _vr.getReason());
		return false;
	}

	/**
	 * 
	 * 
	 * 
	 * @return
	 * @throws Exception 
	 */
	public boolean validateByPolicy() throws Exception {
		/* check */
		if (_vr == null) {
			// validate before
			return false;
		}
		
		String filename = _dir + File.separator + "policy.conf";
		_vr.setValidationPolicy(filename);
		int result = _vr.validationPropertiesByPolicy();
		
		if (result == VerificationResult.VALID) {
			return true;
		} else {
			return false;
		}
	
	}


	/**
	 * 
	 * write Properties to file
	 * 
	 * ./openpts/UUID/vr.properties
	 * 
	 * @return
	 * @throws Exception 
	 */
	public boolean savePlatformProperties() throws Exception {
		if (_vr == null) {
			// validate before
			return false;
		}
		_vr.saveIntegrityProperties(_integrityPropFilename);
		_vr.savePlatformProperties(_prop_filename);
		return true;
	}


	private boolean savePropertiesAsPolicy() throws Exception {
		if (_vr == null) {
			log.error("_vr is null");
			return false;
		}
		
		// _policy_filename = _dir + File.separator + "policy.conf";
		
		PlatformProperties p =  _vr.getPlatformProperties();
		
		// C version did not store
		// tpm.pcr.0 - tpm.pcr.23
		// bios.nvram
		// bios.postbios.rom.strings
		// bios.escd
		// bios.cmos
		// bios.optionrom.execute
		// linux.kernel.cmdline./vmlinuz-2.6.32-71.14.1.el6.x86_64
		// linux.kernel.cmdline.rd_LVM_LV
		
		for (int i = 0; i < 24; i ++) {
			String name = "tpm.pcr." + i;
			p.remove(name);
		}
		
		p.remove("bios.nvram");
		p.remove("bios.postbios.rom.strings");
		p.remove("bios.escd");
		p.remove("bios.cmos");
	
		p.remove("bios.optionrom.execute");
		p.remove("integrity");
		
		p.remove("linux.kernel.cmdline.rd_LVM_LV");
		
		//p.saveProperties(_policy_filename, 0);
		p.saveProperties(_policy_filename);
		
		log.error("write policy file :" + _policy_filename);
		
		return true;
	}


	/**
	 * 
	 * Hello over IF-M
	 * 
	 * @return
	 * @throws Exception 
	 */
	public boolean helloUsingPtscIFM() throws Exception {
		boolean rc = true;
		
		try {
			_ptsc.open();
			
			/* Hello */
			_ptsc.sendCapability();
			_ptsc.receiveMessage();
			if (_ptsc.type == IFM.OPENPTS_CAPABILITY) {
				_ptsc.getCapablity();
				/* check UUIDs */
				if (_uuid == null) {
					_uuid = _ptsc.colector_uuid;
				} else if (_ptsc.colector_uuid.equals(_uuid) != true ) {
					/* UUID missmatch, bad target?  */
					return false;
				}
				if (_rm_uuid == null) {
					_rm_uuid = _ptsc.rm_uuid;
				} else if (_ptsc.rm_uuid.equals(_rm_uuid) != true ) {
					/* UUID missmatch, bad manifest?  */
					return false;
				}
			}
			_ptsc.close();	
		} catch (UnknownHostException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (InterruptedException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		
		return rc;
	}

	
	/**
	 *  Engagement over IF-M
	 *  
	 * @param force (1: override)
	 * @return
	 * @throws Exception 
	 */
	public boolean enroll(int force) throws Exception {
		boolean rc = true;
		
		log.info("enrollment start");
		
		try {
			//ifm.open(_ssh_out, _ssh_in);	
			_ptsc.open();
			/* Hello */
			_ptsc.sendCapability();
			_ptsc.receiveMessage();
			if (_ptsc.type == IFM.OPENPTS_CAPABILITY) {
				_ptsc.getCapablity();
				/* check UUIDs */
				if (_ptsc.colector_uuid.equals(_uuid) != true ) {
					/* UUID missmatch, bad target?  */
					// TODO fatal error
					_ptsc.close();
					log.error("UUID missmatch, bad target");
					return false;
				}
				if (_ptsc.rm_uuid.equals(_rm_uuid) != true ) {
					/* UUID missmatch, bad manifest?  */
					// TODO ask update or not 
					_ptsc.close();
					log.error("RM UUID missmatch, bad target");
					return false;
				}
			}
			
			/* get PubKey */
			byte[] pubkey_hex = null;
			_ptsc.sendRequestTpmPubkey();
			_ptsc.receiveMessage();
			if (_ptsc.type == IFM.TPM_PUBKEY) {
				log.info("length = " + _ptsc.length);
				
				pubkey_hex = _ptsc.value;
				//_pubkey = _ptsc.value;
				_pubkey = new byte[1 + 256]; // TODO 
				_pubkey[0] = 0; // set MSB = 0
				for (int i = 0;i < 256; i++) {
					_pubkey[i + 1] = pubkey_hex[28 + i];
				}

			} else {
				System.err.println("BAD message type : " + _ptsc.type);
			}
			log.info("get pubkey");
			
			/* get manifests */
			_ptsc.sendRequestManifestSets();
			_ptsc.receiveMessage();
			if (_ptsc.type == IFM.RIMM_SET) {
				//System.out.println("length = " + ifm.length);
				// rum num
				_rm_num = ba2int(_ptsc.value, 0);
				int ptr = 4;
				log.info("rm num  : " + _rm_num);
				
				_rm_files = new String[_rm_num];
				
				for (int i = 0; i < _rm_num; i++) {
					int len = ba2int(_ptsc.value, ptr);
					ptr += 4;
					
					/* rm filename */
					_rm_files[i] =   _dir + File.separator + _rm_uuid.toString() + File.separator + "rm" + i + ".xml";
					log.info("rm len  : " + len);
					log.info("rm file : " + _rm_files[i]);
					FileOutputStream out = new FileOutputStream(_rm_files[i]);
					out.write(_ptsc.value, ptr, len);
					out.flush();
					out.close();
					ptr += len;
					
					log.info("Manifest file : " + _rm_files[i]);					
				}
			} else {
				System.err.println("BAD message type : " + _ptsc.type);
			}
			log.info("get rm done");
			
			/**/
			String rm_uuid_filename = _dir + File.separator +  "rm_uuid";
			UuidTool.saveUuidFile(_rm_uuid, rm_uuid_filename);
			
			/* write conf */
			_target_conf = new Properties();
			_target_conf.setProperty("target.uuid", _uuid.toString());
			_target_conf.setProperty("target.pubkey", Base64Tool.encode(pubkey_hex));  // Base64
			_target_conf.setProperty("verifier.logging.dir", "./");
			_target_conf.setProperty("policy.file", "./policy.conf");
			_target_conf.setProperty("rm.basedir", "./");
			_target_conf.setProperty("rm.num", Integer.toString(_rm_num));  // TODO
			_target_conf.setProperty("rm.uuid.file", "./rm_uuid");
			_target_conf.setProperty("newrm.uuid.file", "./newrm_uuid");
			_target_conf.setProperty("oldrm.uuid.file", "./oldrm_uuid");
			_target_conf.setProperty("ir.file", "./ir.xml");
			_target_conf.setProperty("integrity.file", "./ir.properties");
			_target_conf.setProperty("prop.file", "./vr.properties");
			_target_conf.setProperty("ima.validation.mode", "none");
			//_target_conf.setProperty("ssh.mode", "on"); 
			_target_conf.setProperty("ssh.port", Integer.toString(_ssh_port));  // TODO
			_target_conf.setProperty("ssh.username", _ssh_username);  // TODO
			//} else {
			//	_target_conf.setProperty("ssh.mode", "off");  // TODO
			//}
			_target_conf.setProperty("hostname", _hostname);
			// TODO _target_conf.setProperty("port", Integer.toString(_port));
			
			/* save */
			String conf_filename = _dir + File.separator +  "target.conf";
			_target_conf.store(new FileOutputStream(conf_filename), "created by jOpenPTS");
			log.info("conf stored");
			
			// load conf 
			//_target_conf = new Properties();
			//_target_conf.load(new FileInputStream(conf_filename));
			
			/* Nonce */
			_nonce = _ptsc.sendNonce();
			
			/* get IR */
			_ptsc.sendRequestIntegrityReport();
			_ptsc.receiveMessage();
			if (_ptsc.type == IFM.INTEGRITY_REPORT) {
				//System.out.println("length = " + ifm.length);
				
				/* save to file */
				// ./openpts/UUID/ir.xml
				_ir_file = _dir + File.separator + "ir.xml";
				
				FileOutputStream out = new FileOutputStream(_ir_file);
				out.write(_ptsc.value, 0,_ptsc.length);
				out.flush();
				out.close();

			} else {
				System.err.println("BAD message type : " + _ptsc.type);
			}
			
			/* validate */
			
			if (setManifestFiles() != true) {
				System.err.println("loadManifest - fail ");
				rc = false;
			}
			
			if (setIntegrityReportFile() != true) {
				System.err.println("loadIntegrityReport - fail ");
				rc = false;
			}
			
			if (validateByManifest() != true) {
				System.err.println("validateByManifest - fail ");
				rc = false;
			}
			
			_policy_filename = _dir + File.separator +  "policy.conf";  // TTODO
			if (savePropertiesAsPolicy() != true) {
				System.err.println("savePolicy - fail ");
				rc = false;
			}
			
			
			//String policy_filename = _dir + File.separator +  "policy.conf";
			//p.store(new FileOutputStream(policy_filename), "created by jOpenPTS");
			//log.info("policy_filename stored");
			
			/*
			# generated by openpts. do not edit this file
			target.uuid=583e608e-32a2-11e0-a516-001f160c9c28
			target.pubkey=AAAAAQABAAIAAAAMAAAIAAAAAAIAAAAAAAABAMhg5f3JMz18qsruzCu/HRFMVhqqjgwnc4OOuNjJ9AXFgikpe90J3HB7FA9ox/Z01Z8cOgXVUhAcCL2VT/2QnGzkM25ou17xOUb0mPojrJD2sevuW4RGhyjkY24O5n5W+oYWAraJOADSrQPOjeZKoCvRl27QpLzYTU4Tc7v8tD0zSM4JGTjVkZ88yt/PhcLczx0GsK2PkQZCfVH4i+wsqlSgn63fqPD/wQX3DdpHGYZ0u4lmLUFwztqsS7jXdafzCU9sKlAkGih3xeji05w9CzW5xBfnFWTWheLG53wQTZhhJ7IkO8HaxAI/f7n80pOXMEys3XociQPkO6Q/0BFa6ns=
			verifier.logging.dir=./
			policy.file=./policy.conf
			rm.basedir=./
			rm.num=2
			rm.uuid.file=./rm_uuid
			newrm.uuid.file=./newrm_uuid
			oldrm.uuid.file=./oldrm_uuid
			ir.file=./ir.xml
			prop.file=./vr.properties
			ima.validation.mode=none
			ssh.mode=off
			hostname=localhost
			port=6678
	*/
			
			_ptsc.close();
		} catch (java.net.ConnectException e) {
			System.err.println("Conenction fail");
			System.err.println("  hostname : " + _hostname);
			System.err.println("  port     : " + _ssh_port);
			_reasons.addReason("[NETWORK] Connection to the collector at " + _hostname + ":" +  _ssh_port + " was faild");
			rc = false;
			e.printStackTrace();
		} catch (UnknownHostException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		} catch (InterruptedException e) {
			e.printStackTrace();
		}
		
		return rc;
	}
	
	
	
	/**
	 * 
	 * Verify over IF-M
	 * 
	 * @return
	 * @throws Exception 
	 */
	public int verify() throws Exception {
		int rc = VerificationResult.VALID;
		
		if (_ptsc == null) {
			_ptsc = new PTS(_hostname, _ssh_username, _ssh_port);
		}
		
		try {
			_ptsc.open();
			
			/* Hello */
			_ptsc.sendCapability();
			_ptsc.receiveMessage();
			if (_ptsc.type == IFM.OPENPTS_CAPABILITY) {
				_ptsc.getCapablity();
				/* check UUIDs */
				if (_ptsc.colector_uuid.equals(_uuid) != true ) {
					/* UUID missmatch, bad target?  */
					// TODO fatal error
					_ptsc.close();
					_reasons.addReason("[TARGET] bad UUID of collector");
					return VerificationResult.UNVERIFIED;
				}
				if (_ptsc.rm_uuid.equals(_rm_uuid) != true ) {
					/* UUID missmatch, bad manifest?  */
					// TODO ask update or not 
					_ptsc.close();
					_reasons.addReason("[TARGET] bad UUID of manifest");
					return VerificationResult.UNVERIFIED;
				}
			}
			
			/* Nonce */
			_nonce = _ptsc.sendNonce();
			
			/* get IR */
			_ptsc.sendRequestIntegrityReport();
			_ptsc.receiveMessage();
			if (_ptsc.type == IFM.INTEGRITY_REPORT) {
				/* save to file */
				// ./openpts/UUID/ir.xml
				_ir_file = _dir + File.separator + "ir.xml";
				
				FileOutputStream out = new FileOutputStream(_ir_file);
				out.write(_ptsc.value, 0,_ptsc.length);
				out.flush();
				out.close();
	
			} else {
				System.err.println("BAD message type : " + _ptsc.type);
			}
		
			_ptsc.close();
			
			/* validate IR */
			_integrity = "VALID";
			
			// TODO Public Key to validate the quote!
			
			/* load manifests */
			if (setManifestFiles() != true) {
				System.err.println("loadManifest - fail ");
				_reasons.addReason("[MANIFEST] can not load the manifest");
				rc = VerificationResult.UNVERIFIED;
				_integrity = "UNVERIFIED";
			}
			
			/* this just set the ir filename */
			if (setIntegrityReportFile() != true) {
				System.err.println("loadIntegrityReport - fail ");
				_reasons.addReason("[REPORT] can not load the integrity report");
				rc = VerificationResult.UNVERIFIED;;
				_integrity = "UNVERIFIED";
			}
			
			/* Validate IR by Manifest(FSM) - validation 1/2 */
			if (validateByManifest() != true) {
				System.err.println("validateByManifest - fail ");
				_reasons.addReason("[MANIFEST] validation was fail");
				rc = VerificationResult.INVALID;;
				_integrity = "INVALID";
			}
			
			/* Save vr.properties and ir.properties */
			if (savePlatformProperties() != true) {
				System.err.println("saveProperties - fail ");
				_reasons.addReason("[PROPERTY] save was fail");
				rc = VerificationResult.UNVERIFIED; // TODO ???
				_integrity = "UNVERIFIED";
			}
			
			/* reload integrity prop */
			_integrityProp = null; 
			loadIntegrityProperties();
			
			// TODO  Ignore  tpm.quote.pcr.N = XXX 
			// Do quote 
			/* validate By Policy - validation 2/2 */
			if (validateByPolicy() != true) {
				System.err.println("validateByPolicy - fail ");
				// TODO get reasons duaring validateByPolicy
				_reasons.addReason("[POLICY] validation was fail");
				rc = VerificationResult.INVALID;	
				_integrity = "INVALID";
			}			
		} catch (java.net.ConnectException e) {
			_reasons.addReason("[NETWORK] Connection to the collector at " + _hostname + ":" +  _ssh_port + " was faild");
			rc = VerificationResult.UNVERIFIED;
		} catch (UnknownHostException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		} catch (InterruptedException e) {
			e.printStackTrace();
		}
	
		return rc;
	}


	public void delete() throws IOException {
		// delete this target files
		// .openpts/UUID
		log.info("Delete target " + _dir);
		System.out.println("Delete target " + _dir);
		File f = new File(_dir);
		rmfile(f);
	}


	/**
	 * Byte[] -> int 
	 * @param value
	 * @param i offset
	 * @return
	 */
	private int ba2int(byte[] value, int offset) {
		int i32;
		
		log.debug("ba2int " + (value[offset] & 0xff));
		log.debug("ba2int " + (value[offset+1] & 0xff));
		log.debug("ba2int " + (value[offset+2] & 0xff));
		log.debug("ba2int " + (value[offset+3] & 0xff));
		
		i32 = value[offset] & 0xff;
		i32 = i32 << 8;
		i32 += value[offset + 1] & 0xff;
		i32 = i32 << 8;
		i32 += value[offset + 2] & 0xff;
		i32 = i32 << 8;
		i32 += value[offset + 3] & 0xff;
		return i32;
	}

	public void printReason() {
		System.out.println("Reason");
		
		_reasons.printReason();
	}


	public String getIntegrity() {
		return _integrity;
	}


	public Object getRemediation() {
		return _reasons.toString();
	}

	private void rmfile(File f) throws IOException {
		if (f.isDirectory()) {
		    File[] fl=f.listFiles();
		    for (int i=0;i<fl.length;i++) {
		    	rmfile(fl[i]);
		    }
		}
		if (!(f.delete())) {
		    throw new java.io.IOException("Delete "+f.getPath() + " was failed");
		}
    }


}
