/*
 * 
 * Licensed Materials - Property of IBM
 *
 * Open Platform Trust Services - An open source TCG PTS
 *
 * (C) Copyright International Business Machines Corp. 2011
 *
 */

package com.ibm.trl.tcg.pts.verifier;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
//import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.Scanner;
import java.util.UUID;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.ibm.trl.tcg.pts.tools.UuidTool;
import com.jcraft.jsch.JSchException;

/**
 * 
 * Class for the target list
 * 
 * 
 * 
 * @author Seiji Munetoh (munetoh@users.sourceforge.jp)
 *
 */
public class TargetList {
	/* Logger */
	private Log log = LogFactory.getLog(this.getClass());
	
	/* Verifier - base */
	private List<Target> _target_list = null;
	
	/* config */
	private String _dir = null;
	private File _configDir;
	
	/* status */
	private boolean _initialized = false;
	private boolean _force_create = false;

	
	/**
	 * HOME/.openpts
	 * @throws IOException 
	 * @throws FileNotFoundException 
	 * @throws JSchException 
	 * @throws InterruptedException 
	 */
	public TargetList() throws FileNotFoundException, IOException, InterruptedException, JSchException {
		_dir = System.getProperty("user.home") + File.separator +  ".openpts";	
		init(_dir);
	}
	
	public TargetList(boolean force) throws FileNotFoundException, IOException, InterruptedException, JSchException {
		_dir = System.getProperty("user.home") + File.separator +  ".openpts";	
		_force_create  = true;
		init(_dir);
	}


	/**
	 * 
	 * Init target list
	 * if config dir is missing -> create new dir
	 * 
	 * @param dirname
	 * @throws FileNotFoundException
	 * @throws IOException
	 * @throws JSchException 
	 * @throws InterruptedException 
	 */
	public void init(String dirname) throws FileNotFoundException, IOException, InterruptedException, JSchException {
		_configDir = new File(dirname);
		
		/* if dir is missing => create new dir, and conf*/
		if ( _configDir.exists() == false ) {

			if (_force_create == false) {
				/* ask */
				System.out.println("config dir " + dirname + " is missing create? [Y/n]");
				Scanner s = new Scanner(System.in);
				String str = s.next();
				if (str.equals("Y")) {
					_configDir.mkdirs();
				} else if (str.equals("y")) {
					_configDir.mkdirs();
				} else {
					System.out.println("abort");
					_initialized = false;
					return;
				}
			} else {
				/* force create */
				_configDir.mkdirs();
			}
			log.info("mkdir : " + dirname);
			
			/* uuid gen -> file */
			UUID u = UUID.randomUUID();  // TODO type 4
			String uuid_filename = _configDir.getAbsolutePath() + File.separator +  "uuid";		
			UuidTool.saveUuidFile(u, uuid_filename);
			
			//File uuid_file = new File(uuid_filename);
			//FileWriter filewriter = new FileWriter(uuid_file);
			//filewriter.write(u.toString());
			//filewriter.close();

			/* conf -> file */
			Properties p = new Properties();
			p.setProperty("uuid.file", "./uuid");
			p.setProperty("verifier.logging.dir","./");
			
			String conf_filename = _configDir.getAbsolutePath() + File.separator +  "openpts.conf";	
			
			p.store(new FileOutputStream(conf_filename), "created by jOpenPTS");
			

			log.info("create new dir :" + _configDir.getAbsolutePath());
			log.info("create new uuid :" + uuid_filename);
			log.info("create new openpts.config :" + conf_filename);
		}
		
		/* Log4j setting */
		// if verifier was initialized by C version of OpenPTS, it does not contains the settings for Log4J 
		initLog4j(_configDir);
		
		
		// load targets
		refresh();
		
		_initialized = true;
	}

	/**
	 * 
	 * @param configDir
	 * @throws FileNotFoundException
	 * @throws IOException
	 */
	private void initLog4j(File configDir) throws FileNotFoundException, IOException {
		String log4j_filename = configDir.getAbsolutePath() + File.separator +  "log4j.properties";	
		File log4j_file = new File(log4j_filename);
		if (log4j_file.exists() == false ) {
			/* log4j.properties */
			String logFilename =  configDir.getAbsolutePath() +  File.separator + "openpts.log";
			Properties log4j_prop = new Properties();
			log4j_prop.setProperty("log4j.rootCategory", "WARN, LOGFILE");
			log4j_prop.setProperty("log4j.appender.LOGFILE", "org.apache.log4j.FileAppender");
			log4j_prop.setProperty("log4j.appender.LOGFILE.File", logFilename);

			log4j_prop.setProperty("log4j.appender.LOGFILE.Append", "true");
			log4j_prop.setProperty("log4j.appender.LOGFILE.layout", "org.apache.log4j.PatternLayout");
			log4j_prop.setProperty("log4j.appender.LOGFILE.layout.ConversionPattern", "%p [%t] (%c:%L) - %m%n");
			
			log4j_prop.setProperty("log4j.appender.CONSOLE", "org.apache.log4j.ConsoleAppender");
			log4j_prop.setProperty("log4j.appender.CONSOLE.layout", "org.apache.log4j.PatternLayout");
			log4j_prop.setProperty("log4j.appender.CONSOLE.layout.ConversionPattern", "%5p [%t] (%F:%L) - %m%n");

			log4j_prop.store(new FileOutputStream(log4j_filename), "created by jOpenPTS");
			
		}
		
		String comlog_filename = configDir.getAbsolutePath() + File.separator +  "commons-logging.properties";	
		File comlog_file = new File(comlog_filename);
		if (comlog_file.exists() == false ) {
			/* commons-logging.properties  */
			Properties comlog_prop = new Properties();
			comlog_prop.setProperty("org.apache.commons.logging.log", "org.apache.commons.logging.impl.Log4JLogger");

			comlog_prop.store(new FileOutputStream(comlog_filename), "created by jOpenPTS");			
		}
	}

	public void refresh() throws FileNotFoundException, IOException, InterruptedException, JSchException {
		
		// TODO free
		
		try {
			/* read opnpts.conf */			
			String openpts_config_filename =  _configDir.getAbsolutePath() + File.separator + "openpts.conf";
			Properties verifier_conf = new Properties();
			verifier_conf.load(new FileInputStream(openpts_config_filename));
			String uuid_filename = _configDir.getAbsolutePath() + File.separator + verifier_conf.getProperty("uuid.file");
			
			//System.out.println(uuid_filename);
			//System.out.println(verifier_conf.getProperty("verifier.logging.dir"));
			log.info("uuid_filename        :" + uuid_filename);
			log.info("verifier.logging.dir :" + verifier_conf.getProperty("verifier.logging.dir"));
			
			/* read verifier UUID */
			//UUID uuid = UuidTool.readUuidFile(uuid_filename);
			
			/* search target config */
			File[] file_list = _configDir.listFiles();
			_target_list = new ArrayList<Target>();
			
			
			/* count dirs */
			for (int i = 0; i < file_list.length; i++) {
				// System.out.println(i + " " + file_list[i].getAbsolutePath());
				if (file_list[i].isDirectory()) {
					//System.out.println(i + " " + file_list[i].getAbsolutePath());
					log.info("target " + i + " : " + file_list[i].getAbsolutePath());
					Target t = new Target(file_list[i]);
					_target_list.add(t);
				}
			}
			// System.out.println("Target " + _target_list.size());
			log.info("target num :" + _target_list.size());
			
			
		} catch (FileNotFoundException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		_initialized = true;
	}
	
	
	
	
	public String getConfigDir() {
		return _dir;
	}

	/**
	 * 
	 * 
	 * 
	 * @return
	 */
	public List<Target> listTargets() {
		//System.out.println("Target " + _target_list.size());
		return _target_list;
	}

	/**
	 * target manifest  return
	 * -------------------------
	 *    X       -     null
	 *    O       X     null?
	 *    O       O     Target
	 * 
	 * 
	 * @param targetUuid
	 * @param rmUuid
	 * @return
	 */
	public Target getTarget(UUID targetUuid, UUID rmUuid) {
		/* lock up target */
		
		for (int i = 0; i < _target_list.size(); i++) {
			Target t = _target_list.get(i);
			if (t._uuid.equals(targetUuid)) {
				if (t._rm_uuid.equals(rmUuid)) {
					//System.out.println("getTarget() - HIT");					
					return t;
				}
			}			
		}
		return null;
	}
	
	
	public Target getTarget(String hostname) {
		/* check */
		if (_target_list == null) {
			// TODO not init.
			return null;
		}
		
		for (int i = 0; i < _target_list.size(); i++) {
			Target t = _target_list.get(i);
			if (t._hostname.equals(hostname)) {
					return t;
			}
		}
		return null;
	}
	
	public Target getTarget(int index) {
		return _target_list.get(index);
	}
	
	public void printAll() {
		System.out.println("number of target   : " + _target_list.size());
				
		for (int i = 0; i < _target_list.size(); i++) {
			Target t = _target_list.get(i);
			
			System.out.println("-----------------------------------------------------");
			System.out.println("Target         : " + t._hostname);
			System.out.println("port           : " + t._ssh_port);
			System.out.println("Collector UUID : " + t._uuid.toString());
			System.out.println("Manifest UUID  : " + t._rm_uuid.toString());
		
		}
		System.out.println("-----------------------------------------------------");
	}

	public int size() {
		return _target_list.size();
	}

	/**
	 * Add new target
	 * @param hostname
	 * @param username
	 * @param port
	 * @return
	 * @throws Exception
	 */
	public Target addTarget(String hostname, String username, int port) throws Exception {
		// new target
		Target t = new Target(_dir , hostname, username, port);
		// TODO add to the list
		// after enrollment was done, call refresh to add this target to the list
		return t;
	}

	public boolean isInitialized() {
		return _initialized;
	}

	public void deleteTarget(int index) throws IOException {
		// remove local data
		Target t = _target_list.get(index);
		t.delete();
		// remove target[i] from the list
		_target_list.remove(index);
		
	}


}
