/** @ingroup backend
 * @file  backendWrapperBidi.c
 * @brief definition of wrapper object for Bidi.
 *
 * This file defines BidiWrapperObj which is used by any backend program.
 * BidiWrapperObj is an object containing bidiC,bidi module name,XML parser...
 *
 * @date $Date: 2004/10/22 08:32:53 $
 * @version $Revision: 1.1.1.1 $
 *
 * Copyright (C) 2004 by Turbolinux,Inc.
 */

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifndef _GNU_SOURCE
#define _GNU_SOURCE
#endif

#include <string.h>
#include <signal.h>
#include <unistd.h>

#include "backendCommon.h"

//Prototypes
void signal_handler(int);
void initSignal(void);
BidiWrapperObj *newBidiObj(void);
void *deleteBidiObj(BidiWrapperObj*);
BidiWrapperObj *initBidi(const char *,int,const char*);
void deleteBidi(BidiWrapperObj*);
int startBidiJob(BidiWrapperObj*,int);
int endBidiJob(BidiWrapperObj*);
int endBidiRead(BidiWrapperObj*);
int startBidiRead(BidiWrapperObj*,int);
int readBidiStatus(BidiWrapperObj*);
int cancelBidiJob(BidiWrapperObj*);

/**
 * Function for signal handler.
 *
 */
void signal_handler(int sig)
{
  DEBUGPRINT(("got signal...%d",sig));
  switch(sig){
  case SIGALRM:
    bidiTimeout=1;
    break;
  case SIGTERM:
    bidiCancel=1;
    break;
  }
}

/**
 * Initialize signal functions.
 *
 */
void initSignal(void)
{
  struct sigaction action;
  memset(&action, 0, sizeof(action));
  sigemptyset(&action.sa_mask);
  action.sa_handler = signal_handler;
  sigaction(SIGALRM, &action, NULL);
  sigaction(SIGTERM, &action, NULL);
}

/**
 * Constructor that creates an empty BidiWrapperObj.
 *
 * @return Pointer to the initalized BidiWrapperObj.
 */
BidiWrapperObj *newBidiObj(void)
{
  DEBUGPRINT(("%s: start",__func__));
  BidiWrapperObj *self = (BidiWrapperObj *)malloc(sizeof(BidiWrapperObj));
  if(!self){
    DEBUGPRINT(("make BidiWrapperObj error!"));
    return NULL;
  }
  memset(self,0,sizeof(BidiWrapperObj));
  
  self->bidiC=NULL;
  self->bidiLang=NULL;
  self->bidiReadFD=0;
  self->bidiModuleName=NULL;
  self->bidiReadInterval=0;

  self->bidiCapJob=0;
  self->bidiDoJob=0;
  self->bidiEprogress=0;
  self->bidiReadStatus=0;

  self->XMLinfo=NULL;
  self->XMLParser = newParser();
  DEBUGPRINT(("make BidiWrapperObj success...%p",self));
  DEBUGPRINT(("%s: end",__func__));
  return self;
}

/**
 * Destructor that delete the BidiWrapperObj.
 *
 * @param bidiobj Pointer to BidiWrapperObj.
 * @return None.
 */
void *deleteBidiObj(BidiWrapperObj *bidiobj)
{
  DEBUGPRINT(("%s: start",__func__));
  if(bidiobj != NULL){
    free(bidiobj);
  }
  DEBUGPRINT(("%s: end",__func__));
}

/**
 * Function for initalize BidiWrapperObj.
 *
 * @param PPDfile  File path to the PPD file.
 * @param FD       File descriptor of printer device.
 * @param bidiLang Pointer to the language string.
 * @return Pointer to the new BidiWrapperObj.
 */
BidiWrapperObj *initBidi(const char *PPDfile, int FD,const char *bidiLang)
{
  DEBUGPRINT(("%s: start",__func__));
  int capRet;
  BidiWrapperObj *self;

  self= newBidiObj();
  if(fetchPPDEntry(PPDfile,self) < 0){
    DEBUGPRINT(("cannot read PPD file correctly"));
    return NULL;
  }

  DEBUGPRINT(("bidiLanguage=%s",bidiLang));
  DEBUGPRINT(("module=%s",self->bidiModuleName));
  DEBUGPRINT(("read interval=%d",self->bidiReadInterval));

  alarm(BIDI_TIMEOUT);
  self->bidiC = bidiNew(self->bidiModuleName, FD, FD, NULL);
  alarm(0);
  if(bidiTimeout){
    DEBUGPRINT(("%s:timeout",__func__));
    bidiTimeout=0;
  }
  if(!self->bidiC){
    DEBUGPRINT(("cannot make bidi object"));
    return NULL;
  }

  alarm(BIDI_TIMEOUT);
  if( (capRet = bidiGetCap(self->bidiC,BIDI_CAP_JOB)) == BIDI_ERROR ){
    bidiDestroy(self->bidiC);
    self->bidiC=NULL;
    alarm(0);
    if(bidiTimeout){
      DEBUGPRINT(("%s:timeout",__func__));
      bidiTimeout=0;
    }
  }

  self->bidiCapJob = ( capRet == BIDI_TRUE ? 1:0 );
  self->bidiReadFD = bidiGetReadFD(self->bidiC);
  self->bidiLang = (char *)malloc(strlen(bidiLang)+1);
  self->bidiLang = strndup(bidiLang,strlen(bidiLang));

  DEBUGPRINT(("%s: end",__func__));
  return self;
}

/**
 * Wrapper function of bidiDestroy.
 *
 * @param bidiobj Pointer to BidiWrapperObj.
 * @return None.
 */
void deleteBidi(BidiWrapperObj *bidiobj)
{
  DEBUGPRINT(("%s: start",__func__));
  if(bidiobj){
    if(bidiobj->bidiC){
      alarm(BIDI_TIMEOUT);
      bidiDestroy(bidiobj->bidiC);
      alarm(0);
      if(bidiTimeout){
	DEBUGPRINT(("%s:timeout",__func__));
	bidiTimeout=0;
      }
      bidiobj->bidiC = NULL;
      free(bidiobj->XMLinfo);
      deleteParser(bidiobj->XMLParser);
    }
  }
  DEBUGPRINT(("%s: end",__func__));
}

/**
 * Wrapper function of bidiStartJob.
 *
 * @param bidiobj Pointer to BidiWrapperObj.
 * @param jobid Id number of Job.
 * @retval < 0 Fail.
 * @retval > 0 Success.
 */
int startBidiJob(BidiWrapperObj *bidiobj,int jobid)
{
  DEBUGPRINT(("%s: start",__func__));

  int ret;
  if(!bidiobj) return -1;
  if(!bidiobj->bidiCapJob){
    DEBUGPRINT(("bidi object does not support BIDI_CAP_JOB"));
    return -1;
  }
  if(bidiobj->bidiEprogress){
    DEBUGPRINT(("bidiEprogress is already set"));
    return -1;
  }

  ret = bidiStartJob(bidiobj->bidiC,jobid);
  switch(ret){
  case BIDI_ERROR:
    DEBUGPRINT(("bidiStartJob return BIDI_ERROR"));
    deleteBidi(bidiobj);
    break;
  case BIDI_EPROGRESS:
    DEBUGPRINT(("bidiStartJob return BIDI_EPROGRESS"));
    bidiobj->bidiEprogress = 1;
    break;
  }
  bidiobj->bidiJobID=jobid;
  bidiobj->bidiDoJob=1;

  DEBUGPRINT(("%s: end",__func__));
  return ret;
}

/**
 * Wrapper function of bidiEndJob.
 *
 * @param bidiobj Pointer to BidiWrapperObj.
 * @retval < 0 Fail.
 * @retval > 0 Success.
 */
int endBidiJob(BidiWrapperObj *bidiobj)
{
  DEBUGPRINT(("%s: start",__func__));
  int ret;
  if(!bidiobj) return -1;
  if(!bidiobj->bidiCapJob){
    DEBUGPRINT(("bidi object does not support BIDI_CAP_JOB"));
    return -1;
  }
  if(!bidiobj->bidiDoJob){
    DEBUGPRINT(("bidiDoJob is not set"));
    return -1;
  }

  if(bidiobj->bidiC)
    ret=bidiEndJob(bidiobj->bidiC);
  switch(ret){
  case BIDI_EPROGRESS:
    DEBUGPRINT(("bidiEndJob return BIDI_EPROGRESS"));
    bidiobj->bidiEprogress=1;
    break;
  case BIDI_ERROR:
    DEBUGPRINT(("bidiEndJob return BIDI_ERROR"));
    deleteBidi(bidiobj);
    break;
  default:
    bidiobj->bidiDoJob=0;
    break;
  }
  DEBUGPRINT(("%s: end",__func__));
  return 0;
}

/**
 * Wrapper function of bidiEndRead.
 *
 * @param bidiobj Pointer to BidiWrapperObj.
 * @retval < 0 Fail.
 * @retval > 0 Success.
 */
int endBidiRead(BidiWrapperObj *bidiobj)
{
  DEBUGPRINT(("%s: start",__func__));
  int ret=0;
  if(!bidiobj) return -1;
  if(bidiobj->bidiReadStatus){
    if ( (ret=bidiEndRead(bidiobj->bidiC)) == BIDI_ERROR ){
      DEBUGPRINT(("bidiEndRead return BIDI_ERROR"));
      deleteBidi(bidiobj);
    }
  }
  bidiobj->bidiReadStatus=0;
  bidiobj->XMLinfo=NULL;

  DEBUGPRINT(("%s: end",__func__));
  return ret;
}

/**
 * Wrapper function of bidiStartRead.
 *
 * @param bidiobj Pointer to BidiWrapperObj.
 * @param bidiReadMode Enum of the Bidi reading mode.
 * @retval < 0 Fail.
 * @retval > 0 Success.
 */
int startBidiRead(BidiWrapperObj *bidiobj,int bidiReadMode)
{
  DEBUGPRINT(("%s: start",__func__));
  int ret;

  if(!bidiobj) return -1;
  if(bidiobj->bidiReadStatus){
    DEBUGPRINT(("bidiReadStatus is already set"));
    return -1;
  }

  //  alarm(10);
  ret=bidiStartRead(bidiobj->bidiC,bidiReadMode,bidiobj->bidiLang);
  if ( ret == BIDI_ERROR ){
    DEBUGPRINT(("bidiStartRead return BIDI_ERROR"));
    deleteBidi(bidiobj);
    return -1;
  }
  if ( ret == BIDI_EINTR ){
    DEBUGPRINT(("bidiStartRead return BIDI_EINTR"));
  }
  //  alarm(0);
  if(bidiTimeout){
    DEBUGPRINT(("%s:timeout",__func__));
    bidiTimeout=0;
  }

  bidiobj->bidiReadStatus=1;
  bidiobj->infoSize=0;
  bidiobj->XMLinfo = (char *)malloc(sizeof(char));
  memset(bidiobj->XMLinfo,'\0',sizeof(char));

  DEBUGPRINT(("%s: end",__func__));
  return ret;
}

/**
 * Wrapper function of bidiReadStatus.
 *
 * @param bidiobj Pointer to BidiWrapperObj.
 * @retval < 0 Fail.
 * @retval > 0 Success.
 */
int readBidiStatus(BidiWrapperObj *bidiobj)
{
  DEBUGPRINT(("%s: start",__func__));
  int readsize;
  int buf_size=512;
  char buf[buf_size+1];

  if(!bidiobj) return -1;
  if(!bidiobj->bidiReadStatus){
    startBidiRead(bidiobj,BIDI_READ_PRT_MIB_SUMMARY);
  }

  memset((void *)buf,0,buf_size+1);
  alarm(10);
  if ( (readsize=bidiRead(bidiobj->bidiC,buf,buf_size)) == BIDI_EINTR){
    DEBUGPRINT(("bidiRead return BIDI_EINTR"));
    return -1;
  }
  alarm(0);
  if(bidiTimeout){
    DEBUGPRINT(("%s:timeout",__func__));
    bidiTimeout=0;
  }

  if(readsize > 0){
    bidiobj->infoSize+=readsize;
    bidiobj->XMLinfo = (char*)realloc(bidiobj->XMLinfo,bidiobj->infoSize);
    strncat(bidiobj->XMLinfo,buf,readsize); 
  }
  DEBUGPRINT(("%s: end",__func__));
  return readsize;
}

/**
 * Wrapper function of bidiCancelJob.
 *
 * @param bidiobj Pointer to BidiWrapperObj.
 * @retval < 0 Fail.
 * @retval > 0 Success.
 */
int cancelBidiJob(BidiWrapperObj *bidiobj)
{
  DEBUGPRINT(("%s: start",__func__));
  int ret=0;

  if(!bidiobj) return -1;
  if(bidiobj->bidiCapJob == 0 ){
    DEBUGPRINT(("not support CancelJob"));
    return -1;
  }

  ret = bidiCancelJob(bidiobj->bidiC,bidiobj->bidiJobID);
  if ( ret == BIDI_ERROR ){
    DEBUGPRINT(("bidiCancelJob return BIDI_ERROR"));
    deleteBidi(bidiobj);
    return -2;
  }else if ( ret == BIDI_ENOJOB ){
    DEBUGPRINT(("bidiCancelJob return BIDI_ENOJOB"));
    return -3;
  }else if ( ret == BIDI_EPROGRESS ){
    DEBUGPRINT(("bidiCancelJob return BIDI_EPROGRESS"));
    return -4;
  }else if ( ret == BIDI_OK ){
    DEBUGPRINT(("bidiCancelJob OK"));
    bidiCancel=0;
    return 0;
  }
  DEBUGPRINT(("%s: end",__func__));
  return -5;
}

