/**@ingroup backend
 * @file  backendWriteData.c
 * @brief Functions for outputting print data to printer,fetching printer status from bidi module.
 *
 * @date $Date: 2004/10/22 08:32:53 $ \n
 * @version $Revision: 1.1.1.1 $
 *
 * 2004/02/13 Modified by Canon.INC \n
 * Copyright CANON INC. 2004 \n
 * 1) Modified the select loop to set only the necessary fd into the fdset under each condition.\n
 * 2) Modified to read the printer status data every interval time.\n
 * 3) Read the entire printer status data at one select loop.\n
 * 4) Change the buffer byte size to "PIPE_BUF".\n
 * 5) Set an alarm signal before writing the printer data.\n
 * 6) Set an appropriate fd into the "bidiFD".\n
 * 7) Set an appropriate timeout value into the select() function.\n
 *
 * Copyright (C) 2004 by Turbolinux,Inc.\n
 */

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <fcntl.h>
#include <stdio.h>
#include <errno.h>
#include <signal.h>
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <limits.h>
#include <string.h>
 
#include <bidiEntry.h>
#include <bidiParser.h>
#include <bidiEntryCUPS.h>

#include "backendCommon.h"

#define STATE_MESG_SIZE     1024 /**< size of status message buffer.*/ 
#define BIDI_WRITE_TIMEOUT  2    /**< number of writing data timeout.*/ 
#define DATA_BUF_SIZE       (PIPE_BUF)

//Prototypes
void writeData(BidiWrapperObj*,int,int);

/**
 * Function for outputting print data to outFD,fetching printer status from inFD via BidiWrapperObj.
 * 
 * @param objBidi Pointer to BidiWrapperObject.
 * @param inFD    File descriptor of input data.
 * @param outFD   File descriptor of printer device.
 */
void writeData(BidiWrapperObj *objBidi,int inFD,int outFD)
{
  ssize_t in_bytes=0;         /* number of bytes of input data.*/
  ssize_t out_bytes=0;        /* number of bytes of output data.*/
  ssize_t out_bytes_total=0;  /* number of total bytes of output data.*/
  char buffer[DATA_BUF_SIZE]; /* buffer data of output data*/
  char *bufptr=NULL;               /* pointer to buffer data */
  int bidiFD=0;               /* file descriptor of bidi module for read printer status*/
  int doRead=1;               /* can data be read from bidi? */
  fd_set readfds;
  fd_set writefds;
  int maxfd;
  struct timeval tout;
  time_t tread;
  time_t tprev;
  infoCUPS *info;
  char state_mesg[STATE_MESG_SIZE];

  tprev = time(NULL); /* get start time */

  if( startBidiRead(objBidi,BIDI_READ_PRT_MIB_SUMMARY) < 0){ /* can status data be read ? */
    fputs("INFO: BidiRead error\n", stderr);
    doRead=0;
  }else{
    doRead=1;
  }

  if (objBidi)
    bidiFD = objBidi->bidiReadFD;

  maxfd = (bidiFD > outFD ? bidiFD : outFD); /* check maximum file descriptor for select() */
  maxfd++;

  setbuf(stderr, NULL);

  while(1){
    FD_ZERO(&readfds);
    FD_ZERO(&writefds);

    if (in_bytes > 0)
      FD_SET(outFD,&writefds);
    else
      FD_SET(inFD,&readfds);

    tread = time(NULL);

    if(objBidi){
      if (tread - tprev > objBidi->bidiReadInterval ){
	tprev = tread;
	FD_SET(bidiFD,&readfds);
      }
    }

    if(objBidi){
      tout.tv_sec = objBidi->bidiReadInterval;
    }else{
      tout.tv_sec = 2;
    }
    tout.tv_usec = 0;

    if ( select(maxfd,&readfds,&writefds,NULL,&tout) < 0){
      perror("select error");
      continue;
    }

    /* perform cancel */
    if(bidiCancel){
      DEBUGPRINT(("Cancel Job"));
      if(cancelBidiJob(objBidi) == 0){
	break;
      }
    }

    /* if can read data from inFD */
    if(FD_ISSET(inFD,&readfds)){
      in_bytes = read(inFD, buffer, sizeof(buffer));
      if(in_bytes < 0){
	perror("read error");
	break;
      }
      if(in_bytes == 0) break;
      bufptr = buffer;
    }

    /* if can out data to outFD */
    if(FD_ISSET(outFD,&writefds)){
      alarm(BIDI_WRITE_TIMEOUT);
      out_bytes = write(outFD,bufptr,in_bytes);
      alarm(0);

      if(out_bytes < 0){
	if ( errno == ENOTTY ){
	  out_bytes = write(outFD,bufptr,in_bytes);
	}
	else if( errno == EINTR ){
	  out_bytes = 0;
	}
      }
      if(out_bytes >= 0){
//	perror("ERROR: Unable to send print file to printer");
//	break;
        out_bytes_total += out_bytes;
        in_bytes -=out_bytes;
        bufptr += out_bytes;
      }
    }

    /* if can read status data from inFD */
    if(objBidi && doRead && FD_ISSET(bidiFD,&readfds)){
      int status_size = 0;
      int size;

      while ( (size = readBidiStatus(objBidi)) > 0 )
	status_size += size;

      if (status_size > 0) {
	DEBUGPRINT(("read well-formed XML info"));
	writeXMLdata(objBidi,getenv("PRINTER"));

	if( info = parseXML(objBidi) ){
	  char mesg_buf[256];

	  snprintf(state_mesg,6,"INFO:");
	  fprintf(stderr,"STATE:\n");
	  while(info){
	    fprintf(stderr,"STATE:+%s\n",info->state_reason);
	    if( strlen(state_mesg)+1+strlen(info->state_messages)+1+2 < STATE_MESG_SIZE ){
	      snprintf(mesg_buf,256,"%s\t",info->state_messages);
	      strncat(state_mesg,mesg_buf,strlen(info->state_messages)+1);
	    }
	    info = info->next;
	  }
	  if(strlen(state_mesg) > 6){
	    fprintf(stderr,"INFO: \n"); /* clear print status */
	    fprintf(stderr,"%s\n",state_mesg);
	  }
	}

	endBidiRead(objBidi);
	if( startBidiRead(objBidi,BIDI_READ_PRT_MIB_SUMMARY) < 0){
	  fputs("INFO: BidiRead error\n", stderr);
	}else{
	  doRead=1;
	}
      }
    }
    //  fprintf(stderr, "INFO: printing %d bytes\n",(unsigned long)out_bytes_total);
  }
  endBidiRead(objBidi);
}
