/** @file
 * @brief HTTP client
 *
 * This file is part of the `Printer Status Utility Type B' program.
 *
 * @author Copyright (C) 2003,2004 EPSON KOWA Corporation
 * @date 2004/02/06
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif


#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <stdarg.h>
#include <string.h>
#include <netdb.h>

#include <signal.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netinet/in.h>

#include "httpc.h"
#include "psub_debug.h"

#define CR '\r'
#define LF '\n'

/* for Signal */
typedef void Sigfunc(int);

Sigfunc *Signal(int signo, Sigfunc *func);
static void connect_alarm(int signo);

/* utils */
static int set_sock_timeout(int sock_fd, int timeout_sec);
static int connect_timeout(int sock_fd, const struct sockaddr *serv_addr, socklen_t addrlen, int timeout_sec);
static int read_line(int fd, char *buff, int maxLen);
static int read_all(int fd, const void *buf, size_t buf_size);
static int throw_out_all(int fd, size_t size);
static int write_all(int fd, const void *buf, size_t nBytes);
static int parse_a_header(char *header, char** value);
static char to_hex(unsigned char c);
static char to_char(unsigned char upper, unsigned char lower);
static char h2c(unsigned char c);


/* debug */
#if defined(DEBUG)
static void dump_hostent(struct hostent *rhe);
#endif


httpc_handle *
http_open(const char *server_name,
	  int port,
          int timeout_sec)
{
        httpc_handle *handle;
        struct hostent *server;
        struct sockaddr_in server_addr;
        int sock_fd;

        ASSERT(NULL != server_name);
        ASSERT(port > 0);
        ASSERT(timeout_sec > 0);

        handle = (httpc_handle*)malloc(sizeof(httpc_handle));
        if (handle == NULL) {
                goto END_OPEN_NOOP;
        }

        server = gethostbyname(server_name);
        if (NULL == server) {
                dbgp(dbg, "failed to resolv hostname failed(%s).\n",
                     server_name);
                goto END_OPEN_FREE;
        }

#if defined(DEBUG)
        /* for debug only */
        dump_hostent(server);
#endif

        /* Socket */
        memset((char *)&server_addr, 0, sizeof(server_addr));
        server_addr.sin_family = AF_INET;
        server_addr.sin_port = htons(port);
        memcpy((char *)&server_addr.sin_addr,
               server->h_addr, server->h_length);

        sock_fd = socket(AF_INET, SOCK_STREAM, 0);
        if (sock_fd < 0) {
                dbgp(dbg, "failed to create a socket.\n");
                goto END_OPEN_FREE;
        }

        /* socket timeout */
        if (OK != set_sock_timeout(sock_fd, timeout_sec)) {
                dbgp(dbg, "failed to set socket timeout.\n");
                goto END_OPEN_CLOSE;
        }

        /* ³ */
        {
                int result;

                result = connect_timeout(sock_fd,
                                         (struct sockaddr *)&server_addr,
                                         sizeof(server_addr),
                                         timeout_sec);
                if (-1 == result) {
                        dbgp(dbg, "failed to connect to server(errno=%d).\n",
                             errno);
                        goto END_OPEN_CLOSE;
                }
        }

        /* λ */
        handle->fd = sock_fd;
        handle->timeout_sec = timeout_sec;
        handle->server_name = server_name;
        handle->code = 0;
        handle->content_type = NULL;
        handle->location = NULL;
        handle->content_length = -1;

        return handle;

  END_OPEN_CLOSE:
        close(sock_fd);
  END_OPEN_FREE:
        free(handle);
  END_OPEN_NOOP:
        return NULL;
}

int
http_get(httpc_handle *handle,
	 const char *path,
	 char *response_entity,
	 int buff_size)
{

        int fd;

        ASSERT(NULL != handle);
	
        dbgp(dbg, "handle=%p, path=[%s], response_entity=%p, buff_size=%d\n",
             handle,
             path,
             response_entity,
             buff_size);

        fd = handle->fd;

        /* handleresponseϰö */
        {
	  
                handle->content_length = -1;
                handle->code = 0;
                if (NULL != handle->content_type) {
                        free(handle->content_type);
                        handle->content_type = NULL;
                }
                if (NULL != handle->location) {
                        free(handle->location);
                        handle->location = NULL;
                }
        }
  
        /* reset sock timeout */
        /* http_open()塢桼handletimeoutͤѤǽ */
        set_sock_timeout(fd, handle->timeout_sec);

        /* write request */
        {
                char *request_buf;
                int size_of_request_buf;
                int len;
                int write_result;

                /* ꥯѤΥХåե */
                size_of_request_buf = strlen(handle->server_name) + strlen(path) + 26;
                request_buf = (char*)malloc(sizeof(char) * size_of_request_buf);
                if (NULL == request_buf) {
                        return SOCK_ERR;
                }

                sprintf(request_buf, "GET %s HTTP/1.1\r\nHost: %s\r\n\r\n",
                        path,
                        handle->server_name);
                len = strlen(request_buf);

                write_result = write_all(fd, request_buf, len);
                free(request_buf);
                request_buf = 0;

                if (OK != write_result) {
                        /* Barkley SocketǤϸwriteΥॢȤȯʤ */
                        /* fctl()ioctl()ǥǡ줿Ȥݾڤ륪ץϤ뤫⤷ʤ */
                        dbgp(dbg, "send request failed(errno=%d).\n", errno);
                        return SOCK_ERR;
                }
        }
	

        /* read response header */
        {
#define LINE_BUFFERSIZE 1024 /* TBD äRFC줿ͤǤϤʤ */

                char line_buf[LINE_BUFFERSIZE];
                int result;

                /* status line */
                result = read_line(fd, line_buf, LINE_BUFFERSIZE);
                if (result < 0) {
                        if (EWOULDBLOCK == errno) {
                                dbgp(dbg, "read status line timedout(errno=%d).\n", errno);
                        }
                        dbgp(dbg, "read status line failed(errno=%d).\n", errno);
                        return SOCK_ERR;
                } else if (result == 0) {
                        /* Ĺ0Υǡ֤ä뤳ȤϤꤨʤ */
                        dbgp(dbg, "No Response-Line.(%d) errno=%d\n", result, errno);
                        return SOCK_ERR;
                }

                /* dbgp(dbg, "ResponseStatus: (%d)%s", result, line_buf); */

                /* parsing status code */
                {
                        char code[4];
                        strncpy(code, line_buf + 9, 3);
                        handle->code = atoi(code);
			
                        /* dbgp(dbg, "\tStatus Code = %d\n", handle->code); */
                }

                /* ñ̤ǥإåɤߤǤ*/
                /* TBD : ʣԤϤإåΤˤߤʤ
                   ưΤʤϤ */
                do {
                        char *value;
                        int parse_result;
			
                        result = read_line(fd, line_buf, LINE_BUFFERSIZE);
                        if (result < 0) {
                                return SOCK_ERR;
                        }

                        /* dbgp(dbg, "ResponseHeader: (%d)%s", result, line_buf); */

                        /* إåΥѡ */
                        /* ɬפʥإåξȴƽ */
                        /* buffer overflowȯ硢read_lineɤߤȤФ */
                        parse_result = parse_a_header(line_buf, &value);
                        if (0 != parse_result) {
                                if (0 == strcmp(line_buf, "Content-Length:")) {
                                        /* Content-Length */
                                        /* TBD Apacheϡ200ʳξ硢Entitychunk֤
                                           ΤΥɤǤchunk򰷤ʤᡢApacheΥ顼Entityɤߤʤ
                                           ʤ */
                                        handle->content_length = atoi(value);
                                } else if (0 == strcmp(line_buf, "Content-Type:")) {
                                        /* Content-Type */
#if 0
                                        if (NULL != handle->content_type) {
                                                free(handle->content_type);
                                                handle->content_type = NULL;
                                        }
#endif
                                        handle->content_type = (char*)malloc(sizeof(char) * strlen(value) + 1);
                                        strcpy(handle->content_type, value);
                                } else if (0 == strcmp(line_buf, "Location:")) {
#if 0
                                        if (NULL != handle->location) {
                                                free(handle->location);
                                                handle->location = NULL;
                                        }
#endif
                                        handle->location = (char*)malloc(sizeof(char) * strlen(value) + 1);
                                        strcpy(handle->location, value);
                                }
	
                        }
                } while ((CR != line_buf[0]) || (LF != line_buf[1])); /*  */
        }

        /* dbgp(dbg, "end of header, head of entity.\n"); */
	
        /* read response entity */
        /* TBD entity ɤߤΥॢȤΥƥ */
        /* TBD entity ɤߤbuffer overflowˤϡɤ߼ΤƤԤʤʤ
           get/post˺꤬ */
        {
                if (handle->content_length == -1) {
                        /* Content-Length̵ */
                        /* TBD ϤchunkξHTTP/1.0ξǤ櫓 */
                        int size;
                        int read_size;

                        read_size = buff_size - 1;
                        dbgp(dbg, "try reading entity size %d.\n", read_size);

                        size = read_all(fd, response_entity, read_size);
                        response_entity[size] = '\0'; /* NULLߥ͡ */

                        if (size < 0) {
                                return SOCK_ERR;
                        } else if (size < handle->content_length) {
                                /* Хåե­ʤɤ߼ΤƤ */
                                if (OK != throw_out_all(fd, handle->content_length - size)) {
                                        return SOCK_ERR;
                                }
                                return APP_ERR;
                        }
		
		
                } else {
                        /* Content-Lengthͭ */
                        int size;
                        int read_size;
	
                        read_size = (handle->content_length < buff_size - 1) ? handle->content_length : buff_size - 1;
                        dbgp(dbg, "reading entity size %d.\n", read_size);

                        size = read_all(fd, response_entity, read_size);
                        response_entity[size] = '\0'; /* NULLߥ͡ */
                        if (size < 0) {
                                return SOCK_ERR;
                        } else if (size < handle->content_length) {
                                /* Хåե­ʤɤ߼ΤƤ */
                                if (OK != throw_out_all(fd, handle->content_length - size)) {
                                        return SOCK_ERR;
                                }
                                return APP_ERR;
                        }
                        /* dbgp(dbg, "entity size=%d\n", size); */
                }
	
        }

        /* HTTP顼ΥǤ⡢Entityɤ߹ߤϹԤʤ
           âApacheΤ褦˥顼chunk֤̤ݡ */
        if (200 != handle->code) {
                return HTTP_ERR;
        }

        /* λ */
        return OK;
}


int
http_post(httpc_handle *handle, const char* path, const char* request_entity, char* response_entity, int buff_size)
{
        ASSERT(NULL != handle);
	
        ASSERT(0); /* not supported */
        return OK;
}

int
http_close(httpc_handle *handle)
{
        ASSERT(NULL != handle);
	
        if (NULL != handle->content_type) {
                free(handle->content_type);
                handle->content_type = NULL;
        }

        if (NULL != handle->location) {
                free(handle->location);
                handle->location = NULL;
        }
  
        close(handle->fd);
        free(handle);

        return OK;
}

int
http_set_timeout(httpc_handle *handle, int timeout_sec)
{
        ASSERT(handle != NULL);
        handle->timeout_sec = timeout_sec;
        return (OK);
}


char *
http_get_content_type(httpc_handle *handle)
{
        ASSERT(handle != NULL);
        return (handle->content_type);
}

char *
http_get_location(httpc_handle *handle)
{
        ASSERT(handle != NULL);
        return (handle->location);
}


int
http_get_content_length(httpc_handle *handle)
{
        ASSERT(handle != NULL);
        return (handle->content_length);
}


int http_get_status_code(httpc_handle *handle)
{
        ASSERT(handle != NULL);
        return (handle->code);
}

	
	

#if defined(DEBUG)
/** DEBUG: ̤̾ɽ롣
 *
 * @param rhe \c hostent ¤ΤؤΥݥ
 */
static void
dump_hostent(struct hostent * rhe) {

        char **pptr;
        struct in_addr ia;

        /* hostentƳǧ */
        dbgp(dbg, "Dumping hostent -- \n");

        /* ̾ */
        dbgp(dbg, "\th_name %s\n", rhe->h_name);
        /* ̾ */
        {
                int j = 0;
                for (pptr = rhe->h_aliases; *pptr != NULL; pptr++) {
                        dbgp(dbg, "\th_aliases[%d] %s\n", j, *pptr);
                        j++;
                }
        }
	
        /* ɥ쥹 */
        dbgp(dbg, "\th_addrtype %d\n", rhe->h_addrtype);
        /* ɥ쥹Ĺ */
        dbgp(dbg, "\th_length %d\n", rhe->h_length);

        /* ɥ쥹 */
        {
                int j = 0;
                for (pptr = rhe->h_addr_list; *pptr != NULL; pptr++) {
                        ia.s_addr = **((unsigned long int **)pptr);
                        dbgp(dbg, "\th_addr_list[%d] %s\n", j, inet_ntoa(ia));
                        j++;
                }
        }
        return;
}
#endif

/** connect_timeout()signalϥɥ
 *
 * @param signo ʥ
 */
static void
connect_alarm(int signo)
{
        /* noop */
        return;
}

/** sigactionΥåѡؿ
 *
 * @param signo ʥμ
 * @param func ʥϥɥؤδؿݥ
 * @retval ʥϥɥؤδؿݥ󥿡顼SIG_ERR
 */
Sigfunc *
Signal(int signo, Sigfunc *func)
{
        struct sigaction act;
        struct sigaction oact;

        act.sa_handler = func;
        sigemptyset(&act.sa_mask);
        act.sa_flags = 0;
        if (signo == SIGALRM) {
                act.sa_flags |= SA_INTERRUPT; // undeclared on cygwin
        } else {
                act.sa_flags |= SA_RESTART;
        }
        if (sigaction(signo, &act, &oact) < 0) {
                return SIG_ERR;
        }
        return oact.sa_handler;
}

/** åȤΥॢȤꤹ
 *
 * setsockopt(2)ѤơåȤread/writeξΥॢȻ֤
 * ꤹ롣socketμˤäƤϡǻѤ륽åȡץ
 * ¸ߤʤ/read onlyΤΤ⤢롣
 *
 * @param sock_fd оݤȤ륽åȡեǥץ
 * @param timeout_sec ॢȻ()
 * @retval ॢȤ꤬(OK)/(SOCK_ERR)
 */
static int
set_sock_timeout(int sock_fd, int timeout_sec)
{
        struct timeval tv;
        int result;

        dbgp(dbg, "set_sock_timeout %d sec.\n", timeout_sec);
	
        tv.tv_sec = timeout_sec;
        tv.tv_usec = 0;

        result = setsockopt(sock_fd, SOL_SOCKET, SO_RCVTIMEO, &tv, sizeof(tv));
        if (-1 == result) {
                return (SOCK_ERR);
        }
	
        result = setsockopt(sock_fd, SOL_SOCKET, SO_SNDTIMEO, &tv, sizeof(tv));
        if (-1 == result) {
                return (SOCK_ERR);
        }

        return (OK);
}


/** ॢȵǽդconnect(2)
 *
 * alarm()/sigaction()Ѥॢդconnect.
 *
 * @param sock_fd åȡեǥץ
 * @param server_addr Хɥ쥹
 * @param addrlen sizeof(server_addr)
 * @param timeout_sec ॢȻ(ñ)
 * 
 * @retval OK ｪλ
 * @retval SOCK_ERR alarm˼Ԥconnect()Ԥ
 */
static int
connect_timeout(int sock_fd,
		const struct sockaddr *server_addr,
		socklen_t addrlen,
		int timeout_sec)
{
        Sigfunc *sigfunc;
        int conn_result;
        int result;

        /* signal(ꤷƤäϥɥsigfuncȤݴ */
        sigfunc = Signal(SIGALRM, connect_alarm);

        /* alarm */
        if (alarm(timeout_sec) != 0) {
                dbgp(dbg, "alarm was already set.\n");
                result = SOCK_ERR;
                goto END_CONNECT_SIGNAL;
        }

        dbgp(dbg, "set alarm %d sec.\n", timeout_sec);

        /* connect */
        conn_result = connect(sock_fd, server_addr, addrlen);
        if (-1 == conn_result) {
                if (errno == EINTR) {
                        dbgp(dbg, "connection timedout.\n");
                }
                result = SOCK_ERR;
                goto END_CONNECT_ALARM;
        }

        dbgp(dbg, "connection ok.\n");

        result = OK;
        goto END_CONNECT_ALARM;


        /*  */
  END_CONNECT_ALARM:
        /* alarmǥեȤ(0sec)ˤɤ */
        alarm(0);
  END_CONNECT_SIGNAL:
        /* ʥϥɥꤷƤäϥɥˤɤ */
        Signal(SIGALRM, sigfunc);

        return result;
}

/** ХåեƤ \c socket ˽񤭽Ф
 *
 * @param fd ץѤߥåȤΥեǥץ꥿
 * @param buf ǡѥХåե
 * @param buf_size 񤭹ॵ
 *
 * @retval 0 ｪλ\nͤξ硢顼Ǥ롣\n
 */
static int
write_all(int fd, const void *buf, size_t buf_size)
{
        while (0 < buf_size) {
                ssize_t wBytes;

                wBytes = write(fd, buf, buf_size);

                if (wBytes < 0) {
                        /* ƥॳ륨顼 */
                        /* EINTR, EIOˤĤƤϡߺѽ֤Ԥʤ */
                        if((errno != EINTR) && (errno != EIO)) {
                                /* ʤ顼ȯ */
                                if (EWOULDBLOCK == errno) {
                                        /* timeout */
                                        dbgp(dbg, "write_all timeout.\n");
                                }
                                return SOCK_ERR;
                        }
                } else {
                        /* 񤭹 */
                        buf += wBytes;
                        buf_size -= wBytes;
                }
        }
        return (OK);
}


/** \c socket Хåեɤ߹
 *
 * @param fd ץѤߥåȤΥեǥץ꥿
 * @param buf ǡѥХåե
 * @param buf_size \a buf Υ
 *
 * @retval 0 ｪλ\nͤξ硢顼Ǥ롣\n
 */
static int
read_all(int fd, const void *buf, size_t buf_size)
{
        int rest_buf;
        rest_buf = buf_size;

        ASSERT(fd >= 0);
        ASSERT(buf != NULL);

        /* read (2)0֤ޤǥХåեɤ߽ФƤ
           read-1֤Ƥϥ顼λ */
        while (0 < rest_buf) {
                ssize_t rBytes;
                rBytes = read(fd, (char*)buf, rest_buf);
                if (rBytes < 0) {
                        /* TBD ߺѤǤ륱Ϥ뤫 */
                        if (EWOULDBLOCK == errno) {
                                dbgp(dbg, "read_all timeouted.\n");
                        }
                        return SOCK_ERR;
                } else if (0 == rBytes) {
                        /* ɤ߽ */
                        return (buf_size - rest_buf);
                }
                rest_buf -= rBytes;
                buf += rBytes;
        }
        /* Хåե­ */
        ASSERT(rest_buf == 0);
        return (buf_size - rest_buf); /* must be equal buf_size */
}


/** \c socket Σԥǡɤ߼
 *
 * TBD nullߥ͡Ȥ٤
 *
 * @param fd ץѤߥåȤΥեǥץ꥿
 * @param buff ǡѥХåե
 * @param buflen \a buf Υ
 *
 * @retval n ɤ߼äǡΥХȥ\n
 * ͤξ硢顼Ǥ롣
 */
int
read_line(int fd, char *buff, int buflen)
{
        int len = 0;
        int status;

        ASSERT(buff != NULL);
        ASSERT(fd >= 0);

        if (buflen == 0) {
                return 0;
        }

        /* ХåեΥꥢ */
        memset((void *)buff, 0, buflen);

        /* socketɤ߹ */
        while (len < buflen) {

                /* 1ʸɤ */
                status = read(fd, buff + len, 1);
                if (status < 0) {
                        dbgp(dbg, "read_line failed.\n");
                        if (EWOULDBLOCK == errno) {
                                dbgp(dbg, "read_line timeouted.\n");
                        }
                        return -1;
                }
                if (status == 0) {
                        return len;
                }

                /* ɤ߹ */
                /* Ԥνλ */
                if (*(buff + len) == CR) {
                        len++;
                        //status = recv(fd, buff + len, 1, 0);
                        status = read(fd, buff + len, 1);
                        if (status < 0) {
                                dbgp(dbg, "read_line failed.\n");
                                if (EWOULDBLOCK == errno) {
                                        dbgp(dbg, "read_line timeouted.\n");
                                }
                                return -1; /* ERROR */				
                        }
                        if (status == 0) {
                                return len; /* Data END */
                        }
                        if (*(buff + len) == LF) {
                                /* end of line */
                                len++;
                                return len;
                        }
                }
                len++;
        }
        /* Хåեʤ */
        ASSERT(len == buflen);
        return len;
}

/** socketХȿʬΥǡɤ߼ΤƤ
 *
 * @param fd Socket file descpriter.
 * @param size ɤ߼ΤƤ륵.
 * @retval OK ｪλ
 * @retval SOCK_ERR 顼ȯ
 */
static int
throw_out_all(int fd, size_t size)
{
        char tmp[1024];
        ssize_t rest;
	
        ASSERT(fd >= 0);
        ASSERT(size > 0);
	
        rest = size;
	
        /* ꥵʬɤߤԤʤ뤫read(2)0֤ޤ
           Хåեɤ߽ФƤϼΤƤ롣
           read-1֤Ƥϥ顼λ */
        while (0 < rest) {
                ssize_t rBytes;
                ssize_t read_size;
		
                read_size = (rest < 1024) ? rest : 1024;
                rBytes = read(fd, tmp, read_size);
                if (rBytes < 0) {
                        /* TBD ߺѤǤ륱Ϥ뤫 */
                        if (EWOULDBLOCK == errno) {
                                dbgp(dbg, "throw_out_all timeouted.\n");
                        }
                        return SOCK_ERR;
                } else if (0 == rBytes) {
                        /* ɤ߽ */
                        dbgp(dbg, "%d bytes data read and throw outed.\n", size - rest + rBytes);
                        return OK; //(buf_size - rest_buf);
                }
                rest -= rBytes;
        }
        dbgp(dbg, "%d bytes data read and throw outed.\n", size - rest);
        return OK; /* must be equal buf_size */
	
}


/** ХåեHTTPإå˲Ūѹᤷ䤹
 *
 * @param header HTTP إå
 * @param value إåγͤؤΥݥ󥿤Ǽ뤿Υݥ󥿥ꥹ
 *
 * @retval 1 ᤹٤ǡä
 * @retval 0 ᤹٤ǡʤä
 */
static int
parse_a_header(char *header, char** value)
{
        char *p;
	
        ASSERT(header != NULL);

        if (*header == 9 || *header == 32) {
                /* ԤΥإå³ */
                return 0;
        }

        p = header;
	
        /* SP(32)ΰ֤õ'\0'ۤꤳࡣSP(32)μΰ֤value */
        p = strchr(header, ' ');
        if (p == NULL) {
                return 0;
        }
	
        *p = '\0';
        p++;

        *value = p;

        /* valueμCRõ'\0'ۤꤳࡣ */
        p = strchr(p, CR);
        if (p == NULL) {
                return 0;
        }
        *p = '\0';
	
        return 1;
}


int
url_encode(const char *in, int in_len, char* out, int out_len)
{
	int i;
	int o;
	unsigned char c;
	int err;

	err = 0;
	for (i = 0, o = 0; i < in_len; i++) {
		c = in[i];

		if ((('a' <= c) && ('z' >= c))
                    || (('A' <= c) && ('Z' >= c))
                    || ('-' == c)
                    || ('_' == c)
                    || ('*' == c)
                    || ('.' == c))
		{
			/* 󥳡ɤɬפʤ饯 */
			if (err || (NULL == out)) {
				o++;
				continue;
			} else if ((out_len < o + 1)) {
				o++;
				err = 1;
				continue;
			} else {
				out[o++] = c;
			}
		} else if (' ' == c) {
			/* SP̤ʥ󥳡ɤ򤹤 */
			if (err || (NULL == out)) {
				o++;
				continue;
			} else if ((out_len < o + 1)) {
				o++;
				err = 1;
				continue;
			} else {
				out[o++] = '+';
			}
		} else {
			/* 󥳡ɤɬפʥ饯 */
			if (err || (NULL == out)) {
				o += 3;
				continue;
			} else if ((out_len < o + 3)) {
				o += 3;
				err = 1;
				continue;
			} else {
				out[o++] = '%';
				out[o++] = to_hex(((c >> 4) & 0x0f));
				out[o++] = to_hex((c & 0x0f));
			}

		}
	}
	if (err) {
		return -1;
	}
	return o;
}


int
url_decode(const char *in, int in_len, char *out, int out_len)
{
	int i;
	int o;
	unsigned char c;
	unsigned char d = '\0';
	char u = '\0';
	char l;

	int st;
	int err;

	/**
	 * note:
	 * Ϥϰϥåϼ֤ʤΤ1Ťɤࡣ
	 * ؤState Machineˤ롣
	 * ϤϤ1饯ʤΤǰѿȤ
	 * ɤΥǤޤȤƥ롼ǽ롣
	 * ϤΤʤStateǤcontinueǽФ
	 */
	i = 0;
	o = 0;
	st = 0;
	err = 0;
	while (i < in_len) {
		c = in[i++];

		switch (st) {
                    case 0:
                            if ('%' == c) {
                                    st = 1; /* change state */
                                    continue; /* no output */
                            } else if('+' == c) {
                                    d = ' ';
                            } else {
                                    d = c;
                            }
                            break;
                    case 1:
                            u = c;
                            st = 2; /* change state */
                            continue; /* no output */
                    case 2:
                            l = c;
                            d = to_char(u, l);
                            st = 0; /* change state */
                            break;
		}

		/*  */
		if ((NULL != out) && !err) {
			if (o < out_len) {
				out[o] = d;
			} else {
				err = 1;
			}
		}
		/* 顼ǤʸΥȤϤ */
		/* 1饯ФơϤϤ1饯 */
		o++;
	}
	if (err) {
		return -1;
	}
	return o;
}

/** 饯Υإɽ
 *
 * @param c 饯(016ޤǤ)
 * @return cΥإɽ
 */
static char
to_hex(unsigned char c)
{
	/* 0 - 9 */
	if (c <= 0x09) {
		return (unsigned char)(c + '0');
	}
	/* A - F */
	if (0x0a <= c && 0x0f >= c) {
		return (unsigned char)(c + '7');
	}
	return '0';
}


/** إɽʸ顢1ХȤοͤ
 *
 * @param upper 4bitΥإʸ
 * @param lower 4bitΥإʸ
 *
 * @return 
 */
static char
to_char(unsigned char upper, unsigned char lower)
{
	return (h2c(upper) << 4) + h2c(lower);
}


/** إʸ(0-9, A-F, a-F)бͤ
 *
 * @param c إʸ
 *
 * @return  \n
 *		   ϰϳͤ褿0֤
 */
static char
h2c(unsigned char c)
{
	unsigned char r;

	r = 0;

	if ('0' <= c && c <= '9') {
		r = (unsigned char)(c - '0');
	}
	if ('a' <= c && c <= 'f') {
		r = (unsigned char)(c - 'a' + 10);
	}
	if ('A' <= c && c <= 'F') {
		r = (unsigned char)(c - 'A' + 10);
	}

	return r;
}

/* end of file */
