/** @file
 * @brief Printer status handling functions.
 *
 * This file is part of the `Printer Status Utility Type B' program.
 *
 * @author Copyright (C) 2003,2004 EPSON KOWA Corporation
 * @date 2004/02/06
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * Header Include section...
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>		/* for fprintf() */
#include <stdlib.h>		/* for exit() */
#include <string.h>
#include <getopt.h>		/* for getopt()*/

#include "psub_callbacks.h"
#include "psub_interface.h"
#include "psub_support.h"

#include "psub_common.h"
#include "psub_debug.h"
#include "httpc.h"
#include "psub_xmlutil.h"

#include "psub_status_proc.h"
#include "psub_daemonio.h"


/*
 * Define and Macros...
 */

#ifdef DEBUG
#define SOCK_TIMEOUT_SEC 20
#else
#define SOCK_TIMEOUT_SEC 60
#endif

/** GTK Window refresh interval time in Sec. */
#define DISP_REFLESH_SEC 8

/** GTK polling timer control switch. */
#define TIMER_ON  1
#define TIMER_OFF 0

/** Communication buffer size for printer daemon. */
#define RECV_BUFF_SIZE (1024 * 64)
#define SEND_BUFF_SIZE (1024 *  1)

/* Others */

/*
 * Global Params...
 */

/** HTTP communication handle */
httpc_handle *Httpc;

/** Printer daemon I/Obuffer */
char  Send_buf  [SEND_BUFF_SIZE];
char  Recv_buf  [RECV_BUFF_SIZE];

char _pr_name   [MAX_PRINTER_LIST * MAX_PRINTER_NAME];
char *pr_name   [MAX_PRINTER_LIST                   ];
char _pr_comment[MAX_PRINTER_LIST * MAX_COMMENT_SIZE];
char *pr_comment[MAX_PRINTER_LIST                   ];

/** GTK CallBack handle (user_data) */
gp_t  Gparams = {
        0,                      /**< GtkWidget * #top_w */
        0,                      /**< GtkWidget * #sel_w */
        0,                      /**< GtkWidget * #err_w */
        "C",			/**< Display Language #lang */
        { '\0', },              /**< #Hostname */
        { '\0', },              /**< #Printer_name */
        { '\0', },              /**< #Printer_description */
        0,                      /**< #is_http_opend */
        -1                      /**< #timeout_handler_id */
};

/*
 * Function define...
 */
LOCAL_FUNC int http_err_msg( const httpc_handle *hp, char *buff, const int buff_size );
LOCAL_FUNC int psub_http_open ( gp_t *gp, const char *server_name, const int port, const int timeout_sec);
LOCAL_FUNC int psub_http_close( gp_t *gp );
LOCAL_FUNC int switch_server  ( gp_t *gp );
LOCAL_FUNC int specify_printer( gp_t *gp, const int port, const char *printer);

/*
 * Programs...
 */


FUNCTION void init_status_proc(int argc, char **argv)
{
        LOCAL_FUNC void setup_printer_name( gp_t *gp, int argc, char **argv);

        gp_t *gp = &Gparams;

        /* Init global params */
        Recv_buf[0]         = '\0';
        Send_buf[0]         = '\0';
        gp->Printer_name[0] = '\0';

        setup_printer_name( gp, argc, argv);

        if ( '\0' == gp->Printer_name[0] ) {
                gtk_widget_show(gp->sel_w);

        } else {
                int err;

                err = specify_printer( gp, DAEMON_PORT, gp->Printer_name );

                if ( -10 == err ) {
                        msgp(Critical, "Printer \"%s\" not found on \"%s\".\n",
                             gp->Printer_name, gp->Hostname);
                }

                if ( 0 > err ) {
                        exit(1);
                }

                start_monitor();
        }
}

/** Command line parse and specify target printer name.
 *
 * Default hostname is 'localhost'.
 *
 * @param gp   PSUB Global parameters.             See #gp_t
 * @param argc Number of command line argument(s). See #main()
 * @param argv Command line argument list.         See #main()
 */
LOCAL_FUNC void setup_printer_name( gp_t *gp, int argc, char **argv)
{
        LOCAL_FUNC void usage(void);

        overwrite_Hostname( gp, "localhost" );

        while(1){
                int c = getopt(argc, argv, "h:v:?");
                if(c < 0)
                        break;

                switch (c) {

                    case 'h':		/* Hostname */
                            overwrite_Hostname( gp, optarg );
                            break;

                    case 'v':		/* Verbosity Level */
                            shift_info_level(atoi(optarg));
                            break;

                    case '?':		/* Usage */
                            usage();
                            exit(1);
                            break;

                    default:
                            break;
                }
        }

        if( optind < argc )
                strcpy(gp->Printer_name, argv[optind++]);
}

/** Useage
 */
LOCAL_FUNC void usage(void)
{
        msgp(Err, "\nUsage: printerStatusUtilityTypeB [-h hostname]  printer\n\n" );
}


FUNCTION void end_status_proc(void)
{
        psub_http_close( &Gparams );
}


LOCAL_FUNC void debug_disp_http_err( void )
{
#ifdef DEBUG
        dbgp(dbg, "HTTP errorCode=%d\n", Httpc->code);
        dbgp(dbg, "   server_name=%s\n", Httpc->server_name);
        dbgp(dbg, "      location=%s\n", Httpc->location);
        dbgp(dbg, "content_length=%d\n", Httpc->content_length);
        dbgp(dbg, "  content_type=%s\n", Httpc->content_type);
        dbgp(dbg, "      Recv_buf=[%s]\n", Recv_buf);
#endif
}



#define SWITCH_SERVER_ERR (-11)

/** #http_get() ֤ͤɾ롣
 *
 * @param status #http_get() 
 * @param httpc  #httpc_handle
 * @param msgbuf error message buffer.
 * @param buf_size Byte size of #msgbuf.
 *
 * @return   0 ³ Printer Ȥ̿³֤Ǥ뤳Ȥɽ
 * @return  -1 ʾ Printer Ȥ̿Ǥʤ֤ˤ뤳Ȥɽ
 * @return 301 HTTP ERROR 301 "Moved Permanently" ֤줿Ȥɽ\n
 *             301ξϡprint serverڤؤɬפ롣
 */
LOCAL_FUNC int judge_http_err( const int status, const httpc_handle *httpc, char *msgbuf, const int buf_size )
{
        ASSERT( NULL != msgbuf );
        ASSERT( 0 < buf_size );

        msgbuf[0] = '\0';

        if( OK == status )
        {
                return 0;
        }

        if ( ( status < -1 ) && ( NULL != httpc ) )
                debug_disp_http_err();

        if ( HTTP_ERR == status )
        {
                ASSERT( 301 != httpc->code );

                if( 500 == http_err_msg( httpc, msgbuf, buf_size ) )
                        return 0;
                else
                        return -1;

        }
        else if( SOCK_ERR == status )
        {
                strncpy( msgbuf, _("Server connection error."), MAX_MESSAGE_SIZE );
                return -1;

        }

        /* Unknown error. */
        strncpy( msgbuf, _("Printer daemon error."), MAX_MESSAGE_SIZE );
        return -1;
}

FUNCTION gboolean updateStatus( gpointer *data )
{
        LOCAL_FUNC void disp_multiline_msg(const gp_t *gp, const char *buf, const char *msg);

        gp_t *gp = (gp_t *)data;
        GtkWidget *widget;	/*  */
        char msgbuf[MAX_MESSAGE_SIZE +1] = { '\0', };
        void *doc;		/* xmlϥɥ */
        int ret = 0;

        if ( 0 == gp->is_http_opend ) {
                dbgp(dbg, "Not opend\n");
                return (gboolean) TIMER_OFF;
        }

        widget = lookup_widget (gp->top_w, "NameValue");
        gtk_label_set_text(GTK_LABEL(widget), gp->Printer_name);

        widget = lookup_widget (gp->top_w, "LocationValue");
        gtk_label_set_text(GTK_LABEL(widget), gp->Hostname);

        widget = lookup_widget (gp->top_w, "DescriptionValue");
        gtk_label_set_text(GTK_LABEL(widget), gp->Printer_description);

        msgbuf[0] = '\0';
        disp_multiline_msg ( gp, msgbuf, "InkLevelTextView" );
        disp_multiline_msg ( gp, msgbuf, "WarningTextView" );

        /* idReadMode(BiDi Read Mode ID) =  BIDI_READ_PRT_MIB_SUMMARY */
        snprintf(Send_buf, SEND_BUFF_SIZE, "/status/%s?idReadMode=%d&LANG=%s",
                 gp->Printer_name, 1, gp->lang);

        ret = http_get( Httpc, Send_buf, Recv_buf, RECV_BUFF_SIZE );

        if( OK == ret )
        {
                msgp(Noisy, "\n* Recive Data\n[%s]\n", Recv_buf);
                doc = xml_start(Recv_buf, strlen(Recv_buf));

                get_printer_status( doc, msgbuf, MAX_MESSAGE_SIZE);
                widget = lookup_widget (gp->top_w, "StateValue");
                gtk_label_set_text(GTK_LABEL(widget), msgbuf);

                if ( OK != get_printer_alert(doc, msgbuf, MAX_MESSAGE_SIZE) )
                        msgbuf[0] = '\0';

                disp_multiline_msg ( gp, msgbuf, "WarningTextView" );

                if ( OK != get_supplies_status(doc, msgbuf, MAX_MESSAGE_SIZE) )
                        msgbuf[0] = '\0';

                disp_multiline_msg ( gp, msgbuf, "InkLevelTextView" );

                xml_end(doc);

                return (gboolean) TIMER_ON;
        }

        if( ( HTTP_ERR == ret ) && ( 301 == Httpc->code ) ) /* Moved Permanently */
        {
                widget = lookup_widget (gp->top_w, "StateValue");
                gtk_label_set_text(GTK_LABEL(widget),  _("Please wait..."));

                if ( 0 == switch_server( gp ) )
                {
                        return (gboolean) TIMER_ON;
                }
                ret = SWITCH_SERVER_ERR;
                strncpy( msgbuf, _("Remote printer could not access."), MAX_MESSAGE_SIZE );
        }


        widget = lookup_widget (gp->top_w, "StateValue");
        gtk_label_set_text(GTK_LABEL(widget),  _("Error"));

        if( SWITCH_SERVER_ERR == ret )
        {
                ret = -1;
        }
        else {
                ret = judge_http_err( ret, Httpc, msgbuf, MAX_MESSAGE_SIZE );
        }

        if( -1 == ret )          /* Stop polling. */
        {
                if ( -1 != gp->timeout_handler_id ) {
                        gtk_timeout_remove( gp->timeout_handler_id );
                        gp->timeout_handler_id = -1;
                }
                psub_http_close( gp );

                gtk_widget_hide(
                        lookup_widget (GTK_WIDGET(gp->sel_w), "PrSel_frame") );
                gtk_widget_set_sensitive(
                        lookup_widget (GTK_WIDGET(gp->sel_w), "PrSel_okbutton"),
                        FALSE );
                gtk_widget_show(gp->sel_w);

                disp_info ( msgbuf );
                return (gboolean) TIMER_OFF;
        }

        if( '\0' != msgbuf )
        {
                disp_multiline_msg ( gp, msgbuf, "WarningTextView" );
        }

        return (gboolean) TIMER_ON;
}

LOCAL_FUNC void disp_multiline_msg(const gp_t *gp, const char *buf, const char *msg)
{
        GtkWidget *widget;
        GtkTextBuffer *buffer;
        GtkTextIter start, end;

        widget = lookup_widget (gp->top_w, msg);
        buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (widget));
        gtk_text_buffer_get_start_iter (buffer, &start);
        gtk_text_buffer_get_end_iter(buffer, &end);
        gtk_text_buffer_delete (buffer, &start, &end);
        if ( 0 != strlen(buf) ) {
		gtk_text_buffer_insert_at_cursor(buffer, buf, -1);
        }
}

/** Daemon error parser.
 *
 * Create daemon error message string from the daemon error status.
 *
 * @param hp        Pointer to #httpc_handle.
 * @param buff      Error message store buffer.
 * @param buff_size Size of buff.
 *
 * @retval  0 printer daemon error.
 * @retval other HTTP error code.
 */
LOCAL_FUNC int http_err_msg( const httpc_handle *hp, char *buff, const int buff_size )
{
        LOCAL_FUNC int get_derr_subcode( const char *str );

        typedef struct daemon_error_code_tag {
                int code;
                char *msg;
        } dm_err_t;

        const dm_err_t derr[] = {
                { 301, N_("Moved Permanently") },
                { 400, N_("Bad Request"), },
                { 404, N_("Not Found"), },
                { 405, N_("Method Not Allowed"), },
                { 408, N_("Request Timeout"), },
                { 411, N_("Length Required"), },
                { 413, N_("Request Entity Too Large"), },
                { 414, N_("Request-URI Too Long"), },
                { 501, N_("Not Implemented"), },
                { 503, N_("Service Unavailable"), },
                { 505, N_("HTTP Version Not Supported"), },
                { 500, N_("Internal Server Error"), },
                {   0, "" } };

        if ( 500 == hp->code ) {
                char s[256];
                int sub_code = get_derr_subcode( Recv_buf );

                if ( ( -110 ) == sub_code )
                        strcpy( s, _("/etc/oppd.conf could not access.") );
                else if ( ( -111 ) == sub_code )
                        strcpy( s, _("/etc/oppd.conf format error. ( empty? )") );
                else if ( ( (-1) >= sub_code ) && ( (-9) <= sub_code ) )
                        strcpy( s, _("BiDi error.") );
                else
                        strcpy( s, _("Printer daemon error.") );

                snprintf( buff, buff_size, _("%1$s\nhttp_error_code: %2$d\nDetail: %3$s (%4$d)"),
                          _("Internal Server Error"), hp->code, s, sub_code );

        } else {
                int i;
                const int n = sizeof(derr) / sizeof(dm_err_t);

                for ( i = 0; i < n; i++ )
                        if ( derr[i].code == hp->code )
                                break;

                if ( i < n ) {
                        strncpy( buff, _(derr[i].msg), buff_size );
                } else {
                        snprintf( buff, buff_size, "%s\nhttp_error_code: %d",
                                  _("Printer daemon error."), hp->code );
                }
        }
        return  hp->code;
}

/** Daemon error sub-code paser.
 *
 * Get daemon error sub-code from the string.
 *
 * @param str      daemon response HTTP body.
 *
 * @retval  0      complete.
 * @retval -1      error.
 *
 * @note           Caller must be allocate these buffers.
 */
LOCAL_FUNC int get_derr_subcode( const char *str )
{
        char *cp = strstr( str, "ecode:" );

        if ( NULL == cp ) {
                return 0;
        }

        return ( atoi( &cp[ strlen("ecode:") ] ) );
}




FUNCTION void start_monitor( void )
{
        gp_t *gp = &Gparams;

        updateStatus( (gpointer*) gp );

        gp->timeout_handler_id = gtk_timeout_add(
                (DISP_REFLESH_SEC * 1000), (GtkFunction) updateStatus, gp );

        gtk_widget_show( gp->top_w );
}


FUNCTION int overwrite_Hostname( gp_t *gp, const char *hostname )
{
        if ( 0 != gp->is_http_opend ) {
                dbgp(dbg, "Hostname not changed.\n");
                return -1;
        }

        strncpy( gp->Hostname, hostname, MAX_HOSTNAME);

        return 0;
}


FUNCTION int open_get_plist( const char *hostname, const int port, char ***plist, char ***comment )
{
        gp_t *gp = &Gparams;
        int timeout_sec = SOCK_TIMEOUT_SEC;
        int pr_num;
        int ret;

        if ( 0 != psub_http_open( gp, hostname, DAEMON_PORT, timeout_sec ) )
        {
                msgp(Critical, "Server \"%s\" connection error.\n", hostname);
                return -1;
        }

        ret = http_get( Httpc, "/list", Recv_buf ,RECV_BUFF_SIZE );
        if( OK != ret )
        {
                msgp(Critical, "\"%s\" printer daemon communication error(%d).\n", hostname, ret);
                psub_http_close( gp );
                return ret;
        }

        /* Init the printer list. */
        {
                int i;
                for( i = 0; i < MAX_PRINTER_LIST; i++ ) {
                        pr_name[i] = &_pr_name[(MAX_PRINTER_NAME * i)];
                        pr_name[i][0] = '\0';
                        pr_comment[i] = &_pr_comment[(MAX_COMMENT_SIZE * i)];
                        pr_comment[i][0] = '\0';
                }
        }

        /* Set printer list from server responce data. */
        pr_num = get_pr_list(Recv_buf, pr_name, pr_comment);
        if ( pr_num <= 0 ) {
                msgp(Critical, "No printer on \"%s\".\n", hostname);
                psub_http_close( gp );
                return 0;
        }

        *plist = pr_name;
        *comment = pr_comment;

        return pr_num;
}


/** HTTP connection open.
 *
 * #http_open() wrapper.
 *
 * @param gp PSUB Global parameters. See #gp_t
 * @param server_name See #http_open()
 * @param port        See #http_open()
 * @param timeout_sec See #http_open()
 *
 * @retval  0 OK
 * @retval -1 ERROR
 */
LOCAL_FUNC int psub_http_open ( gp_t *gp, const char *server_name, const int port, const int timeout_sec)
{
        if ( 0 != gp->is_http_opend ) {
                psub_http_close(gp);
        }

        ASSERT ( 0 == gp->is_http_opend );
        ASSERT ( NULL == Httpc );

        Httpc = http_open(server_name, port, timeout_sec);
        if ( NULL == Httpc ){
                return -1;
        }
        gp->is_http_opend = 1;

        return 0;
}



/** HTTP connection close.
 *
 * #http_close() wrapper.
 *
 * @param gp PSUB Global parameters. See #gp_t
 *
 * @retval  0 OK
 * @retval -1 ERROR
 */
LOCAL_FUNC int psub_http_close( gp_t *gp )
{
        int ret;

        if ( ( 0 == gp->is_http_opend ) || ( NULL == Httpc )) {
                dbgp(dbg, "Not opend.\n");
                gp->is_http_opend = 0;
                return -1;
        }

        ret = http_close( Httpc );

        if ( 0 == ret ) {
                gp->is_http_opend = 0;
                Httpc = NULL;
        }
        return ret;
}


LOCAL_FUNC int switch_server  ( gp_t *gp )
{
        LOCAL_FUNC int split_uri( const char *uri, char *hostname, int *port, char *pr_name );

        char hostname[MAX_HOSTNAME];
        char printer[MAX_PRINTER_NAME];
        int port;

        if ( ( 0 == split_uri(Httpc->location, hostname, &port, printer) ) &&
             ( 0 == psub_http_close( gp ) ) &&
             ( 0 == overwrite_Hostname( gp, hostname ) ) &&
             ( 0 == specify_printer( gp, port, printer) ) ) {
                strcpy( gp->Printer_name, printer );
        } else
                return -1;

        return 0;
}

/** OpenPrintingPrinterDaemon(oppd) URI parser
 *
 * Split oppd format URI.
 *
 * @param uri      oppd style URI string.
 * @param hostname Hostname store buffer.
 * @param port     Daemon communication port number store buffer.
 * @param pr_name  Printer name store buffer.
 *
 * @retval  0      complete.
 * @retval -1      error.
 *
 * @note           Caller must be allocate these buffers.
 */
LOCAL_FUNC int split_uri( const char *uri, char *hostname, int *port, char *pr_name )
{
        char buf[1024], *cp;
        char *host, *pt, *pr;

        strncpy( buf, uri, 1023 );

        if ( NULL == ( cp = strstr( buf, "oppd://" ) ) )
                return -1;

        host = &cp[ strlen("oppd://") ];
        if ( NULL != ( cp = strchr( host, ' '  ) ) )  *cp = '\0';
        if ( NULL != ( cp = strchr( host, '\t' ) ) )  *cp = '\0';
        if ( NULL != ( cp = strchr( host, '\r' ) ) )  *cp = '\0';
        if ( NULL != ( cp = strchr( host, '\n' ) ) )  *cp = '\0';

        if ( NULL == ( pt = strchr( host, ':' ) ) ) {
                port[0] = DAEMON_PORT;

                if ( NULL == ( pr = strchr( pt, '/' ) ) )
                        return -1;
        } else {

                *pt++ = '\0';

                if ( NULL == ( pr = strchr( pt, '/' ) ) )
                        return -1;

                *pr++ = '\0';
                port[0] = atoi(pt);
        }

        strcpy( hostname, host );
        strcpy( pr_name, pr );

        return 0;
}


LOCAL_FUNC int specify_printer( gp_t *gp, const int port, const char *printer)
{
        char **plist;
        char **comment;
        int pr_num;
        int i;

        pr_num = open_get_plist( gp->Hostname, port, &plist, &comment );
        if ( 0 >= pr_num )
        {
                return pr_num;
        }

        /* Search printer name in the printer list. */
        for( i = 0; i < pr_num; i++ ){
                if( strcmp(plist[i], printer) == 0 )
                        break;
        }

        if( pr_num == i ){
                psub_http_close( gp );
                return -10;
        } else {
                strcpy( gp->Printer_description, comment[i] );
        }

        return 0;
}

/* end of file */
