/*
 * Copyright (c) 2008-2009 OrangeSignal.com All rights reserved.
 */

package jp.sourceforge.orangesignal.trading;

import java.util.LinkedList;

import jp.sourceforge.orangesignal.trading.commission.Commission;
import jp.sourceforge.orangesignal.trading.order.Order;

/**
 * 売買の管理インターフェースを提供します。
 * 
 * @author 杉澤 浩二
 */
public interface Trader {

	/**
	 * 手数料情報を返します。
	 * 
	 * @return 手数料情報
	 */
	public Commission getCommission();

	/**
	 * 口座情報を返します。
	 * 
	 * @return 口座情報
	 */
	public Account getAccount();

	/**
	 * デフォルトの数量を返します。
	 * 
	 * @return デフォルトの数量
	 */
	public int getDefaultQuantity();

	/**
	 * デフォルトの数量を設定します。
	 * 
	 * @param defaultQuantity デフォルトの数量
	 */
	public void setDefaultQuantity(int defaultQuantity);

	// ----------------------------------------------------------------------

	/**
	 * 指定したシンボルのポジション状態の種類を返します。
	 * 
	 * @param symbol シンボル
	 * @return ポジション状態の種類
	 */
	public MarketPositionType getMarketPositionType(String symbol);

	/**
	 * <p>エントリー中の最後のポジションを返します。</p>
	 * <p>エントリー中のポジションがない場合(フラットな状態)は <code>null</code> を返します。</p>
	 * 
	 * @param symbol シンボル
	 * @return エントリー中の最後のポジション。又は <code>null</code>
	 */
	public Position getCurrentPosition(String symbol);

	/**
	 * エントリー中のポジションのリストを返します。
	 * 
	 * @param symbol シンボル
	 * @return エントリー中のポジションのリスト
	 */
	public LinkedList<Position> getCurrentPositions(String symbol);

//	/**
//	 * 指定されたIDと一致する決済済みポジションを返します。
//	 * 
//	 * @param id ID
//	 * @return 決済済みポジション
//	 */
//	public Position getPositionById(int id);

	/**
	 * 決済済みポジションのリストを返します。
	 * 
	 * @return 決済済みポジションのリスト
	 */
	public LinkedList<Position> getPositions();

	/**
	 * 指定されたシンボルと一致する決済済みポジションのリストを返します。
	 * 
	 * @param symbol シンボル
	 * @return 決済済みポジションのリスト
	 */
	public LinkedList<Position> getPositionsBySymbol(String symbol);

	/**
	 * 指定されたエントリーラベルと一致する決済済みポジションのリストを返します。
	 * 
	 * @param label エントリーラベル
	 * @return 決済済みポジションのリスト
	 */
	public LinkedList<Position> getPositionsByEntryLabel(String label);

	/**
	 * 指定されたイグジットラベルと一致する決済済みポジションのリストを返します。
	 * 
	 * @param label イグジットラベル
	 * @return 決済済みポジションのリスト
	 */
	public LinkedList<Position> getPositionsByExitLabel(String label);

	/**
	 * 指定されたラベルと一致する決済済みポジションのリストを返します。
	 * 
	 * @param label ラベル
	 * @return 決済済みポジションのリスト
	 */
	public LinkedList<Position> getPositionsByLabel(String label);

	// ----------------------------------------------------------------------

	/**
	 * 指定された注文情報から買い注文を発行します。
	 * 
	 * @param order 注文情報
	 */
	public void buy(Order order);

	/**
	 * 指定された注文情報から空売り注文を発行します。
	 * 
	 * @param order 注文情報
	 */
	public void sellShort(Order order);

	/**
	 * 指定された注文情報から売り(売戻し)注文を発行します。
	 * 
	 * @param order 注文情報
	 */
	public void sell(Order order);

	/**
	 * 指定された注文情報から買戻し注文を発行します。
	 * 
	 * @param order 注文情報
	 */
	public void buyToCover(Order order);

}