/*
 * Copyright (c) 2009 OrangeSignal.com All rights reserved.
 * 
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 * 
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 * 
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.ta.data;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.Set;

import jp.sf.orangesignal.ta.util.ArrayUtils;
import jp.sf.orangesignal.ta.util.Assert;

/**
 * データ項目群情報からデータセットへデータを設定する機能を提供します。
 * 
 * @param <T> データセットの型
 * @author 杉澤 浩二
 * @since 2.1
 */
public class DatasetBinder<T> {

	/**
	 * データ項目群情報を保持します。
	 */
	private DatasetSource source;

	/**
	 * データセットを保持します。
	 */
	private T dataset;

	/**
	 * データセットへデータを設定する名前群を保持します。
	 */
	private String[] includeNames;

	/**
	 * データセットへデータを設定しない名前群を保持します。
	 */
	private String[] excludeNames;

//	private boolean excludesNull;

	/**
	 * 指定されたデータ項目群情報とデータセットからこのクラスを構築するコンストラクタです。
	 * 
	 * @param source データ項目群情報
	 * @param dataset データセット
	 * @throws IllegalArgumentException パラメータに <code>null</code> が指定された場合
	 */
	public DatasetBinder(final DatasetSource source, final T dataset) {
		Assert.notNull(source, "DatasetSource must not be null");
		Assert.notNull(dataset, "Dataset must not be null");
		this.source = source;
		this.dataset = dataset;
	}

	/**
	 * データセットへデータを設定する名前群を設定します。
	 * 
	 * @param names データセットへデータを設定する名前群
	 * @return このオブジェクトへの参照
	 * @throws IllegalArgumentException データセットへデータを設定しない名前群が存在する場合
	 */
	public DatasetBinder<T> includes(final String... names) {
		Assert.isEmpty(excludeNames, "Only includes or excludes may be specified.");
		this.includeNames = names;
		return this;
	}

	/**
	 * データセットへデータを設定しない名前群を設定します。
	 * 
	 * @param names データセットへデータを設定しない名前群
	 * @return このオブジェクトへの参照
	 * @throws IllegalArgumentException データセットへデータを設定する名前群が存在する場合
	 */
	public DatasetBinder<T> excludes(final String... names) {
		Assert.isEmpty(includeNames, "Only includes or excludes may be specified.");
		this.excludeNames = names;
		return this;
	}

	/**
	 * 指定された名前がデータセットとしてデータを設定すべき名前かどうかを返します。
	 * 
	 * @param name 名前
	 * @return データセットとしてデータを設定すべき名前の場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	private boolean isTargetName(final String name) {
		if (ArrayUtils.isNotEmpty(excludeNames)) {
			for (final String propertyName : excludeNames) {
				if (propertyName.equals(name))
					return false;
			}
			return true;
		}
		if (ArrayUtils.isNotEmpty(includeNames)) {
			for (final String propertyName : includeNames) {
				if (propertyName.equals(name))
					return true;
			}
			return false;
		}
		return true;
	}

	/**
	 * データ項目群情報からデータセットへデータを設定して返します。
	 * 
	 * @return データが設定されたデータセット
	 * @throws DatasetBindException データの設定に失敗した場合
	 */
	@SuppressWarnings("unchecked")
	public T execute() throws DatasetBindException {
		try {
			final Class<?> cls = dataset.getClass();

			// 日時データをコピーします。
			for (final Map.Entry<String, Date[]> entry : source.getDateMap().entrySet()) {
				final String name = entry.getKey();
				if (isTargetName(name)) {
					final Field field = cls.getDeclaredField(name);
					final Class<?> fieldType = field.getType();
					final Object value;

					// 文字列配列の場合
					if (fieldType.isArray() && String.class.isAssignableFrom(fieldType.getComponentType())) {
						value = DataConvertUtils.toStringArray(entry.getValue(), source.getDateFormatConfig(name));
					// コレクションの場合
					} else if (Collection.class.isAssignableFrom(fieldType)) {
						final Collection<Date> c = Arrays.asList(entry.getValue());
						if (!fieldType.isInterface()) {
							final Collection list = (Collection) fieldType.newInstance();
							list.addAll(c);
							value = list;
						} else if (List.class.isAssignableFrom(fieldType)) {
							value = new ArrayList<Date>(c);
						} else if (Set.class.isAssignableFrom(fieldType)) {
							value = new LinkedHashSet<Date>(c);
						} else if (Queue.class.isAssignableFrom(fieldType)) {
							value = new LinkedList<Date>(c);
						} else {
							throw new IllegalStateException("this dataset field type unsupported: " + fieldType.getName());
						}
					} else {
						value = entry.getValue();
					}
					if (!field.isAccessible()) {
						field.setAccessible(true);
					}
					field.set(dataset, value);
				}
			}

			// 数値データをコピーします。
			for (final Map.Entry<String, Number[]> entry : source.getNumberMap().entrySet()) {
				final String name = entry.getKey();
				if (isTargetName(name)) {
					final Field field = cls.getDeclaredField(name);
					final Class<?> fieldType = field.getType();
					final Object value;

					// 文字列配列の場合
					if (fieldType.isArray() && String.class.isAssignableFrom(fieldType.getComponentType())) {
						value = DataConvertUtils.toStringArray(entry.getValue(), source.getNumberFormatConfig(name));
						// コレクションの場合
					} else if (Collection.class.isAssignableFrom(fieldType)) {
						final Collection<Number> c = Arrays.asList(entry.getValue());
						if (!fieldType.isInterface()) {
							final Collection list = (Collection) fieldType.newInstance();
							list.addAll(c);
							value = list;
						} else if (List.class.isAssignableFrom(fieldType)) {
							value = new ArrayList<Number>(c);
						} else if (Set.class.isAssignableFrom(fieldType)) {
							value = new LinkedHashSet<Number>(c);
						} else if (Queue.class.isAssignableFrom(fieldType)) {
							value = new LinkedList<Number>(c);
						} else {
							throw new IllegalStateException("this dataset field type unsupported: " + fieldType.getName());
						}
					} else {
						value = entry.getValue();
					}
					if (!field.isAccessible()) {
						field.setAccessible(true);
					}
					field.set(dataset, value);
				}
			}

			return dataset;
		} catch (Exception e) {
			throw new DatasetBindException(e.getMessage(), e);
		}
	}

}
