/*
 * Copyright (c) 2006-2009 OrangeSignal.com All rights reserved.
 * 
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 * 
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 * 
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.ta.util;

import java.lang.reflect.Array;

/**
 * 配列操作に関するユーティリティを提供します。
 * 
 * @author 杉澤 浩二
 */
public abstract class ArrayUtils {

	/**
	 * 不変の空の文字列配列です。
	 */
	public static final String[] EMPTY_STRING_ARRAY = new String[0];

	/**
	 * 適切なインデックスが見つからない事を表します。
	 */
	public static final int INDEX_NOT_FOUND = -1;

	//-----------------------------------------------------------------------

	/**
	 * <p>指定された配列群の中で一番長さの短い配列の長さを返します。<code>null</code> の配列は無視されます。
	 * 
	 * @param arrays 配列群
	 * @return 一番長さの短い配列の長さ
	 */
	public static int getMinLength(final Object[]... arrays) {
		int result = 0;
		for (final Object[] array : arrays) {
			if (array == null) {
				continue;
			}
			final int length = array.length;
			if (result == 0 || result > length) {
				result = length;
			}
		}
		return result;
	}

	/**
	 * 指定された配列群の中で一番長さの長い配列の長さを返します。<code>null</code> の配列は無視されます。
	 * 
	 * @param arrays 配列群
	 * @return 一番長さの長い配列の長さ
	 */
	public static int getMaxLength(final Object[]... arrays) {
		int result = 0;
		for (final Object[] array : arrays) {
			if (array == null) {
				continue;
			}
			final int length = array.length;
			if (result == 0 || result < length) {
				result = length;
			}
		}
		return result;
	}

	/**
	 * 指定された配列群の長さが全て同じかどうかを返します。<code>null</code> の配列は無視されます。
	 * 
	 * @param arrays 配列群
	 * @return 全ての配列の長さが等しい場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public static boolean isSameLength(final Object[]... arrays) {
		return getMinLength(arrays) == getMaxLength(arrays);
	}

	// ----------------------------------------------------------------------

	/**
	 * 指定された配列を、前から検索し、<code>null</code> ではない最初の位置を返します。
	 * 
	 * @param array 配列
	 * @return <code>null</code> ではない最初の位置。見つからない場合は <code>-1</code>
	 */
	public static int notNullIndexOf(final Object[] array) {
		return notNullIndexOf(array, 0);
	}

	/**
	 * 指定された配列を、前から検索し、<code>null</code> ではない最初の位置を返します。
	 * 
	 * @param array 配列
	 * @param start 開始位置
	 * @return <code>null</code> ではない最初の位置。見つからない場合は <code>-1</code>
	 */
	public static int notNullIndexOf(final Object[] array, final int start) {
		final int length = getMinLength(array);
		for (int i = Math.max(start, 0); i < length; i++) {
			if (array[i] != null)
				return i;
		}
		return INDEX_NOT_FOUND;
	}

	/**
	 * 指定された配列を、後ろから検索し、<code>null</code> ではない最初の位置を返します。
	 * 
	 * @param array 配列
	 * @return <code>null</code> ではない最初の位置。見つからない場合は <code>-1</code>
	 */
	public static int notNullLastIndexOf(final Object[] array) {
		return notNullLastIndexOf(array, Integer.MAX_VALUE);
	}

	/**
	 * 指定された配列を、後ろから検索し、<code>null</code> ではない最初の位置を返します。
	 * 
	 * @param array 配列
	 * @param start 開始位置
	 * @return <code>null</code> ではない最初の位置。見つからない場合は <code>-1</code>
	 */
	public static int notNullLastIndexOf(final Object[] array, final int start) {
		final int length = getMinLength(array);
		final int _start = start < length ? start : length - 1;
		for (int i = _start; i >= 0; i--) {
			if (array[i] != null)
				return i;
		}
		return INDEX_NOT_FOUND;
	}

	//-----------------------------------------------------------------------

	/**
	 * <p>指定された配列を、指定された長さだけ拡張して返します。</p>
	 * 指定された配列が <code>null</code> の場合は <code>null</code> を返します。<br>
	 * 拡張する長さに正数が指定された場合は配列の後部を拡張します。<br>
	 * 拡張する長さに負数が指定された場合は配列の前部を拡張します。<br>
	 * 拡張する長さに <code>0</code> が指定された場合は、指定された配列をそのまま返します。
	 * 
	 * @param array 配列
	 * @param space 拡張する長さ
	 * @return 拡張された配列
	 */
	@SuppressWarnings("unchecked")
	public static <T> T[] extend(final T[] array, final int space) {
		if (array == null)
			return null;
		if (space == 0)
			return array;

		final int length = array.length;
		final int abs = Math.abs(space);
		final int newSize = length + abs;
		final Class type = array.getClass().getComponentType();

		final T[] results = (T[]) Array.newInstance(type, newSize);
		if (space > 0)
			System.arraycopy(array, 0, results, 0, length);
		else
			System.arraycopy(array, 0, results, abs, length);
		return results;
	}

	//-----------------------------------------------------------------------
/*
	public static boolean isEmpty(final byte[] array) {
		return array == null || array.length == 0;
	}
	public static boolean isEmpty(final short[] array) {
		return array == null || array.length == 0;
	}
	public static boolean isEmpty(final int[] array) {
		return array == null || array.length == 0;
	}
	public static boolean isEmpty(final long[] array) {
		return array == null || array.length == 0;
	}
	public static boolean isEmpty(final float[] array) {
		return array == null || array.length == 0;
	}
	public static boolean isEmpty(final double[] array) {
		return array == null || array.length == 0;
	}
	public static boolean isEmpty(final char[] array) {
		return array == null || array.length == 0;
	}
	public static boolean isEmpty(final boolean[] array) {
		return array == null || array.length == 0;
	}
*/
	public static boolean isEmpty(final Object[] array) {
		return array == null || array.length == 0;
	}
/*
	public static boolean isNotEmpty(final byte[] array) {
		return array != null && array.length > 0;
	}
	public static boolean isNotEmpty(final short[] array) {
		return array != null && array.length > 0;
	}
	public static boolean isNotEmpty(final int[] array) {
		return array != null && array.length > 0;
	}
	public static boolean isNotEmpty(final long[] array) {
		return array != null && array.length > 0;
	}
	public static boolean isNotEmpty(final float[] array) {
		return array != null && array.length > 0;
	}
	public static boolean isNotEmpty(final double[] array) {
		return array != null && array.length > 0;
	}
	public static boolean isNotEmpty(final char[] array) {
		return array != null && array.length > 0;
	}
	public static boolean isNotEmpty(final boolean[] array) {
		return array != null && array.length > 0;
	}
*/
	public static boolean isNotEmpty(final Object[] array) {
		return array != null && array.length > 0;
	}

}
