/*
 * Copyright (c) 2006-2009 OrangeSignal.com All rights reserved.
 * 
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 * 
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 * 
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.trading;

/**
 * ポジション状態の種類を表す列挙型を提供します。
 * 
 * @author 杉澤 浩二
 */
public enum MarketPositionType {

	/**
	 * フラットポジションである事を表します。
	 */
	FLAT {
		@Override public boolean isFlat() { return true; }
		@Override public boolean isLong() { return false; }
		@Override public boolean isShort() { return false; }
	},

	/**
	 * 買いポジションである事を表します。
	 */
	LONG {
		@Override public boolean isFlat() { return false; }
		@Override public boolean isLong() { return true; }
		@Override public boolean isShort() { return false; }
	},

	/**
	 * 売りポジションである事を表します。
	 */
	SHORT {
		@Override public boolean isFlat() { return false; }
		@Override public boolean isLong() { return false; }
		@Override public boolean isShort() { return true; }
	},

	/**
	 * ヘッジ(両建て)ポジションである事を表します。
	 * 
	 * @deprecated 現在サポートしていません。
	 */
	HEDGE {
		@Override public boolean isFlat() { return false; }
		@Override public boolean isLong() { return true; }
		@Override public boolean isShort() { return true; }
	};

	/**
	 * このポジションの種類がポジションを含んでいないかどうかを返します。
	 * 
	 * @return ポジションを含んでいない場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public abstract boolean isFlat();

	/**
	 * このポジションの種類が買いポジションを含んでいるかどうかを返します。
	 * 
	 * @return 買いポジションを含んでいる場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public abstract boolean isLong();

	/**
	 * このポジションの種類が売りポジションを含んでいるかどうかを返します。
	 * 
	 * @return 売りポジションを含んでいる場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public abstract boolean isShort();

	/**
	 * 指定されたパラメーターからポジションの種類を返します。
	 * 
	 * @param l 買いポジションを含んでいるかどうか
	 * @param s 売りポジションを含んでいるかどうか
	 * @return ポジションの種類
	 */
	public static MarketPositionType valueOf(final boolean l, final boolean s) {
		if (l && s)
			return HEDGE;
		if (l)
			return LONG;
		if (s)
			return SHORT;
		return FLAT;
	}

}
