/*
 * Copyright (c) 2009 OrangeSignal.com All rights reserved.
 *
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 *
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.csv.manager;

import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.io.Writer;
import java.text.Format;
import java.util.List;
import java.util.zip.ZipOutputStream;

import jp.gr.java_conf.dangan.util.lha.LhaOutputStream;
import jp.sf.orangesignal.csv.Csv;
import jp.sf.orangesignal.csv.CsvConfig;
import jp.sf.orangesignal.csv.handlers.BeanListHandler;

/**
 * Java プログラム要素のリストと区切り文字形式データの統合出力インタフェースの実装クラスを提供します。
 *
 * @author 杉澤 浩二
 */
public class CsvBeanSaver<T> implements CsvSaver<T> {

	/**
	 * 区切り文字形式情報を保持します。
	 */
	private CsvConfig cfg;

	/**
	 * Java プログラム要素のリストを保持します。
	 */
	private List<T> beans;

	/**
	 * データアクセスハンドラを保持します。
	 */
	private BeanListHandler<T> handler;

	/**
	 * コンストラクタです。
	 * 
	 * @param cfg 区切り文字形式情報
	 * @param beans Java プログラム要素のリスト
	 * @param beanClass Java プログラム要素の型
	 * @throws IllegalArgumentException パラメータが不正な場合
	 */
	public CsvBeanSaver(final CsvConfig cfg, final List<T> beans, final Class<T> beanClass) {
		if (cfg == null) {
			throw new IllegalArgumentException("CsvConfig must not be null");
		}
		this.cfg = cfg;
		this.beans = beans;
		this.handler = new BeanListHandler<T>(beanClass);
	}

	/**
	 * Java プログラム要素へデータを設定する名前群を設定します。
	 * 
	 * @param names Java プログラム要素へデータを設定する名前群
	 * @return このオブジェクトへの参照
	 * @throws IllegalArgumentException Java プログラム要素へデータを設定しない名前群が存在する場合
	 */
	public CsvBeanSaver<T> includes(final String...names) {
		handler.includes(names);
		return this;
	}

	/**
	 * Java プログラム要素へデータを設定しない名前群を設定します。
	 * 
	 * @param names Java プログラム要素へデータを設定しない名前群
	 * @return このオブジェクトへの参照
	 * @throws IllegalArgumentException Java プログラム要素へデータを設定する名前群が存在する場合
	 */
	public CsvBeanSaver<T> excludes(final String...names) {
		handler.excludes(names);
		return this;
	}

	/**
	 * 指定された Java プログラム要素のフィールドを処理するフォーマットオブジェクトを設定します。
	 * 
	 * @param name Java プログラム要素のフィールド名
	 * @param format フィールドを処理するフォーマットオブジェクト
	 * @return このオブジェクトへの参照
	 * @since 1.2
	 */
	public CsvBeanSaver<T> format(final String name, final Format format) {
		handler.format(name, format);
		return this;
	}

	@Override
	public void to(final Writer writer) throws IOException {
		Csv.save(beans, writer, cfg, handler);
	}

	@Override
	public void to(final OutputStream out, final String encoding) throws IOException {
		Csv.save(beans, out, encoding, cfg, handler);
	}

	@Override
	public void to(final OutputStream out) throws IOException {
		Csv.save(beans, out, cfg, handler);
	}

	@Override
	public void to(final File file, final String encoding) throws IOException {
		Csv.save(beans, file, encoding, cfg, handler);
	}

	@Override
	public void to(final File file) throws IOException {
		Csv.save(beans, file, cfg, handler);
	}

	@Override
	public void to(final LhaOutputStream out, final String entryName) throws IOException {
		Csv.save(beans, out, cfg, handler, entryName);
	}

	@Override
	public void to(final ZipOutputStream out, final String entryName) throws IOException {
		Csv.save(beans, out, cfg, handler, entryName);
	}

}
