/*
 * Copyright (c) 2009 OrangeSignal.com All rights reserved.
 *
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 *
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.csv.handlers;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import jp.sf.orangesignal.csv.CsvListHandler;
import jp.sf.orangesignal.csv.CsvReader;
import jp.sf.orangesignal.csv.CsvWriter;

/**
 * 項目名と項目値のマップのリストで区切り文字形式データアクセスを行うハンドラを提供します。
 *
 * @author 杉澤 浩二
 * @since 1.1
 */
public class ColumnNameMapListHandler implements CsvListHandler<Map<String, String>> {

	/**
	 * 項目名のリストを保持します。
	 */
	private List<String> columnNames;

	/**
	 * 取得データの開始位置を保持します。
	 */
	private int offset;

	/**
	 * 取得データの限度数を保持します。
	 */
	private int limit;

	/**
	 * デフォルトコンストラクタです。
	 */
	public ColumnNameMapListHandler() {}

	/**
	 * 指定された項目名を項目名の一覧へ追加します。
	 * 
	 * @param columnName 項目名
	 * @return このオブジェクトへの参照
	 */
	public ColumnNameMapListHandler addColumn(final String columnName) {
		if (columnNames == null) {
			columnNames = new ArrayList<String>();
		}
		columnNames.add(columnName);
		return this;
	}

	/**
	 * 指定された項目名の一覧を設定します。
	 * 
	 * @param columnNames 項目名の一覧
	 * @return このオブジェクトへの参照
	 */
	public ColumnNameMapListHandler columnNames(final Collection<String> columnNames) {
		this.columnNames = new ArrayList<String>(columnNames);
		return this;
	}

	/**
	 * 取得データの開始位置を設定します。
	 * 
	 * @param offset 取得データの開始位置
	 * @return このオブジェクトへの参照
	 * @since 1.2.2
	 */
	public ColumnNameMapListHandler offset(final int offset) {
		this.offset = offset;
		return this;
	}

	/**
	 * 取得データの限度数を設定します。
	 * 
	 * @param limit 取得データの限度数
	 * @return このオブジェクトへの参照
	 * @since 1.2.2
	 */
	public ColumnNameMapListHandler limit(final int limit) {
		this.limit = limit;
		return this;
	}

	@Override
	public List<Map<String, String>> load(final CsvReader reader) throws IOException {
		// TODO: 必要があればここに入力パラメータを検証するコードを記述します。

		if (columnNames == null) {
			columnNames = reader.readValues();
			if (columnNames == null) {
				// ヘッダがない場合は例外をスローします。
				throw new IOException("No header is available");
			}
		}

		// ヘッダ部を処理します。
		final int columnCount = columnNames.size();
		final Map<String, String> m = new LinkedHashMap<String, String>(columnCount);
		for (final String columnName : columnNames) {
			m.put(columnName, null);
		}

		// データ部を処理します。
		final List<Map<String, String>> results = new ArrayList<Map<String, String>>();
		int offset = 0;
		List<String> values;
		while ((values = reader.readValues()) != null && (limit <= 0 || results.size() < limit)) {
			if (offset < this.offset) {
				offset++;
				continue;
			}
			final Map<String, String> map = new LinkedHashMap<String, String>(m);
			final int len = Math.min(columnCount, values.size());
			for (int pos = 0; pos < len; pos++) {
				map.put(columnNames.get(pos), values.get(pos));
			}
			results.add(map);
		}
		return results;
	}

	@Override
	public void save(final List<Map<String, String>> list, final CsvWriter writer) throws IOException {
		// TODO: 必要があればここに入力パラメータを検証するコードを記述します。

		if (columnNames == null) {
			for (final Map<String, String> map : list) {
				if (map != null) {
					columnNames = new ArrayList<String>(map.keySet());
					break;
				}
			}
		}
		if (columnNames == null) {
			// ヘッダがない場合は例外をスローします。
			throw new IOException("No header is available");
		}

		// ヘッダ部を処理します。
		writer.writeValues(columnNames);

		// データ部を処理します。
		final int columnCount = columnNames.size();
		for (final Map<String, String> map : list) {
			// 要素が null の場合は null 出力します。
			if (map == null) {
				writer.writeValues(null);
				continue;
			}

			final String[] values = new String[columnCount];
			for (int i = 0; i < columnCount; i++) {
				values[i] = map.get(columnNames.get(i));
			}
			writer.writeValues(Arrays.asList(values));
		}
	}

}
