/*
 * Copyright (c) 2009 OrangeSignal.com All rights reserved.
 *
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 *
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.csv.manager;

import java.text.Format;

import jp.sf.orangesignal.csv.CsvConfig;
import jp.sf.orangesignal.csv.handlers.BeanListHandler;

/**
 * Java プログラム要素のリストと区切り文字形式データの統合入力インタフェースの実装クラスを提供します。
 *
 * @author 杉澤 浩二
 */
public class CsvBeanLoader<T> extends CsvLoaderSupport<T> {

	/**
	 * データアクセスハンドラを保持します。
	 */
	private BeanListHandler<T> handler;

	/**
	 * コンストラクタです。
	 *
	 * @param cfg 区切り文字形式情報
	 * @param beanClass JavaBean の型
	 * @throws IllegalArgumentException パラメータが不正な場合
	 */
	public CsvBeanLoader(final CsvConfig cfg, final Class<T> beanClass) {
		super(cfg);
		this.handler = new BeanListHandler<T>(beanClass);
	}

	/**
	 * Java プログラム要素へデータを設定する名前群を設定します。
	 * 
	 * @param names Java プログラム要素へデータを設定する名前群
	 * @return このオブジェクトへの参照
	 * @throws IllegalArgumentException Java プログラム要素へデータを設定しない名前群が存在する場合
	 * @see BeanListHandler#includes(String...)
	 */
	public CsvBeanLoader<T> includes(final String... names) {
		handler.includes(names);
		return this;
	}

	/**
	 * Java プログラム要素へデータを設定しない名前群を設定します。
	 * 
	 * @param names Java プログラム要素へデータを設定しない名前群
	 * @return このオブジェクトへの参照
	 * @throws IllegalArgumentException Java プログラム要素へデータを設定する名前群が存在する場合
	 * @see BeanListHandler#excludes(String...)
	 */
	public CsvBeanLoader<T> excludes(final String... names) {
		handler.excludes(names);
		return this;
	}

	/**
	 * 指定された Java プログラム要素のフィールドを処理するフォーマットオブジェクトを設定します。
	 * 
	 * @param name Java プログラム要素のフィールド名
	 * @param format フィールドを処理するフォーマットオブジェクト
	 * @return このオブジェクトへの参照
	 * @since 1.2
	 */
	public CsvBeanLoader<T> format(final String name, final Format format) {
		handler.format(name, format);
		return this;
	}

	/**
	 * 取得データの開始位置を設定します。
	 * 
	 * @param offset 取得データの開始位置
	 * @return このオブジェクトへの参照
	 * @since 1.2.1
	 */
	public CsvBeanLoader<T> offset(final int offset) {
		handler.offset(offset);
		return this;
	}

	/**
	 * 取得データの限度数を設定します。
	 * 
	 * @param limit 取得データの限度数
	 * @return このオブジェクトへの参照
	 * @since 1.2.1
	 */
	public CsvBeanLoader<T> limit(final int limit) {
		handler.limit(limit);
		return this;
	}

	@Override protected BeanListHandler<T> getCsvListHandler() { return handler; }

}
