/*
 * Copyright 2005 Jenia org.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jenia.faces.chart.util;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Paint;
import java.awt.geom.Rectangle2D;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.text.DecimalFormat;
import java.text.NumberFormat;

import org.apache.batik.dom.GenericDOMImplementation;
import org.apache.batik.svggen.SVGGraphics2D;
import org.jfree.chart.ChartFactory;
import org.jfree.chart.ChartUtilities;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.axis.CategoryAxis;
import org.jfree.chart.axis.CategoryLabelPositions;
import org.jfree.chart.axis.NumberAxis;
import org.jfree.chart.axis.NumberTickUnit;
import org.jfree.chart.labels.ItemLabelPosition;
import org.jfree.chart.labels.StandardCategoryItemLabelGenerator;
import org.jfree.chart.plot.CategoryPlot;
import org.jfree.chart.plot.PiePlot;
import org.jfree.chart.plot.PiePlot3D;
import org.jfree.chart.plot.PlotOrientation;
import org.jfree.chart.plot.XYPlot;
import org.jfree.chart.renderer.category.AreaRenderer;
import org.jfree.chart.renderer.category.CategoryItemRenderer;
import org.jfree.chart.renderer.category.LineAndShapeRenderer;
import org.jfree.data.category.CategoryDataset;
import org.jfree.data.general.PieDataset;
import org.jfree.data.xy.IntervalXYDataset;
import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;

public abstract class ChartBean {
	protected int width = 376;

	protected int height = 240;

	protected abstract JFreeChart getChart();

	public final void writeChart(OutputStream out) throws IOException {
		ChartUtilities.writeChartAsPNG(out, getChart(), width, height);
	}

	public final void writeChartJPEG(OutputStream out) throws IOException {
		ChartUtilities.writeChartAsJPEG(out, getChart(), width, height);
	}

	public void saveChartToSVG(PrintWriter pw) throws Exception {
		if (getChart() != null) {
			DOMImplementation domImpl = GenericDOMImplementation
					.getDOMImplementation();
			Document document = domImpl.createDocument(null, "svg", null);
			SVGGraphics2D svgGenerator = new SVGGraphics2D(document);
			getChart().draw(svgGenerator,
					new Rectangle2D.Double(0, 0, width, height), null);

			svgGenerator.stream(pw, false);
		}
	}

	protected JFreeChart createBarChart3D(String title, String xAxisLabel,
			String yAxisLabel, CategoryDataset dataset, NumberFormat formatter,
			double upperMargin, double labelAngle, boolean includeZero,
			boolean labelVisibility, BasicStroke[] basicStroke,
			boolean includeLegend, boolean includeTooltips,
			boolean includeUrls, PlotOrientation orientation, Color color,
			float transparency, Color plotColor, String xAxisOrientation,
			double unit, Paint[] paint) {

		if (orientation == null)
			orientation = PlotOrientation.VERTICAL;

		if (formatter == null)
			formatter = new DecimalFormat("#,##0.00");

		JFreeChart chart = ChartFactory.createBarChart3D(title, xAxisLabel,
				yAxisLabel, dataset, // data
				orientation, // orientation
				includeLegend, // include legend
				includeTooltips, // tooltips
				includeUrls // urls
				);

		chart.setBackgroundPaint(color);
		setLegend(chart, includeLegend);

		CategoryPlot plot = chart.getCategoryPlot();
		if (xAxisOrientation.equals("vertical")) {
			CategoryAxis axis = plot.getDomainAxis();
			axis.setCategoryLabelPositions(CategoryLabelPositions.UP_90);
			axis.setMaximumCategoryLabelWidthRatio(100f);
		}
		plot.setForegroundAlpha(transparency);
		plot.setBackgroundPaint(plotColor);
		CategoryItemRenderer renderer = plot.getRenderer();

		if (basicStroke == null) {
			int column = dataset.getColumnCount();
			for (int i = 0; i < column; i++) {
				renderer.setSeriesStroke(i, new BasicStroke());
			}
		} else {
			setSeriesStrokes(renderer, basicStroke);
		}
		
		if (paint != null) {
			int column = dataset.getRowCount();
			for (int i=0; i<column; i++) {
				renderer.setSeriesPaint(i,paint[i]);
			}
		}

		setRangeAxis(plot, renderer, formatter, upperMargin, labelAngle,
				includeZero, labelVisibility, unit);

		return chart;
	}

	protected JFreeChart createHistogramChart(String title, String xAxisLabel,
			String yAxisLabel, IntervalXYDataset dataset,
			NumberFormat formatter, double upperMargin, double labelAngle,
			boolean includeZero, boolean labelVisibility,
			BasicStroke[] basicStroke, boolean includeLegend,
			boolean includeTooltips, boolean includeUrls,
			PlotOrientation orientation, Color color, float transparency,
			Color plotColor, String xAxisOrientation, double unit) {

		if (orientation == null)
			orientation = PlotOrientation.VERTICAL;

		if (formatter == null)
			formatter = new DecimalFormat("#,##0.00");

		JFreeChart chart = ChartFactory.createHistogram(title, xAxisLabel,
				yAxisLabel, dataset, // data
				orientation, // orientation
				includeLegend, // include legend
				includeTooltips, // tooltips
				includeUrls // urls
				);

		chart.setBackgroundPaint(color);
		setLegend(chart, includeLegend);

		// XYPlot plot = (XYPlot)chart.getPlot();
		// if (xAxisOrientation.equals("vertical")) {
		// NumberAxis axis = (NumberAxis)plot.getDomainAxis();
		// axis.setCategoryLabelPositions(CategoryLabelPositions.UP_90);
		// axis.setMaximumCategoryLabelWidthRatio(100f);
		// }
		// plot.setForegroundAlpha(transparency);
		// plot.setBackgroundPaint(plotColor);
		// CategoryItemRenderer renderer = plot.getRenderer();
		//
		// if (basicStroke == null) {
		// int column = dataset.getSeriesCount();
		// for (int i = 0; i < column; i++) {
		// renderer.setSeriesStroke(i, new BasicStroke());
		// }
		// } else {
		// setSeriesStrokes(renderer, basicStroke);
		// }
		//
		// setRangeAxis(plot, renderer, formatter, upperMargin, labelAngle,
		// includeZero, labelVisibility, unit);

		return chart;
	}

	protected JFreeChart createBarChart(String title, String xAxisLabel,
			String yAxisLabel, CategoryDataset dataset, NumberFormat formatter,
			double upperMargin, double labelAngle, boolean includeZero,
			boolean labelVisibility, BasicStroke[] basicStroke,
			boolean includeLegend, boolean includeTooltips,
			boolean includeUrls, PlotOrientation orientation, Color color,
			float transparency, Color plotColor, String xAxisOrientation,
			double unit, Paint[] paint) {

		if (orientation == null)
			orientation = PlotOrientation.VERTICAL;

		if (formatter == null)
			formatter = new DecimalFormat("#,##0.00");

		JFreeChart chart = ChartFactory.createBarChart(title, xAxisLabel,
				yAxisLabel, dataset, // data
				orientation, // orientation
				includeLegend, // include legend
				includeTooltips, // tooltips
				includeUrls // urls
				);

		chart.setBackgroundPaint(color);
		setLegend(chart, includeLegend);

		CategoryPlot plot = chart.getCategoryPlot();
		if (xAxisOrientation.equals("vertical")) {
			CategoryAxis axis = plot.getDomainAxis();
			axis.setCategoryLabelPositions(CategoryLabelPositions.UP_90);
			axis.setMaximumCategoryLabelWidthRatio(100f);
		}
		plot.setForegroundAlpha(transparency);
		plot.setBackgroundPaint(plotColor);
		CategoryItemRenderer renderer = plot.getRenderer();

		if (basicStroke == null) {
			int column = dataset.getRowCount();
			for (int i = 0; i < column; i++) {
				renderer.setSeriesStroke(i, new BasicStroke());
			}
		} else {
			setSeriesStrokes(renderer, basicStroke);
		}
		
		if (paint != null) {
			int column = dataset.getRowCount();
			for (int i=0; i<column; i++) {
				renderer.setSeriesPaint(i,paint[i]);
			}
		}
		

		setRangeAxis(plot, renderer, formatter, upperMargin, labelAngle,
				includeZero, labelVisibility, unit);

		return chart;
	}

	protected JFreeChart createStackedBarChart3D(String title,
			String xAxisLabel, String yAxisLabel, CategoryDataset dataset,
			NumberFormat formatter, double upperMargin, double labelAngle,
			boolean includeZero, boolean labelVisibility,
			BasicStroke[] basicStroke, boolean includeLegend,
			boolean includeTooltips, boolean includeUrls,
			PlotOrientation orientation, Color color, float transparency,
			Color plotColor, String xAxisOrientation, double unit, Paint[] paint) {

		if (orientation == null)
			orientation = PlotOrientation.VERTICAL;

		if (formatter == null)
			formatter = new DecimalFormat("#,##0.00");

		JFreeChart chart = ChartFactory.createStackedBarChart3D(title,
				xAxisLabel, yAxisLabel, dataset, // data
				orientation, // orientation
				includeLegend, // include legend
				includeTooltips, // tooltips
				includeUrls // urls
				);

		chart.setBackgroundPaint(color);
		setLegend(chart, includeLegend);

		CategoryPlot plot = chart.getCategoryPlot();
		if (xAxisOrientation.equals("vertical")) {
			CategoryAxis axis = plot.getDomainAxis();
			axis.setCategoryLabelPositions(CategoryLabelPositions.UP_90);
			axis.setMaximumCategoryLabelWidthRatio(100f);
		}
		plot.setForegroundAlpha(transparency);
		plot.setBackgroundPaint(plotColor);
		CategoryItemRenderer renderer = plot.getRenderer();

		if (basicStroke == null) {
			int column = dataset.getColumnCount();
			for (int i = 0; i < column; i++) {
				renderer.setSeriesStroke(i, new BasicStroke());
			}
		} else {
			setSeriesStrokes(renderer, basicStroke);
		}
		
		if (paint != null) {
			int column = dataset.getRowCount();
			for (int i=0; i<column; i++) {
				renderer.setSeriesPaint(i,paint[i]);
			}
		}
		

		setRangeAxis(plot, renderer, formatter, upperMargin, labelAngle,
				includeZero, labelVisibility, unit);

		return chart;
	}

	protected JFreeChart createLineChart(String title, String xAxisLabel,
			String yAxisLabel, CategoryDataset dataset, NumberFormat formatter,
			double upperMargin, double labelAngle, boolean includeZero,
			boolean labelVisibility, BasicStroke[] basicStroke,
			boolean includeLegend, boolean includeTooltips,
			boolean includeUrls, PlotOrientation orientation, Color color,
			float transparency, Color plotColor, String xAxisOrientation,
			double unit, Paint[] paint) {

		if (orientation == null)
			orientation = PlotOrientation.VERTICAL;

		if (formatter == null)
			formatter = new DecimalFormat("#,##0.00");

		JFreeChart chart = ChartFactory.createLineChart(title, xAxisLabel,
				yAxisLabel, dataset, // data
				orientation, // orientation
				includeLegend, // include legend
				includeTooltips, // tooltips
				includeUrls // urls
				);

		chart.setBackgroundPaint(color);
		setLegend(chart, includeLegend);

		CategoryPlot plot = chart.getCategoryPlot();
		if (xAxisOrientation.equals("vertical")) {
			CategoryAxis axis = plot.getDomainAxis();
			axis.setCategoryLabelPositions(CategoryLabelPositions.UP_90);
			axis.setMaximumCategoryLabelWidthRatio(100f);
		}

		plot.setForegroundAlpha(transparency);
		plot.setBackgroundPaint(plotColor);
		LineAndShapeRenderer renderer = (LineAndShapeRenderer) plot
				.getRenderer();

		if (basicStroke == null) {
			int column = dataset.getColumnCount();
			for (int i = 0; i < column; i++) {
				renderer.setSeriesStroke(i, new BasicStroke());
			}
		} else {
			setSeriesStrokes(renderer, basicStroke);
		}
		
		if (paint != null) {
			int column = dataset.getRowCount();
			for (int i=0; i<column; i++) {
				renderer.setSeriesPaint(i,paint[i]);
			}
		}		

		setRangeAxis(plot, renderer, formatter, upperMargin, labelAngle,
				includeZero, labelVisibility, unit);

		return chart;
	}

	protected JFreeChart createAreaChart(String title, String xAxisLabel,
			String yAxisLabel, CategoryDataset dataset, NumberFormat formatter,
			double upperMargin, double labelAngle, boolean includeZero,
			boolean labelVisibility, BasicStroke[] basicStroke,
			boolean includeLegend, boolean includeTooltips,
			boolean includeUrls, PlotOrientation orientation, Color color,
			float transparency, Color plotColor, String xAxisOrientation,
			double unit, Paint[] paint) {

		if (orientation == null)
			orientation = PlotOrientation.VERTICAL;

		if (formatter == null)
			formatter = new DecimalFormat("#,##0.00");

		JFreeChart chart = ChartFactory.createAreaChart(title, xAxisLabel,
				yAxisLabel, dataset, // data
				orientation, // orientation
				includeLegend, // include legend
				includeTooltips, // tooltips
				includeUrls // urls
				);

		chart.setBackgroundPaint(color);
		setLegend(chart, includeLegend);

		CategoryPlot plot = chart.getCategoryPlot();
		if (xAxisOrientation.equals("vertical")) {
			CategoryAxis axis = plot.getDomainAxis();
			axis.setCategoryLabelPositions(CategoryLabelPositions.UP_90);
			axis.setMaximumCategoryLabelWidthRatio(100f);
		}

		plot.setForegroundAlpha(transparency);
		plot.setBackgroundPaint(plotColor);
		AreaRenderer renderer = (AreaRenderer) plot.getRenderer();

		if (basicStroke == null) {
			int column = dataset.getColumnCount();
			for (int i = 0; i < column; i++) {
				renderer.setSeriesStroke(i, new BasicStroke());
			}
		} else {
			setSeriesStrokes(renderer, basicStroke);
		}
		
		if (paint != null) {
			int column = dataset.getRowCount();
			for (int i=0; i<column; i++) {
				renderer.setSeriesPaint(i,paint[i]);
			}
		}		

		setRangeAxis(plot, renderer, formatter, upperMargin, labelAngle,
				includeZero, labelVisibility, unit);

		return chart;
	}

	protected JFreeChart createPieChart3D(String title, PieDataset dataset,
			Color color, float transparency, Color plotColor, boolean crush) {
		JFreeChart chart = null;
		if (crush) {
			chart = ChartFactory.createPieChart(title, dataset,
					true, // include
					// legend
					true, // tooltips
					false // urls
					);
		} else {
			chart = ChartFactory.createPieChart3D(title, dataset,
					true, // include
					// legend
					true, // tooltips
					false // urls
					);
		}

		chart.setBackgroundPaint(color);
		if (crush) {
			PiePlot plot = (PiePlot) chart.getPlot();
			plot.setForegroundAlpha(transparency);
			plot.setBackgroundPaint(plotColor);
		} else {
			PiePlot3D plot = (PiePlot3D) chart.getPlot();
			plot.setForegroundAlpha(transparency);
			plot.setBackgroundPaint(plotColor);
			
		}

		return chart;
	}

	protected void setLegend(JFreeChart chart, boolean showShapes) {
		// LegendTitle legend = (LegendTitle)chart.getLegend();
	}

	protected void setRangeAxis(CategoryPlot plot,
			CategoryItemRenderer renderer, NumberFormat formatter,
			double upperMargin, double labelAngle, boolean includeZero,
			boolean labelVisibility, double unit) {
		NumberAxis rangeAxis = (NumberAxis) plot.getRangeAxis();
		rangeAxis.setStandardTickUnits(NumberAxis.createIntegerTickUnits());
		rangeAxis.setNumberFormatOverride(formatter);
		rangeAxis.setAutoRangeIncludesZero(includeZero);
		rangeAxis.setUpperMargin(upperMargin);
		rangeAxis.setLabelAngle(labelAngle);
		if (Double.compare(unit, Double.MIN_VALUE) != 0)
			rangeAxis.setTickUnit(new NumberTickUnit(unit));

		renderer.setItemLabelGenerator(new StandardCategoryItemLabelGenerator(
				StandardCategoryItemLabelGenerator.DEFAULT_LABEL_FORMAT_STRING,
				formatter));
		renderer.setItemLabelsVisible(labelVisibility);
		renderer.setPositiveItemLabelPosition(new ItemLabelPosition());
		renderer.setNegativeItemLabelPosition(new ItemLabelPosition());
	}

	protected void setSeriesStrokes(CategoryItemRenderer renderer,
			BasicStroke[] basicStroke) {
		for (int i = 0; i < basicStroke.length; i++) {
			BasicStroke ba = basicStroke[i];
			renderer.setSeriesStroke(i, ba);
		}
	}

	public static final Color getColor(String htmlColor) {
		if (htmlColor == null)
			htmlColor = "#FFFFFF";
		String color = htmlColor.substring(1);
		Integer rComponent = new Integer(Integer.parseInt(
				color.substring(0, 2), 16));
		Integer gComponent = new Integer(Integer.parseInt(
				color.substring(2, 4), 16));
		Integer bComponent = new Integer(Integer.parseInt(
				color.substring(4, 6), 16));
		Color graphColor = new Color(rComponent.intValue(), gComponent
				.intValue(), bComponent.intValue());

		return graphColor;
	}

	public static final String getFileExt(String fileExt) {
		if (fileExt == null)
			fileExt = "png";
		else if (fileExt.equals("jpeg"))
			fileExt = "jpg";
		else if (!fileExt.equals("png") && !fileExt.equals("jpg")
				&& !fileExt.equals("svg"))
			fileExt = "png";

		return fileExt;
	}

	public static final String getChartType(String type) {
		if (type == null)
			return "Bar";
		if (!type.equals("Bar") && !type.equals("StackedBar")
				&& !type.equals("Line") && !type.equals("Area")
				&& !type.equals("Histogram") && !type.equals("BarChart2D"))
			return "Bar";
		return type;
	}
}
