/*
 * Copyright 2005 Jenia org.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jenia.faces.renderkit.html;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.faces.component.UIComponent;
import javax.faces.component.UIData;
import javax.faces.component.UIForm;
import javax.faces.component.UIInput;
import javax.faces.component.UIParameter;
import javax.faces.component.UIViewRoot;
import javax.faces.component.ValueHolder;
import javax.faces.context.FacesContext;
import javax.faces.context.ResponseWriter;
import javax.faces.convert.Converter;
import javax.faces.convert.ConverterException;
import javax.faces.render.Renderer;

import org.jenia.faces.component.UIPopupBox;
import org.jenia.faces.util.MessageFactory;
import org.jenia.faces.util.Util;

/**
 * @author Andrea Tessaro Porta
 * 
 * The base of this class is com.sun.faces.renderkit.html_basic.HtmlBasicRenderer
 **/
public class HtmlBasicRenderer extends Renderer {

    public HtmlBasicRenderer() {
        super();
    }

    protected void commonJavascript(FacesContext context, UIComponent component) throws IOException {
        // empty by now
    }
    
    public void addGenericErrorMessage(FacesContext facesContext,
                                       UIComponent component,
                                       String messageId, String param) {
        Object[] params = new Object[3];
        params[0] = param;
        facesContext.addMessage(component.getClientId(facesContext),
            MessageFactory.getMessage(facesContext, component,messageId, params));
    }

    public void decode(FacesContext context, UIComponent component) {

        // If the component is disabled, do not change the value of the
        // component, since its state cannot be changed.
        if (Util.componentIsDisabledOnReadonly(component)) {
            return;
        }

        String clientId = component.getClientId(context);
        Map requestMap = context.getExternalContext().getRequestParameterMap();
        // Don't overwrite the value unless you have to!
        if (requestMap.containsKey(clientId)) {
            String newValue = (String) requestMap.get(clientId);
            setSubmittedValue(component, newValue);
        }
    }


    public void encodeEnd(FacesContext context, UIComponent component)
        throws IOException {

        String currentValue = null;

        // suppress rendering if "rendered" property on the component is
        // false.
        if (!component.isRendered()) {
            return;
        }

        currentValue = getCurrentValue(context, component);
        getEndTextToRender(context, component, currentValue);
    }


    /**
     * Gets value to be rendered and formats it if required. Sets to empty
     * string if value is null.
     */
    protected String getCurrentValue(FacesContext context, UIComponent component) {

        if (component instanceof UIInput) {
            Object submittedValue = ((UIInput) component).getSubmittedValue();
            if (submittedValue != null) {
                return (String) submittedValue;
            }
        }

        String currentValue = null;
        Object currentObj = getValue(component);
        if (currentObj != null) {
            currentValue = getFormattedValue(context, component, currentObj);
        }
        return currentValue;
    }


    protected Object getValue(UIComponent component) {
        throw new UnsupportedOperationException();
    }


    /**
     * Renderers override this method to write appropriate HTML content into
     * the buffer.
     */
    protected void getEndTextToRender(FacesContext context, UIComponent component,
                                      String currentValue) throws IOException {
        return;
    }


    /**
     * Renderers override this method to store the previous value
     * of the associated component.
     */
    protected void setSubmittedValue(UIComponent component, Object value) {
    }


    /**
     * Renderers override this method in case output value needs to be
     * formatted
     */
    protected String getFormattedValue(FacesContext context, UIComponent component,
                                       Object currentValue)
        throws ConverterException {

        String result = null;
        // formatting is supported only for components that support
        // converting value attributes.
        if (!(component instanceof ValueHolder)) {
            if (currentValue != null) {
                result = currentValue.toString();
            }
            return result;
        }

        Converter converter = null;

        // If there is a converter attribute, use it to to ask application
        // instance for a converter with this identifer.

        if (component instanceof ValueHolder) {
            converter = ((ValueHolder) component).getConverter();
        }

        // if value is null and no converter attribute is specified, then
        // return a zero length String.
        if (converter == null && currentValue == null) {
            return "";
        }

        if (converter == null) {
            // Do not look for "by-type" converters for Strings
            if (currentValue instanceof String) {
                return (String) currentValue;
            }

            // if converter attribute set, try to acquire a converter
            // using its class type.

            Class converterType = currentValue.getClass();
            converter = Util.getConverterForClass(converterType);

            // if there is no default converter available for this identifier,
            // assume the model type to be String.
            if (converter == null && currentValue != null) {
                result = currentValue.toString();
                return result;
            }
        }

        if (converter != null) {
            result = converter.getAsString(context, component, currentValue);

            return result;
        } else {
            // throw converter exception if no converter can be
            // identified
		    Object [] params = {
				currentValue,
				"null Converter"
			    };
	    
            throw new ConverterException(MessageFactory.getMessage(
                context, component, Util.CONVERSION_ERROR_MESSAGE_ID, params));
        }
    }


    public String convertClientId(FacesContext context, String clientId) {
        return clientId;
    }


    /**
     * <p>Render nested child components by invoking the encode methods
     * on those components, but only when the <code>rendered</code>
     * property is <code>true</code>.</p>
     */
    protected void encodeRecursive(FacesContext context, UIComponent component)
        throws IOException {

        // suppress rendering if "rendered" property on the component is
        // false.
        if (!component.isRendered()) {
            return;
        }

        // Render this component and its children recursively
        component.encodeBegin(context);
        if (component.getRendersChildren()) {
            component.encodeChildren(context);
        } else {
            Iterator kids = getChildren(component);
            while (kids.hasNext()) {
                UIComponent kid = (UIComponent) kids.next();
                encodeRecursive(context, kid);
            }
        }
        component.encodeEnd(context);
    }


    /**
     * <p>Return an Iterator over the children of the specified
     * component, selecting only those that have a
     * <code>rendered</code> property of <code>true</code>.</p>
     *
     * @param component <code>UIComponent</code> for which to extract children
     */
    protected Iterator getChildren(UIComponent component) {

        List results = new ArrayList();
        Iterator kids = component.getChildren().iterator();
        while (kids.hasNext()) {
            UIComponent kid = (UIComponent) kids.next();
            if (kid.isRendered()) {
                results.add(kid);
            }
        }
        return (results.iterator());

    }


    /**
     * <p>Return true if there is a children of the 
     * specified component, those that have a
     * <code>rendered</code> property of <code>true</code>.</p>
     *
     * @param component <code>UIComponent</code> for which to extract children
     */
    protected boolean hasChildren(UIComponent component) {
        Iterator kids = component.getChildren().iterator();
        while (kids.hasNext()) {
            UIComponent kid = (UIComponent) kids.next();
            if (kid.isRendered()) {
                return true;
            }
        }
        return false;
    }


    /**
     * <p>Return the specified facet from the specified component, but
     * <strong>only</strong> if its <code>rendered</code> property is
     * set to <code>true</code>.
     *
     * @param component Component from which to return a facet
     * @param name      Name of the desired facet
     */
    protected UIComponent getFacet(UIComponent component, String name) {

        UIComponent facet = component.getFacet(name);
        if ((facet != null) && !facet.isRendered()) {
            facet = null;
        }
        return (facet);

    }


    /**
     * @return true if this renderer should render an id attribute.
     */
    protected boolean shouldWriteIdAttribute(UIComponent component) {
        String id;
        return (null != (id = component.getId()) &&
            !id.startsWith(UIViewRoot.UNIQUE_ID_PREFIX));
    }


    protected void writeIdAttributeIfNecessary(FacesContext context,
                                               ResponseWriter writer,
                                               UIComponent component) {
        if (shouldWriteIdAttribute(component)) {
            try {
                writer.writeAttribute("id", component.getClientId(context),
                                      "id");
            } catch (IOException e) {
                // nothing to do
            }
        }
    }

    protected UIForm getMyForm(FacesContext context, UIComponent component) {
        UIComponent parent = component.getParent();
        while (parent != null) {
            if (parent instanceof UIForm) {
                break;
            }
            parent = parent.getParent();
        }
        return (UIForm) parent;
    }

    protected UIPopupBox getMyPopupBox(FacesContext context, UIComponent component) {
        UIComponent parent = component.getParent();
        while (parent != null) {
            if (parent instanceof UIPopupBox) {
                break;
            }
            parent = parent.getParent();
        }
        return (UIPopupBox) parent;
    }

    protected UIData getMyDatTable(FacesContext context, UIComponent component) {
        UIComponent parent = component.getParent();
        while (parent != null) {
            if (parent instanceof UIData) {
                break;
            }
            parent = parent.getParent();
        }
        return (UIData) parent;
    }

	protected Param[] getParamList(FacesContext context, UIComponent component) {
        ArrayList parameterList = new ArrayList();

        Iterator kids = component.getChildren().iterator();
        while (kids.hasNext()) {
            UIComponent kid = (UIComponent) kids.next();

            if (kid instanceof UIParameter) {
                UIParameter uiParam = (UIParameter) kid;
                Object value = uiParam.getValue();
                Param param = new Param(uiParam.getName(),
                                        (value == null ? null :
                                         value.toString()));
                parameterList.add(param);
            }
        }

        return (Param[]) parameterList.toArray(new Param[parameterList.size()]);
    }

    //inner class to store parameter name and value pairs
    protected class Param {

        public Param(String name, String value) {
            set(name, value);
        }


        private String name;
        private String value;


        public void set(String name, String value) {
            this.name = name;
            this.value = value;
        }


        public String getName() {
            return name;
        }


        public String getValue() {
            return value;
        }
    }

	public void encodeChildren(FacesContext facesContext, UIComponent component) throws IOException {
		renderChildren(facesContext,component);
	}
	
    public static void renderChildren(FacesContext facesContext, UIComponent component) throws IOException
	{
		if (component.getChildCount() > 0)
		{
		    for (Iterator it = component.getChildren().iterator(); it.hasNext(); )
		    {
		        UIComponent child = (UIComponent)it.next();
		        renderChild(facesContext, child);
		    }
		}
	}
	
	
	public static void renderChild(FacesContext facesContext, UIComponent child)
	    throws IOException
	{
		if (!child.isRendered())
		{
		    return;
		}
		
		child.encodeBegin(facesContext);
		if (child.getRendersChildren())
		{
		    child.encodeChildren(facesContext);
		}
		else
		{
		    renderChildren(facesContext, child);
		}
		child.encodeEnd(facesContext);
	}
	
	
}
