/*
 * Copyright 2005 Jenia org.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jenia.faces.util;

import java.io.IOException;
import java.util.Collections;
import java.util.Iterator;
import java.util.Locale;
import java.util.ResourceBundle;

import javax.faces.FactoryFinder;
import javax.faces.application.Application;
import javax.faces.application.ApplicationFactory;
import javax.faces.application.FacesMessage;
import javax.faces.component.NamingContainer;
import javax.faces.component.UIComponent;
import javax.faces.component.UIForm;
import javax.faces.component.UIViewRoot;
import javax.faces.context.FacesContext;
import javax.faces.context.ResponseWriter;
import javax.faces.convert.Converter;
import javax.faces.el.MethodBinding;
import javax.faces.el.ValueBinding;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.apache.myfaces.config.MyfacesConfig;
import org.apache.myfaces.renderkit.html.HtmlFormRendererBase;
import org.apache.myfaces.renderkit.html.HtmlRendererUtils;
import org.apache.myfaces.renderkit.html.util.JavascriptUtils;

import com.sun.faces.renderkit.html_basic.FormRenderer;

/**
 * @author Andrea Tessaro Porta
 **/
public class Util {


    public static final String JENIA_4_FACES = "jenia4faces";

    public static final String JENIA_RESOURCE_PREFIX = "/jenia4faces";

    public static final String CONVERSION_ERROR_MESSAGE_ID = "org.jenia.faces.CONVERSION_ERROR";

    public static final String COMPONENT_NOT_FOUND_ERROR_MESSAGE_ID = "org.jenia.faces.COMPONENT_NOT_FOUND_ERROR";

    public static final String NULL_PARAMETERS_ERROR_MESSAGE_ID = "org.jenia.faces.NULL_PARAMETERS_ERROR";

    public static final String FORM_NEEDED_MESSAGE_ID = "org.jenia.faces.FORM_NEEDED";

    public static final String FACET_NEEDED_MESSAGE_ID = "org.jenia.faces.FACET_NEEDED";

    public static boolean componentIsDisabledOnReadonly(UIComponent component) {
        Object disabledOrReadonly = null;
        boolean result = false;
        if (null !=
            (disabledOrReadonly = component.getAttributes().get("disabled"))) {
            if (disabledOrReadonly instanceof String) {
                result = ((String) disabledOrReadonly).equalsIgnoreCase("true");
            } else {
                result = disabledOrReadonly.equals(Boolean.TRUE);
            }
        }
        if ((result == false) &&
            null !=
            (disabledOrReadonly = component.getAttributes().get("readonly"))) {
            if (disabledOrReadonly instanceof String) {
                result = ((String) disabledOrReadonly).equalsIgnoreCase("true");
            } else {
                result = disabledOrReadonly.equals(Boolean.TRUE);
            }
        }

        return result;
    }

    public static Converter getConverterForClass(Class converterClass) {
        if (converterClass == null) {
            return null;
        }
        try {
            ApplicationFactory aFactory =
                (ApplicationFactory) FactoryFinder.getFactory(
                    FactoryFinder.APPLICATION_FACTORY);
            Application application = aFactory.getApplication();
            return (application.createConverter(converterClass));
        } catch (Exception e) {
            return (null);
        }
    }
    public static synchronized String getExceptionMessageString(String messageId,
                                                          Object params[]) {
        String result = null;

        FacesMessage message = MessageFactory.getMessage(messageId, params);
        if (null != message) {
            result = message.getSummary();
        }


        if (null == result) {
            result = "null MessageFactory";
        }
        return result;
    }

	public static final FacesContext getFacesContext() {
		return FacesContext.getCurrentInstance();
	}

	public static final Application getApplication() {
		return getFacesContext().getApplication();
	}

	/**
	 * @deprecated
	 */
	public static final HttpSession getSession() {
        return (HttpSession)getFacesContext().getExternalContext().getSession(false);
	}

	/**
	 * @deprecated
	 */
	public static final HttpServletRequest getRequest() {
        return (HttpServletRequest)getFacesContext().getExternalContext().getRequest();
	}
	
	/**
	 * @deprecated
	 */
	public static final HttpServletResponse getResponse() {
        return (HttpServletResponse)getFacesContext().getExternalContext().getResponse();
	}
	
	public static final boolean isMyFaces() {
		return getFacesContext().getClass().getName().startsWith("org.apache.myfaces.");
	}
	
	public static final boolean isSUNRI() {
		return getFacesContext().getClass().getName().startsWith("com.sun.faces.");
	}
	
	public static final String onClickPrefix(String formId) {
        StringBuffer buffer = new StringBuffer();
		if (isSUNRI()) {
			String hfn = formId + NamingContainer.SEPARATOR_CHAR + UIViewRoot.UNIQUE_ID_PREFIX + "cl";
	        buffer.append("document.forms[");
	        buffer.append("'");
	        buffer.append(formId);
	        buffer.append("'");
	        buffer.append("]['");
	        buffer.append(hfn);
	        buffer.append("'].value=''; ");
    	    FormRenderer.addNeededHiddenField(getFacesContext(),hfn);
		} else if (isMyFaces()) {
	        //call the clear_<formName> method
	        buffer.append(HtmlRendererUtils.getClearHiddenCommandFormParamsFunctionName(formId)).append("();");
	        //if configured add autoscroll support
	        if (MyfacesConfig.getCurrentInstance(getFacesContext().getExternalContext()).isAutoScroll())
	            JavascriptUtils.appendAutoScrollAssignment(buffer, formId);
		}
		return buffer.toString();
	}
	
	public static final void addHiddenFiled(UIForm form,String field, String value) throws IOException {
		ResponseWriter rw = getFacesContext().getResponseWriter();
		if (isMyFaces()) {
        	HtmlFormRendererBase.addHiddenCommandParameter(form,field);
		} else {
	        rw.startElement("input", null);
	        rw.writeAttribute("type","hidden",null);
	        rw.writeAttribute("name",field,null);
	        if (value!=null) 
	        	rw.writeAttribute("value",value,null);
	        rw.endElement("input");
		}
	}
	
	public static final String internalPath(String path) {
	    // i'm sure that there is a better way to do this...
		return getFacesContext().getExternalContext().getRequestContextPath() + JENIA_RESOURCE_PREFIX + path;
	}

    public static final Locale getLocale(){
        return getFacesContext().getViewRoot().getLocale();
    }
    
    public static ValueBinding getValueBinding(String valueRef) {
        return getApplication().createValueBinding(valueRef);
    }

    public static MethodBinding getMethodBinding(String valueRef,Class[] args) {
        return getApplication().createMethodBinding(valueRef,args);
    }

	public static final Object getFacesBean(String name) {
		return getApplication().createValueBinding("#{"+name+"}").getValue(getFacesContext());
	}
	
	public static final ValueBinding createValueBinding(String name) {
		return getApplication().createValueBinding(name);
	}
	
	public static final MethodBinding createMethodBinding(String name, Class[] args) {
		return getApplication().createMethodBinding(name,args);
	}
	
    public static Iterator getMessageIter(FacesContext context,
                                      String forComponent,
                                      UIComponent component) {
        Iterator messageIter = null;
        // Attempt to use the "for" attribute to locate 
        // messages.  Three possible scenarios here:
        // 1. valid "for" attribute - messages returned
        //    for valid component identified by "for" expression.
        // 2. zero length "for" expression - global errors
        //    not associated with any component returned
        // 3. no "for" expression - all messages returned.
        if (null != forComponent) {
            if (forComponent.length() == 0) {
                messageIter = context.getMessages(null);
            } else {
                UIComponent result = getForComponent(context, forComponent,
                                                     component);
                if (result == null) {
                    messageIter = Collections.EMPTY_LIST.iterator();
                } else {
                    messageIter =
                        context.getMessages(result.getClientId(context));
                }
            }
        } else {
            messageIter = context.getMessages();
        }
        return messageIter;
    }


    /**
     * Locates the component identified by <code>forComponent</code>
     *
     * @param forComponent - the component to search for
     * @param component    - the starting point in which to begin the search
     * @return the component with the the <code>id</code that matches
     *         <code>forComponent</code> otheriwse null if no match is found.
     */
    public static UIComponent getForComponent(FacesContext context,
                                          String forComponent, UIComponent component) {
        if (null == forComponent || forComponent.length() == 0) {
            return null;
        }

        UIComponent result = null;
        UIComponent currentParent = component;
        try {
            // Check the naming container of the current 
            // component for component identified by
            // 'forComponent'
            while (currentParent != null) {
                // If the current component is a NamingContainer,
                // see if it contains what we're looking for.
                result = currentParent.findComponent(forComponent);
                if (result != null)
                    break;
                // if not, start checking further up in the view
                currentParent = currentParent.getParent();
            }                   

            // no hit from above, scan for a NamingContainer
            // that contains the component we're looking for from the root.    
            if (result == null) {
                result =
                    findUIComponentBelow(context.getViewRoot(), forComponent);
            }
        } catch (Throwable t) {
            throw new RuntimeException("Error searching for component : " + forComponent);
        }
        // log a message if we were unable to find the specified
        // component (probably a misconfigured 'for' attribute
        if (result == null) {
            throw new RuntimeException("Wrong for component id specified");
        }
        return result;
    }


    /**
     * <p>Recursively searches for {@link NamingContainer}s from the
     * given start point looking for the component with the <code>id</code>
     * specified by <code>forComponent</code>.
     *
     * @param startPoint   - the starting point in which to begin the search
     * @param forComponent - the component to search for
     * @return the component with the the <code>id</code that matches
     *         <code>forComponent</code> otheriwse null if no match is found.
     */
    public static UIComponent findUIComponentBelow(UIComponent startPoint, String forComponent) {
        UIComponent retComp = null;
        Iterator children = startPoint.getFacetsAndChildren();
        while (children.hasNext()) {
            UIComponent comp = (UIComponent) children.next();

            if (comp instanceof NamingContainer) {
                retComp = comp.findComponent(forComponent);
            }

            if (retComp == null) {
            	retComp = findUIComponentBelow(comp, forComponent);
            }

            if (retComp != null)
                break;
        }
        return retComp;
    }
	
	public static Object setObjectAttribute(String key, Object object, Object attribute) {
		String k = Util.JENIA_4_FACES + key;
		String hid = (String)getSessionAttribute(k);
		if (hid==null){
			hid = k + object.hashCode();
			setSessionAttribute(k,hid);
			setSessionAttribute(hid,attribute);
			return null;
		} else {
			Object old = getSessionAttribute(hid);
			if (hid.equals(k + object.hashCode())) {
				setSessionAttribute(hid,attribute);
				return old;
			} else {
				removeSessionAttribute(hid);
				hid = k + object.hashCode();
				setSessionAttribute(k,hid);
				setSessionAttribute(hid,attribute);
				return old;
			}
		}
	}

	public static Object getObjectAttribute(String key, Object object) {
		String k = Util.JENIA_4_FACES + key;
		String hid = (String)getSessionAttribute(k);
		if (hid==null){
			return null;
		} else {
			if (hid.equals(k + object.hashCode())) {
				return getSessionAttribute(hid);
			} else {
				removeSessionAttribute(hid);
				return null;
			}
		}
	}

	public static Object removeObjectAttribute(String key, Object object) {
		String k = Util.JENIA_4_FACES + key;
		String hid = (String)getSessionAttribute(k);
		if (hid==null){
			return null;
		} else {
			Object old = getSessionAttribute(hid);
			removeSessionAttribute(hid);
			return old;
		}
	}

    public static final String getLocalizedMessage(String bundleName, String key){
        ResourceBundle bundle = ResourceBundle.getBundle( bundleName, getFacesContext().getViewRoot().getLocale());
        String ret =null;
        try {
             ret = bundle.getString(key);
        } catch (Exception e) {
            ret = "???"+key+"???";
        }
        return ret;
    }
    
	public static final void addErrorMessage(UIComponent component, String  bundleName, String summaryKey, String detailKey) {
	    getFacesContext().addMessage(component.getClientId(getFacesContext()),new FacesMessage(FacesMessage.SEVERITY_ERROR,getLocalizedMessage(bundleName,summaryKey),getLocalizedMessage(bundleName,detailKey)));
	}

	public static final void addErrorMessage(String  bundleName, String summaryKey, String detailKey) {
	    getFacesContext().addMessage(null,new FacesMessage(FacesMessage.SEVERITY_ERROR,getLocalizedMessage(bundleName,summaryKey),getLocalizedMessage(bundleName,detailKey)));
	}

	public static final void addFatalMessage(UIComponent component, String  bundleName, String summaryKey, String detailKey) {
	    getFacesContext().addMessage(component.getClientId(getFacesContext()),new FacesMessage(FacesMessage.SEVERITY_FATAL,getLocalizedMessage(bundleName,summaryKey),getLocalizedMessage(bundleName,detailKey)));
	}

	public static final void addFatalMessage(String  bundleName, String summaryKey, String detailKey) {
	    getFacesContext().addMessage(null,new FacesMessage(FacesMessage.SEVERITY_FATAL,getLocalizedMessage(bundleName,summaryKey),getLocalizedMessage(bundleName,detailKey)));
	}

	public static final void addWarningMessage(UIComponent component, String  bundleName, String summaryKey, String detailKey) {
	    getFacesContext().addMessage(component.getClientId(getFacesContext()),new FacesMessage(FacesMessage.SEVERITY_WARN,getLocalizedMessage(bundleName,summaryKey),getLocalizedMessage(bundleName,detailKey)));
	}

	public static final void addWarningMessage(String  bundleName, String summaryKey, String detailKey) {
	    getFacesContext().addMessage(null,new FacesMessage(FacesMessage.SEVERITY_WARN,getLocalizedMessage(bundleName,summaryKey),getLocalizedMessage(bundleName,detailKey)));
	}

	public static final void addInfoMessage(UIComponent component, String  bundleName, String summaryKey, String detailKey) {
	    getFacesContext().addMessage(component.getClientId(getFacesContext()),new FacesMessage(FacesMessage.SEVERITY_INFO,getLocalizedMessage(bundleName,summaryKey),getLocalizedMessage(bundleName,detailKey)));
	}

	public static final void addInfoMessage(String  bundleName, String summaryKey, String detailKey) {
	    getFacesContext().addMessage(null,new FacesMessage(FacesMessage.SEVERITY_INFO,getLocalizedMessage(bundleName,summaryKey),getLocalizedMessage(bundleName,detailKey)));
	}

	public static final Object getSessionAttribute(String key) {
		return getFacesContext().getExternalContext().getSessionMap().get(key);
	}

	public static final void setSessionAttribute(String key, Object value) {
		getFacesContext().getExternalContext().getSessionMap().put(key, value);
	}

	public static final void removeSessionAttribute(String key) {
		getFacesContext().getExternalContext().getSessionMap().remove(key);
	}
	
	public static final Object getRequestAttribute(String key) {
		return getFacesContext().getExternalContext().getRequestMap().get(key);
	}

	public static final void setRequestAttribute(String key, Object value) {
		getFacesContext().getExternalContext().getRequestMap().put(key, value);
	}
	
	public static final void removeRequestAttribute(String key) {
		getFacesContext().getExternalContext().getRequestMap().remove(key);
	}
	
	public static final Object getRequestParameter(String key) {
		return getFacesContext().getExternalContext().getRequestParameterMap().get(key);
	}
}
