/*
 * Copyright 2005-2006 The Portal Application Laboratory Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.sf.pal.blog.bean;

import java.io.NotSerializableException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Map;

import javax.faces.context.ExternalContext;
import javax.faces.context.FacesContext;
import javax.faces.model.DataModel;

import jp.sf.pal.blog.BlogConstants;
import jp.sf.pal.blog.model.BlogComment;
import jp.sf.pal.blog.model.BlogMessage;
import jp.sf.pal.blog.model.BlogTrackback;
import jp.sf.pal.blog.util.BlogDaoUtil;
import jp.sf.pal.blog.util.BlogMessaging;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.seasar.portlet.util.PortletResourceBundleUtil;
import org.seasar.portlet.util.RenderResponseUtil;

import com.marevol.utils.faces.application.FacesMessageUtil;
import com.marevol.utils.faces.model.S2HibernateDataModel;
import com.marevol.utils.faces.util.LocaleUtil;

public class BlogViewPageBean
{
    /**
     * Logger for this class
     */
    private static final Log log = LogFactory.getLog(BlogViewPageBean.class);

    private BlogViewSessionBean viewSession;

    private BlogMessage blogMessage;

    private String mode;

    private String commentName;

    private String commentEmail;

    private String commentUrl;

    private String commentMessage;

    public BlogViewPageBean()
    {
        mode = BlogConstants.BLOG_LIST_VIEW;

        Long id = null;
        try
        {
            id = (Long) BlogMessaging
                    .receiveForView(BlogConstants.BLOG_MESSAGE_ID);
            if (log.isDebugEnabled())
            {
                log.debug("BlogViewPageBean() -  : id=" + id);
            }

            if (id != null)
            {
                blogMessage = BlogDaoUtil.getBlogMessageDao()
                        .getBlogMessageById(id.longValue());
                mode = BlogConstants.BLOG_CONTENT_VIEW;
            }
        }
        catch (ClassCastException e)
        {
            log.warn(e);
        }

    }

    public String listView()
    {
        if (log.isDebugEnabled())
        {
            log.debug("listView() - start");
        }

        BlogMessaging.cancelForView(BlogConstants.BLOG_MESSAGE_ID);

        if (log.isDebugEnabled())
        {
            log.debug("listView() - end");
        }
        return null;
    }

    public String commentView()
    {
        if (log.isDebugEnabled())
        {
            log.debug("commentView() - start");
        }

        ExternalContext exContext = FacesContext.getCurrentInstance()
                .getExternalContext();
        Map map = exContext.getRequestMap();
        Map blogMap = (Map) map.get("blogMsg");
        BlogMessage blogMsg = (BlogMessage) blogMap.get("0");
        if (blogMsg != null)
        {
            try
            {
                BlogMessaging.publishForView(BlogConstants.BLOG_MESSAGE_ID,
                        blogMsg.getId());
            }
            catch (NotSerializableException e)
            {
                log.error(e);
            }
            catch (ClassCastException e)
            {
                log.error(e);
            }
        }
        else
        {
            log.error("Selected BlogMessage is null.");
        }

        if (log.isDebugEnabled())
        {
            log.debug("commentView() - end");
        }
        return null;
    }

    public String postComment()
    {
        Long id = null;
        try
        {
            //TODO replace with request parameter. because session might be expired..
            id = (Long) BlogMessaging
                    .receiveForView(BlogConstants.BLOG_MESSAGE_ID);

            if (id != null)
            {
                blogMessage = BlogDaoUtil.getBlogMessageDao()
                        .getBlogMessageById(id.longValue());
                if (getCommentName() != null && getCommentMessage() != null)
                {
                    BlogComment blogComment = new BlogComment();
                    blogComment.setName(getCommentName());
                    blogComment.setEmail(getCommentEmail());
                    blogComment.setUrl(getCommentUrl());
                    blogComment.setMessage(getCommentMessage());
                    blogComment
                            .setCreatedtime(Calendar.getInstance().getTime());
                    blogComment.setBlogMessage(blogMessage);

                    BlogDaoUtil.getBlogCommentDao().save(blogComment);

                    // I18n
                    FacesMessageUtil.addInfoMessage(PortletResourceBundleUtil
                            .getString(LocaleUtil.getLocale(),
                                    "blogview.PostedTheComment"));

                    return null;
                }
                else
                {
                    // I18n
                    FacesMessageUtil.addErrorMessage(PortletResourceBundleUtil
                            .getString(LocaleUtil.getLocale(),
                                    "blogview.NameOrCommentsAreNull"));
                    log.error(PortletResourceBundleUtil.getString(LocaleUtil
                            .getLocale(), "blogview.NameOrCommentsAreNull"));
                }
            }
            else
            {
                // I18n
                FacesMessageUtil.addErrorMessage(PortletResourceBundleUtil
                        .getString(LocaleUtil.getLocale(),
                                "blogview.CannotFindTheTargetBlogMessage"));
                log.error(PortletResourceBundleUtil
                        .getString(LocaleUtil.getLocale(),
                                "blogview.CannotFindTheTargetBlogMessage"));
            }
        }
        catch (ClassCastException e)
        {
            log.warn(e);
        }

        return null;
    }

    public String deleteComment()
    {
        ExternalContext exContext = FacesContext.getCurrentInstance()
                .getExternalContext();
        Map map = exContext.getRequestMap();
        BlogComment blogCmt = (BlogComment) map.get("blogCmt");
        if (blogCmt != null)
        {
            BlogDaoUtil.getBlogCommentDao().delete(blogCmt);

            // I18n
            FacesMessageUtil.addInfoMessage(PortletResourceBundleUtil
                    .getString(LocaleUtil.getLocale(),
                            "blogview.DeletedTheComment"));

        }
        else
        {
            // I18n
            FacesMessageUtil.addErrorMessage(PortletResourceBundleUtil
                    .getString(LocaleUtil.getLocale(),
                            "blogview.CannotFindTheTargetBlogComment"));
            log.error(PortletResourceBundleUtil.getString(LocaleUtil
                    .getLocale(), "blogview.CannotFindTheTargetBlogComment"));
        }
        return null;
    }

    public String deleteTrackback()
    {
        ExternalContext exContext = FacesContext.getCurrentInstance()
                .getExternalContext();
        Map map = exContext.getRequestMap();
        BlogTrackback blogTb = (BlogTrackback) map.get("blogTb");
        if (blogTb != null)
        {
            BlogDaoUtil.getBlogTrackbackDao().delete(blogTb);

            // I18n
            FacesMessageUtil.addInfoMessage(PortletResourceBundleUtil
                    .getString(LocaleUtil.getLocale(),
                            "blogview.DeletedTheTrackback"));

        }
        else
        {
            // I18n
            FacesMessageUtil.addErrorMessage(PortletResourceBundleUtil
                    .getString(LocaleUtil.getLocale(),
                            "blogview.CannotFindTheTargetBlogTrackback"));
            log.error(PortletResourceBundleUtil.getString(LocaleUtil
                    .getLocale(), "blogview.CannotFindTheTargetBlogTrackback"));
        }
        return null;
    }

    // Setter and Getter

    /**
     * @return Returns the blogMessages.
     */
    public DataModel getBlogMessages()
    {
        StringBuffer query = new StringBuffer(
                "select msg from BlogMessage msg ");
        String param = "where ";
        ArrayList paramList = new ArrayList();

        if (!getViewSession().getOwner().equals(""))
        {
            query.append(param);
            query.append("msg.owner = ? ");
            param = "and ";
            paramList.add(getViewSession().getOwner());
        }

        query.append("order by msg.createdtime desc");

        if (log.isDebugEnabled())
        {
            log.debug("query=" + query.toString());
        }

        Object[] params = new Object[paramList.size()];
        for (int i = 0; i < paramList.size(); i++)
        {
            params[i] = paramList.get(i);
        }

        return new S2HibernateDataModel(query.toString(), params);
    }

    /**
     * @return Returns the viewSession.
     */
    public BlogViewSessionBean getViewSession()
    {
        if (log.isDebugEnabled())
        {
            log.debug("getViewSessionBean() - start");
        }

        if (log.isDebugEnabled())
        {
            log.debug("getViewSessionBean() - end");
        }
        return viewSession;
    }

    /**
     * @param viewSession The viewSession to set.
     */
    public void setViewSession(BlogViewSessionBean viewPortlet)
    {
        if (log.isDebugEnabled())
        {
            log.debug("setViewSessionBean(BlogViewSessionBean) - start");
        }

        this.viewSession = viewPortlet;

        if (log.isDebugEnabled())
        {
            log.debug("setViewSessionBean(BlogViewSessionBean) - end");
        }
    }

    /**
     * @return Returns the blogMessage.
     */
    public BlogMessage getBlogMessage()
    {
        if (log.isDebugEnabled())
        {
            log.debug("getBlogMessage() - start");
        }

        if (log.isDebugEnabled())
        {
            log.debug("getBlogMessage() - end");
        }

        return blogMessage;
    }

    /**
     * @param blogMessage The blogMessage to set.
     */
    public void setBlogMessage(BlogMessage blogMessage)
    {
        if (log.isDebugEnabled())
        {
            log.debug("setBlogMessage(BlogMessage) - start");
        }

        this.blogMessage = blogMessage;

        if (log.isDebugEnabled())
        {
            log.debug("setBlogMessage(BlogMessage) - end");
        }
    }

    /**
     * @return Returns the mode.
     */
    public String getMode()
    {
        if (log.isDebugEnabled())
        {
            log.debug("getMode() - start");
        }

        if (log.isDebugEnabled())
        {
            log.debug("getMode() - end");
        }
        return mode;
    }

    /**
     * @param mode The mode to set.
     */
    public void setMode(String mode)
    {
        if (log.isDebugEnabled())
        {
            log.debug("setMode(String) - start");
        }

        this.mode = mode;

        if (log.isDebugEnabled())
        {
            log.debug("setMode(String) - end");
        }
    }

    public String actionUrl()
    {
        return RenderResponseUtil.createActionURL().toString();
    }

    /**
     * @return Returns the commentEmail.
     */
    public String getCommentEmail()
    {
        return commentEmail;
    }

    /**
     * @param commentEmail The commentEmail to set.
     */
    public void setCommentEmail(String commentEmail)
    {
        this.commentEmail = commentEmail;
    }

    /**
     * @return Returns the commentMessage.
     */
    public String getCommentMessage()
    {
        return commentMessage;
    }

    /**
     * @param commentMessage The commentMessage to set.
     */
    public void setCommentMessage(String commentMessage)
    {
        this.commentMessage = commentMessage;
    }

    /**
     * @return Returns the commentName.
     */
    public String getCommentName()
    {
        return commentName;
    }

    /**
     * @param commentName The commentName to set.
     */
    public void setCommentName(String commentName)
    {
        this.commentName = commentName;
    }

    /**
     * @return Returns the commentUrl.
     */
    public String getCommentUrl()
    {
        return commentUrl;
    }

    /**
     * @param commentUrl The commentUrl to set.
     */
    public void setCommentUrl(String commentUrl)
    {
        this.commentUrl = commentUrl;
    }

    public String getJavaScriptForRedirect()
    {
        if (log.isDebugEnabled())
        {
            log.debug("getJavaScriptForRedirect()");
        }

        ExternalContext exContext = FacesContext.getCurrentInstance()
                .getExternalContext();
        return "<script type=\"text/javascript\">"
                + "\n"
                + "<!--"
                + "\n"
                + "        function "
                + exContext.encodeNamespace("redirect")
                + "() {"
                + "\n"
                + "            var index = window.location.search.indexOf(\"?\");"
                + "\n"
                + "            if (index < window.location.search.length) {"
                + "\n"
                + "                var params = window.location.search.substr(index + 1).split(\"&\");"
                + "\n" + "                var i = 0;" + "\n"
                + "                for (i = 0; i < params.length; i++ ) {"
                + "\n"
                + "                    var pair = params[i].split(\"=\");"
                + "\n" + "                    if (pair[0] == \""
                + exContext.encodeNamespace(BlogConstants.BLOG_REDIRECT_ID)
                + "\" || pair[0] ==\"" + BlogConstants.BLOG_REDIRECT_ID
                + "\") {" + "\n" + "                        document.forms['"
                + exContext.encodeNamespace("redirectForm")
                + "'].elements[0].value = pair[1];" + "\n"
                + "                        document.forms['"
                + exContext.encodeNamespace("redirectForm") + "'].submit();"
                + "\n" + "                    }" + "\n" + "                }"
                + "\n" + "            }" + "\n" + "        }" + "\n"
                + "        " + exContext.encodeNamespace("redirect") + "();"
                + "\n" + "//-->" + "\n" + "</script>" + "\n";
    }
}
