/*
 * Copyright 2005-2006 The Portal Application Laboratory Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.sf.pal.blog.servlet;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.util.Calendar;
import java.util.Enumeration;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import jp.sf.pal.blog.BlogRuntimeException;
import jp.sf.pal.blog.dao.BlogMessageDao;
import jp.sf.pal.blog.dao.BlogTrackbackDao;
import jp.sf.pal.blog.model.BlogMessage;
import jp.sf.pal.blog.model.BlogTrackback;
import jp.sf.pal.blog.util.BlogDaoUtil;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.validator.UrlValidator;
import org.seasar.framework.container.S2Container;
import org.seasar.framework.container.factory.S2ContainerFactory;

public class TrackbackServlet extends HttpServlet
{
    /**
     * 
     */
    private static final long serialVersionUID = 299981574720975561L;

    /**
     * Logger for this class
     */
    private static final Log log = LogFactory.getLog(TrackbackServlet.class);

    /**
     * The title of the entry.
     * 
     * Optional.
     */
    public static final String TITLE = "title";

    /**
     * An excerpt of the entry.
     * 
     * Optional.
     */
    public static final String EXCERPT = "excerpt";

    /**
     * The permalink for the entry. Like any permalink, this should point as 
     * closely as possible to the actual entry on the HTML page, as it will 
     * be used when linking to the entry in question.
     * 
     * Required. If a client neglects to send a url, the server MUST respond 
     * with an error message.
     */
    public static final String URL = "url";

    /**
     * The name of the weblog to which the entry was posted.
     * 
     * Optional.
     */
    public static final String BLOG_NAME = "blog_name";

    /* (non-Javadoc)
     * @see javax.servlet.http.HttpServlet#doPost(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
     */
    protected void doPost(HttpServletRequest request, HttpServletResponse response) throws ServletException,
            IOException
    {
        String contentType = request.getContentType();
        String charset = "UTF-8";
        int start = contentType.indexOf("charset=");
        if (start >= 0)
        {
            int end = contentType.indexOf(" ", start + 1);
            if (end >= 0)
            {
                charset = contentType.substring(start + 8, end);
            }
            else
            {
                charset = contentType.substring(start + 8);
            }
        }

        if (log.isDebugEnabled())
        {
            log.debug("doPost(HttpServletRequest, HttpServletResponse) -  : contentType=" + contentType + ", charset="
                    + charset);
        }

        try
        {
            request.setCharacterEncoding(charset);
        }
        catch (UnsupportedEncodingException e)
        {
            request.setCharacterEncoding("UTF-8");
        }

        // getPathInfo
        String pathInfo = request.getPathInfo();
        if (pathInfo == null)
        {
            if (log.isDebugEnabled())
            {
                log.debug("doPost(HttpServletRequest, HttpServletResponse) -  : pathInfo=" + pathInfo);
            }

            printOutputXml(response, 1, "Invalid URL.");
            return;
        }

        start = pathInfo.indexOf("/");
        if (start >= 0)
        {
            int end = pathInfo.indexOf("/", start + 1);
            String blogIdString;
            if (end >= 0)
            {
                blogIdString = pathInfo.substring(start + 1, end);
            }
            else
            {
                blogIdString = pathInfo.substring(start + 1);
            }

            if (log.isDebugEnabled())
            {
                log.debug("doPost(HttpServletRequest, HttpServletResponse) -  : blogidString=" + blogIdString);
            }

            try
            {
                long blogId = Long.parseLong(blogIdString);
                String title = request.getParameter(TITLE);
                String blogName = request.getParameter(BLOG_NAME);
                String excerpt = request.getParameter(EXCERPT);
                String url = request.getParameter(URL);

                if (log.isDebugEnabled())
                {
                    for (Enumeration e = request.getParameterNames(); e.hasMoreElements();)
                    {
                        String name = (String) e.nextElement();
                        log.debug("doPost(HttpServletRequest, HttpServletResponse) -  : name=" + name + ", value="
                                + request.getParameter(name));
                    }
                }
                if (excerpt != null)
                {
                    // changed to 255 bytes or less
                    while (excerpt.getBytes("UTF-8").length > 255)
                    {
                        excerpt = excerpt.substring(0, excerpt.length() - 1);
                    }
                }

                // check url
                if (url == null || url.equals(""))
                {
                    printOutputXml(response, 1, "The given url is null or empty.");
                    log.error("The given url is null or empty.");
                    return;
                }

                // check url using commons validator
                String[] schemes = { "http", "https" };
                UrlValidator urlValidator = new UrlValidator(schemes);
                if (!urlValidator.isValid(url))
                {
                    printOutputXml(response, 1, "The given url is invalid.");
                    log.error("The given url is invalid.");
                    return;
                }

                if (title == null)
                {
                    title = "";
                }
                if (excerpt == null)
                {
                    excerpt = "";
                }
                if (blogName == null)
                {
                    blogName = "";
                }

                if (log.isDebugEnabled())
                {
                    log.debug("doPost(HttpServletRequest, HttpServletResponse) -  : blogId=" + blogId + ", title="
                            + title + ", blogName=" + blogName + ", excrpt=" + excerpt + ", url=" + url);
                }

                try
                {
                    //TODO move config to web.xml
                    S2Container container = S2ContainerFactory.create("jp/sf/pal/blog/Blog.dicon");
                    container.init();

                    BlogMessageDao bmDao = BlogDaoUtil.getBlogMessageDao(container);
                    BlogMessage blogMsg = bmDao.getBlogMessageById(blogId);
                    if (blogMsg == null)
                    {
                        printOutputXml(response, 1, "The target Blog message does not exist. BLOG ID=" + blogId);
                        log.error("The target Blog message does not exist. BLOG ID=" + blogId);
                        return;
                    }

                    BlogTrackbackDao btDao = BlogDaoUtil.getBlogTrackbackDao(container);

                    BlogTrackback bt = new BlogTrackback();
                    bt.setTitle(title);
                    bt.setBlogname(blogName);
                    bt.setExcerpt(excerpt);
                    bt.setUrl(url);
                    bt.setCreatedtime(Calendar.getInstance().getTime());
                    bt.setBlogMessage(blogMsg);

                    btDao.save(bt);

                    container.destroy();
                }
                catch (BlogRuntimeException e)
                {
                    log.error(e);
                    printOutputXml(response, 1, "Runtime Exception occurs");
                    return;
                }
                catch (IOException e)
                {
                    log.error(e);
                    printOutputXml(response, 1, "Runtime Exception occurs");
                    return;
                }

                printOutputXml(response, 0, null);

                // Completed
                return;
            }
            catch (NumberFormatException e)
            {
                log.error(e);
                printOutputXml(response, 1, "Invalid Number Format.");
                return;
            }

        }
        else
        {
            if (log.isDebugEnabled())
            {
                log.debug("doPost(HttpServletRequest, HttpServletResponse) -  : pathInfo=" + pathInfo);
            }

            printOutputXml(response, 1, "Invalid URL.");
            return;
        }
    }

    /* (non-Javadoc)
     * @see javax.servlet.http.HttpServlet#doDelete(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
     */
    protected void doDelete(HttpServletRequest request, HttpServletResponse response) throws ServletException,
            IOException
    {
        printNonSupported(request, response);
    }

    /* (non-Javadoc)
     * @see javax.servlet.http.HttpServlet#doGet(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
     */
    protected void doGet(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException
    {
        printNonSupported(request, response);
    }

    /* (non-Javadoc)
     * @see javax.servlet.http.HttpServlet#doHead(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
     */
    protected void doHead(HttpServletRequest request, HttpServletResponse response) throws ServletException,
            IOException
    {
        printNonSupported(request, response);
    }

    /* (non-Javadoc)
     * @see javax.servlet.http.HttpServlet#doOptions(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
     */
    protected void doOptions(HttpServletRequest request, HttpServletResponse response) throws ServletException,
            IOException
    {
        printNonSupported(request, response);
    }

    /* (non-Javadoc)
     * @see javax.servlet.http.HttpServlet#doPut(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
     */
    protected void doPut(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException
    {
        printNonSupported(request, response);
    }

    /* (non-Javadoc)
     * @see javax.servlet.http.HttpServlet#doTrace(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
     */
    protected void doTrace(HttpServletRequest request, HttpServletResponse response) throws ServletException,
            IOException
    {
        printNonSupported(request, response);
    }

    protected void printNonSupported(HttpServletRequest request, HttpServletResponse response) throws IOException
    {
        printOutputXml(response, 1, "The method is not supported.");

    }

    protected void printOutputXml(HttpServletResponse response, int code, String message) throws IOException
    {
        response.setContentType("application/xhtml+xml; charset=UTF-8");

        PrintWriter out = response.getWriter();
        out.println("<?xml version=\"1.0\" encoding=\"utf-8\"?>");
        out.println("<response>");
        out.println("<error>" + code + "</error>");
        if (message != null)
        {
            out.println("<message>" + message + "</message>");
        }
        out.println("</response>");
    }
}
