/*
 * Copyright 2005-2006 Portal Application Laboratory project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.pal.facesdevfilter;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.Writer;

import javax.portlet.Portlet;
import javax.portlet.PortletConfig;
import javax.portlet.PortletException;
import javax.portlet.PortletMode;
import javax.portlet.WindowState;
import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import jp.sf.pal.facesdevfilter.http.HttpServletPortletRequestWrapper;
import jp.sf.pal.facesdevfilter.http.HttpServletPortletResponseWrapper;
import jp.sf.pal.facesdevfilter.portlet.PortletConfigImpl;
import jp.sf.pal.facesdevfilter.portlet.descriptors.PortletNode;
import jp.sf.pal.facesdevfilter.services.PortletConfigService;

public class FacesDevFilter implements Filter {
    /**
     * Logger for this class
     */
    private static final Log log = LogFactory.getLog(FacesDevFilter.class);

    private static final String PORTLET_XML = "/WEB-INF/portlet.xml";

    public static final String ENCODING = "encoding";

    public static final String ADD_HTML_TAGS = "addHtmlTags";

    public static String DEFAULT_ENCODING = "UTF-8";

    private ServletContext servletContext;

    private PortletConfigService portletConfigService;

    private String encoding;

    private boolean addHtmlTags;

    public void init(FilterConfig filterConfig) throws ServletException {
        servletContext = filterConfig.getServletContext();

        String portletXmlFilename = filterConfig.getServletContext()
                .getRealPath(PORTLET_XML);

        if (log.isDebugEnabled()) {
            log.debug("init(FilterConfig) - portletXmlFilename="
                    + portletXmlFilename);
        }
        try {
            portletConfigService = new PortletConfigService(
                    new FileInputStream(portletXmlFilename));
        } catch (FileNotFoundException e) {
            log.error("File Not Found. The file name is " + portletXmlFilename,
                    e);
            throw new ServletException(e);
        }

        // encoding
        encoding = filterConfig.getInitParameter(ENCODING);
        if (encoding == null) {
            encoding = DEFAULT_ENCODING;
        }

        // header&footer: default is true
        String v = filterConfig.getInitParameter(ADD_HTML_TAGS);
        if (v != null) {
            if (v.equalsIgnoreCase("true")) {
                addHtmlTags = true;
            } else {
                addHtmlTags = false;
            }
        } else {
            addHtmlTags = true;
        }

    }

    public void destroy() {
        // TODO Auto-generated method stub

    }

    public void doFilter(ServletRequest request, ServletResponse response,
            FilterChain chain) throws IOException, ServletException {
        if (request instanceof HttpServletRequest
                && response instanceof HttpServletResponse) {
            HttpServletPortletRequestWrapper wrappedRequest = new HttpServletPortletRequestWrapper(
                    (HttpServletRequest) request,
                    (HttpServletResponse) response);
            HttpServletPortletResponseWrapper wrappedResponse = new HttpServletPortletResponseWrapper(
                    (HttpServletRequest) request,
                    (HttpServletResponse) response);
            storePortalEnvironment(wrappedRequest, wrappedResponse);
            // set PortletConfig to attribute
            wrappedRequest.setAttribute(FacesDevConstants.PORTLET_CONFIG,
                    wrappedRequest.getSession().getAttribute(
                            FacesDevConstants.PORTLET_CONFIG));
            chain.doFilter(wrappedRequest, wrappedResponse);
            // TODO set title
            postRender(wrappedResponse);
        } else {
            chain.doFilter(request, response);
        }
    }

    private void postRender(HttpServletPortletResponseWrapper wrappedResponse)
            throws IOException {
        HttpServletResponse response = (HttpServletResponse) wrappedResponse
                .getResponse();
        Writer writer = response.getWriter();

        if (addHtmlTags) {
            StringBuffer buf = new StringBuffer();
            buf.append("<html>");
            buf.append("<head>");
            buf
                    .append("<meta http-equiv=\"Content-type\" content=\"text/html; charset="
                            + encoding + "\" />");
            buf.append("<title>Faces Development Filter</title>");
            buf.append("</head>");
            buf.append("<body>");
            writer.write(buf.toString());
        }
        writer.write(wrappedResponse.getString());
        if (addHtmlTags) {
            StringBuffer buf = new StringBuffer();
            buf.append("</body>");
            buf.append("</html>");
            writer.write(buf.toString());
        }
        writer.flush();
    }

    public void addHtmlFooter(ServletResponse response) {
        StringBuffer buf = new StringBuffer();
        buf.append("</body>");
        buf.append("</html>");
        try {
            response.getWriter().write(buf.toString());
            response.getWriter().flush();
        } catch (IOException e) {
            log.error("Could not add footer.", e);
        }
    }

    protected void storePortalEnvironment(HttpServletRequest request,
            HttpServletResponse response) {
        HttpSession session = request.getSession(true);
        // set portletconfig
        String portletName = request
                .getParameter(FacesDevConstants.PORTLET_NAME_REQUEST_KEY);
        if (portletName == null) {
            if (session.getAttribute(FacesDevConstants.PORTLET_CONFIG) == null) {
                PortletNode portletNode = portletConfigService
                        .getDefaultPortlet();
                if (portletNode == null) {
                    throw new IllegalArgumentException(
                            "Could not find a default portlet in the portlet descriptor. ");
                }
                PortletConfig portletConfig = new PortletConfigImpl(
                        servletContext, portletNode);
                session.setAttribute(FacesDevConstants.PORTLET_CONFIG,
                        portletConfig);
                // init Portlet
                String portletClassName = portletNode.getPortletClass();
                // if (portletClassName != null
                // && portletClassName
                // .equals(FacesDevConstants.APACHE_PORTALS_FILTERPORTLET)) {
                // portletClassName = (String) portletNode.getInitParams()
                // .get("portlet-class");
                // }
                Portlet portlet = createPortlet(portletClassName, portletConfig);
                session.setAttribute(FacesDevConstants.PORTLET, portlet);
            }
        } else {
            PortletNode portletNode = portletConfigService
                    .getPortletByName(portletName);
            if (portletNode == null) {
                throw new IllegalArgumentException("Could not find "
                        + portletName + " in the portlet descriptor. ");
            }
            PortletNode prevPortletNode = (PortletNode) session
                    .getAttribute(FacesDevConstants.PORTLET_CONFIG);
            if (!portletNode.equals(prevPortletNode)) {
                PortletConfig portletConfig = new PortletConfigImpl(
                        servletContext, portletNode);
                session.setAttribute(FacesDevConstants.PORTLET_CONFIG,
                        portletConfig);
                // destroy previous Portlet
                Portlet prevPortlet = (Portlet) session
                        .getAttribute(FacesDevConstants.PORTLET);
                if (prevPortlet != null) {
                    prevPortlet.destroy();
                }
                // init Portlet
                String portletClassName = portletNode.getPortletClass();
                // if (portletClassName != null
                // && portletClassName
                // .equals(FacesDevConstants.APACHE_PORTALS_FILTERPORTLET)) {
                // portletClassName = (String) portletNode.getInitParams()
                // .get("portlet-class");
                // }
                Portlet portlet = createPortlet(portletClassName, portletConfig);
                session.setAttribute(FacesDevConstants.PORTLET, portlet);
            }
        }

        // set portlet mode
        String portletModeName = request
                .getParameter(FacesDevConstants.PORTLET_MODE_REQUEST_KEY);
        if (portletModeName == null) {
            if (session.getAttribute(FacesDevConstants.PORTLET_MODE) == null) {
                session
                        .setAttribute(FacesDevConstants.PORTLET_MODE,
                                new PortletMode(
                                        FacesDevConstants.DEFAULT_PORTLET_MODE));
            }
        } else {
            session.setAttribute(FacesDevConstants.PORTLET_MODE,
                    new PortletMode(portletModeName));
        }

        // set window state
        String windowStateName = request
                .getParameter(FacesDevConstants.WINDOW_STATE_REQUEST_KEY);
        if (windowStateName == null) {
            if (session.getAttribute(FacesDevConstants.WINDOW_STATE) == null) {
                session
                        .setAttribute(FacesDevConstants.WINDOW_STATE,
                                new WindowState(
                                        FacesDevConstants.DEFAULT_WINDOW_STATE));
            }
        } else {
            session.setAttribute(FacesDevConstants.WINDOW_STATE,
                    new WindowState(windowStateName));
        }

        // set namespace
        String namespaceName = request
                .getParameter(FacesDevConstants.NAMESPACE_REQUEST_KEY);
        if (namespaceName == null) {
            if (session.getAttribute(FacesDevConstants.NAMESPACE) == null) {
                session.setAttribute(FacesDevConstants.NAMESPACE,
                        FacesDevConstants.DEFAULT_NAMESPACE);
            }
        } else {
            session.setAttribute(FacesDevConstants.NAMESPACE, namespaceName);
        }
    }

    private Portlet createPortlet(String portletClassName,
            PortletConfig portletConfig) {
        try {
            Class portletClass = Class.forName(portletClassName);
            Object portletObj = portletClass.newInstance();
            if (portletObj instanceof Portlet) {
                Portlet portlet = (Portlet) portletObj;
                portlet.init(portletConfig);
                return portlet;
            } else {
                throw new IllegalArgumentException(portletClassName
                        + " is not Portlet instance.");
            }
        } catch (ClassNotFoundException e) {
            throw new IllegalArgumentException(portletClassName
                    + " is not Portlet instance.", e);
        } catch (InstantiationException e) {
            throw new IllegalArgumentException(portletClassName
                    + " is not Portlet instance.", e);
        } catch (IllegalAccessException e) {
            throw new IllegalArgumentException(portletClassName
                    + " is not Portlet instance.", e);
        } catch (PortletException e) {
            throw new IllegalStateException(
                    "An exception occurrs during calling init() in "
                            + portletClassName, e);
        }

    }
}
