package jp.sf.pal.jstock.reader.impl;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.URL;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.oro.text.perl.Perl5Util;

import jp.sf.pal.jstock.dto.StockDataDto;
import jp.sf.pal.jstock.exception.JStockRuntimeException;
import jp.sf.pal.jstock.reader.StockDataReader;
import jp.sf.pal.jstock.util.MessageUtil;
import jp.sf.pal.jstock.util.PriceUtil;

public class StockDataReaderImpl implements StockDataReader {
    
    private static final Log log = LogFactory.getLog(StockDataReaderImpl.class);
    
    /** 情報を取得する際の URL の固定部分 */
    //private static final String URL_BASE = "http://money.www.infoseek.co.jp/MnStock/%s/"; for JDK 1.5
    public static final String URL_BASE = "http://money.www.infoseek.co.jp/MnStock/{0}/";
    
    /** 銘柄 */
    public static final int STOCK_NAME_INDEX = 1;
    /** コード */
    public static final int STOCK_CODE_INDEX = 3;
    /** 市場 */
    public static final int STOCK_MARKET_INDEX = 4;
    /** 業種 */
    public static final int STOCK_CATEGORY_INDEX = 4;
    /** 取引値*/
    public static final int PRICE_INDEX = 5;
    /** 時刻 */
    public static final int TIME_INDEX = 6;
    /** 前日比 */
    public static final int CHANGE_INDEX = 7;
    /** 前日比 (パーセント) */
    public static final int CHANGE_PERCENT_INDEX = 9;
    /** 高値 */
    public static final int HIGH_PRICE_INDEX = 11;
    /** 安値 */
    public static final int LOW_PRICE_INDEX = 13;
    /** 始値 */
    public static final int START_PRICE_INDEX = 15;
    /** 出来高 */
    public static final int VOLUME_INDEX = 17;
    /** 52週高値 */
    public static final int HIGH_PRICE_52_INDEX = 19;
    /** 52週安値 */
    public static final int LOW_PRICE_52_INDEX = 21;
    /** 時価総額 */
    public static final int AGGREGATE_PRICE_INDEX = 23;
    /** ストップ高／安 */
    public static final int STOP_INDEX = 0;
    /** 前日終値 */
    public static final int YESTERDAY_PRICE_INDEX = 0;
    /** 気配値 */
    public static final int INDICATIVE_PRICE_INDEX = 0;
    
    /** 株価情報表示部分の正規表現 */
    public static final String INFO_REGEXP =
        "m`"
        + "<span id=\"dname\">(.+)</span>.*"    // 銘柄
        + "<small>.*(&lt;(.+)&gt;)(.*)</small>.+"   // コード，市場
        + "<!-- /銘柄ヘッダ -->.*"
        + "<table width=\"100%\".*summary=\"株価詳細\" style=\"margin-bottom:10px;\">.*"
        + "<tr align=\"right\">.*"
        + "<th scope=\"col\">現在値</th>.*"
        + "<td align=\"center\"><b>(.+)</b>.*"  // 現在値
        + "<nobr>\\((.+)\\)</nobr></td>.*"       // 時刻       
        + "<th scope=\"col\" style=\"border-bottom:0;\">前日比</th>"
        + "<td align=\"center\" style=\"border-bottom:0;\"><font color.+>(.+)</font>.*&nbsp;\\|&nbsp;</span>" // 前日比
        + "(.*<font color.*>)(.+%)?</font>.*"   // 前日比 %
        + ">高値</th>.*(<td style=\".+\">(.+)</td>).*"
        + ">安値</th>.*(<td>(.+)</td>).*"
        + ">始値</th>.*(<td>(.+)</td>).*"
        + ">出来高</th>.*(<td>(.+)</td>).*"
        + ">52週高値</th>.*(<td>(.+)</td>).*"
        + ">52週安値</th>.*(<td>(.+)</td>).*"
        + ">時価総額</th>.*(<td>(.+)</td>).*"
        + ">単元株数</th>.*(<td>(.+)</td>).*"
        + "<!-- ====== /株価 ====== -->"
        + "`is";
    
    /** 取得する株の証券コードのリスト */
    private List codes;
    
    public StockDataReaderImpl(List codes) {
        this.codes = codes;
    }

    public void setCodes(List codes) {
        this.codes = codes;
    }

    public StockDataDto getStockData(String code) 
        throws JStockRuntimeException {
        
        Object[] args = {code};
        
        try {

            //String urlstr = String.format(URL_BASE, code);    for JDK 1.5
            String urlstr = MessageFormat.format(URL_BASE, args);
            URL url = new URL(urlstr);
            
            // Connect Server
            HttpURLConnection urlconn = (HttpURLConnection)url.openConnection();
            urlconn.setRequestMethod("GET");
            urlconn.setInstanceFollowRedirects(false);
            urlconn.connect();
            
            // Charset の取り出し
            String contentType = urlconn.getContentType();
            int index = contentType.toLowerCase().indexOf("charset");
            String charSet = null;
            if (index >= 0) {
                String[] charSets = contentType.substring(index).split("=");
                charSet = charSets[1];
            }
            
            BufferedReader reader =
                new BufferedReader(new InputStreamReader(urlconn.getInputStream(), charSet));
            
            StringBuffer buf = new StringBuffer();
            String line;
            while((line = reader.readLine()) != null) {
                buf.append(line);
            }
            
            // 正規表現
            Perl5Util regexpUtil = new Perl5Util();
            if (regexpUtil.match(INFO_REGEXP, buf.toString()) == true) {
                StockDataDto stockDataDto = new StockDataDto();
                stockDataDto.setName(regexpUtil.group(STOCK_NAME_INDEX));
                stockDataDto.setTime(regexpUtil.group(TIME_INDEX));
                try {
                    stockDataDto.setPrice(PriceUtil.convertToBigDecimal(regexpUtil.group(PRICE_INDEX)));
                } catch(NumberFormatException e) {
                    stockDataDto.setPrice(null);
 
                }
                try {
                    stockDataDto.setChange(PriceUtil.convertToBigDecimal(regexpUtil.group(CHANGE_INDEX)));
                } catch(NumberFormatException e) {
                    stockDataDto.setChange(null);
                }
                return stockDataDto;
            }
        
        } catch (Exception e) {
            throw new JStockRuntimeException("jp.sf.pal.jstock.ReadStockInfoError", args);
        }
        
        return null;
    }

    public List getStockDataList() {
        
        List dataList = new ArrayList();
        
        for (int i = 0; i < this.codes.size(); i++) {
            String[] args = {(String)this.codes.get(i)};
            try {
                dataList.add(this.getStockData((String)this.codes.get(i)));
            } catch (JStockRuntimeException e) {
                log.error(MessageUtil.getErrorMessage(e.getMessageId(), e.getArgs()));
            }
        }
        
        return dataList ;
    }

}
