/*
 * Copyright 2004-2006 The Portal Application Laboratory Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.pal.wiki.service;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.faces.context.FacesContext;

import jp.sf.pal.wiki.WikiConstants;
import jp.sf.pal.wiki.callback.WikiLinkGeneratorCallback;
import jp.sf.pal.wiki.entity.WikiPage;
import jp.sf.pal.wiki.util.WikiUtil;
import jp.sf.stconv.pipeline.PipelineException;
import jp.sf.stconv.pipeline.StreamConverterPipeline;
import jp.sf.stconv.storage.StreamStorage;
import jp.sf.stconv.storage.StreamStorageException;
import jp.sf.stconv.storage.impl.FileStreamStorageImpl;
import jp.sf.stconv.wiki.pipeline.callback.LinkGeneratorCallback;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.seasar.framework.container.S2Container;

import jp.sf.pal.common.faces.application.FacesMessageUtil;

/**
 * @author shinsuke
 * 
 */
public class WikiContentService {
    /**
     * Logger for this class
     */
    private static final Log log = LogFactory.getLog(WikiContentService.class);

    private String currentPageName;

    private S2Container container;

    private String encoding;

    public WikiContentService(S2Container container) {
        setContainer(container);
        setEncoding(WikiUtil.getEncoding());
    }

    public String getCurrentPageName() {
        if (currentPageName != null) {
            return currentPageName;
        }

        String currentPageName = getCurrentPageNameFromSession();
        if (log.isDebugEnabled()) {
            log.debug("getCurrentPageName() - pageName=" + currentPageName);
        }
        if (currentPageName == null) {
            // TODO send fallback message to view
            currentPageName = WikiConstants.FRONT_PAGE;
        }

        return currentPageName;
    }

    public void setCurrentPageName(String pageName) {
        // if (isPageName(pageName)) {
        // throw new IllegalArgumentException("Invalid page name: " + pageName);
        // }
        setCurrentPageNameToSession(pageName);
    }

    public String getCurrentContent() {
        String pageName = getCurrentPageName();
        if (log.isDebugEnabled()) {
            log.debug("getCurrentContent() - pageName=" + pageName);
        }
        return getContent(pageName);
    }

    public String getContent(String pageName) {
        File pageFile = getPageFile(pageName);
        if (pageFile == null || !pageFile.exists()) {
            // TODO i18n
            return "The Page Not Found.";
        }

        StringBuffer buffer = new StringBuffer();
        BufferedReader r = null;
        try {
            r = new BufferedReader(new InputStreamReader(new FileInputStream(
                    pageFile), getEncoding()));
            String l = null;
            while ((l = r.readLine()) != null) {
                buffer.append(l);
                buffer.append("\n");
            }
        } catch (UnsupportedEncodingException e) {
            log.error("Unsupported Encoding. ", e);
            // TODO i18n
            FacesMessageUtil.addErrorMessage("Unsupported Encoding: "
                    + getEncoding());
        } catch (IOException e) {
            log.error("I/O Exception. ", e);
            // TODO i18n
            FacesMessageUtil.addErrorMessage("I/O Exception. ", e);
        } finally {
            if (r != null) {
                try {
                    r.close();
                } catch (IOException e) {
                }
            }
        }

        return buffer.toString();
    }

    public String getCurrentConvertedContent() {
        String pageName = getCurrentPageName();
        if (log.isDebugEnabled()) {
            log.debug("getCurrentConvertedContent() - pageName=" + pageName);
        }
        return getConvertedContent(pageName);
    }

    public String getConvertedContent(String pageName) {
        File pageFile = getPageFile(pageName);
        if (pageFile == null || !pageFile.exists()) {
            // TODO i18n
            return "The Page Not Found.";
        }

        StreamConverterPipeline pipeline = (StreamConverterPipeline) container
                .getComponent(WikiConstants.WIKI_CONVERTER_PIPELINE);

        StringBuffer buffer = new StringBuffer();
        try {
            StreamStorage storage = new FileStreamStorageImpl(
                    new FileInputStream(pageFile), getEncoding());
            storage.addCallback(LinkGeneratorCallback.LINK_GENERATOR_CALLBACK,
                    new WikiLinkGeneratorCallback());
            pipeline.invoke(storage);

            BufferedReader r = null;
            try {
                r = new BufferedReader(new InputStreamReader(storage
                        .getResultInputStream(), storage.getEncoding()));
                String l = null;
                while ((l = r.readLine()) != null) {
                    buffer.append(l);
                    buffer.append("\n");
                }
            } catch (UnsupportedEncodingException e) {
                log.error("Unsupported Encoding. ", e);
                // TODO i18n
                FacesMessageUtil.addErrorMessage("Unsupported Encoding: "
                        + getEncoding());
            } catch (StreamStorageException e) {
                log.error("Could not read the stream. ", e);
                // TODO i18n
                FacesMessageUtil.addErrorMessage("Could not read the stream. ",
                        e);
            } catch (IOException e) {
                log.error("I/O Exception. ", e);
                // TODO i18n
                FacesMessageUtil.addErrorMessage("I/O Exception. ", e);
            } finally {
                if (r != null) {
                    try {
                        r.close();
                    } catch (IOException e) {
                    }
                }
            }
            storage.destroy();
        } catch (FileNotFoundException e) {
            log.error("Could not find: " + pageFile, e);
            // TODO i18n
            FacesMessageUtil.addErrorMessage("Could not find: " + pageFile, e);
        } catch (StreamStorageException e) {
            log.error("StreamStorage Exception.", e);
            // TODO i18n
            FacesMessageUtil.addErrorMessage("StreamStorage Exception.", e);
        } catch (PipelineException e) {
            log.error("Pipeline Exception.", e);
            // TODO i18n
            FacesMessageUtil.addErrorMessage("Pipeline Exception.", e);
        }

        return buffer.toString();
    }

    public boolean setContent(String pageName, String content) {
        File pageFile = getPageFile(pageName);
        boolean write = true;
        Writer writer = null;
        try {
            writer = new FileWriter(pageFile);
            writer.write(content);
            writer.flush();
        } catch (IOException e) {
            write = false;
            log.error("Could not writer the content on " + pageName, e);
        } finally {
            if (writer != null) {
                try {
                    writer.close();
                } catch (IOException e) {
                    write = false;
                    log.error("Could not close " + pageName, e);
                }
            }
        }
        return write;
    }

    public boolean isPageName(String pageName) {
        File pageFile = getPageFile(pageName);
        if (pageFile == null || !pageFile.exists()) {
            return false;
        }
        return true;
    }

    protected File getPageFile(String pageName) {
        File dataDir = getDataDir();
        return new File(dataDir, pageName + WikiConstants.DATA_SUFFIX);
    }

    public List getPages() {
        ArrayList list = new ArrayList();
        File dataDir = getDataDir();

        File[] files = dataDir.listFiles();
        for (int i = 0; i < files.length; i++) {
            String filename = files[i].getName();
            if (log.isDebugEnabled()) {
                log.debug("getPages() - filename=" + filename);
            }

            if (filename != null
                    && filename.lastIndexOf(WikiConstants.DATA_SUFFIX) >= 0) {
                WikiPage page = new WikiPage();
                page.setFile(files[i]);
                page.setPageName(filename.substring(0, filename.length()
                        - WikiConstants.DATA_SUFFIX.length()));
                page.setUpdated(new Date(files[i].lastModified()));
                list.add(page);
            }

        }
        //TODO sort?

        return list;
    }

    protected File getDataDir() {
        String dataDirName = WikiUtil.getWikiDataDir();
        File dataDir = new File(dataDirName);
        if (!dataDir.exists()) {
            if (!dataDir.mkdirs()) {
                throw new IllegalAccessError("Could not create a directory:"
                        + dataDirName);
            }
            if (log.isDebugEnabled()) {
                log.debug("Created directory: " + dataDirName);
            }
        }
        return dataDir;
    }

    protected void setCurrentPageNameToSession(String pageName) {
        getFacesContext().getExternalContext().getSessionMap().put(
                WikiConstants.CURRENT_PAGE_NAME, pageName);
    }

    protected String getCurrentPageNameFromSession() {
        return (String) getFacesContext().getExternalContext().getSessionMap()
                .get(WikiConstants.CURRENT_PAGE_NAME);
    }

    protected FacesContext getFacesContext() {
        FacesContext context = FacesContext.getCurrentInstance();
        if (context == null) {
            throw new IllegalStateException("FacesContext instance is null.");
        }
        return context;
    }

    /**
     * @return Returns the container.
     */
    public S2Container getContainer() {
        return container;
    }

    /**
     * @param container
     *            The container to set.
     */
    public void setContainer(S2Container container) {
        this.container = container;
    }

    /**
     * @return Returns the encoding.
     */
    public String getEncoding() {
        return encoding;
    }

    /**
     * @param encoding
     *            The encoding to set.
     */
    public void setEncoding(String encoding) {
        this.encoding = encoding;
    }

}
