# testconfiguration.rb: testcases for BlazerProxy::Configuration

require 'test/unit'
load 'blazer-ja-proxy'

require 'tempfile'

class Testconfiguration < Test::Unit::TestCase
	include BlazerProxy

	def test_parse_ok
		c = Configuration.new([])
		assert_equal(nil, c.path)
		assert_equal(8080, c['bind port'])	# default
		f = StringIO.new(<<'_END')
# conf file
 # strange comment 
proxy user: moge:	# user name: moge, password: nil
proxy user: fuga:hoge
BIND ADDRESS: 1111	# down cased
_END
		c.parse(f)
		c.check
		assert_equal({'moge' => nil, 'fuga' => 'hoge'}, c['proxy user'])	# set
		assert_equal(8080, c['bind port'])	# default
		assert_equal('1111', c['bind address'])	# set
	end

	def test_parse_ng
		c = Configuration.new([])
		f = StringIO.new("not-allowed-keyword:")
		c.parse(f)
		assert_raise(ConfigurationError, ":1: keyword `not-allowed-keyword' not known"){c.check}
		f = StringIO.new("proxt username: ok keyword\nnot-allowed-keyword")
		c.parse(f)
		assert_raise(ConfigurationError, ":2: keyword `not-allowed-keyword' not known"){c.check}
	end

	def test_conf
		t = Tempfile.new(File.basename(__FILE__))
		t.write(<<'_END')
# test configuration file
proxy user: moge: 
bind port: 1234
_END
		t.close
		u = Tempfile.new(File.basename(__FILE__))
		u.close(true)	# should no longer be exist
		c = Configuration.new([u.path, t.path])
		assert_equal(t.path, c.path)
		{
			'proxy user' => {'moge' => nil},
			'bind port' => '1234',
		}.each_pair do |k,v|
			assert_equal(v, c[k])
		end
	end

	def test_special
		c = Configuration.new([])
		f = StringIO.new("proxy users: moge:hoge")
		c.parse(f)
		assert_raise(ConfigurationError, ":1: keyword `proxy users' not known"){c.check}
	end

	def test_verbose
		t = Tempfile.new(File.basename(__FILE__))
		t.write(<<'_END')
# test configuration file
proxy user: moge: 
bind port: 1234
_END
		t.close
		# loud
		message_target = <<"_END"
added user:`moge' to configuration `proxy user'
set configuration `bind port' as `1234'
read configuration from #{t.path}
_END
		stderr_orig = $stderr
		$stderr = StringIO.new
		c = Configuration.new([t.path], 10)
		message = $stderr.string
		$stderr = stderr_orig
		assert_equal(message_target, message)
		# quiet
		stderr_orig = $stderr
		$stderr = StringIO.new
		c = Configuration.new([t.path], 0)
		message = $stderr.string
		assert_equal('', message)
		$stderr = stderr_orig
	end

	def test_entry_typo
		c = Configuration.new([])
		assert_raise(ConfigurationError, "keyowrd `no kw' not known"){c['no kw']}
	end

	def test_server_log_depth
		c = Configuration.new([])
		f = StringIO.new('server log depth: info')
		c.parse(f)
		c.check
		assert_equal(::WEBrick::Log::INFO, c['server log depth'])
	end

	def test_browser_charset
		c = Configuration.new([])
		f = StringIO.new('browser charset: sjis')
		c.parse(f)
		c.check
		assert_equal(NKF::SJIS, c['browser charset'])
	end

	def test_ng_browser_charset
		c = Configuration.new([])
		f = StringIO.new('browser charset: info')
		c.parse(f)
		assert_equal(NKF::EUC, c['browser charset'])
		assert_raise(ConfigurationError, ":1: value `info' not allowed for keyword `browser charset'"){c.check}
	end

end
