/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.issuelist;

import jp.valtech.bts.data.Issue;
import jp.valtech.bts.util.BTSUtility;

import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;

/**
 * {@link IssueListView 課題票一覧テーブル}のソータクラスです。
 * 
 * @author		<A href="mailto:s_imai@valtech.jp">S.Imai</A>
 * @version	Ver.0.8
 */
public class IssueListViewSorter extends ViewerSorter {

	/** ソート反転フラグ */
	private boolean reversed = false;
	
	/** カラム番号 */
	private int columnNumber;
	
	/**
	 * コンストラクタ
	 * @param		columnNumber		カラム番号
	 */
	public IssueListViewSorter(int columnNumber){
		this.columnNumber = columnNumber;
	}

	/*
	 *  (非 Javadoc)
	 * @see org.eclipse.jface.viewers.ViewerSorter#compare(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
	 */
	public int compare(Viewer viewer, Object e1, Object e2) {
		// 比較結果
		int result = 0;
		// 課題票オブジェクトの比較を行う
		result = compareColumnValue((Issue)e1, (Issue)e2);
		// ソート反転処理
		if (reversed){
			result = -result;
		}
		return result;
	}

	/**
	 * 各カラムごとに課題票データの比較を行います。
	 * @param		e1 		最初の課題票オブジェクト
	 * @param		e2 		次の課題票オブジェクト
	 * @return		比較結果
	 */
	int compareColumnValue(Issue e1, Issue e2) {
		switch (columnNumber) {
			// 1カラム目：完了期限アイコン
			case 0 :
				return compareTo(Integer.valueOf(BTSUtility.judgeLimit(e1)), Integer.valueOf(BTSUtility.judgeLimit(e2)));

			// 2カラム目：課題票ID
			case 1 :
				return e1.getDisplayIssueID().compareTo(e2.getDisplayIssueID());

			// 3カラム目：タイトル
			case 2 :
				return compareTo(e1.getTitle(), e2.getTitle());
				
			// 4カラム目：カテゴリ
			case 3 :
				return compareTo(e1.getCategory(), e2.getCategory());
				
			// 5カラム目：優先度
			case 4 :
				return compareTo(e1.getPriority(), e2.getPriority());
				
			// 6カラム目：ステータス
			case 5 :
				return compareTo(e1.getStatus(), e2.getStatus());
				
			// 7カラム目：担当者
			case 6 :
				return compareTo(e1.getAssigned(), e2.getAssigned());
				
			// 8カラム目：起票日
			case 7 :
				return compareTo(e1.getCreateDate(), e2.getCreateDate());
				
			default:
				return 0;
		}
	}
	
	
	/**
	 * 指定された２つのオブジェクトの順序を比較します。
	 * 比較対照のオブジェクトはかならず{@link Comparable}を実装している必要があります。
	 * 
	 * @param		obj1			比較対象のオブジェクト
	 * @param		obj2			比較対象のオブジェクト
	 * @return		負の数：obj1の方が小さい。
	 * 				ゼロ　：obj1,obj2両方等しい。
	 * 				正の数：obj2の方が小さい。
	 */
	private int compareTo(Object obj1, Object obj2) {
		if(obj1 == null && obj2 == null) {
			return 0;
		} else if(obj1 != null && obj2 == null) {
			return 1;
		} else if(obj1 == null && obj2 != null) {
			return -1;
		} else {
			return ((Comparable)obj1).compareTo((Comparable)obj2);
		}
	}

	/**
	 * ソート反転フラグを返します。
	 * @return		ソート反転フラグ
	 */
	public boolean isReversed() {
		return reversed;
	}

	/**
	 * ソート反転フラグを設定します。
	 * @param		reversed		ソート反転フラグ
	 */
	public void setReversed(boolean reversed) {
		this.reversed = reversed;
	}

	/**
	 * カラム番号を返します。
	 * @return		カラム番号
	 */
	public int getColumnNumber() {
		return columnNumber;
	}
	
}
