/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.msglist;

import java.io.File;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.List;

import jp.valtech.bts.command.client.SendMessage;
import jp.valtech.bts.data.CurrentProject;
import jp.valtech.bts.data.MessagePacket;
import jp.valtech.bts.data.MessageType;
import jp.valtech.bts.data.NetworkConfig;
import jp.valtech.bts.facade.SendMessageFacade;
import jp.valtech.bts.network.ClientInfo;
import jp.valtech.bts.network.ClientList;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.ui.navigator.ProjectConfig;
import jp.valtech.bts.util.BTSUtility;
import jp.valtech.bts.util.IOUtility;
import jp.valtech.bts.util.Logging;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.Text;

/**
 * メッセージ送信用のダイアログです。
 * {@link jp.valtech.bts.ui.msglist.MessageListView メッセージ一覧ビュー}から呼ばれるダイアログです。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class MessageSenderDialog extends Dialog implements Logging {
	
	/** リモート端末テーブルビューア */
	private CheckboxTableViewer viewer;

    /** 「ファイルPATH入力」用テキストボックス */
    private Text pathTxt;
    
    /** 「メッセージ入力」用テキストボックス */
    private Text msgText;
    
    /** 送信メッセージ（受信メッセージの引用符） */
    private String message = "";
	
	/** あて先情報：送信先情報 */
	private ClientInfo toClient;

	/** あて先情報：送信先を全員にするかどうか。true--全員、false--全員でない*/
	private boolean sendAll = false;
	
    /**
	 * 親クラスにシェルを設定します。
	 * 
	 * @param			parentShell
	 */
	public MessageSenderDialog(Shell parentShell) {
		super(parentShell);
	}
	
	/**
	 * 返信メッセージ文字列を設定
	 * 
	 * @param		message			返信メッセージ文字列		
	 */
	public void setReplyMessage(String message) {
		this.message= message;
	}
	
	/**
	 * あて先情報設定：{@link ClientInfo 接続ユーザ情報}を格納します。
	 * 
	 * @param		clientInfo		接続ユーザ情報
	 */
	public void setClientInfo(ClientInfo clientInfo) {
		this.toClient = clientInfo;
	}

	
	/**
	 * あて先情報設定：あて先を全員にするかどうか
	 * 
	 * @param		sendAll		
	 */
	public void setSendAll(boolean sendAll) {
		this.sendAll = sendAll;
	}
	
	/**
     * 当ダイアログの設定します。タイトルを設定します。
     * 
     * @param			newShell			該ダイアログのシェル
     */
    protected void configureShell(Shell newShell) {
        super.configureShell(newShell);
        newShell.setText(Messages.getString("MessageSenderDialog.0"));  //$NON-NLS-1$
    }



    /**
     * 送信メッセージ作成用のダイアログを生成します。
     * 
     * @param			parent					親コンポジット
     * @return			当ダイアログのコントロールインスタンス
     * @see				Dialog#createDialogArea(org.eclipse.swt.widgets.Composite)
     */
    protected Control createDialogArea(Composite parent) {
    	
		// 左右分割したパネルを使う
		SashForm sashForm = new SashForm(parent, SWT.HORIZONTAL);
		sashForm.setLayoutData(new GridData(GridData.FILL_BOTH));
		
        try {
        	
    		// 左ペインにリモート端末選択テーブルを配置
    		createTableArea(sashForm);
    		
    		// 右ペインに「メッセージ入力」と「添付ファイル」を配置
    		createRightPain(sashForm);
    		
			// SashFormの左右の比率を設定
			sashForm.setWeights(new int[]{40, 60});
        	
            // セパレータ生成
            createSeparatorLine(parent);
            
        }catch (Exception e) {
			String msg = Messages.getString("MessageSenderDialog.20"); //$NON-NLS-1$
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);
		}
        return sashForm;
    }
    
    
	/**
	 * リモート端末選択テーブルのUIを生成します。
	 * 
	 * @param parent 親コンポジット
	 */
	private void createTableArea(Composite parent) {

		// このパーツのレイアウト設定。グリッドレイアウトを使う
		Composite thisArea = new Composite(parent, SWT.NONE);
		thisArea.setLayout(new GridLayout());

		// テーブルタイトルの設定
		Label label = new Label(thisArea, SWT.NONE);
		label.setText(Messages.getString("MessageSenderDialog.1")); //$NON-NLS-1$

		// テーブルの表示設定
		Table table = new Table(thisArea, SWT.CHECK | SWT.H_SCROLL | SWT.V_SCROLL | SWT.FULL_SELECTION | SWT.BORDER);
		table.setLinesVisible(false);
		table.setHeaderVisible(true);

		// テーブルレイアウトの設定
		TableLayout tableLayout = new TableLayout();
		table.setLayout(tableLayout);

		// 1カラム目（アドレス）の設定
		tableLayout.addColumnData(new ColumnWeightData(100, true));
		TableColumn tcAddress = new TableColumn(table, SWT.NONE, 0);
		tcAddress.setText(Messages.getString("MessageSenderDialog.2")); //$NON-NLS-1$

		// 2カラム目（ユーザ）の設定
		tableLayout.addColumnData(new ColumnWeightData(100, true));
		TableColumn tcUser = new TableColumn(table, SWT.NONE, 1);
		tcUser.setText(Messages.getString("MessageSenderDialog.3")); //$NON-NLS-1$

		// テーブルのビューアオブジェクト設定
		viewer = new CheckboxTableViewer(table);

		// テーブルの横幅・高さの調整
		GridData gridData = new GridData(GridData.FILL_HORIZONTAL);
		gridData.heightHint = 210;
		gridData.widthHint = 230;
		viewer.getControl().setLayoutData(gridData);

		// テーブルにプロバイダークラスを設定
		viewer.setContentProvider(getTableContentProvider());
		viewer.setLabelProvider(getTableLabelProvider());
		
		// ユーザ情報取得
		ClientInfo[] clientInfo = ClientList.getClientInfoList();
		
		// 自分自身を除外したユーザ情報
		List clientInfoExceptMe = new ArrayList();
		
		// 取得したユーザ分処理する
		for(int i=0; i<clientInfo.length; i++) {
			
			// ユーザ情報取得
			ClientInfo info = clientInfo[i];
			
			// ユーザ情報が自分自身では無い場合
			if(!BTSUtility.isLocalHost(info)) {
				
				// 自分自身を除外したユーザ情報にセット
				clientInfoExceptMe.add(info);
				
			}
		}
		// テーブルにデータをセット
		viewer.setInput(clientInfoExceptMe);
		
        // 送信先のチェックON／OFF設定
		if(this.toClient ==null) {
			viewer.setAllChecked(this.sendAll);
		} else {
			viewer.setCheckedElements(new ClientInfo[]{toClient});
		}
		
		// ビューアに選択変更リスナを追加
		viewer.addSelectionChangedListener(new ISelectionChangedListener(){

			/*
			 *  (非 Javadoc)
			 * @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
			 */
			public void selectionChanged(SelectionChangedEvent event) {
				
				// 「送信」ボタンの活性・非活性切換え
				getButton(IDialogConstants.OK_ID).setEnabled(viewer.getCheckedElements().length != 0);
				
			}
			
		});
		
	}


	/**
	 * 「リモート端末テーブル」のコンテントプロバイダを返します。
	 * 
	 * @return 「リモート端末テーブル」のコンテントプロバイダ
	 */
	private IStructuredContentProvider getTableContentProvider() {
		return new IStructuredContentProvider() {

			public Object[] getElements(Object inputElement) {
				return ((List)inputElement).toArray();
			}

			public void dispose() {

			}

			public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {

			}

		};
	}

	/**
	 * {@link TableLabelProvider 「リモート端末一覧テーブル」のラベルプロバイダ}を返します。
	 * 
	 * @return 「リモート端末一覧テーブル」のラベルプロバイダ
	 */
	private ILabelProvider getTableLabelProvider() {
		return new TableLabelProvider();
	}

	/**
	 * 「リモート端末一覧テーブル」のラベルプロバイダです。
	 */
	private class TableLabelProvider extends LabelProvider implements ITableLabelProvider {

		private Image image = BtsPlugin.getInstance().getImage("msg_list_view.gif");
		
		/*
		 * (非 Javadoc)
		 * 
		 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnImage(java.lang.Object,
		 *      int)
		 */
		public Image getColumnImage(Object element, int columnIndex) {
			switch (columnIndex) {
			case 0:
				return image;
			}
			return null;
		}

		/*
		 * (非 Javadoc)
		 * 
		 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnText(java.lang.Object,
		 *      int)
		 */
		public String getColumnText(Object element, int columnIndex) {
			switch (columnIndex) {
			case 0:
				return ((ClientInfo) element).getClientAddress();
			case 1:
				return ((ClientInfo) element).getClientName();
			}
			return "";
		}
	}
	
	
	/**
	 * 右ペインを生成します。
	 * 右ペインには「メッセージ入力」と「添付」を配置します。
	 * 
	 * @param			parent			親コンポジット
	 * @throws 		Exception
	 */
	private void createRightPain(Composite parent) throws Exception {
		
		// 右ペインのコンポジット生成
		Composite thisArea = new Composite(parent, SWT.NONE);
		
    	// レイアウト設定。3列表示のグリッドレイアウトを使う
        GridLayout gridLayout = new GridLayout();
        gridLayout.numColumns = 3;
        thisArea.setLayout(gridLayout);

    	// 「メッセージ入力」のUIを生成
    	createMessageArea(thisArea);

        // 「添付ファイル」設定のUIを生成
    	createAttachmentArea(thisArea);
	}

    
    /**
     * このダイアログにメッセージ入力の部品を生成します。
     * 
     * @param			thisArea						右ペインのコンポジット
     * @throws			Exception
     */
    private void createMessageArea(Composite thisArea) throws Exception {

        // ラベル生成
        Label label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("MessageSenderDialog.4")); //$NON-NLS-1$
		GridData gridData = new GridData(GridData.FILL); 
		gridData.horizontalSpan = 3;
		label.setLayoutData(gridData);
        
        
        // メッセージ入力用テキストの生成
        msgText = new Text(thisArea, SWT.MULTI|SWT.H_SCROLL|SWT.V_SCROLL|SWT.BORDER | SWT.WRAP);
        // レイアウト設定
		gridData = new GridData(GridData.FILL_HORIZONTAL); 
        gridData.widthHint = 300;
        gridData.heightHint = 200;
        gridData.horizontalSpan = 3;
        msgText.setLayoutData(gridData);

        if(message != null) {
        	msgText.setText(message);
        }
    }
    

    /**
     * このダイアログに添付ファイル指定の部品を生成します。
     * 
     * @param			thisArea					右ペインのコンポジット
     * @throws			Exception
     */
    private void createAttachmentArea(final Composite thisArea) throws Exception {

        // ラベル生成
        Label label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("MessageSenderDialog.5")); //$NON-NLS-1$

        // ファイルPATH入力ようテキストボックスの生成
        pathTxt = new Text(thisArea,SWT.BORDER);
        // レイアウトを設定
		GridData gridData = new GridData(GridData.FILL_HORIZONTAL); 
        gridData.widthHint = 240;
        pathTxt.setLayoutData(gridData);

        // ファイル選択ダイアログを開くボタンを生成
        Button exportBtn = new Button(thisArea, SWT.PUSH);
        exportBtn.setText(Messages.getString("MessageSenderDialog.6")); //$NON-NLS-1$
        
        // ボタン押下時の動作を設定。
        exportBtn.addSelectionListener(new SelectionAdapter(){
        	public void widgetSelected(SelectionEvent e){
        		// ファイル選択ダイアログを開く
        		FileDialog dialog = new FileDialog(thisArea.getShell(), SWT.OPEN);
        		// ファイル選択ダイアログで選択したファイルの振るパスをテキストボックスに表示する。
        		String path = dialog.open();
        		if(path != null) {
            		pathTxt.setText(path);
        		}
        	}
        });
    }
    
    
    /**
     * ボタンバーにボタンを生成します。
     * 左側のボタンのラベルを「送信」にするためにオーバーライドしてます。
     * この「送信」ボタンは送信先が選択されているときのみ活性します。
     * 
     * @param			parent						親コンポジット
     */
    protected void createButtonsForButtonBar(Composite parent) {
        createButton(parent, IDialogConstants.OK_ID, Messages.getString("MessageSenderDialog.7"), true); //$NON-NLS-1$
		getButton(IDialogConstants.OK_ID).setEnabled(viewer.getCheckedElements().length != 0);
        createButton(parent, IDialogConstants.CANCEL_ID,
                IDialogConstants.CANCEL_LABEL, false);
    }

    
    /**
     * セパレータラインを生成します。
     * 親コンポジットに対してセパレータラインを生成します。
     * 
     * @param			parent						親コンポジット
     */
    protected void createSeparatorLine(Composite parent) {
        // 親コンポーネントにラベルを生成
        Label separator = new Label(parent, SWT.HORIZONTAL | SWT.SEPARATOR);
        // レイアウトを設定。グリッドレイアウトを使う。
        GridData gd = new GridData(GridData.FILL_HORIZONTAL);
        gd.horizontalSpan = 1;
        separator.setLayoutData(gd);
    }

    /*
     *  (非 Javadoc)
     * @see org.eclipse.jface.dialogs.Dialog#okPressed()
     */
    protected void okPressed() {


    	// 送信メッセージ生成
    	MessagePacket packet = createMessagepacket();
    	if(packet == null) {
    		return ;
    	}
    	
    	// 送信メッセージをローカルDBに保存
    	SendMessageFacade facade = new SendMessageFacade();
    	facade.addMessage(packet);

    	// 送信
    	boolean res = sendMessage(packet);
    	if(!res) {
    		return;
    	}
    	
    	// メッセージ保存
    	super.okPressed();
    }
    
    
    /**
     * 送信オブジェクトを生成して返します。
     * 
     * @return		送信オブジェクト
     */
    private MessagePacket createMessagepacket() {
    	
    	MessagePacket packet = new MessagePacket();
    	
    	// メッセージ本文設定
    	String sendMessage = this.msgText.getText();
    	packet.setMessage(sendMessage);

    	// 添付ファイル設定
    	String attachmentPath = pathTxt.getText().trim();
    	if( !attachmentPath.equals("") ) {
        		
    		// 存在するファイルかどうかのチェック
    		File file = new File(attachmentPath);
    		if(!file.exists()) {
    			MessageDialog.openInformation
				( BtsPlugin.getInstance().getShell()
				, Messages.getString("MessageSenderDialog.8") //$NON-NLS-1$
				, Messages.getString("MessageSenderDialog.9")); //$NON-NLS-1$
    			return null;
    		}
    			
    		try {
    			byte[] entity = IOUtility.read(attachmentPath);
        		packet.setAttachFile(entity);
        		packet.setAttachFileName(file.getName());
    		} catch (Exception e) {
    			BtsPlugin.getInstance().errorlog(e);
    			MessageDialog.openInformation
				( BtsPlugin.getInstance().getShell()
				, Messages.getString("MessageSenderDialog.10") //$NON-NLS-1$
				, Messages.getString("MessageSenderDialog.11")); //$NON-NLS-1$
    			return null;
			}
    	} else {
    		packet.setAttachFile(null);
    		packet.setAttachFileName(null);
    	}
    	
    	// 送信者設定
    	ProjectConfig config = CurrentProject.getInsance().getProjectConfig();
    	packet.setFromUser( config.getUserName() );
    	
    	// 送信日時設定
    	packet.setSent(System.currentTimeMillis());
    	
    	// メッセージ種別設定
    	packet.setMessageType(MessageType.USER_VALUE);
    	
    	return packet;
    }
    
    
	/**
	 * 指定の課題票を送信します。
	 * 
	 * @param		newIssue			対象となる課題票
	 */
	protected boolean sendMessage(final MessagePacket messagePacket) {
		try {
			// チェックONにしたユーザ情報取得
	    	final Object[] clientInfos = viewer.getCheckedElements();

	    	if(clientInfos==null || clientInfos.length==0) {
				MessageDialog.openInformation
					( BtsPlugin.getInstance().getShell()
					, Messages.getString("MessageSenderDialog.12") //$NON-NLS-1$
					, Messages.getString("MessageSenderDialog.13")); //$NON-NLS-1$
	    		return false;
	    	}


	    	// プログレスダイアログを生成
			IRunnableWithProgress progress = new IRunnableWithProgress() {

				/*
				 *  (非 Javadoc)
				 * @see org.eclipse.jface.operation.IRunnableWithProgress#run(org.eclipse.core.runtime.IProgressMonitor)
				 */
				public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
					
					// プログレスバーを設定
					monitor.beginTask(Messages.getString("MessageSenderDialog.14"), clientInfos.length); //$NON-NLS-1$

					// チェックONにしたユーザに送信する
					for (int i = 0; i < clientInfos.length; i++) {
						
						ClientInfo info = (ClientInfo)clientInfos[i];

						// キャンセルボタンを押した場合
						if (monitor.isCanceled()) {
							throw new InterruptedException();
						}

						// 課題票送信コマンド生成
						SendMessage send= new SendMessage( new NetworkConfig()
														 , info.getClientAddress()
														 , info.getPort().intValue());
						send.setMessagePacket(messagePacket);
						
						// ダイアログにメッセージ表示
						monitor.subTask(Messages.getString("MessageSenderDialog.15") + info.getClientName() + Messages.getString("MessageSenderDialog.16")); //$NON-NLS-1$ //$NON-NLS-2$

						// 課題票送信
						try {
							send.execute();
						} catch (Exception e) {
							logger.fatal(Messages.getString("MessageSenderDialog.17") + info, e); //$NON-NLS-1$
							throw new InterruptedException(e.getMessage());
						}
						
						// プログレスバーのメモリを上げる
						monitor.worked(1);
					}
					monitor.done();
				}
			};

			// プログレスダイアログの表示
			ProgressMonitorDialog dialog = new ProgressMonitorDialog(BtsPlugin.getInstance().getShell());
			dialog.run(true, true, progress);

			return true;
		} catch (InvocationTargetException e) {
			BtsPlugin.getInstance().errorlog(e);
		} catch (InterruptedException e) {
			// キャンセル後の処理
			MessageDialog.openInformation(BtsPlugin.getInstance().getShell(), Messages.getString("MessageSenderDialog.18"), Messages.getString("MessageSenderDialog.19")); //$NON-NLS-1$ //$NON-NLS-2$
		}
		return false;
	}

}
