/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.navigator;

import jp.valtech.bts.ui.BtsPlugin;

import org.eclipse.jface.dialogs.TitleAreaDialog;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Text;

/**
 * プロジェクト設定で入力された値のエラーチェックをします。
 * 
 * @author		<A href="mailto:s_imai@valtech.jp">S.Imai</A>
 * @version	Ver.0.8
 */
public class ProjectConfigDialogValidate {
	
	/** タイトルエリアダイアログ */
	private TitleAreaDialog dialog;
	
	/**
	 * タイトルエリアダイアログを取得します。
	 * @param		dialog		タイトルエリアダイアログ
	 */
	public ProjectConfigDialogValidate(TitleAreaDialog dialog) {
		this.dialog = dialog;
	}

	/**
	 * プロジェクト名のエラーチェックを行います。
	 * 
	 * @param 		text		プロジェクト名のテキストボックス
	 * @return		エラーの有無
	 */
	public boolean validateProjectName(Text text){
		// プロジェクト名が空ならエラー
		if(text.getText().trim().length() == 0){
			dialog.setErrorMessage(Messages.getString("ProjectConfigDialogValidate.0")); //$NON-NLS-1$
			return false;
		} else {
			// エラーなし
			dialog.setErrorMessage(null);
			return true;
		}
	}
	
	/**
	 * ユーザ名のエラーチェックを行います。
	 * 
	 * @param 		text		ユーザ名のテキストボックス
	 * @return		エラーの有無
	 */
	public boolean validateUserName(Text text){
		// ユーザ名が空ならエラー
		if(text.getText().trim().length() == 0){
			dialog.setErrorMessage(Messages.getString("ProjectConfigDialogValidate.1")); //$NON-NLS-1$
			return false;
		} else {
			// エラーなし
			dialog.setErrorMessage(null);
			return true;
		}
	}
	
	/**
	 * SSIDのエラーチェックを行います。
	 * 
	 * @param 		text		SSIDのテキストボックス
	 * @return		エラーの有無
	 */
	public boolean validateSsid(Text text) {

		// ダイアログセッティング取得
		IProjectDialogSettings setting = BtsPlugin.getInstance().getProjectDialogSettings();

		// SSIDが空ならエラー
		if (text.getText().trim().length() == 0) {
			dialog.setErrorMessage(Messages.getString("ProjectConfigDialogValidate.2")); //$NON-NLS-1$
			return false;
		} else if (setting != null) {
			// XMLから全てのセクション取得
			IProjectDialogSettings[] sections = setting.getSections();
			for (int i = 0; i < sections.length; i++) {
				// SSIDが既に存在するならエラー
				if ((text.getText().trim()).equals(sections[i]
						.get(ProjectConfigManager.KEY_SSID))) {
					dialog.setErrorMessage(Messages.getString("ProjectConfigDialogValidate.3")); //$NON-NLS-1$
					return false;
				}
			}

		}
		// エラーなし
		dialog.setErrorMessage(null);
		return true;

	}
	
	/**
	 * 優先度のエラーチェックを行います。
	 * 
	 * @param 		text		優先度のテキストボックス
	 * @return		エラーの有無
	 */
	public boolean validatePriority(Text text){
		// 優先度が空ならエラー
		if(text.getText().trim().length() == 0){
			dialog.setErrorMessage(Messages.getString("ProjectConfigDialogValidate.4")); //$NON-NLS-1$
			return false;
		} else {
			// エラーなし
			dialog.setErrorMessage(null);
			return true;
		}
	}
	
	/**
	 * 投票数のエラーチェックを行います。
	 * 
	 * @param		check		投票数のチェックボックス
	 * @param 		text		投票数のテキストボックス
	 * @return		エラーの有無
	 */
	public boolean validateVote(Button check, Text text){
		if (check.getSelection()) {
			if (text.getText().trim().length() == 0) {
				// 投票数が空ならエラー
				dialog.setErrorMessage(Messages.getString("ProjectConfigDialogValidate.5")); //$NON-NLS-1$
				return false;
			} else {
				try {
					if(Integer.parseInt(text.getText().trim()) < 0){
						dialog.setErrorMessage(Messages.getString("ProjectConfigDialogValidate.6")); //$NON-NLS-1$
						return false;
					}
				} catch (NumberFormatException e) {
					dialog.setErrorMessage(Messages.getString("ProjectConfigDialogValidate.7")); //$NON-NLS-1$
					return false;
				}
				// エラーなし
				dialog.setErrorMessage(null);
				return true;
			}
		} else {
			// エラーなし
			dialog.setErrorMessage(null);
			return true;
		}
	}
	
	/**
	 * 完了期限のエラーチェックを行います。
	 * 
	 * @param		check		完了期限のチェックボックス
	 * @param 		text		完了期限のテキストボックス
	 * @return		エラーの有無
	 */
	public boolean validateUntilDeadLine(Button check, Text text){
		if (check.getSelection()) {
			if (text.getText().trim().length() == 0) {
				// 完了期限が空ならエラー
				dialog.setErrorMessage(Messages.getString("ProjectConfigDialogValidate.8")); //$NON-NLS-1$
				return false;
			} else {
				try {
					if(Integer.parseInt(text.getText().trim()) <= 0){
						dialog.setErrorMessage(Messages.getString("ProjectConfigDialogValidate.9")); //$NON-NLS-1$
						return false;
					}
				} catch (NumberFormatException e) {
					dialog.setErrorMessage(Messages.getString("ProjectConfigDialogValidate.10")); //$NON-NLS-1$
					return false;
				}
				// エラーなし
				dialog.setErrorMessage(null);
				return true;
			}
		} else {
			// エラーなし
			dialog.setErrorMessage(null);
			return true;
		}
	}
	
	/**
	 * SMTPサーバ名のエラーチェックを行います。
	 * 
	 * @param		text		SMTPサーバ名のテキストボックス
	 * @return		エラーの有無
	 */
	public boolean validateSmtp(Button check, Text text) {
		
		// メール通知のチェックがされていない場合
		if(!check.getSelection()) {
			// エラーなし
			dialog.setErrorMessage(null);
			return true;
		}
		
		// SMTPサーバ名が空ならエラー
		if(text.getText().trim().length() == 0){
			dialog.setErrorMessage(Messages.getString("ProjectConfigDialogValidate.11")); //$NON-NLS-1$
			return false;
		} else {
			// エラーなし
			dialog.setErrorMessage(null);
			return true;
		}
		
	}
	
	
	/**
	 * 差出人メールアドレスのエラーチェックを行います。
	 * 
	 * @param 		text		差出人メールアドレスのテキストボックス
	 * @return		エラーの有無
	 */
	public boolean validateSender(Button check, Text text) {
		
		// メール通知のチェックがされていない場合
		if(!check.getSelection()) {
			// エラーなし
			dialog.setErrorMessage(null);
			return true;
		}
		
		// SMTPサーバ名が空ならエラー
		if(text.getText().trim().length() == 0){
			dialog.setErrorMessage(Messages.getString("ProjectConfigDialogValidate.12")); //$NON-NLS-1$
			return false;
		} else {
			// エラーなし
			dialog.setErrorMessage(null);
			return true;
		}

	}
	
	/**
	 * 送信先メールアドレスのエラーチェックを行います。
	 * 
	 * @param 		text		送信先メールアドレスのテキストボックス
	 * @return		エラーの有無
	 */
	public boolean validateRecipient(Button check, Text text) {
		
		// メール通知のチェックがされていない場合
		if(!check.getSelection()) {
			// エラーなし
			dialog.setErrorMessage(null);
			return true;
		}
		
		// SMTPサーバ名が空ならエラー
		if(text.getText().trim().length() == 0){
			dialog.setErrorMessage(Messages.getString("ProjectConfigDialogValidate.13")); //$NON-NLS-1$
			return false;
		} else {
			// エラーなし
			dialog.setErrorMessage(null);
			return true;
		}
		
	}
	
	/**
	 * アドレステキストのエラーチェックを行います。
	 * 
	 * @param 		text		アドレステキストボックス
	 * @return		エラーの有無
	 */
	public boolean validateAddress(Button fixedSyncChk, Text text){
		
		// 「固定端末への同期を使用可にする」チェックがされていない場合
		if(!fixedSyncChk.getSelection()) {
			// エラーなし
			dialog.setErrorMessage(null);
			return true;
		}
		
		try {
			
			if(Integer.parseInt(text.getText()) >= 0) {
				// エラーなし
				dialog.setErrorMessage(null);
				return true;
			}
			else {
				// エラー
				dialog.setErrorMessage(Messages.getString("ProjectConfigDialogValidate.15")); //$NON-NLS-1$
				return false;
			}
			
		} catch (NumberFormatException e) {
			// エラー
			dialog.setErrorMessage(Messages.getString("ProjectConfigDialogValidate.15")); //$NON-NLS-1$
			return false;
		}

	}
	
	/**
	 * ポート番号テキストのエラーチェックを行います。
	 * 
	 * @param 		text		ポート番号テキストボックス
	 * @return		エラーの有無
	 */
	public boolean validatePort(Button fixedSyncChk, Text text){
		
		// 「固定端末への同期を使用可にする」チェックがされていない場合
		if(!fixedSyncChk.getSelection()) {
			// エラーなし
			dialog.setErrorMessage(null);
			return true;
		}
		
		try {
			
			if(Integer.parseInt(text.getText()) >= 0) {
				// エラーなし
				dialog.setErrorMessage(null);
				return true;
			}
			else {
				// エラー
				dialog.setErrorMessage(Messages.getString("ProjectConfigDialogValidate.16")); //$NON-NLS-1$
				return false;
			}
			
		} catch (NumberFormatException e) {
			// エラー
			dialog.setErrorMessage(Messages.getString("ProjectConfigDialogValidate.16")); //$NON-NLS-1$
			return false;
		}

	}

}
