/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.util;

import java.beans.XMLDecoder;
import java.beans.XMLEncoder;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetAddress;
import java.nio.channels.FileChannel;
import java.sql.Timestamp;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.ResourceBundle;

import jp.valtech.bts.data.CurrentProject;
import jp.valtech.bts.data.Issue;
import jp.valtech.bts.data.IssueStatus;
import jp.valtech.bts.network.ClientInfo;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.ui.navigator.ProjectConfig;

import org.eclipse.jface.dialogs.MessageDialog;

/**
 * 
 * <DL><DT><B>BTS共通ユーティリティ処理を集めたクラス</B></DT>
 *  <DD>
 *  このユーティリティは、
 *  暗号化＋圧縮＋BASE64エンコーディング({@link #encode(String, String, boolean)})および複合化＋伸長＋BASE64デコーディング({@link #decode(String, String, boolean)})、
 *  オブジェクトのXML化({@link #serialize(Object)})およびXMLからオブジェクトの変換({@link #deserialize(String)})
 *  が定義されています。
 *  </DD>
 * </DL>
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class BTSUtility implements Logging {

	/** ファイルシステムのファイル区切り文字列 */
	private static final String SEPARATOR	= System.getProperty( "file.separator" );

	/** リソースファイルアクセスクラス */
	private static ResourceBundle bundle	= ResourceBundle.getBundle( "command" );

	/**
	 * <DL><DT><B>コンストラクタ</B>
	 *  <DD></DD>
	 * </DL>
	 */
	private BTSUtility() {
		super();
	}


	/**
	 * <DL><DT><B>オブジェクトをXMLへシリアライズします。</B></DL>
	 *  <DD>
	 *  {@link XMLEncoder}を利用してオブジェクトをXMLドキュメントへ変換します。
	 *  変換されたオブジェクトは文字列として出力されます。2バイト文字が含まれている場合のエンコーディングはUTF-8です。<br><br>
	 *  {@link XMLEncoder}で出力可能なオブジェクトはデフォルトコンストラクタが存在しなければなりません。
	 *  {@link File}、{@link java.sql.Timestamp}などは、デフォルトコンストラクタが無いため、シリアライズできません。
	 *  これらはXML化処理時にスキップされるためXMLドキュメントとして残りません。</DD>
	 * </DL>
	 * @param obj
	 * @return		シリアライズされたオブジェクトのXMLドキュメント
	 * @throws IOException 
	 */
	public static String serialize( Object obj ) throws IOException{
		String result = null;

		ByteArrayOutputStream baos = null;
		try{
			baos = new ByteArrayOutputStream();
			serialize( baos, obj );
			baos.flush();
			result = new String( baos.toByteArray( ), CHARSET.UTF_8 );
		}finally{
			if( baos != null ){
				try {
					baos.close();
				} catch ( IOException e ) {
					logger.fatal( e.getMessage(), e );
				}
			}
		}
		
		return result;
	}

	/**
	 * <DL><DT><B>オブジェクトをシリアライズし、指定されたファイルに出力します。</B></DL>
	 *  <DD>
	 *  {@link XMLEncoder}を利用してオブジェクトをXMLドキュメントへ変換します。
	 *  変換されたオブジェクトは指定された出力先の{@link File}に出力されます。<br><br>
	 *  {@link XMLEncoder}で出力可能なオブジェクトはデフォルトコンストラクタが存在しなければなりません。
	 *  {@link File}、{@link java.sql.Timestamp}などは、デフォルトコンストラクタが無いため、シリアライズできません。
	 *  これらはXML化処理時にスキップされるためXMLドキュメントとして残りません。</DD>
	 * </DL>
	 * @param f
	 * @param obj
	 * @throws IOException
	 */
	public static void serialize( File f, Object obj ) throws IOException{
		OutputStream os = new FileOutputStream( f );
		try{
			serialize( os, obj );
			os.flush();
		}finally{
			if( os != null ){
				try {
					os.close();
				} catch ( IOException e ) {
					logger.fatal( e.getMessage(), e );
				}
			}
		}
	}

	/**
	 * <DL><DT><B>オブジェクトをシリアライズし、指定された出力ストリームに出力します。</B></DL>
	 *  <DD>
	 *  {@link XMLEncoder}を利用してオブジェクトをXMLドキュメントへ変換します。
	 *  変換されたオブジェクトは指定された出力先の{@link OutputStream}に出力されます。<br><br>
	 *  {@link XMLEncoder}で出力可能なオブジェクトはデフォルトコンストラクタが存在しなければなりません。
	 *  {@link File}、{@link java.sql.Timestamp}などは、デフォルトコンストラクタが無いため、シリアライズできません。
	 *  これらはXML化処理時にスキップされるためXMLドキュメントとして残りません。
	 *  </DD>
	 * </DL>
	 * @see		java.beans.XMLEncoder
	 * @param	os	出力先
	 * @param	obj	対象オブジェクト
	 */
	public static void serialize( OutputStream os, Object obj ){
		XMLEncoder e = new XMLEncoder( os );
		e.writeObject( obj );
		e.close();
	}

	/**
	 * <DL><DT><B>入力ストリームからのデータを逆シリアライズしてオブジェクトを取得します。</B></DL>
	 *  <DD></DD>
	 * </DL>
	 * @param is				逆シリアライズするための対象入力ストリーム
	 * @return					XMLから逆シリアライズされたオブジェクト
	 */
	public static Object deserialize( InputStream is ){
		Object result = null;

		XMLDecoder d = new XMLDecoder( is );
		result = d.readObject();
		d.close();

		return result;
	}

	/**
	 * <DL><DT><B>指定されたStringオブジェクトを逆シリアライズしてオブジェクトを取得します。</B></DL>
	 *  <DD></DD>
	 * </DL>
	 * @param document			逆シリアライズするためのString文字列
	 * @return					XMLから逆シリアライズされたオブジェクト
	 * @throws IOException 
	 */
	public static Object deserialize( String document ) throws IOException{
		InputStream is	= null;
		Object result	= null;
		try{
			is = new ByteArrayInputStream( document.getBytes( CHARSET.UTF_8 ) );
			result = deserialize( is );
		}finally{
			if( is != null ){
				try {
					is.close();
				} catch ( IOException e ) {
					logger.fatal( e.getMessage(), e );
				}
			}
		}
		return result;
	}

	/**
	 * <DL><DT><B>ファイルからXMLデータを読み込み、逆シリアライズしてオブジェクトを取得します。</B></DL>
	 *  <DD></DD>
	 * </DL>
	 * @param f				逆シリアライズするための対象ファイル(XMLフォーマット)
	 * @return					XMLから逆シリアライズされたオブジェクト
	 * @throws IOException
	 */
	public static Object deserialize( File f ) throws IOException{
		InputStream is	= null;
		Object result	= null;
		try{
			is = new FileInputStream( f );
			result = deserialize( is );
		}finally{
			if( is != null ){
				try {
					is.close();
				} catch ( IOException e ) {
					logger.fatal( e.getMessage(), e );
				}
			}
		}
		return result;
	}


	/**
	 * 日付フォーマット。
	 * {@link java.sql.Timestamp}オブジェクトを指定のフォーマット文字列でフォーマットして返します。
	 * 
	 * @param			date 				タイムスタンプ型オブジェクト
	 * @param			formatString 		フォーマット文字列
	 * @return			フォーマット済みのDate文字列
	 */
	public static String formatDate(Timestamp date, String formatString) {
		if(date ==null) {
			return null;
		}
		return formatDate(date.getTime(), formatString);
	}

	/**
	 * 日付フォーマット。
	 * {@link java.util.Date}オブジェクトを指定のフォーマット文字列でフォーマットして返します。
	 * 
	 * @param			date 				タイムスタンプ型オブジェクト
	 * @param			formatString 		フォーマット文字列
	 * @return			フォーマット済みのDate文字列
	 */
	public static String formatDate(Date date, String formatString) {
		if(date ==null) {
			return null;
		}
		return formatDate(date.getTime(), formatString);
	}

	/**
	 * 日付フォーマット。
	 * long値を指定のフォーマット文字列でフォーマットして返します。
	 * 
	 * @param			date 				タイムスタンプ型オブジェクト
	 * @param			formatString 		フォーマット文字列
	 * @return			フォーマット済みのDate文字列
	 */
	public static String formatDate(long date, String formatString) {
		Date tmp = new Date(date);
		return (new SimpleDateFormat(formatString)).format(tmp);
	}

	/**
	 * 日付フォーマット。
	 * {@link java.util.Date}オブジェクトを"yyyy/MM/dd HH:mm"形式でフォーマットして返します。
	 * 
	 * @param			date 				タイムスタンプ型オブジェクト
	 * @param			formatString 		フォーマット文字列
	 * @return			フォーマット済みのDate文字列
	 */
	public static String formatDate(Timestamp date) {
		return formatDate(date, "yyyy/MM/dd HH:mm");
	}

	/**
	 * 日付フォーマット。
	 * {@link java.util.Date}オブジェクトを"yyyy/MM/dd HH:mm"形式でフォーマットして返します。
	 * 
	 * @param			date 				タイムスタンプ型オブジェクト
	 * @param			formatString 		フォーマット文字列
	 * @return			フォーマット済みのDate文字列
	 */
	public static String formatDate(Date date) {
		return formatDate(date, "yyyy/MM/dd HH:mm");
	}

	/**
	 * 日付フォーマット。
	 * long値を"yyyy/MM/dd HH:mm"形式でフォーマットして返します。
	 * 
	 * @param			date 				long型タイムスタンプ
	 * @param			formatString 		フォーマット文字列
	 * @return			フォーマット済みのDate文字列
	 */
	public static String formatDate(long date) {
		
		return formatDate(new Timestamp(date), "yyyy/MM/dd HH:mm");
	}
	
	/**
	 * {@link java.util.Dateオブジェクト生成。
	 * 指定の日付文字列を元にDateオブジェクトを生成して返します。
	 * 生成に失敗した場合はnullを返します。
	 * 
	 * @param			dateString			日付の文字列
	 * @return			日付文字列をjava.util.Dateに変換したオブジェクト
	 */
	public static Date createDate(String dateString) {
		return createDate(dateString, "yyyy/MM/dd");
	}
	
	
	/**
	 * {@link java.util.Dateオブジェクト生成。
	 * 指定の日付文字列とフォーマット文字列元にDateオブジェクトを生成して返します。
	 * 生成に失敗した場合はnullを返します。
	 * 
	 * @param			dateString			日付の文字列
	 * @param			format				フォーマット文字列
	 * @return			日付文字列をjava.util.Dateに変換したオブジェクト
	 */
	public static Date createDate(String dateString, String format) {
		if(dateString==null) {
			return null;
		}
		
		SimpleDateFormat dateFormat = new SimpleDateFormat(format);
		// 厳密な解析
		dateFormat.setLenient(false);
		
		try {
			return dateFormat.parse(dateString);
		}catch (Exception e) {
			return null;
		}
	}
	
	
	/**
	 * 日付のエラーチェックを行います。
	 * @param		frmString		日付の文字列（from）
	 * @param		toString		日付の文字列（to）
	 * @return		エラーの有無
	 */
	public static boolean dateTextError(String frmString, String toString) {
		
		if (!"".equals(frmString) && !"".equals(toString)) {
			// 期間のエラーチェック
			return fromtoError(frmString, toString);
		} else {
			// フォーマットのエラーチェック
			return formatError(frmString, toString);
		}
		
	}
	
	
	/**
	 * 日付の期間チェックを行います。
	 * @param		frmString		日付の文字列（from）
	 * @param 		toString		日付の文字列（to）
	 * @return		エラーの有無
	 */
	public static boolean fromtoError(String frmString, String toString) {
		
		if(createDate(frmString) == null || createDate(toString) == null) {
			// 変換エラー
			MessageDialog.openError(null, Messages.getString("BTSUtility.0"), Messages.getString("BTSUtility.1")); //$NON-NLS-1$ //$NON-NLS-2$
			return false;
		}

		if(createDate(toString).before(createDate(frmString))) {
			// 期間が FROM<=TO になっていない場合
			MessageDialog.openError(null, Messages.getString("BTSUtility.2"), Messages.getString("BTSUtility.3")); //$NON-NLS-1$ //$NON-NLS-2$
			return false;
		}
		
		return true;
		
	}
	
	
	/**
	 * 日付のフォーマットチェックを行います。
	 * @param		frmString		日付の文字列（from）
	 * @param 		toString		日付の文字列（to）
	 * @return		エラーの有無
	 */
	public static boolean formatError(String frmString, String toString) {
		
		if(createDate(frmString) == null && createDate(toString) == null) {
			// 変換エラー
			MessageDialog.openError(null, Messages.getString("BTSUtility.4"), Messages.getString("BTSUtility.5")); //$NON-NLS-1$ //$NON-NLS-2$
			return false;
		}
		
		return true;
		
	}

	
	/**
	 * 日付のフォーマットチェックを行います。
	 * 
	 * @param 		dateStr		日付の文字列
	 * @return		エラーの有無
	 */
	public static boolean formatError(String dateStr) {
		
		if(createDate(dateStr) == null) {
			// 変換エラー
			MessageDialog.openError(null, Messages.getString("BTSUtility.6"), Messages.getString("BTSUtility.7")); //$NON-NLS-1$ //$NON-NLS-2$
			return false;
		}
		
		return true;
		
	}

	/**
	 * 自分のIPアドレスを返します。
	 * 
	 * @return		自分のIPアドレス
	 */
	public static String getMyAddress() {
		try {
			InetAddress inetAddress = InetAddress.getLocalHost();
			return inetAddress.getHostAddress();
		}catch (Exception e) {
			logger.fatal(e);
			return null;
		}
	}


	/**
     * <p>rightPad pad a String with spaces (' ').</p>
     *
     * <p>The String is padded to the size of <code>size<code>.</p>
     *
     * <pre>
     * rightPad(null, *)   = null
     * rightPad("", 3)     = "   "
     * rightPad("bat", 3)  = "bat"
     * rightPad("bat", 5)  = "bat  "
     * rightPad("bat", 1)  = "bat"
     * rightPad("bat", -1) = "bat"
     * </pre>
     *
     * @param str  the String to pad out, may be null
     * @param size  the size to pad to
     * @return left padded String or original String if no padding is necessary,
     *  <code>null</code> if null String input
     */
    public static String rightPad(String str, int size) {
        return rightPad(str, size, ' ');
    }

    
    /**
     * <p>rightPad pad a String with a specified character.</p>
     *
     * <p>Pad to a size of <code>size</code>.</p>
     *
     * <pre>
     * rightPad(null, *, *)     = null
     * rightPad("", 3, 'z')     = "zzz"
     * rightPad("bat", 3, 'z')  = "bat"
     * rightPad("bat", 5, 'z')  = "batzz"
     * rightPad("bat", 1, 'z')  = "bat"
     * rightPad("bat", -1, 'z') = "bat"
     * </pre>
     *
     * @param str  the String to pad out, may be null
     * @param size  the size to pad to
     * @param padChar  the character to pad with
     * @return left padded String or original String if no padding is necessary,
     *  <code>null</code> if null String input
     * @since 2.0
     */
    public static String rightPad(String str, int size, char padChar) {
        if (str == null) {
            return null;
        }
        if(size < 1) {
        	return str;
        }
        
        int strSize = str.getBytes().length;
        if(strSize > size) {
        	return str;
        }
        
        int padding = size - strSize;
        StringBuffer buf = new StringBuffer();
        for (int i = 0; i < padding ; i++) {
            buf.append(" ");
		}
        return str + buf.toString();
    }
    
    
	/**
	 * 指定の{@link ClientInfo クライアント情報}が自分自身かどうか判別します。
	 * 
	 * @param		clientInfo		クライアント情報
	 * @return		true-- 指定のクライアント情報が自分自身。
	 * 				false--指定のクライアント情報が自分自身でない
	 */
	public static boolean isLocalHost(ClientInfo clientInfo) {
		try {
			String remoteAddress = clientInfo.getClientAddress();
			String localAddress = InetAddress.getLocalHost().getHostAddress();

			return localAddress.equals(remoteAddress);
		} catch (Exception e) {
			return false;
		}
	}
	
	
	/**
	 * 完了期限を判定します。
	 * 
	 * @param			issue			課題票
	 * @return			判定 ({@link Issue#DEADLINE_SAFE}:余裕がある　
	 * 						  {@link Issue#DEADLINE_WARNING}:迫っている　　　
	 * 						  {@link Issue#DEADLINE_OVER}:切れている)
	 */
	public static int judgeLimit(Issue issue) {
		
		return judgeLimit(issue.getDeadline(), issue.getStatus());
	
	}

	/**
	 * 完了期限を判定します。
	 * 
	 * @param			deadLine			課題票の完了期限
	 * @param			status				課題票のステータス
	 * @return			判定 ({@link Issue#DEADLINE_SAFE}:余裕がある　
	 * 						  {@link Issue#DEADLINE_WARNING}:迫っている　　　
	 * 						  {@link Issue#DEADLINE_OVER}:切れている)
	 */
	public static int judgeLimit(Date deadLine, String status) {
		
		// 完了期限種別(0:余裕がある　1:迫っている　2:切れている)
		int limitType = Issue.DEADLINE_SAFE;

		// 「ステータス：完了」の場合
		if(IssueStatus.CLOSED_VALUE.equals(status)) {
			return limitType;
		}

		
		// 完了期限が設定されている場合
		if(deadLine != null) {
			
			// 現在開いているプロジェクト設定情報取得
			ProjectConfig config = CurrentProject.getInsance().getProjectConfig();
			
			// 現在の日付カレンダ取得
			Calendar calendarNow = Calendar.getInstance();
			
			// 完了期限カレンダ取得
			Calendar calendarDead = Calendar.getInstance();
			calendarDead.setTime(deadLine);
			
			// 完了期限までの残り日数
			int untilLimit = (calendarDead.get(Calendar.YEAR) - calendarNow.get(Calendar.YEAR)) * 365
								+ calendarDead.get(Calendar.DAY_OF_YEAR) - calendarNow.get(Calendar.DAY_OF_YEAR);
			
			// 完了期限が切れている場合
			if(untilLimit < 0) {
				limitType = Issue.DEADLINE_OVER;
			}
			
			// 完了期限が迫っているアイコンを表示する設定である場合
			else if(config.isUntilDeadLineCheck()) {
				
				// 完了期限警告日数取得
				int limit = Integer.parseInt(config.getUntilDeadLine());
				
				// 完了期限が迫っている場合
				if(untilLimit <= limit) {
					limitType = Issue.DEADLINE_WARNING;
				}

			}
	
		}
		
		return limitType;
	}
    
	
	/**
	 * 指定のSSIDのホームディレクトリを取得します。
	 * 
	 * @param			ssid		SSID
	 * @return			指定のSSIDのホームディレクトリのPATH文字列
	 */
	public static String getBaseDir(String ssid) {
		// ワークスペースのディレクトリ取得
		String baseDir = BtsPlugin.getInstance().getStateLocation().toString();

		// 当該プロジェクトのファイルを保管するディレクトリのPATH文字列生成
		String ssidDir = baseDir + SEPARATOR + ssid + SEPARATOR;

		return ssidDir;
	}
	
	
	/**
	 * ファイルをコピーします。
	 * 当プラグインの中で持つファイルのコピーを行う場合は{@link BtsPlugin#copy(String, String)}の方を使ってください。
	 * 
	 * @param		inFilepath		コピー元ファイルのPATH文字列
	 * @param		outFilepath		生成するファイルのPATH文字列
	 * @throws		IOException
	 * @see			jp.valtech.bts.ui.BtsPlugin#copy(String, String)
	 */
	public static void copy(String inFilepath, String outFilepath) throws IOException {
		FileInputStream fis = null;
		FileOutputStream fos = null;
		try {
			// ファイルのオープン
			fis = new FileInputStream(inFilepath);
			fos = new FileOutputStream(outFilepath);
			
			// コピー準備
			FileChannel inChannel = fis.getChannel();
			FileChannel outChannel = fos.getChannel();

			// コピー実行
			inChannel.transferTo(0, inChannel.size(), outChannel);
		} finally {
			// ファイルクローズ
			try {
				if(fis != null) {
					fis.close();
				}
			}
			catch(IOException e) {
			}
			try {
				if(fos != null) {
					fos.close();
				}
			}
			catch(IOException e) {
			}
		}
	}

	/**
	 * <DL>
	 *  <DT><B>受信したコマンドからサーバコマンドをルックアップします</B></DT>
	 *  <DD>
	 *    引数のコマンド名から実行するコマンドクラスをルックアップします。コマンド名が見つからない場合や適切な基底クラスを継承/実装していない場合などは、このメソッドで取得できるコマンドクラスはnullとなります。
	 * 	  <ul>	
	 *	  <li>ソースのルートディレクトリ直下に<code>command.properties</code>を作成します。
	 *	  <li><code>キー=値</code>形式でコマンドに対応する、コマンド実装クラスをFQCN(Fully Qualified Class Name/パッケージ名を含めた完全修飾クラス名)で指定します。
	 *		  また、キーを指定する際には、コマンド名の先頭に<code>"cmd."</code>を必ず付加してください。
	 *		  {@link jp.valtech.bts.network.command.client.ClientCommand}クラスからの呼び出しの際には、<code>"cmd."</code>をはずしたコマンド名を利用してください。
	 *	  </ul>
	 * 	</dd>
	 * </DL>
	 * @param cmd		コマンド(ルックアップ対象)
	 * @param name		コマンドインスタンスの名称
	 * @return			サーバコマンドのインスタンス。インスタンス化に失敗した場合はnullが返ります。
	 */
	public static Object getCommand(String cmd){
		try {
			String clazzName = bundle.getString( "cmd." + cmd );

			if(logger.isDebugEnabled()) {
				logger.debug( "loockup target command name=" + cmd + "/class name=" + clazzName );
			}

			if( clazzName != null ){
				//	指定されたFQCNを使用し、標準クラスローダからクラスオブジェクトを取得します。
				Class clazz = Class.forName( clazzName );
				return clazz.newInstance();
			}

		} catch ( Exception e ) {
			logger.error( e );
		}

		return null;
	}
}
