/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.command.client;

import java.lang.reflect.InvocationTargetException;
import java.util.List;

import jp.valtech.bts.data.Issue;
import jp.valtech.bts.data.NetworkConfig;
import jp.valtech.bts.facade.SyncClientFacade;
import jp.valtech.bts.facade.SyncHeaderFacade;
import jp.valtech.bts.network.Request;
import jp.valtech.bts.network.Response;
import jp.valtech.bts.network.command.TCPClientCommand;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.ui.action.RefreshAction;
import jp.valtech.bts.util.Logging;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;

/**
 * 同期リクエストを送信します。
 * このコマンドで同期リクエストを送信します。同期リクエストの送信にはTCP通信が使用されます。
 * 
 * @see			jp.valtech.bts.command.server.SyncIssueServer
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class SyncIssue extends TCPClientCommand implements Logging {

	/** 同期先のユーザ名 */
	private String userName;

	/** 同期コマンドが成功したかどうか */
	private boolean finished = false;
	
	/**
	 * クライアントコマンド生成
	 * 
	 * @param		config 		設定情報
	 * @param		host		送信先ホスト名
	 * @param		port 		送信先ポート番号
	 */
	public SyncIssue( NetworkConfig myconfig, String host, int port ) {
		super( myconfig, host, port );
		setTimeout(myconfig.getSyncTimeout());
	}

	
	/**
	 * 同期リクエストを送信します。
	 * 送信結果は{@link TCPClientCommand#isSuccessful()}にて取得します。
	 * 
	 */
	public void execute(){
		try {
			
	    	// プログレスダイアログを生成
			IRunnableWithProgress progress = new IRunnableWithProgress() {

				/*
				 *  (非 Javadoc)
				 * @see org.eclipse.jface.operation.IRunnableWithProgress#run(org.eclipse.core.runtime.IProgressMonitor)
				 */
				public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
					
					// 送信オブジェクトを生成
					Request request = new Request();
					request.setCommand( "syncIssue" ); 

					//### TASK1: ローカルの課題票情報を取得 ###
					{
						monitor.beginTask(Messages.getString("SyncIssue.1"), 5); //$NON-NLS-1$
						
						SyncHeaderFacade syncHeaderFacade = new SyncHeaderFacade();

						// 課題票ヘッダ情報をリクエストに格納
						List issueHeaderList = syncHeaderFacade.getIssueHeaderList();
						request.addParameter( "issueHeader",  issueHeaderList);
						monitor.worked(1);

						// ごみ箱ヘッダ情報をリクエストに格納
						List garbageHeaderList = syncHeaderFacade.getGarbageHeaderList();
						request.addParameter( "garbageHeader",  garbageHeaderList);
						monitor.worked(1);

						// キャンセルボタンを押した場合
						if (monitor.isCanceled()) {
							throw new InterruptedException(Messages.getString("SyncIssue.4")); //$NON-NLS-1$
						}
					}

					//### TASK2: ローカルの課題票情報送信～同期データ受信 ###
					try {
						monitor.subTask ( "[ " + userName + " (" + host + Messages.getString("SyncIssue.2"));  //$NON-NLS-1$ 

						// 送信～受信処理
						Response response = send( request );
						monitor.worked(2);
					
						// キャンセルボタンを押した場合
						if (monitor.isCanceled()) {
							throw new InterruptedException(Messages.getString("SyncIssue.4")); //$NON-NLS-1$
						}

						// サーバ側で同期失敗
						if (!isSuccessful()) {
							throw new InterruptedException(response.getMessage());
						}

						// 同期対象の課題票を取得
						Issue[] syncIssues 	 = (Issue[])response.getParameterObject("syncIssues");
						// ごみ箱に入る課題票を取得
						Issue[] syncGarbages = (Issue[])response.getParameterObject("syncGarbages");

						
					//### TASK3: ローカルDBに反映 ###
						SyncClientFacade syncFacade = new SyncClientFacade();
						syncFacade.applySyncIssues(syncIssues, syncGarbages, monitor);
						syncFacade.saveMessage(userName, host);
						
						monitor.done();
	
					}catch (Exception e) {
						logger.fatal(Messages.getString("SyncIssue.5"), e);//$NON-NLS-1$
						throw new InterruptedException(e.getMessage());
					}
				}
			};

			// プログレスダイアログの表示
			ProgressMonitorDialog dialog = new ProgressMonitorDialog(BtsPlugin.getInstance().getShell());
			dialog.run(true, true, progress);
			
			// UI表示を最新にする。 
			RefreshAction refreshAction = new RefreshAction();
			refreshAction.run();

			// 成功フラグをtrueにする
			this.finished = true;
			
		} catch (InvocationTargetException e) {
			BtsPlugin.getInstance().errorlog(e);
		} catch (InterruptedException e) {
			// キャンセル後の処理
			MessageDialog.openWarning( BtsPlugin.getInstance().getShell()
									 , Messages.getString("SyncIssue.3")   //$NON-NLS-1$
									 , e.getMessage()); 
		}
	}
	
	
	/**
	 * 同期相手のユーザ名を設定します。	
	 * 
	 * @param userName	同期相手のユーザ名
	 */
	public void setRemoteUserName(String userName)  {
		this.userName = userName;
	}

	/**
	 * 同期処理の実行結果を取得する。
	 * 
	 * @return			true --同期処理実行成功。
	 * 					false--同期処理実行失敗。
	 */
	public boolean isFinished() {
		return this.finished;
	}

}

