/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.editor;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.util.List;

import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import jp.valtech.bts.data.Issue;
import jp.valtech.bts.data.IssueStatus;
import jp.valtech.bts.data.OutputDetails;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.util.BTSUtility;
import jp.valtech.bts.util.Logging;

/**
 * 課題票詳細をファイルに出力します。
 * 
 * @author		<A href="mailto:s_imai@valtech.jp">S.Imai</A>
 * @version	Ver.0.8
 */
public class IssueDetailsExport implements Logging {
	
	/** 帳票出力ダイアログ */
	private IssueDetailsExportDialog dialog;
	
    /** 帳票出力先のファイルパス */
    private String fileName = null;
    
    /** 帳票出力先のフォルダパス */
    private String filterPath;

    
    /**
     * 初期設定を行います。
     * 
     * @param		dialog			課題票詳細出力ダイアログ
     * @param 		fileName		出力先パス
     * @param 		filterPath		出力先フォルダパス
     */
	public IssueDetailsExport(IssueDetailsExportDialog dialog, String fileName, String filterPath) {
		this.dialog = dialog;
		this.fileName = fileName;
		this.filterPath = filterPath;
	}
	
	
	/**
	 * 現在開いている課題票の詳細を帳票に出力します。
	 */
	public void exportRun() {
		
		// cssファイルを出力
		BtsPlugin.getInstance().copy("doc/output.css", filterPath + "/output.css");

		// OutputDetailsオブジェクトに値をセットし、指定された形式で出力
		outputFile(setDetails());
	
	}
	
	/**
	 * オブジェクトを指定された形式でファイルに出力します。
	 * Transformer#transform(Source, Result)を用いる際に、
	 * StreamSourceとStreamResultの入出力元ストリームは明示的にクローズする必要があります。
	 * 
	 * @param		object		出力するオブジェクト
	 */
	private void outputFile(Object object) {
		
		try {
			
			// XMLファイルに出力する場合
			if (dialog.fileXMLRdi.getSelection()) {

				// オブジェクトをXMLファイルに出力
				BTSUtility.serialize(new File(fileName), object);

			}
			
			// HTMLファイルに出力する場合
			else if (dialog.fileHTMRdi.getSelection()) {
				
				// オブジェクトをXMLファイルに出力
				BTSUtility.serialize(new File(filterPath + "/tmp.xml"), object);

				// TransformerFactoryインスタンスを取得
				TransformerFactory factory = TransformerFactory.newInstance();
				
				// XSLファイルのURL取得
				URL url = BtsPlugin.getInstance().getURL("doc/outputDetails.xsl");

				// XSLファイルからtranceformerを取得
				InputStream inputStream = url.openStream();
				Transformer transformer = factory.newTransformer(new StreamSource(inputStream));

				// 出力するエンコーディングを設定
				transformer.setOutputProperty("encoding", "UTF-8");

				// XSLTを用いてXMLファイルをHTMLファイルに変換して出力
				InputStream in = new FileInputStream(filterPath + "/tmp.xml");
				OutputStream out = new FileOutputStream(fileName);
				transformer.transform(new StreamSource(in), new StreamResult(out));
				
				// クローズ
				inputStream.close();
				in.close();
				out.close();
				
			}
			
		} catch (Exception e) {
			String msg = Messages.getString("IssueDetailsExport.3"); //$NON-NLS-1$
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);

		}	
		
	}
	
    
	/**
	 * 課題票詳細出力用の情報をOutputDetailsオブジェクトにセットします。
	 * 
	 * @return		出力するオブジェクト
	 */
    private OutputDetails setDetails() {
    	
		// アクティブな課題票エディタを取得する
    	IssueMultiPageEditor editorPart = BtsPlugin.getInstance().getActiveEditor();
    	
    	// 課題票取得
    	Issue issue = editorPart.getModifiedIssue();  	
    	
    	// OutputDetailsオブジェクトに課題票詳細情報をセット
    	OutputDetails outputDetails = new OutputDetails();
    	outputDetails.setIssueId(issue.getDisplayIssueID());										// 課題票ID
    	outputDetails.setCreateUser(issue.getCreateUser());											// 起票者
    	outputDetails.setCategory(issue.getCategory());												// カテゴリ
    	outputDetails.setTitle(issue.getTitle());													// タイトル
    	outputDetails.setDescription(issue.getDescription());										// 詳細
    	outputDetails.setPriority(issue.getPriority());												// 優先度
    	outputDetails.setStatus(IssueStatus.getEnumFromValue(issue.getStatus()).getDescription());	// ステータス
    	outputDetails.setAssigned(issue.getAssigned());												// 担当者 	
    	outputDetails.setDeadLine(BTSUtility.formatDate(issue.getDeadline(), "yyyy/MM/dd"));		// 完了期限
    	outputDetails.setCreateDate(BTSUtility.formatDate(issue.getCreateDate()));					// 起票日
    	outputDetails.setUpdateDate(BTSUtility.formatDate(issue.getUpdateDate()));					// 更新日
    	outputDetails.setHeader(dialog.headerTxt.getText().trim());									// ヘッダ
    	outputDetails.setFooter(dialog.footerTxt.getText().trim());									// フッタ
   
    	// 内緒のバグ情報をセット
    	if(issue.getSecretBug().booleanValue()) {
    		outputDetails.setSecretBug("ON");
    	} else {
    		outputDetails.setSecretBug("OFF");
    	}
    	
    	// 一時保留情報をセット
    	if(issue.getSuspend().booleanValue()) {
    		outputDetails.setSuspend("ON");
    	} else {
    		outputDetails.setSuspend("OFF");
    	}
    	
    	// 添付情報をセット
    	if(((List)dialog.attachViewer.getInput()).size() == 0) {
    		outputDetails.setAttach(Messages.getString("IssueDetailsExport.0")); //$NON-NLS-1$
    	} else {
    		outputDetails.setAttach(Messages.getString("IssueDetailsExport.1")); //$NON-NLS-1$
    	}
    	
    	return outputDetails;
    	
    }

	
}
