/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.issuelist;


import java.util.ArrayList;
import java.util.List;

import jp.valtech.bts.dao.BtsDBException;
import jp.valtech.bts.data.CurrentProject;
import jp.valtech.bts.data.Issue;
import jp.valtech.bts.data.IssueType;
import jp.valtech.bts.facade.ModifyIssueFacade;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.ui.IBtsViewPart;
import jp.valtech.bts.ui.action.DeleteIssueAction;
import jp.valtech.bts.ui.action.SendGarbageIssueAction;
import jp.valtech.bts.ui.action.OpenIssueAction;
import jp.valtech.bts.ui.action.OpenIssueFilterDlgAction;
import jp.valtech.bts.util.Logging;

import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.part.ViewPart;


/**
 * 課題票一覧ビューです。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class IssueListView extends ViewPart implements IBtsViewPart, Logging{
	
	/** 課題票一覧テーブルのビューアインスタンス */
	private TableViewer viewer;

	/** 課題票を開くアクション */
	private OpenIssueAction openIssueAction;
	/** 一覧のフィルターアクション */
	private OpenIssueFilterDlgAction issueFilterAction;
	/** 課題票をゴミ箱に移動するアクション */
	private SendGarbageIssueAction moveGarbageBoxAction;
	/** 課題票を削除するアクション */
	private DeleteIssueAction deleteIssueAction;
	
	
	/**
	 * ビュー生成時に最初に呼ばれるメソッドです。
	 * 
	 * @see			org.eclipse.ui.part.WorkbenchPart#createPartControl(org.eclipse.swt.widgets.Composite)
	 * @param		parent				親コンポジット
	 */
	public void createPartControl(Composite parent) {

		try {
			// ビューア取得
			createIssueListTable(parent);
			
			// このビューに結びつくアクションを登録
			makeActions();
			
			// 右クリックメニューを設定
			hookContextMenu();
			
			// ダブルクリックの動作を設定
			hookDoubleClickAction();
			
			// ツールバーを設定
			fillLocalToolBar();

			// フィルタ実行。ビューオープン時は１件も表示しない。ツリークリックで表示されるようにする。
			IDialogSettings section 
				= BtsPlugin.getInstance().getDialogSettings().getSection(IssueFilterManager.SECTION_NAME);
			if(section != null) {
				IssueListViewFilter filter = new IssueListViewFilter(this);
				filter.setClearFilter();
				filter.addIssueFilters();
			}

		}catch (Exception e) {
			String msg = Messages.getString("IssueListView.0"); //$NON-NLS-1$
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);
		}
	}


	/**
	 * 課題票一覧の表示件数を表示します。
	 * 
	 * @param		filterListSize			課題票一覧の表示件数
	 */
	public void updateDescription(int filterListSize) {
		setContentDescription(Messages.getString("IssueListView.1") + filterListSize + Messages.getString("IssueListView.2")); //$NON-NLS-1$ //$NON-NLS-2$
	}

	/**
	 * テーブルを右クリックしたときの動作です。
	 * 右クリックメニューを表示します。
	 */
	private void hookContextMenu() {
		MenuManager menuMgr = new MenuManager("#PopupMenu"); 
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(new IMenuListener() {
			public void menuAboutToShow(IMenuManager manager) {
				fillContextMenu(manager);
			}
		});
		Menu menu = menuMgr.createContextMenu(viewer.getControl());
		viewer.getControl().setMenu(menu);
		getSite().registerContextMenu(menuMgr, viewer);
	}

	
	/**
	 * テーブルを右クリックしたときの動作です。
	 * 
	 * @param		manager			右クリックメニューのマネージャインスタンス
	 */
	private void fillContextMenu(IMenuManager manager) {
		
		// 選択した課題票を取得する
		Issue issue = getSelectionRecord();
		if(issue != null) {
			// 課題票オープンのアクションをメニューに登録
			openIssueAction.setIssue(issue);
			manager.add(openIssueAction);
			
			// 課題票の種別によりメニューを分ける
			if( IssueType.GARBAGE_VALUE.equals(issue.getType()) ) {
				//=== ごみ箱の課題票 ===
			
			} else if( IssueType.DRAFT_VALUE.equals(issue.getType()) ) {
				//=== 下書きの課題票 ===
				// めにゅー： 削除
				deleteIssueAction.setIssue(issue);
				manager.add(deleteIssueAction); 
				
			} else if( IssueType.CONFLICT_VALUE.equals(issue.getType()) ) {
				//=== 競合の課題票 ===
				// めにゅー： 課題票削除
				deleteIssueAction.setIssue(issue);
				manager.add(deleteIssueAction); 
	
			} else {
				//=== 通常の課題票 ===
				// めにゅー： ゴミ箱へ移動
				moveGarbageBoxAction.setIssue(issue);
				manager.add(moveGarbageBoxAction);  
			}
		}
			
		// Other plug-ins can contribute there actions here
		manager.add(new Separator(IWorkbenchActionConstants.MB_ADDITIONS));
	
	}


	/**
	 * ツールバーの設定です。
	 * 「フィルターボタン」を表示します。
	 */
	private void fillLocalToolBar() {
		IActionBars bars = getViewSite().getActionBars();
		// ぼたん：フィルター
		IToolBarManager toolbar = bars.getToolBarManager(); 
		toolbar.add(issueFilterAction);
	}


	/**
	 * テーブルをダブルクリックしたときの動作
	 *
	 */
	private void hookDoubleClickAction() {
		viewer.addDoubleClickListener(new IDoubleClickListener() {
			public void doubleClick(DoubleClickEvent event) {

				// 選択したを取得する
				Issue issue = getSelectionRecord();
				openIssueAction.setIssue(issue);

				// 課題票を開く
				openIssueAction.run();
			}
		});
	}
	

	/**
	 * アクションクラスの生成。
	 * このビューから呼び出されるアクションクラスを生成します。
	 *
	 */
	private void makeActions() {
		
		// 課題票を開くアクション
		openIssueAction = new OpenIssueAction(OpenIssueAction.OPEN_EXIST_ISSUE);
		
		// 課題表一覧のフィルターアクション
		issueFilterAction = new OpenIssueFilterDlgAction(this);

		// 課題票をゴミ箱に移動するアクション 
		moveGarbageBoxAction = new SendGarbageIssueAction();

		// 課題票を削除するアクション
		deleteIssueAction = new DeleteIssueAction(); 

		// カレントプロジェクトに以下のアクションを加える
		CurrentProject project = CurrentProject.getInsance();
		project.addAction(issueFilterAction);
		
    	// カレントプロジェクトのオープン／クローズにより活性／非活性の表示を分ける
		if(project.isOpen()) {
			issueFilterAction.setEnabled(true);
		} else {
			issueFilterAction.setEnabled(false);
		}
	}


	/**
	 * 選択しているレコードを返します。
	 * 選択しているレコードを{@link Issue}にキャストして返します。
	 * 
	 * @return			キャスト済み選択レコード
	 */
	private Issue getSelectionRecord() {
		
		// 選択しているレコードを取得します。
		ISelection selection = viewer.getSelection();
		Object obj = ((IStructuredSelection)selection).getFirstElement();
		if(obj instanceof Issue) {
			// キャストして返します。
			return (Issue)obj;
		}
		return null;
	}

	
	/**
	 * 課題票一覧テーブルにフォーカスします。
	 */
	public void setFocus() {
		viewer.getControl().setFocus();
	}

	/**
	 * 当インスタンスで持つテーブルビューアインスタンスを返します。
	 * 
	 * @return			当インスタンスで持つテーブルビューアインスタンス
	 */
	public TableViewer getViewer() {
		return this.viewer;
	}
	
	
	
	/**
	 * 課題票一覧ビューアを生成します。
	 * 
	 * @param			parent				親コンポジットです。
	 */
	private void createIssueListTable(Composite parent)  throws BtsDBException {

		Table table= new Table(parent, SWT.SINGLE | SWT.H_SCROLL | SWT.V_SCROLL | SWT.FULL_SELECTION | SWT.BORDER);
		table.setLayoutData(new GridData(GridData.FILL_BOTH));
		
		TableLayout layout = new TableLayout();
		table.setLayout(layout);

		table.setHeaderVisible(true);
		table.setLinesVisible(true);
		
		// 課題票一覧テーブルからビューアインスタンスを取得
		viewer = new TableViewer(table);
		// リスナー取得
		SelectionListener headerListener = getColumnListener(viewer);
		
		// 1カラム目（期限のアイコン）
		TableColumn col1= new TableColumn(table, SWT.NONE, 0);
		col1.setText(""); 
		col1.setWidth(20);
		col1.setAlignment(SWT.LEFT);
		col1.addSelectionListener(headerListener);

		// 2カラム目（課題票ID）
		TableColumn col2= new TableColumn(table, SWT.NONE, 1);
		col2.setText("ID"); 
		col2.setWidth(100);
		col2.setAlignment(SWT.LEFT);
		col2.addSelectionListener(headerListener);
		
		// 3カラム目（タイトル）
		TableColumn col3= new TableColumn(table, SWT.NONE, 2);
		col3.setText(Messages.getString("IssueListView.3"));  //$NON-NLS-1$
		col3.setWidth(300);
		col3.setAlignment(SWT.LEFT);
		col3.addSelectionListener(headerListener);
		
		// 4カラム目（カテゴリ）
		TableColumn col4= new TableColumn(table, SWT.NONE, 3);
		col4.setText(Messages.getString("IssueListView.4"));  //$NON-NLS-1$
		col4.setWidth(55);
		col4.setAlignment(SWT.LEFT);
		col4.addSelectionListener(headerListener);
		
		// 5カラム目（優先度）
		TableColumn col5= new TableColumn(table, SWT.NONE, 4);
		col5.setText(Messages.getString("IssueListView.5"));  //$NON-NLS-1$
		col5.setWidth(50);
		col5.setAlignment(SWT.LEFT);
		col5.addSelectionListener(headerListener);

		// 6カラム目（ステータス）
		TableColumn col6= new TableColumn(table, SWT.NONE, 5);
		col6.setText(Messages.getString("IssueListView.6"));  //$NON-NLS-1$
		col6.setWidth(75);
		col6.setAlignment(SWT.LEFT);
		col6.addSelectionListener(headerListener);

		// 7カラム目（担当者）
		TableColumn col7= new TableColumn(table, SWT.NONE, 6);
		col7.setText(Messages.getString("IssueListView.7"));  //$NON-NLS-1$
		col7.setWidth(100);
		col7.setAlignment(SWT.LEFT);
		col7.addSelectionListener(headerListener);

		// 8カラム目（起票日）
		TableColumn col8= new TableColumn(table, SWT.NONE, 7);
		col8.setText(Messages.getString("IssueListView.8"));  //$NON-NLS-1$
		col8.setWidth(105);
		col8.setAlignment(SWT.LEFT);
		col8.addSelectionListener(headerListener);
		
		// ビューアにプロバイダを設定
		IssueListViewContentProvider provider = new IssueListViewContentProvider(this);
		viewer.setContentProvider(provider);
		viewer.setLabelProvider(new IssueListViewLabelProvider());

		// ソートのデフォルト設定（起票日の降順）
		IssueListViewSorter sorter = new IssueListViewSorter(7);
		sorter.setReversed(true);
		viewer.setSorter(sorter);

		// テーブルの初期データ（空）を設定
		viewer.setInput(new ArrayList());

		// カレントプロジェクトがある場合はそれに属する課題票をテーブルに表示する
		CurrentProject project = CurrentProject.getInsance();
		if(project.isOpen()) {
			ModifyIssueFacade issueModifyFacade = new ModifyIssueFacade(); 
			List issueList = issueModifyFacade.getIssueList() ;
			provider.refresh(issueList);
		}
	}
	

	/**
	 * ソータを設定するリスナーです。
	 * @param		tableViewer		テーブルビューア
	 * @return		セレクションアダプタ
	 */
	private SelectionListener getColumnListener(final TableViewer tableViewer){
		
		return new SelectionAdapter(){

			/*
			 *  (非 Javadoc)
			 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			public void widgetSelected(SelectionEvent e) {
				
				// クリックしたカラム番号取得
				int column = tableViewer.getTable().indexOf((TableColumn)e.widget);
				
				// ソータの取得
				IssueListViewSorter oldSorter = (IssueListViewSorter) tableViewer.getSorter();
				
				// 同じカラムをクリックしたときの処理
				if(oldSorter != null && column == oldSorter.getColumnNumber()){
					// ソート反転フラグの切換え
					oldSorter.setReversed(!oldSorter.isReversed());
					// テーブルの再表示
					tableViewer.refresh();
				} else {
					// ソータの設定
					tableViewer.setSorter(new IssueListViewSorter(column));
				}
			}
		};
	}

    /*
     *  (非 Javadoc)
     * @see org.eclipse.ui.IWorkbenchPart#dispose()
     */
    public void dispose() {
    	
    	// カレントプロジェクトに設定したアクションを削除する
		CurrentProject project = CurrentProject.getInsance();
		project.removeAction(issueFilterAction);
    	
    	super.dispose();
    }
}

