/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.navigator;

import java.util.List;

import jp.valtech.bts.data.CurrentProject;
import jp.valtech.bts.data.IssueType;
import jp.valtech.bts.facade.ModifyIssueFacade;
import jp.valtech.bts.network.ClientInfo;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.ui.action.CloseProjectAction;
import jp.valtech.bts.ui.action.DeleteProjectAction;
import jp.valtech.bts.ui.action.ImportIssueAction;
import jp.valtech.bts.ui.action.OpenChartDlgAction;
import jp.valtech.bts.ui.action.OpenClassifiedExportDlgAction;
import jp.valtech.bts.ui.action.OpenIssueAction;
import jp.valtech.bts.ui.action.OpenMessageDlgAction;
import jp.valtech.bts.ui.action.OpenProjectAction;
import jp.valtech.bts.ui.action.OpenProjectConfigDlgAction;
import jp.valtech.bts.ui.action.RefreshAction;
import jp.valtech.bts.ui.action.SyncRemoteAction;
import jp.valtech.bts.ui.action.SyncSelectionRemoteAction;
import jp.valtech.bts.ui.issuelist.IssueListView;
import jp.valtech.bts.ui.issuelist.IssueListViewFilter;
import jp.valtech.bts.util.Logging;

import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.part.ViewPart;

/**
 * BTSナビゲータののツリービューです。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class NavigatorView extends ViewPart implements Logging {

    /** ツリーを表示するビューアインスタンスです。 */
    private TreeViewer viewer = null;

	/** プロジェクトを開くアクション(新規登録) */
	private OpenProjectAction openProjectAction;
	/** 「ファイル出力ボタン」を押した時のアクション */
	private OpenClassifiedExportDlgAction openClassifiedExportDlgAction;
	/** 「統計グラフ表示」ボタンを押した時のアクション */
	private OpenChartDlgAction openChartDialogAction;
	/** プロジェクト設定画面を表示するアクション(新規登録) */
	private OpenProjectConfigDlgAction registConfigDlgAction;
	/** プロジェクト設定画面を表示するアクション(編集) */
	private OpenProjectConfigDlgAction editConfigDlgAction;
	/** 最新表示にするアクション */
    private RefreshAction refreshAction ;
	/** 同期をとるアクション */
    private SyncRemoteAction syncRemoteAction ;
	/** 同期をとるアクション */
    private SyncSelectionRemoteAction syncSelectionHostAction ;
	/** 課題票を開くアクション(新規課題票) */
	private OpenIssueAction openNewIssueAction;
	/** 課題票を開くアクション(指定のカテゴリで) */
	private OpenIssueAction openCtgryIssueAction;
	/** プロジェクトを閉じるアクション */
	private CloseProjectAction closeProjectAction;
	/** プロジェクトを削除するアクション */
	private DeleteProjectAction deleteProjectAction;
	/** 課題票をインポートするアクション */
	private ImportIssueAction importIssueAction ;
	/** メッセージを送信するアクション */
	private OpenMessageDlgAction openMessageDlgAction;
	
	
	/**
     * ツリー画面を生成します。
     * 
     * @param		parent				親コンポジット
     */
    public void createPartControl(Composite parent) {
    	try {
    		
    		// 親コンポジットにレイアウトを設定
            parent.setLayout(new FillLayout());
            viewer = new TreeViewer(parent, SWT.NONE);
            
            // プロバイダを設定
            NavigatorViewContentProvider provider = new NavigatorViewContentProvider(); 
            viewer.setContentProvider(provider);
            viewer.setLabelProvider(new FolderTreeLabelProvider());
            
            // データ表示
			viewer.setInput(createDefaultTree());

			CurrentProject project = CurrentProject.getInsance();
			if(project.isOpen()) {
				ModifyIssueFacade issueModifyFacade = new ModifyIssueFacade(); 
				List issueList = issueModifyFacade.getIssueList() ;
				provider.refresh(issueList);
			}

            // ソータを設定
            viewer.setSorter(new ViewerSorter(){
            	public int compare(Viewer viewer, Object e1, Object e2) {

            		BtsTreeNode treeNode1 = (BtsTreeNode)e1;
            		BtsTreeNode treeNode2 = (BtsTreeNode)e2;
            		
            		return treeNode1.compareTo(treeNode2);
            	}
            });
            
            // ダブルクリックのメニューを設定
            hookDoubleClick();
            
            // シングルクリックのメニューを設定
            hookSingleClick();

            // アクションの生成
            makeActions();

            // 右クリックメニューの登録
            hookContextMenu();
            
            // ツールバーにボタンを登録
            fillLocalToolBar();
            
    	} catch (Exception e) {
			String msg = Messages.getString("NavigatorView.0"); //$NON-NLS-1$
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);
    	}
    }
    
    
	/**
	 * アクションクラスの生成。
	 * このビューから呼び出されるアクションクラスを生成します。
	 */
	private void makeActions() {
		// ファイル出力ダイアログをポップアップ表示するアクション
		openClassifiedExportDlgAction = new OpenClassifiedExportDlgAction(this);
		
		// 統計情報ダイアログをポップアップ表示するアクション
		openChartDialogAction = new OpenChartDlgAction(this);
		
		// 他の端末と同期をとるアクション
		syncRemoteAction = new SyncRemoteAction();

		// 新規課題票を開くアクション
		openNewIssueAction = new OpenIssueAction(OpenIssueAction.OPEN_NEW_ISSUE);

		// 他の端末と同期をとるアクション
		syncSelectionHostAction = new SyncSelectionRemoteAction();

		// 最新表示にするアクション
		refreshAction = new RefreshAction();

		
		// 新規課題票を開くアクション（カテゴリを指定）
		openCtgryIssueAction = new OpenIssueAction(OpenIssueAction.OPEN_CATEGORY_ISSUE);

		// プロジェクトを開くアクション
		openProjectAction = new OpenProjectAction();

		// プロジェクトを閉じるアクション
		closeProjectAction = new CloseProjectAction();
		
		// プロジェクトを削除するアクション
		deleteProjectAction = new DeleteProjectAction();
	
		// プロジェクトを新規追加するアクション					
		registConfigDlgAction = new OpenProjectConfigDlgAction(this, OpenProjectConfigDlgAction.NEW);

		// プロジェクト情報を編集するアクション
		editConfigDlgAction = new OpenProjectConfigDlgAction(this, OpenProjectConfigDlgAction.EDIT);
		
		// 課題票をインポートするアクション
		importIssueAction = new ImportIssueAction(this);
		
		// メッセージを送信するアクション
		openMessageDlgAction = new OpenMessageDlgAction(this) ;

		// カレントプロジェクトに以下の４つのアクションを加える
		CurrentProject project = CurrentProject.getInsance();
		project.addAction(openClassifiedExportDlgAction);
		project.addAction(openChartDialogAction);
		project.addAction(syncSelectionHostAction);
		project.addAction(syncRemoteAction);
    	project.addAction(openNewIssueAction);
    	
    	// カレントプロジェクトのオープン／クローズにより活性／非活性の表示を分ける
		if(project.isOpen()) {
			openClassifiedExportDlgAction.setEnabled(true);
			openChartDialogAction.setEnabled(true);
			syncSelectionHostAction.setEnabled(true);
			syncRemoteAction.setEnabled(true);
			openNewIssueAction.setEnabled(true);
		} else {
			openClassifiedExportDlgAction.setEnabled(false);
			openChartDialogAction.setEnabled(false);
			syncSelectionHostAction.setEnabled(false);
			syncRemoteAction.setEnabled(false);
			openNewIssueAction.setEnabled(false);
		}
		
	
	}

	/**
	 * ツリーノードを右クリックしたときの動作です。
	 * @throws		Exception
	 */
	private void hookContextMenu() {
		MenuManager menuMgr = new MenuManager("#PopupMenu"); 
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(new IMenuListener() {
			public void menuAboutToShow(IMenuManager manager) {
				fillContextMenu(manager);
			}
		});
		Menu menu = menuMgr.createContextMenu(viewer.getControl());
		viewer.getControl().setMenu(menu);
		getSite().registerContextMenu(menuMgr, viewer);
	}

	
	/**
	 * ツリーノードをダブルクリックしたときの動作を定義
	 *
	 */
	private void hookDoubleClick() {
		viewer.addDoubleClickListener(new IDoubleClickListener() {
			public void doubleClick(DoubleClickEvent event) {

				Viewer view = event.getViewer();
				ISelection selection = event.getSelection();

				if (view instanceof TreeViewer && selection instanceof StructuredSelection) {
					TreeViewer viewer = (TreeViewer) view;
					Object element = ((StructuredSelection) selection).getFirstElement();
					
					// IFolderNodeを実装するクラスは、フォルダの閉じる／開くを行う
					if (element instanceof IFolderNode) {
						if (!viewer.getExpandedState(element)) {
							viewer.expandToLevel(element, 1);
						} else {
							viewer.collapseToLevel(element, 1);
						}
					}
				}
			}
		});
		
	}

	
	/**
	 * ツリーノードをシングルクリックしたときの動作を定義
	 *
	 */
	private void hookSingleClick() {
		viewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				
				IssueListView view = BtsPlugin.getInstance().getIssueView();
				if(view == null) {
					return;
				}
				IssueListViewFilter filter = new IssueListViewFilter(view);
				
				BtsTreeNode selectedNode = getSelectionRecord(); 

				// 選択したノードごとに右クリックメニューを生成する。
				if(selectedNode ==null) {
					// 選択がない場合
					
				} else if(selectedNode instanceof CategoryNode) {
					// 「選択したカテゴリの課題票」のみ表示
					String categoryName = ((CategoryNode)selectedNode).getTextName();
					filter.setCategoryFilter(categoryName);

				} else if(selectedNode instanceof ReleaseNode) {
					// 「配布済み課題票」を全て表示
					filter.setIssueTypeFilter(IssueType.RELEASE_VALUE);
					
				} else if(selectedNode instanceof DraftNode) {
					// 「下書き課題票」のみ表示
					filter.setIssueTypeFilter(IssueType.DRAFT_VALUE);

				} else if(selectedNode instanceof ConflictNode) {
					// 「競合課題票」のみ表示
					filter.setIssueTypeFilter(IssueType.CONFLICT_VALUE);
				
				} else if(selectedNode instanceof GarbageBoxNode) {
					// 「ごみばこ課題票」のみ表示
					filter.setIssueTypeFilter(IssueType.GARBAGE_VALUE);
				
				} else {
					filter.setClearFilter();
				}
				
			}
		});
	}

	/**
	 * 選択しているレコードを返します。
	 * 選択しているレコードを{@link BtsTreeNode}にキャストして返します。
	 * 
	 * @return			キャスト済み選択レコード
	 */
	private BtsTreeNode getSelectionRecord() {
		
		// 選択しているレコードを取得します。
		ISelection selection = viewer.getSelection();
		Object obj = ((IStructuredSelection)selection).getFirstElement();
		if(obj instanceof BtsTreeNode) {
			// キャストして返します。
			return (BtsTreeNode)obj;
		}
		return null;
	}

	/**
	 * ツリーノードを右クリックしたときの動作です。
	 * 
	 * @param		manager			右クリックメニューのマネージャインスタンス
	 */
	private void fillContextMenu(IMenuManager manager) {
		
		// カレントプロジェクト情報を取得
		CurrentProject current = CurrentProject.getInsance();

		// 選択したノードを取得
		BtsTreeNode selectedNode = getSelectionRecord(); 

		// 選択したノードごとに右クリックメニューを生成する。
		if(selectedNode ==null) {
			// 選択がない場合
			manager.add(registConfigDlgAction);			// めにゅー：プロジェクト新規登録 
		} else if(selectedNode instanceof ProjectNode) {
			//■プロジェクトノードの右クリックメニュー
			
			// SSIDを取得してアクションに設定する。
			String ssid = ((ProjectNode)selectedNode).getSSID();
			closeProjectAction.setSSID(ssid);
			deleteProjectAction.setSSID(ssid);
			editConfigDlgAction.setSSID(ssid);
			openProjectAction.setSSID(ssid);

			// カレントプロジェクトかどうかで「開く／閉じる」を決める
			if( current.isCurrentProject(ssid) ) {
				// （プロジェクトがオープン中のメニュー）
				manager.add(closeProjectAction);		// めにゅー：プロジェクトを閉じる 
				manager.add(editConfigDlgAction);		// めにゅー：プロジェクト情報の編集 
				manager.add(new Separator()); 
				manager.add(openNewIssueAction);		// めにゅー：課題票を起票 
				manager.add(importIssueAction);			// めにゅー：課題票をインポート
				manager.add(new Separator()); 
				manager.add(syncRemoteAction);			// めにゅー：固定端末と同期 
				manager.add(syncSelectionHostAction);	// めにゅー：端末を指定して同期 

			} else {
				// （プロジェクトがクローズ中のメニュー）
				manager.add(openProjectAction);			// めにゅー：プロジェクトを開く 
				manager.add(editConfigDlgAction);		// めにゅー：プロジェクト情報の編集 
				manager.add(new Separator()); 
				manager.add(deleteProjectAction);		// めにゅー：プロジェクトを削除
			}
		
		} else if(selectedNode instanceof ReleaseNode) {
			//■配布済みノードの右クリックメニュー
			manager.add(openNewIssueAction);			// めにゅー：課題票を起票
			
		} else if(selectedNode instanceof CategoryNode) {
			//■カテゴリノードの右クリックメニュー

			// カテゴリ名を取得してアクションに設定する。
			String ctgry =  ((CategoryNode)selectedNode).getTextName();
			openCtgryIssueAction.setCategory(ctgry);

			manager.add(openCtgryIssueAction);			// めにゅー：指定のカテゴリで課題票を起票 
			
		} else if(selectedNode instanceof MemberListNode) {
			//■メンバー一覧ノードの右クリックメニュー
			openMessageDlgAction.setSendAll(true);
			openMessageDlgAction.setClientInfo(null);
			manager.add(openMessageDlgAction);			// めにゅー：メッセージ送信
			
		} else if(selectedNode instanceof MemberNode) {
			//■メンバーノードの右クリックメニュー

			ClientInfo info = ((MemberNode)selectedNode).getClientInfo();
			openMessageDlgAction.setSendAll(false);
			openMessageDlgAction.setClientInfo(info);
			manager.add(openMessageDlgAction);			// めにゅー：メッセージ送信
			
		} else if(selectedNode instanceof DraftNode) {
			//■下書き保存ノードの右クリックメニュー
			manager.add(openNewIssueAction);			// めにゅー： 課題票を開く

		} else if(selectedNode instanceof GarbageBoxNode) {
			//■ゴミ箱ノードの右クリックメニュー
		}
		
		// ■オープン中のプロジェクト共通メニュー

		// メニューが登録されている場合はセパレータを入れる
		if( !manager.isEmpty()) {
			manager.add(new Separator());
		}

		if(selectedNode != null) {
			manager.add(refreshAction);		// めにゅー：更新
		}
		
		// Other plug-ins can contribute there actions here
		manager.add(new Separator(IWorkbenchActionConstants.MB_ADDITIONS));
	}

	/**
	 * ツールバーにボタンを登録します。
	 * 
	 * @throws		Exception
	 */
	private void fillLocalToolBar() throws Exception {
		IActionBars bars = getViewSite().getActionBars();
		IToolBarManager toolbar = bars.getToolBarManager(); 
		
		// ぼたん：プロジェクトの新規登録
		toolbar.add(registConfigDlgAction);
		// ぼたん：課題票の新規作成
		toolbar.add(openNewIssueAction);
		toolbar.add(new Separator()); 
		
		// ぼたん：帳票出力
		toolbar.add(openClassifiedExportDlgAction);
		// ぼたん：グラフ表示
		toolbar.add(openChartDialogAction);
		toolbar.add(new Separator()); 

		// ぼたん：指定した端末と同期をとる
		toolbar.add(syncSelectionHostAction);
		// ぼたん：固定端末と同期をとる
		toolbar.add(syncRemoteAction);

	}

	/**
	 * このビューが持つビューアインスタンスを取得します。
	 * 
	 * @return		このビューが持つビューアインスタンス
	 */
	public TreeViewer getViewer() {
		return (TreeViewer)viewer;
	}

	/**
	 * 画面初期表示時のツリーを生成します。
	 * 画面初期表示時は1階層（プロジェクトフォルダ）のみしか表示しません。
	 * 
	 * @return		画面初期表示時のツリー
	 */
	private BtsTreeNode createDefaultTree() {

		// 登録されているプロジェクトを取得
		ProjectConfig[] prjct = ProjectConfigManager.getProjectConfigs();
		
		// プロジェクトが1件も無い場合は空のルートノードを返す
		if(prjct==null) {
			return new BtsTreeNode();
		}

		// ツリーを生成して返す。
		TreeBuilder treeBuilder = new TreeBuilder();
		for (int i = 0; i < prjct.length; i++) {
			treeBuilder.addProjectNode(prjct[i].getProjectName(), prjct[i].getSsid());
		}
		
		return treeBuilder.getRootNode();
	}
	
	
    /**
     * Set focus on the table viewer when this view gets focus.
     */
    public void setFocus() {
        viewer.getControl().setFocus();
    }

    
    /*
     *  (非 Javadoc)
     * @see org.eclipse.ui.IWorkbenchPart#dispose()
     */
    public void dispose() {
    	
    	// カレントプロジェクトに設定したアクションを削除する
		CurrentProject project = CurrentProject.getInsance();
		project.removeAction(openChartDialogAction);
		project.removeAction(syncRemoteAction);
		project.removeAction(syncSelectionHostAction);
		project.removeAction(openClassifiedExportDlgAction);
		project.removeAction(openNewIssueAction);
    	
    	super.dispose();
    }
    
//  _/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/
//  _/　　　　　　　　　　　ここから下はインナークラス群　　　　　　　　　　　　　　_/
//  _/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/
      
    /**
     * このビューのラベルプロバイダです。
     */
    private class FolderTreeLabelProvider extends LabelProvider {

    	/*
    	 *  (非 Javadoc)
    	 * @see org.eclipse.jface.viewers.ILabelProvider#getText(java.lang.Object)
    	 */
        public String getText(Object element) {
        	
        	if( element instanceof IChildCountNode ) {
        		// 子ノードの数を表示するノードは子ノードカウントを付け加える
        		int cnt = ((IChildCountNode)element).getChildCount();
        		return ((BtsTreeNode)element).getTextName() + " (" + cnt + ")";
        	}
            return ((BtsTreeNode)element).getTextName();
        }

        /*
         *  (非 Javadoc)
         * @see org.eclipse.jface.viewers.ILabelProvider#getImage(java.lang.Object)
         */
        public Image getImage(Object element) {
            return  ((BtsTreeNode)element).getImage();
        }
    }

}
