/*
 * Paraselene
 * Copyright (c) 2009-2012  Akira Terasaki
 * このファイルは同梱されているLicense.txtに定めた条件に同意できる場合にのみ
 * 利用可能です。
 */
package paraselene.dyna;

import java.util.*;
import java.io.*;
import java.net.*;


abstract class Loader extends Thread {
	private HashMap<CreateParam, LinkedList<DynamicPage>>	map
		= new HashMap<CreateParam, LinkedList<DynamicPage>>();
	private volatile boolean	add_f = false;
	private volatile boolean started = false;

	Loader() {
		setDaemon( true );
		setPriority( MIN_PRIORITY );
	}

	void add( CreateParam cp ) {
		add_f = true;
		synchronized( map ) {
			add_f = false;
			if ( map.get( cp ) != null )	return;
			map.put( cp, new LinkedList<DynamicPage>() );
			if ( !started ) {
				start();
				started = true;
			}
		}
	}
	DynamicPage getPage( CreateParam cp ) {
		LinkedList<DynamicPage>	list = map.get( cp );
		if ( list == null ) {
			add( cp );
		}
		else {
			DynamicPage	page = null;
			synchronized( list ) {
				page = list.poll();
			}
			if ( page != null )	return page;
		}
		return cp.newPage();
	}

	abstract void load( CreateParam cp, LinkedList<DynamicPage> list );
	public void run() {
		while( true ) {
			try {
				synchronized( map ) {
					for ( CreateParam cp: map.keySet() ) {
						LinkedList<DynamicPage>	list = map.get( cp );
						synchronized( list ) {
							try {
								load( cp, list );
							}
							catch( Exception e ) {
								cp.setError( e );
							}
						}
						if ( add_f )	break;
					}
				}
			}
			catch( Exception e ){}
			try {
				sleep( 5 * 1000 );
			}
			catch( Exception e ){}
		}
	}

	void loopset( CreateParam cp, LinkedList<DynamicPage> list ) {
		if ( list.size() > DynamicPageReloader.limit )	return;
		for ( int i = 0; i < DynamicPageReloader.inc; i++ ) {
			list.offer( cp.newPage() );
		}
	}
	void loopset( CreateParam cp, LinkedList<DynamicPage> list, long now ) {
		if ( list.size() > DynamicPageReloader.limit && !cp.isOld( now ) )	return;
		for ( int i = 0; i < DynamicPageReloader.inc; i++ ) {
			list.offer( cp.newPage( now, list ) );
		}
	}
}

class StringLoader extends Loader {
	void load( CreateParam cp, LinkedList<DynamicPage> list ) {
		loopset( cp, list );
	}
}

class FileLoader extends Loader {
	private HashMap<File, Long>	lasttime = new HashMap<File, Long>();
	private HashMap<File, Long>	timestamp = new HashMap<File, Long>();

	void load( CreateParam cp, LinkedList<DynamicPage> list ) {
		long	now = new Date().getTime();
		long	fix = now - DynamicPageReloader.file_interval;
		File	key = cp.file;

		Long	old = lasttime.get( key ), tm;
		if ( old == null ) {
			tm = key.lastModified();
			lasttime.put( key, now );
			timestamp.put( key, tm );
			loopset( cp, list );
			return;
		}
		tm = timestamp.get( key );
		if ( old > fix ) {
			lasttime.put( key, now );
			tm = key.lastModified();
			if ( tm == 0 ) {
				cp.setError( "cannot access " + key.toString() );
				loopset( cp, list );
				return;
			}
			timestamp.put( key, tm );
		}
		loopset( cp, list, tm );
	}
}

class URLLoader extends Loader {
	private HashMap<URL, Long>	lasttime = new HashMap<URL, Long>();
	private HashMap<URL, Long>	timestamp = new HashMap<URL, Long>();

	private long getTime( URL url ) throws Exception {
		HttpURLConnection	con = null;
		try {
			con = (HttpURLConnection)url.openConnection();
			con.setInstanceFollowRedirects( true );
			con.setRequestMethod( "HEAD" );
			con.connect();
			long	ret = con.getLastModified();
			if ( ret == 0 )	return new Date().getTime();
			return ret;
		}
		finally {
			if ( con != null )	con.disconnect();
		}
	}

	void load( CreateParam cp, LinkedList<DynamicPage> list ) {
		long	now = new Date().getTime();
		long	fix = now - DynamicPageReloader.url_interval;
		URL		key = cp.url;

		Long	old = lasttime.get( key ), tm;
		try {
			if ( old == null ) {
				tm = getTime( key );
				lasttime.put( key, now );
				timestamp.put( key, tm );
				loopset( cp, list );
				list.wait( 1000 * 10 );
				return;
			}
			tm = timestamp.get( key );
			if ( old > fix ) {
				tm = getTime( key );
				lasttime.put( key, now );
				timestamp.put( key, tm );
			}
			loopset( cp, list, tm );
			if ( old > fix )	list.wait( 1000 * 10 );
			return;
		}
		catch( Exception e ) {
			cp.setError( e );
			lasttime.put( key, now );
		}
		loopset( cp, list );
	}
}

/**
 * DynamicPage の自動ロード機構。
 * バックグラウンドで create まで完了した DynamicPage インスタンスを作成します。<br>
 * ユーザー入力に先行して、ファイルアクセスまたはネットワークアクセスを行うため
 * レスポンス時間の短縮に繋がります。<br><br>
 * バックグラウンドでは、次のように動作します。
 * <dl>
 * <dt>作成されるインスタンス
 * <dd>issue メソッドに渡されたインスタンスの
 *   <ul>
 *   <li>コンストラクタ
 *   <li>createメソッド
 *   </ul>の引数をそのまま使い、新しいインスタンスの作成、createを実施します。
 * <dt>{@link DynamicPage#create(String)}
 * <dd>そのまま作成します。
 * <dt>{@link DynamicPage#create(java.io.File,String)}
 * <dd>通常、メモリ上の内容だけを参照し、ファイルIOは発生しません。<br>
 * ただし、1分に1回ファイルの更新時間を参照し、更新されていればファイルを読み直します。
 * <dt>{@link DynamicPage#create(java.net.URL,paraselene.supervisor.RequestParameter.Method,String)}
 * <dd>通常、メモリ上の内容だけを参照し、ネットワークIOは発生しません。<br>
 * ただし、1時間に1回ネットワークリソースの更新時間を参照し、更新されていれば
 * ネットワークリソースを読み直します。<br>
 * また、URLが示すホスト単位で並列アクセスし、同一ホストへのアクセス過多を防止しつつ、
 * 各ホストへ万遍なくアクセスを行います。
 * <dt>IOエラー発生時
 * <dd>上記のうち、外部リソースを参照するものは IO エラー発生の可能性があります。<br>
 * リソース参照(または解析)に失敗した場合は引き続き、メモリ上の情報を使い続けます。
 * </dl>
 */
public class DynamicPageReloader {
	static volatile int limit = 5, inc = 10;
	static volatile int file_interval = 60 * 1000, url_interval = 60 * 60 * 1000;

	private static StringLoader	str_loader = new StringLoader();
	private static FileLoader	file_loader = new FileLoader();
	private static HashMap<String, URLLoader> url_loader = new HashMap<String, URLLoader>();

	private DynamicPageReloader(){}

	/**
	 * 自動ローダー。DynamicPage インスタンスの集合です。<br>
	 * このインスタンスを通して create 済みインスタンスが取得できます。
	 */
	public interface Magazine {
		/**
		 * バックグラウンドでcreate済みのインスタンスを返します。
		 * @return create済みインスタンス。
		 */
		DynamicPage getPage();
		/**
		 * 最終エラーの取得。バックグラウンド処理で発生した例外を取得します。
		 * エラーを一度参照すると、エラー情報はクリアされます。
		 * @return 最後に発生した例外。未発生またはクリア後であればnull。
		 */
		DynamicPageException getLastError();
	}

	private static URLLoader getLoader( URL url ) {
		URLLoader	loader = null;
		synchronized( url_loader ) {
			String	host = url.getHost();
			loader = url_loader.get( host );
			if ( loader == null ) {
				loader = new URLLoader();
				url_loader.put( host, loader );
			}
		}
		return loader;
	}

	static DynamicPage getPage( CreateParam cp ) {
		if ( cp.file != null ) {
			return file_loader.getPage( cp );
		}
		if ( cp.url != null ) {
			return getLoader( cp.url ).getPage( cp );
		}
		return str_loader.getPage( cp );
	}

	/**
	 * 自動ローダーの発行。
	 * 引数インスタンスと同内容のインスタンスを生成するためのローダーを
	 * 提供します。
	 * @param page 複製したいインスタンス。
	 * このインスタンスは create された後でなければなりません。
	 * @return 自動ローダー。このインスタンスを static 変数等に保持して下さい。
	 */
	public static Magazine issue( DynamicPage page ) {
		CreateParam	cp = page.param;
		if ( cp == null )	return null;
		if ( cp.file != null ) {
			file_loader.add( cp );
		}
		else if ( cp.url != null ) {
			getLoader( cp.url ).add( cp );
		}
		else {
			str_loader.add( cp );
		}
		return cp;
	}
}

